"""Mouse support utilities for terminals using Xterm SGR encoding.

This module enables and decodes mouse events via ANSI escape sequences.
"""
from __future__ import annotations

import re
from dataclasses import dataclass

_MOUSE_SGR_RE = re.compile(r"\x1b\[<(?P<btn>\d+);(?P<x>\d+);(?P<y>\d+)(?P<type>[mM])")


@dataclass(slots=True)
class MouseEvent:
    x: int
    y: int
    button: int  # 0 left, 1 middle, 2 right, 64 wheel
    pressed: bool


def enable_mouse_reporting() -> str:
    # Enable SGR extended mouse mode and focus events
    return "\x1b[?1000;1002;1006;1004h"


def disable_mouse_reporting() -> str:
    return "\x1b[?1000;1002;1006;1004l"


def parse_mouse_sequence(seq: str) -> MouseEvent | None:
    m = _MOUSE_SGR_RE.fullmatch(seq)
    if not m:
        return None
    btn = int(m.group("btn"))
    x = int(m.group("x"))
    y = int(m.group("y"))
    pressed = m.group("type") == "M"
    # Normalize button codes: 0 left, 1 middle, 2 right, 64 wheel
    base = btn & 0b11
    wheel = 64 if (btn & 0b1100000) else 0
    button = wheel or base
    return MouseEvent(x=x, y=y, button=button, pressed=pressed)
