"""Thread-safe event bus and common event types for renderers.

This module is renderer-agnostic and may be used by both the CLI and GUI
frameworks. It deliberately keeps the surface small and dependency-free.
"""

from __future__ import annotations

import threading
from dataclasses import dataclass
from typing import TYPE_CHECKING, Any

if TYPE_CHECKING:
    from collections.abc import Callable

__all__ = [
    "EventBus",
    "KeyEvent",
    "TickEvent",
    "ResizeEvent",
    "QuitEvent",
]


@dataclass(slots=True)
class KeyEvent:
    key: str


@dataclass(slots=True)
class TickEvent:
    dt: float


@dataclass(slots=True)
class ResizeEvent:
    width: int
    height: int


@dataclass(slots=True)
class QuitEvent:
    reason: str = "user"


class EventBus:
    """Minimal pub/sub with string event types and payloads.

    Handlers are invoked synchronously in the caller's thread.
    """

    def __init__(self) -> None:
        self._handlers: dict[str, list[Callable[[Any], None]]] = {}
        self._lock = threading.Lock()

    def subscribe(self, event_type: str, handler: Callable[[Any], None]) -> Callable[[], None]:
        with self._lock:
            self._handlers.setdefault(event_type, []).append(handler)

        def unsubscribe() -> None:
            self.unsubscribe(event_type, handler)

        return unsubscribe

    def unsubscribe(self, event_type: str, handler: Callable[[Any], None]) -> None:
        with self._lock:
            handlers = self._handlers.get(event_type)
            if not handlers:
                return
            try:
                handlers.remove(handler)
            except ValueError:
                pass
            if not handlers:
                self._handlers.pop(event_type, None)

    def publish(self, event_type: str, payload: Any | None = None) -> None:  # Any: event payloads can contain any data
        with self._lock:
            handlers = list(self._handlers.get(event_type, ()))
        for handler in handlers:
            try:
                handler(payload)
            except Exception:
                # Best effort: keep other handlers running
                continue
