"""Runtime backends for output streams (TTY, GUI, terminal).

Backends are small factories that hand ``Console`` an appropriate stream-like
object.  The default backend simply returns ``sys.stdout``; the GUI backend
bridges to the runtime in ``rendering_gui.runtime`` which in turn manages the
native window, and the terminal backend provides a diff-friendly live session.
"""

from __future__ import annotations

import sys
from typing import TYPE_CHECKING, Any, TextIO

if TYPE_CHECKING:
    import queue
    from collections.abc import Callable


_TERMINAL_WINDOWS: list[Any] = []  # Any: terminal window objects can be various types


class Backends:
    _registry: dict[str, Callable[[], Any]] = {}  # Any: backends return various stream types

    @classmethod
    def register(cls, name: str, factory: Callable[[], Any]) -> None:  # Any: backends return various stream types
        cls._registry[name] = factory

    @classmethod
    def create_stream(cls, name: str) -> Any:  # Any: backends return various stream types
        factory = cls._registry.get(name) or cls._registry.get("tty")
        if not factory:
            return sys.stdout
        try:
            return factory()
        except Exception:
            return sys.stdout


def _make_tty_stream() -> TextIO:
    return sys.stdout


def _make_gui_stream() -> Any:  # Any: GUI streams can be various types
    try:
        from .rendering_gui.runtime import get_runtime
    except Exception:
        return _make_tty_stream()
    try:
        runtime = get_runtime()
    except Exception:
        runtime = None
    if runtime is None:
        return _make_tty_stream()
    return runtime.create_stream()


def _make_terminal_stream() -> Any:  # Any: terminal streams can be various types
    try:
        from .rendering_cli.window import TerminalStream, TerminalWindow
    except Exception:
        return _make_tty_stream()
    try:
        win = TerminalWindow(stream=sys.stdout)
    except Exception:
        return _make_tty_stream()
    _TERMINAL_WINDOWS.append(win)
    return TerminalStream(win)


# Register default backends
Backends.register("tty", _make_tty_stream)
Backends.register("console", _make_tty_stream)
Backends.register("gui", _make_gui_stream)
Backends.register("terminal", _make_terminal_stream)


def get_gui_key_queue() -> queue.Queue[str] | None:
    try:
        from .rendering_gui.runtime import get_runtime
    except Exception:
        return None
    runtime = get_runtime(create=False)
    if runtime is None:
        return None
    return runtime.get_key_queue()
