"""Visual renderers like GraphCanvas and Waveform."""

from __future__ import annotations

import math
from typing import TYPE_CHECKING

from ..core import render_call
from ..style.colors import Color

if TYPE_CHECKING:
    from collections.abc import Callable

__all__ = [
    "GraphCanvas",
    "Waveform",
    "Spectrogram",
    "ImageBlock",
    "GridMap",
    "MiniMap",
    "graph_canvas",
    "waveform",
    "spectrogram",
    "image_block",
    "mini_map",
]


class GraphCanvas:
    @staticmethod
    def render(
        func: Callable[[float], float],
        x_min: float = -math.pi,
        x_max: float = math.pi,
        width: int = 60,
        height: int = 15,
        color: str = "accent",
    ) -> str:
        col = Color.get_color(color)
        # Sample function
        xs = [x_min + (x_max - x_min) * i / (width - 1) for i in range(width)]
        ys = [func(x) for x in xs]
        y_min, y_max = min(ys), max(ys)
        span = (y_max - y_min) or 1
        grid = [[" " for _ in range(width)] for _ in range(height)]
        axis_y = int(height * (0 - y_min) / span) if y_min <= 0 <= y_max else None
        for x, y in enumerate(ys):
            y_pos = height - 1 - int((y - y_min) / span * (height - 1))
            grid[y_pos][x] = "●"
            if axis_y is not None:
                grid[min(max(axis_y, 0), height - 1)][x] = "─"
        return "\n".join(col + "".join(row) + Color.RESET for row in grid)


class Waveform:
    @staticmethod
    def render(samples: list[float], width: int = 60, height: int = 10, color: str = "primary") -> str:
        if not samples:
            return ""
        # Normalize
        s_min, s_max = min(samples), max(samples)
        span = (s_max - s_min) or 1
        cols = min(width, len(samples))
        step = max(1, len(samples) // cols)
        col = Color.get_color(color)
        grid = [[" " for _ in range(cols)] for _ in range(height)]
        for xi in range(cols):
            val = samples[xi * step]
            y = height - 1 - int((val - s_min) / span * (height - 1))
            grid[y][xi] = "█"
        return "\n".join(col + "".join(row) + Color.RESET for row in grid)


def graph_canvas(*args: object, **kwargs: object):
    return render_call(GraphCanvas.render, *args, **kwargs)


def waveform(*args: object, **kwargs: object):
    return render_call(Waveform.render, *args, **kwargs)


class Spectrogram:
    @staticmethod
    def render(matrix: list[list[float]], shades: str = " .:-=+*#%@", color: str = "fg") -> str:
        if not matrix:
            return ""
        flat = [v for row in matrix for v in row]
        vmin, vmax = min(flat), max(flat)
        span = (vmax - vmin) or 1
        col = Color.get_color(color)
        out_lines: list[str] = []
        for row in matrix:
            line = []
            for v in row:
                idx = int((v - vmin) / span * (len(shades) - 1))
                line.append(shades[idx])
            out_lines.append(col + "".join(line) + Color.RESET)
        return "\n".join(out_lines)


def _read_netpbm_file(p: str) -> tuple[list[list[int]], int, int, int]:
    with open(p, "rb") as f:
        magic = f.readline().strip()
        # Skip comments

        def _read_token() -> bytes:
            b = f.read(1)
            while b == b"#":
                # skip comment line
                while b not in (b"\n", b""):
                    b = f.read(1)
                b = f.read(1)
            while b.isspace():
                b = f.read(1)
            tok = bytearray()
            while b and not b.isspace():
                tok.extend(b)
                b = f.read(1)
            return bytes(tok)

        if magic in (b"P2", b"P3", b"P5", b"P6"):
            w = int(_read_token())
            h = int(_read_token())
            maxv = int(_read_token())
            if magic == b"P2":  # ASCII gray
                vals = [int(_read_token()) for _ in range(w * h)]
                data = [vals[i * w : (i + 1) * w] for i in range(h)]
                return data, w, h, maxv
            if magic == b"P3":  # ASCII RGB
                vals = [int(_read_token()) for _ in range(w * h * 3)]
                gray = []
                for i in range(0, len(vals), 3):
                    r, g, b = vals[i : i + 3]
                    gray.append(int(0.2126 * r + 0.7152 * g + 0.0722 * b))
                data = [gray[i * w : (i + 1) * w] for i in range(h)]
                return data, w, h, maxv
            if magic == b"P5":  # Binary gray
                raw = f.read(w * h)
                vals = list(raw)
                data = [vals[i * w : (i + 1) * w] for i in range(h)]
                return data, w, h, maxv
            if magic == b"P6":  # Binary RGB
                raw = f.read(w * h * 3)
                gray = []
                for i in range(0, len(raw), 3):
                    r, g, b = raw[i], raw[i + 1], raw[i + 2]
                    gray.append(int(0.2126 * r + 0.7152 * g + 0.0722 * b))
                data = [gray[i * w : (i + 1) * w] for i in range(h)]
                return data, w, h, maxv
        raise ValueError("Unsupported or corrupt Netpbm file")


class ImageBlock:
    @staticmethod
    def render(path: str, width: int = 60, shades: str = " .:-=+*#%@") -> str:
        """Render grayscale PGM/PPM (P6/P5 ASCII or binary) images as ASCII.

        This avoids external dependencies by supporting the Netpbm formats.
        """

        try:
            data, w, h, maxv = _read_netpbm_file(path)
        except Exception as e:
            return f"Error reading image: {e}"

        aspect = h / max(1, w)
        new_h = max(1, int(width * aspect * 0.5))
        # Nearest neighbor resize
        out: list[str] = []
        for y in range(new_h):
            src_y = min(h - 1, int(y / new_h * h))
            line_chars = []
            for x in range(width):
                src_x = min(w - 1, int(x / width * w))
                val = data[src_y][src_x]
                idx = int((val / max(1, maxv)) * (len(shades) - 1))
                line_chars.append(shades[idx])
            out.append("".join(line_chars))
        return "\n".join(out)


def spectrogram(*args: object, **kwargs: object):
    return render_call(Spectrogram.render, *args, **kwargs)


def image_block(*args: object, **kwargs: object):
    return render_call(ImageBlock.render, *args, **kwargs)


def grid_map(*args: object, **kwargs: object):
    return render_call(GridMap.render, *args, **kwargs)


class GridMap:
    @staticmethod
    def render(grid: list[list[int]], width: int | None = None, shades: str = " ░▒▓█") -> str:
        """
        Render a grid of 0..N intensity values as ASCII blocks.
        """
        out: list[str] = []
        if not grid:
            out.append("")
            return "\n".join(out)
        max_val = max((max(row) if row else 0) for row in grid) or 1
        for row in grid:
            line = []
            for v in row:
                idx = int(v / max_val * (len(shades) - 1))
                line.append(shades[idx])
            text = "".join(line)
            out.append(text if width is None else text[:width])
        return "\n".join(out)


class MiniMap:
    @staticmethod
    def render(text: str, width: int = 30, height: int = 8, shades: str = " .:-=+*#%@") -> str:
        """Render a miniature overview map of a text block by density.

        The text is split into lines, chunked into a width x height grid, and
        each cell is assigned a shade based on the average visible character density.
        """
        out: list[str] = []
        if not text:
            out.append("")
            return "\n".join(out)
        lines = text.splitlines() or [""]
        # Target grid
        cols, rows = width, height
        # Normalize to rows by slicing the line list
        step = max(1, len(lines) // rows)
        out_rows: list[str] = []
        for r in range(rows):
            start = r * step
            block = lines[start : start + step]
            # Determine density across cols by sampling each line's length
            maxlen = max((len(ln) for ln in block), default=1)
            if maxlen == 0:
                maxlen = 1
            cell_chars = []
            for c in range(cols):
                # Sample position across each line
                x = int(c / max(1, cols - 1) * (maxlen - 1)) if maxlen > 1 else 0
                # Average a small neighborhood density
                count = 0
                for ln in block:
                    if x < len(ln) and not ln[x].isspace():
                        count += 1
                # Map to shade
                frac = count / max(1, len(block))
                idx = int(frac * (len(shades) - 1))
                cell_chars.append(shades[idx])
            out_rows.append("".join(cell_chars))
        for row in out_rows:
            out.append(row)
        return "\n".join(out)


def mini_map(*args: object, **kwargs: object):
    return render_call(MiniMap.render, *args, **kwargs)
