"""Typography-oriented components.

Section 5.1 tasks:
- Replace direct prints with rendering via Console
- Expose data classes for composable blocks with measurement hooks
"""
from __future__ import annotations

from dataclasses import dataclass, field
from typing import TYPE_CHECKING, Any, Literal

from ..core import render_call
from ..core.console import Console, Measure, Renderable
from ..style.colors import Color
from ..symbols import Symbols
from ..utils import get_visible_length, wrap_text

if TYPE_CHECKING:
    from collections.abc import Sequence

    from ..core.console import RenderableLike

__all__ = [
    "Title",
    "Paragraph",
    "ListElement",
    "BlockQuote",
    "Steps",
    "title",
    "paragraph",
    "list_element",
    "block_quote",
    "steps",
]


class Title:
    """Build formatted title elements."""

    @staticmethod
    def render(
        text: str,
        width: int = 80,
        color: str | None = None,
        symbol_style: str = "box",
        align: Literal["left", "center", "right"] = "center",
        padding: int = 1,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        uppercase: bool = False,
        bold: bool = True,
        ascii_mode: bool | None = None
    ) -> None:
        """
        Render a formatted title element.

        Args:
            text: Title text (supports inline markup)
            width: Total width of title box
            color: Text color
            symbol_style: Border style (box, double, rounded, heavy, etc.)
            align: Text alignment
            padding: Padding inside the box
            blank_lines_before: Blank lines before title
            blank_lines_after: Blank lines after title
            uppercase: Convert text to uppercase
            bold: Make text bold
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        console = Console()
        text_color = Color.get_color(color)
        symbols = Symbols.get_symbols(symbol_style, ascii_mode)

        if uppercase:
            text = text.upper()

        # Apply styling
        styled_text = Color.apply_inline_markup(text)
        if bold:
            styled_text = Color.BOLD + styled_text + Color.RESET
        if color:
            styled_text = text_color + styled_text + Color.RESET

        # Calculate dimensions
        visible_text_len = get_visible_length(styled_text)
        padding_total = padding * 2
        inner_width = width - 4 - padding_total  # Account for corners and padding

        # Align text
        if align == "left":
            padded_text = styled_text.ljust(visible_text_len + (inner_width - visible_text_len))
        elif align == "right":
            padded_text = styled_text.rjust(visible_text_len + (inner_width - visible_text_len))
        else:  # center
            spaces_needed = inner_width - visible_text_len
            left_spaces = spaces_needed // 2
            right_spaces = spaces_needed - left_spaces
            padded_text = " " * left_spaces + styled_text + " " * right_spaces

        # Build box
        top = symbols["top_left"] + symbols["horizontal"] * (width - 2) + symbols["top_right"]
        middle = symbols["vertical"] + " " * padding + padded_text + " " * padding + symbols["vertical"]
        bottom = symbols["bottom_left"] + symbols["horizontal"] * (width - 2) + symbols["bottom_right"]

        # Print
        for _ in range(blank_lines_before):
            console.print("")
        console.print(top)
        console.print(middle)
        console.print(bottom)
        for _ in range(blank_lines_after):
            console.print("")


class Paragraph:
    """Build formatted paragraph elements."""

    @staticmethod
    def render(
        text: str,
        width: int = 80,
        color: str | None = None,
        indent: int = 0,
        blank_lines_before: int = 0,
        blank_lines_after: int = 1,
        align: Literal["left", "center", "right", "justify"] = "left",
        line_spacing: int = 0,
        box: bool = False,
        symbol_style: str = "box",
        ascii_mode: bool | None = None
    ) -> None:
        """
        Render a formatted paragraph element.

        Args:
            text: Paragraph text (supports inline markup)
            width: Maximum width
            color: Text color
            indent: Left indentation
            blank_lines_before: Blank lines before paragraph
            blank_lines_after: Blank lines after paragraph
            align: Text alignment
            line_spacing: Blank lines between lines of paragraph
            box: Wrap paragraph in a box
            symbol_style: Box style if box=True
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        console = Console()
        text_color = Color.get_color(color)

        for _ in range(blank_lines_before):
            console.print("")

        lines = wrap_text(text, width - indent)

        if box:
            symbols = Symbols.get_symbols(symbol_style, ascii_mode)
            box_width = width
            top = symbols["top_left"] + symbols["horizontal"] * (box_width - 2) + symbols["top_right"]
            console.print(top)

        for i, line in enumerate(lines):
            indent_str = " " * indent
            colored_line = Color.apply_inline_markup(line)
            if color:
                colored_line = text_color + colored_line + Color.RESET

            effective_width = width - indent
            if align == "left":
                console.print(indent_str + colored_line)
            elif align == "right":
                spaces = max(0, effective_width - get_visible_length(colored_line))
                console.print(indent_str + (" " * spaces) + colored_line)
            elif align == "center":
                spaces = max(0, (effective_width - get_visible_length(colored_line)) // 2)
                console.print(indent_str + (" " * spaces) + colored_line)
            elif align == "justify":
                # Do not justify the final line of the paragraph
                if i == len(lines) - 1 or get_visible_length(colored_line) >= effective_width:
                    console.print(indent_str + colored_line)
                else:
                    # Strip ANSI for measurement, but apply spacing to the original text
                    from ..utils import justify_text, strip_ansi
                    base = strip_ansi(colored_line)
                    just = justify_text(base, effective_width)
                    # Re-apply color codes at start/end if provided
                    if color:
                        just = text_color + just + Color.RESET
                    console.print(indent_str + just)

            if i < len(lines) - 1 and line_spacing > 0:
                for _ in range(line_spacing):
                    console.print("")

        if box:
            bottom = symbols["bottom_left"] + symbols["horizontal"] * (box_width - 2) + symbols["bottom_right"]
            console.print(bottom)

        for _ in range(blank_lines_after):
            console.print("")


@dataclass(slots=True)
class _ListNode:
    text: str
    checked: bool | None = None
    marker: str | None = None
    children: list[_ListNode] = field(default_factory=list)


class ListElement:
    """Build formatted list elements."""

    @staticmethod
    def render(
        items: Sequence[str | dict[str, Any] | tuple[Any, ...]],
        width: int = 80,
        color: str | None = None,
        marker_color: str | None = None,
        list_type: Literal['bullet', 'number', 'dash', 'arrow', 'star', 'check', 'custom'] = 'bullet',
        custom_marker: str = '\u2022',
        indent: int = 0,
        nested_indent: int = 2,
        item_spacing: int = 0,
        blank_lines_before: int = 0,
        blank_lines_after: int = 1,
        box: bool = False,
        symbol_style: str = 'box',
        ascii_mode: bool | None = None
    ) -> None:
        """
        Render a formatted list element.

        Args:
            items: Items to display. Accepts strings, dictionaries with
                keys text, checked, marker, children, or tuples shaped as
                (text, children) or (text, checked, children).
            width: Maximum width
            color: Item text color
            marker_color: Bullet or number color
            list_type: Type of list marker
            custom_marker: Custom marker string if list_type is custom
            indent: Left indentation
            nested_indent: Additional indentation applied per nesting level
            item_spacing: Blank lines between items
            blank_lines_before: Blank lines before list
            blank_lines_after: Blank lines after list
            box: Wrap list in a box
            symbol_style: Box style if box is True
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        console = Console()
        text_color = Color.get_color(color)
        marker_color_code = Color.get_color(marker_color)

        def _coerce_item(raw: str | dict[str, Any] | tuple[Any, ...]) -> _ListNode:
            if isinstance(raw, _ListNode):
                return raw
            if isinstance(raw, str):
                return _ListNode(text=raw)
            if isinstance(raw, dict):
                text_val = str(raw.get('text', ''))
                checked_val = raw.get('checked')
                if isinstance(checked_val, bool):
                    checked = checked_val
                elif isinstance(checked_val, str):
                    checked = checked_val.lower() in {'1', 'true', 'yes', 'on', 'checked', 'x'}
                else:
                    checked = None
                marker_override = raw.get('marker')
                children_raw = raw.get('children') or []
                children_nodes = [_coerce_item(child) for child in children_raw]
                return _ListNode(text=text_val, checked=checked, marker=marker_override, children=children_nodes)
            if isinstance(raw, tuple):
                if not raw:
                    return _ListNode(text='')
                text_val = str(raw[0])
                checked: bool | None = None
                marker_override: str | None = None
                children_nodes: list[_ListNode] = []
                if len(raw) >= 2:
                    second = raw[1]
                    if isinstance(second, bool):
                        checked = second
                    elif isinstance(second, str):
                        marker_override = second
                    elif isinstance(second, (list, tuple)):
                        children_nodes = [_coerce_item(child) for child in second]
                if len(raw) >= 3 and isinstance(raw[2], (list, tuple)):
                    children_nodes = [_coerce_item(child) for child in raw[2]]
                return _ListNode(text=text_val, checked=checked, marker=marker_override, children=children_nodes)
            return _ListNode(text=str(raw))

        nodes = [_coerce_item(item) for item in items]

        markers = {'bullet': '\u2022', 'number': None, 'dash': '\u2500', 'arrow': '\u2192', 'star': '\u2605', 'check': '\u2713', 'custom': custom_marker}
        marker = markers.get(list_type, '\u2022')

        for _ in range(blank_lines_before):
            console.print("")

        if box:
            symbols = Symbols.get_symbols(symbol_style, ascii_mode)
            box_width = width
            top = symbols['top_left'] + symbols['horizontal'] * (box_width - 2) + symbols['top_right']
            console.print(top)

        def _marker_for(node: _ListNode, index_stack: list[int]) -> str:
            if list_type == 'number':
                return '.'.join(str(i) for i in index_stack) + '.'
            if list_type == 'check':
                status = node.checked if node.checked is not None else False
                return '[x]' if status else '[ ]'
            if node.marker:
                return node.marker
            return marker or '\u2022'

        def _render_nodes(nodes_to_render: list[_ListNode], depth: int, index_prefix: list[int]) -> None:
            for idx, node in enumerate(nodes_to_render, start=1):
                current_index = index_prefix + [idx]
                marker_str = _marker_for(node, current_index)
                current_indent = indent + depth * nested_indent
                indent_str = ' ' * current_indent

                marker_space = len(marker_str) + 1
                marker_display = marker_color_code + marker_str + Color.RESET + ' '

                available_width = max(10, width - current_indent - marker_space)
                item_lines = wrap_text(node.text, available_width)

                for line_no, line in enumerate(item_lines):
                    formatted_line = Color.apply_inline_markup(line)
                    if color:
                        formatted_line = text_color + formatted_line + Color.RESET

                    if line_no == 0:
                        console.print(indent_str + marker_display + formatted_line)
                    else:
                        console.print(indent_str + ' ' * marker_space + formatted_line)

                if node.children:
                    child_prefix = current_index if list_type == 'number' else []
                    _render_nodes(node.children, depth + 1, child_prefix)

                if idx < len(nodes_to_render) and item_spacing > 0:
                    for _ in range(item_spacing):
                        console.print("")

        _render_nodes(nodes, 0, [])

        if box:
            bottom = symbols['bottom_left'] + symbols['horizontal'] * (box_width - 2) + symbols['bottom_right']
            console.print(bottom)

        for _ in range(blank_lines_after):
            console.print("")


class BlockQuote:
    """Build block quotes."""

    @staticmethod
    def render(
        text: str,
        width: int = 70,
        author: str = "",
        quote_char: str = "┃",
        color: str = "cyan",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> None:
        """
        Render a block quote.

        Args:
            text: Quote text
            width: Maximum width
            author: Quote author (optional)
            quote_char: Character for quote marker
            color: Quote color
            blank_lines_before: Blank lines before quote
            blank_lines_after: Blank lines after quote
        """
        console = Console()
        for _ in range(blank_lines_before):
            console.print("")

        qcolor = Color.get_color(color)
        lines = wrap_text(text, width - 4)

        for line in lines:
            console.print(f"{qcolor}{quote_char}{Color.RESET} {line}")

        if author:
            console.print(f"{qcolor}{quote_char}{Color.RESET} ")
            console.print(f"{qcolor}{quote_char}{Color.RESET} {Color.DIM}— {author}{Color.RESET}")

        for _ in range(blank_lines_after):
            console.print("")


class Steps:
    """Build step indicators."""

    @staticmethod
    def render(
        steps: list[str],
        current_step: int,
        width: int = 80,
        complete_color: str = "green",
        current_color: str = "cyan",
        pending_color: str = "gray",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> None:
        """
        Render step indicators.

        Args:
            steps: List of step names
            current_step: Index of current step (0-based)
            width: Total width
            complete_color: Color for completed steps
            current_color: Color for current step
            pending_color: Color for pending steps
            blank_lines_before: Blank lines before steps
            blank_lines_after: Blank lines after steps
        """
        console = Console()
        for _ in range(blank_lines_before):
            console.print("")

        comp_color = Color.get_color(complete_color)
        curr_color = Color.get_color(current_color)
        pend_color = Color.get_color(pending_color)

        num_steps = len(steps)
        step_parts = []

        for i, step in enumerate(steps):
            if i < current_step:
                # Completed
                step_str = f"{comp_color}✓ {step}{Color.RESET}"
            elif i == current_step:
                # Current
                step_str = f"{curr_color}● {step}{Color.RESET}"
            else:
                # Pending
                step_str = f"{pend_color}○ {step}{Color.RESET}"

            step_parts.append(step_str)

        # Display steps
        step_width = width // num_steps
        result = ""
        for part in step_parts:
            visible_len = get_visible_length(part)
            padding = (step_width - visible_len) // 2
            result += " " * padding + part + " " * (step_width - visible_len - padding)

        console.print(result)

        # Progress line
        progress = int((current_step / max(num_steps - 1, 1)) * width)
        progress_line = comp_color + "━" * progress + Color.RESET + pend_color + "━" * (width - progress) + Color.RESET
        console.print(progress_line)

        for _ in range(blank_lines_after):
            console.print("")


# Composable dataclasses with measurement hooks

@dataclass(slots=True)
class TitleBlock(Renderable):
    text: str
    width: int = 80
    color: str | None = None
    symbol_style: str = "box"
    align: Literal["left", "center", "right"] = "center"
    padding: int = 1
    uppercase: bool = False
    bold: bool = True
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return console.capture(
            Title.render,
            self.text,
            width=self.width,
            color=self.color,
            symbol_style=self.symbol_style,
            align=self.align,
            padding=self.padding,
            blank_lines_before=0,
            blank_lines_after=0,
            uppercase=self.uppercase,
            bold=self.bold,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


@dataclass(slots=True)
class ParagraphBlock(Renderable):
    text: str
    width: int = 80
    color: str | None = None
    indent: int = 0
    align: Literal["left", "center", "right", "justify"] = "left"
    line_spacing: int = 0
    box: bool = False
    symbol_style: str = "box"
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return console.capture(
            Paragraph.render,
            self.text,
            width=self.width,
            color=self.color,
            indent=self.indent,
            blank_lines_before=0,
            blank_lines_after=0,
            align=self.align,
            line_spacing=self.line_spacing,
            box=self.box,
            symbol_style=self.symbol_style,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


@dataclass(slots=True)
class BlockQuoteBlock(Renderable):
    text: str
    width: int = 70
    author: str = ""
    quote_char: str = "┃"
    color: str = "cyan"

    def render(self, console: Console) -> str:
        return console.capture(
            BlockQuote.render,
            self.text,
            width=self.width,
            author=self.author,
            quote_char=self.quote_char,
            color=self.color,
            blank_lines_before=0,
            blank_lines_after=0,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def title(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable title component."""
    return render_call(Title.render, *args, **kwargs)


def paragraph(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable paragraph component."""
    return render_call(Paragraph.render, *args, **kwargs)


def list_element(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable list component."""
    return render_call(ListElement.render, *args, **kwargs)


def block_quote(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable block quote component."""
    return render_call(BlockQuote.render, *args, **kwargs)


def steps(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable step indicator component."""
    return render_call(Steps.render, *args, **kwargs)
