"""Status and progress components."""
from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING, Any, Literal

from ..core import Console, Measure, render_call
from ..style.colors import Color
from ..symbols import Symbols

if TYPE_CHECKING:
    from ..core.console import RenderableLike

__all__ = [
    "ProgressBar",
    "StatusBar",
    "SegmentedBar",
    "Spinner",
    "Meter",
    "progress_bar",
    "status_bar",
    "segmented_bar",
    "spinner",
    "meter",
    "Thermometer",
    "Compass",
    "GaugeDial",
    "WeatherIndicator",
    "thermometer",
    "compass",
    "gauge_dial",
    "weather_indicator",
]


class ProgressBar:
    """Build progress bars for stats, loading, etc."""

    @staticmethod
    def render(
        current: int | float,
        maximum: int | float,
        width: int = 40,
        label: str = "",
        show_percentage: bool = True,
        show_numbers: bool = True,
        filled_char: str = "#",
        empty_char: str = "-",
        color: str | None = None,
        bracket_style: Literal["square", "round", "angle", "pipe", "none"] = "square",
        indeterminate: bool = False,
        frame: int = 0,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render a progress bar.

        Args:
            current: Current value
            maximum: Maximum value
            width: Width of the bar (excluding label and numbers)
            label: Label text before the bar
            show_percentage: Show percentage after bar
            show_numbers: Show current/max numbers
            filled_char: Character for filled portion
            empty_char: Character for empty portion
            color: Bar color
            bracket_style: Style of brackets around bar
            blank_lines_before: Blank lines before bar
            blank_lines_after: Blank lines after bar
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        if indeterminate:
            # Moving window within the bar
            window = max(1, width // 5)
            start = frame % (width + window)
            bar_chars = [empty_char] * width
            for i in range(window):
                pos = start - i
                if 0 <= pos < width:
                    bar_chars[pos] = filled_char
            bar = "".join(bar_chars)
            percentage = 0.0
        else:
            # Calculate percentage
            percentage = (current / maximum * 100.0) if maximum > 0 else 0.0
            filled_width = int(width * current / maximum) if maximum > 0 else 0
            filled_width = max(0, min(width, filled_width))

            # Build bar
            filled = filled_char * filled_width
            empty = empty_char * (width - filled_width)
            bar = filled + empty

        # Apply color
        if color:
            bar_color = Color.get_color(color)
            bar = bar_color + bar + Color.RESET

        # Apply brackets
        brackets = {
            "square": ("[", "]"),
            "round": ("(", ")"),
            "angle": ("<", ">"),
            "pipe": ("|", "|"),
            "none": ("", "")
        }
        left, right = brackets.get(bracket_style, ("[", "]"))

        # Build output
        output = f"{label + ' ' if label else ''}{left}{bar}{right}"

        if show_percentage and not indeterminate:
            output += f" {percentage:.0f}%"

        if show_numbers and not indeterminate:
            output += f" ({current}/{maximum})"

        out.append(output)

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


def progress_bar(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable progress bar."""
    return render_call(ProgressBar.render, *args, **kwargs)


def status_bar(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable status bar."""
    return render_call(StatusBar.render, *args, **kwargs)


def segmented_bar(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable segmented bar."""
    return render_call(SegmentedBar.render, *args, **kwargs)


def spinner(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable spinner."""
    return render_call(Spinner.render, *args, **kwargs)


def meter(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable meter."""
    return render_call(Meter.render, *args, **kwargs)


class StatusBar:
    """Build status bars for HP, MP, XP, etc."""

    @staticmethod
    def render(
        stats: list[dict[str, Any]],
        width: int = 80,
        bar_width: int = 20,
        layout: Literal["horizontal", "vertical"] = "horizontal",
        show_icons: bool = True,
        live_update: bool = False,
        frames: int = 0,
        fps: float = 8.0,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render status bars for multiple stats.

        Args:
            stats: List of stat dictionaries with keys: label, current, max, color, icon
            width: Total width for horizontal layout
            bar_width: Width of each bar
            layout: Layout direction
            show_icons: Show icons before labels
            blank_lines_before: Blank lines before status bar
            blank_lines_after: Blank lines after status bar

        Example:
            StatusBar.render([
                {"label": "HP", "current": 75, "max": 100, "color": "red", "icon": "<3"},
                {"label": "MP", "current": 50, "max": 80, "color": "blue", "icon": "^"},
                {"label": "XP", "current": 300, "max": 500, "color": "gold", "icon": "*"}
            ])
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        def _render_once(frame: int = 0) -> str:
            from io import StringIO
            buf = StringIO()
            import sys as _sys
            old = _sys.stdout
            try:
                _sys.stdout = buf
                for stat in stats:
                    label = stat.get("label", "")
                    current = stat.get("current", 0)
                    maximum = stat.get("max", 100)
                    color = stat.get("color", "white")
                    icon = stat.get("icon", "")

                    # Build label
                    if show_icons and icon:
                        display_label = f"{icon} {label}"
                    else:
                        display_label = label

                    # Animate indeterminate if max==0
                    indeterminate = maximum == 0
                    ProgressBar.render(
                        current=current,
                        maximum=maximum if maximum else 100,
                        width=bar_width,
                        label=display_label,
                        show_percentage=not indeterminate,
                        show_numbers=not indeterminate,
                        color=color,
                        bracket_style="square",
                        indeterminate=indeterminate,
                        frame=frame,
                    )
            finally:
                _sys.stdout = old
            return buf.getvalue().rstrip("\n")

        if live_update:
            console = Console()
            with console.live() as live:
                if frames <= 0:
                    frames = 40
                for f in range(frames):
                    live.update(_render_once(f))
                    import time as _t
                    _t.sleep(1.0 / max(1.0, fps))
        else:
            out.append(_render_once())

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Spinner:
    """Build loading spinners and indicators."""

    SPINNER_STYLES = {
        "dots": [".", "o", "O", "o"],
        "line": ["-", "\\", "|", "/"],
        "arrow": ["<", "^", ">", "v"],
        "box": ["[ ]", "[=]", "[#]", "[=]"],
        "circle": ["( )", "(o)", "(O)", "(o)"],
        "bouncing": ["*", ".", " ", "."],
    }

    @staticmethod
    def render(
        text: str = "Loading",
        frame: int = 0,
        style: str = "dots",
        color: str = "cyan"
    ) -> str:
        """
        Get a single frame of a spinner (for animation loops).

        Args:
            text: Loading text
            frame: Current frame number
            style: Spinner style
            color: Spinner color

        Returns:
            Formatted spinner string
        """
        frames = Spinner.SPINNER_STYLES.get(style, Spinner.SPINNER_STYLES["dots"])
        spinner_char = frames[frame % len(frames)]

        spinner_color = Color.get_color(color)
        return f"{spinner_color}{spinner_char}{Color.RESET} {text}..."


class SegmentedBar:
    """Build segmented progress bars (health with segments)."""

    @staticmethod
    def render(
        current: int,
        maximum: int,
        segments: int = 10,
        width: int = 40,
        label: str = "",
        color: str = "green",
        segment_char: str = "#",
        separator: str = " ",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render a segmented progress bar.

        Args:
            current: Current value
            maximum: Maximum value
            segments: Number of segments
            width: Total bar width
            label: Label text
            color: Bar color
            segment_char: Character for segments
            separator: Separator between segments
            blank_lines_before: Blank lines before bar
            blank_lines_after: Blank lines after bar
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        bar_color = Color.get_color(color)

        # Calculate segment width
        sep_total = len(separator) * (segments - 1)
        seg_width = (width - sep_total) // segments

        # Calculate filled segments
        filled_segments = int((current / maximum) * segments) if maximum > 0 else 0

        # Build bar
        bar_parts = []
        for i in range(segments):
            if i < filled_segments:
                bar_parts.append(bar_color + segment_char * seg_width + Color.RESET)
            else:
                bar_parts.append(Symbols.BAR_EMPTY() * seg_width)

        bar = separator.join(bar_parts)

        # Output
        if label:
            out.append(f"{label}: [{bar}] {current}/{maximum}")
        else:
            out.append(f"[{bar}] {current}/{maximum}")

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Meter:
    """Build meter/gauge displays."""

    @staticmethod
    def render(
        value: float,
        min_val: float = 0.0,
        max_val: float = 100.0,
        width: int = 50,
        label: str = "",
        thresholds: dict[float, str] | None = None,
        show_value: bool = True,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render a meter/gauge.

        Args:
            value: Current value
            min_val: Minimum value
            max_val: Maximum value
            width: Meter width
            label: Meter label
            thresholds: Dict of threshold:color pairs
            show_value: Show numeric value
            blank_lines_before: Blank lines before meter
            blank_lines_after: Blank lines after meter
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        # Default thresholds
        if thresholds is None:
            thresholds = {0.0: "red", 0.3: "yellow", 0.7: "green"}

        # Calculate percentage
        percentage = (value - min_val) / (max_val - min_val) if max_val > min_val else 0
        percentage = max(0.0, min(1.0, percentage))

        # Determine color based on thresholds; coerce keys to float
        color = "white"
        try:
            th_items = [(float(k), v) for k, v in thresholds.items()]
        except Exception:
            th_items = [(k if isinstance(k, (int, float)) else 0.0, v) for k, v in thresholds.items()]
        for threshold, threshold_color in sorted(th_items, key=lambda kv: kv[0]):
            if percentage >= float(threshold):
                color = threshold_color

        # Build meter
        filled = int(width * percentage)
        meter_color = Color.get_color(color)

        bar = meter_color + Symbols.BAR_FILLED() * filled + Symbols.BAR_EMPTY() * (width - filled) + Color.RESET

        # Output
        if label:
            value_str = f" {value:.1f}" if show_value else ""
            out.append(f"{label}: [{bar}]{value_str}")
        else:
            value_str = f" {value:.1f}" if show_value else ""
            out.append(f"[{bar}]{value_str}")

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Thermometer:
    """Vertical thermometer widget."""

    @staticmethod
    def render(
        value: float,
        *,
        min_val: float = 0.0,
        max_val: float = 100.0,
        height: int = 10,
        color: str = "red",
        label: str | None = None,
    ) -> str:
        out: list[str] = []
        pct = 0.0 if max_val <= min_val else (value - min_val) / (max_val - min_val)
        pct = max(0.0, min(1.0, pct))
        filled = int(round(pct * height))
        col = Color.get_color(color)
        out.append("  ___ ")
        for row in range(height, 0, -1):
            if row <= filled:
                bar = col + "|#|" + Color.RESET
            else:
                bar = "| |"
            out.append(f"  {bar}")
        out.append("  ---")
        if label:
            out.append(f"{label}: {value:.1f}")
        return "\n".join(out)


class Compass:
    """Cardinal compass with direction indicator."""

    @staticmethod
    def render(direction: float | str = 0.0, color: str = "cyan") -> str:
        if isinstance(direction, str):
            mapping = {"N": 0.0, "E": 90.0, "S": 180.0, "W": 270.0}
            deg = mapping.get(direction.upper(), 0.0)
        else:
            deg = float(direction)
        deg = deg % 360.0
        col = Color.get_color(color)
        lines: list[str] = []
        lines.append("   N")
        lines.append(" W + E")
        lines.append("   S")
        if deg < 45 or deg >= 315:
            arrow = '^'
        elif deg < 135:
            arrow = '>'
        elif deg < 225:
            arrow = 'v'
        else:
            arrow = '<'
        lines.append(col + f"  {arrow} {deg:.0f} deg" + Color.RESET)
        return "\n".join(lines)


class GaugeDial:
    """Horizontal gauge with pointer."""

    @staticmethod
    def render(value: float, *, min_val: float = 0.0, max_val: float = 100.0, width: int = 20, color: str = "accent", label: str | None = None) -> str:
        pct = 0.0 if max_val <= min_val else (value - min_val) / (max_val - min_val)
        pct = max(0.0, min(1.0, pct))
        pos = int(pct * (width - 1))
        col = Color.get_color(color)
        scale = "|" + "-" * (width - 2) + "|"
        pointer = " " * pos + "^"
        lines = [col + scale + Color.RESET, pointer]
        if label:
            lines.append(f"{label}: {value:.1f}")
        return "\n".join(lines)


class WeatherIndicator:
    """Simple weather indicator with icon."""

    @staticmethod
    def render(condition: str, temperature_c: float | None = None) -> str:
        cond = condition.lower()
        icon = {
            'sunny': '(sun)',
            'clear': '(sun)',
            'cloudy': '(cloud)',
            'rain': '(rain)',
            'snow': '(snow)',
            'storm': '(storm)',
            'fog': '(fog)',
        }.get(cond, '(weather)')
        text = f"{icon} {condition.title()}"
        if temperature_c is not None:
            text += f" {temperature_c:.1f} degC"
        return text


@dataclass(slots=True)
class _ThermoRenderable:
    value: float
    min_val: float = 0.0
    max_val: float = 100.0
    height: int = 10
    color: str = "red"
    label: str | None = None

    def render(self, console: Console) -> str:
        return Thermometer.render(
            self.value,
            min_val=self.min_val,
            max_val=self.max_val,
            height=self.height,
            color=self.color,
            label=self.label,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def thermometer(
    value: float,
    *,
    min_val: float = 0.0,
    max_val: float = 100.0,
    height: int = 10,
    color: str = "red",
    label: str | None = None,
) -> RenderableLike:
    return _ThermoRenderable(
        value=value,
        min_val=min_val,
        max_val=max_val,
        height=height,
        color=color,
        label=label,
    )


def compass(*args: object, **kwargs: object) -> RenderableLike:
    return render_call(Compass.render, *args, **kwargs)


def gauge_dial(*args: object, **kwargs: object) -> RenderableLike:
    return render_call(GaugeDial.render, *args, **kwargs)


def weather_indicator(*args: object, **kwargs: object) -> RenderableLike:
    return render_call(WeatherIndicator.render, *args, **kwargs)
