"""Navigation and menu components."""
from __future__ import annotations

from dataclasses import dataclass

from ..core import Console, Measure, render_call
from ..style.colors import Color
from ..symbols import Symbols
from ..utils import get_visible_length
from ..utils.input import read_key

__all__ = [
    "Menu",
    "HorizontalMenu",
    "Breadcrumbs",
    "Tabs",
    "Accordion",
    "ContextMenu",
    "HotkeyOverlay",
    "InputPrompt",
    "ConfirmPrompt",
    "NumericPrompt",
    "SearchBox",
    "FileBrowser",
    "menu",
    "horizontal_menu",
    "breadcrumbs",
    "tabs",
    "accordion",
    "context_menu",
    "hotkey_overlay",
    "input_prompt",
    "confirm_prompt",
    "numeric_prompt",
    "search_box",
    "file_browser",
]


class Menu:
    """Build interactive-looking menus."""

    @staticmethod
    def render(
        title: str,
        options: list[str],
        selected_index: int | None = None,
        width: int = 60,
        show_numbers: bool = True,
        border_style: str = "double",
        title_color: str = "cyan",
        selected_color: str = "gold",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None
    ) -> str:
        """
        Render a menu with options.

        Args:
            title: Menu title
            options: List of menu options
            selected_index: Index of selected option (None for no selection)
            width: Menu width
            show_numbers: Show option numbers
            border_style: Border style
            title_color: Title color
            selected_color: Selected option color
            blank_lines_before: Blank lines before menu
            blank_lines_after: Blank lines after menu
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        symbols = Symbols.get_symbols(border_style, ascii_mode)
        tcolor = Color.get_color(title_color)
        scolor = Color.get_color(selected_color)

        # Top border
        top = symbols["top_left"] + symbols["horizontal"] * (width - 2) + symbols["top_right"]
        out.append(top)

        # Title
        title_text = title.center(width - 4)
        title_line = symbols["vertical"] + " " + tcolor + title_text + Color.RESET + " " + symbols["vertical"]
        out.append(title_line)

        # Separator
        separator = symbols["t_right"] + symbols["horizontal"] * (width - 2) + symbols["t_left"]
        out.append(separator)

        # Options
        for i, option in enumerate(options):
            marker = "► " if i == selected_index else "  "
            number = f"{i + 1}. " if show_numbers else ""

            option_text = f"{marker}{number}{option}"

            if i == selected_index:
                option_text = scolor + option_text + Color.RESET

            visible_len = get_visible_length(option_text)
            padding_needed = width - 4 - visible_len
            option_line = symbols["vertical"] + " " + option_text + " " * padding_needed + " " + symbols["vertical"]
            out.append(option_line)

        # Bottom border
        bottom = symbols["bottom_left"] + symbols["horizontal"] * (width - 2) + symbols["bottom_right"]
        out.append(bottom)

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)

    @staticmethod
    def read_choice(
        title: str,
        options: list[str],
        *,
        prompt: str = "Choose an option: ",
        show_numbers: bool = True,
        interactive: bool = False,
        width: int = 60,
        border_style: str = "double",
    ) -> int:
        """Read a menu choice from stdin or via an interactive focus loop.

        Returns the selected index (0-based). Retries on invalid input.
        """
        if not options:
            raise ValueError("Menu options cannot be empty.")

        if interactive:
            return Menu.interactive(
                title,
                options,
                width=width,
                border_style=border_style,
                show_numbers=show_numbers,
            )

        Menu.render(title, options, show_numbers=show_numbers)
        while True:
            try:
                user = input(prompt)
                idx = int(user) - 1 if show_numbers else options.index(user)
                if 0 <= idx < len(options):
                    return idx
            except (ValueError, EOFError):
                pass
            print("Invalid selection. Try again.")

    @staticmethod
    def interactive(
        title: str,
        options: list[str],
        *,
        width: int = 60,
        border_style: str = "double",
        show_numbers: bool = True,
        selected_index: int = 0,
        instructions: bool = True,
        timeout: float = 0.2,
    ) -> int:
        """Run an arrow-key driven menu interaction loop.

        Returns the selected index, or ``-1`` if the session is cancelled.
        """
        if not options:
            return -1

        selected = max(0, min(selected_index, len(options) - 1))
        try:
            while True:
                print("\x1b[2J\x1b[H", end="")
                Menu.render(
                    title,
                    options,
                    selected_index=selected,
                    width=width,
                    border_style=border_style,
                    show_numbers=show_numbers,
                )
                if instructions:
                    print("Use ↑/↓ (or j/k) to move, Enter to select, q to cancel.")
                key = read_key(timeout=timeout)
                if not key:
                    continue
                if key in {"up", "k"}:
                    selected = (selected - 1) % len(options)
                elif key in {"down", "j"}:
                    selected = (selected + 1) % len(options)
                elif key in {"enter", "space"}:
                    return selected
                elif key in {"escape", "q"}:
                    return -1
        finally:
            print("\x1b[2J\x1b[H", end="")

    @staticmethod
    def run(title: str, options: list[str], *, timeout: float = 0.2) -> str | None:
        """High-level stateful run that returns the selected option or None."""
        idx = Menu.interactive(title, options, timeout=timeout)
        return (options[idx] if 0 <= idx < len(options) else None) if idx != -1 else None


@dataclass(slots=True)
class _MenuRenderable:
    title: str
    options: list[str]
    selected_index: int | None = None
    width: int = 60
    show_numbers: bool = True
    border_style: str = "double"
    title_color: str = "cyan"
    selected_color: str = "gold"
    blank_lines_before: int = 0
    blank_lines_after: int = 0
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return Menu.render(
            self.title,
            self.options,
            selected_index=self.selected_index,
            width=self.width,
            show_numbers=self.show_numbers,
            border_style=self.border_style,
            title_color=self.title_color,
            selected_color=self.selected_color,
            blank_lines_before=self.blank_lines_before,
            blank_lines_after=self.blank_lines_after,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def menu(
    title: str,
    options: list[str],
    selected_index: int | None = None,
    width: int = 60,
    show_numbers: bool = True,
    border_style: str = "double",
    title_color: str = "cyan",
    selected_color: str = "gold",
    blank_lines_before: int = 0,
    blank_lines_after: int = 0,
    ascii_mode: bool | None = None,
) -> _MenuRenderable:
    """Return a renderable menu."""
    return _MenuRenderable(
        title=title,
        options=options,
        selected_index=selected_index,
        width=width,
        show_numbers=show_numbers,
        border_style=border_style,
        title_color=title_color,
        selected_color=selected_color,
        blank_lines_before=blank_lines_before,
        blank_lines_after=blank_lines_after,
        ascii_mode=ascii_mode,
    )


def horizontal_menu(*args: object, **kwargs: object):
    """Return a renderable horizontal menu."""
    return render_call(HorizontalMenu.render, *args, **kwargs)


def breadcrumbs(*args: object, **kwargs: object):
    """Return a renderable breadcrumb trail."""
    return render_call(Breadcrumbs.render, *args, **kwargs)


def tabs(*args: object, **kwargs: object):
    """Return a renderable tabs bar."""
    return render_call(Tabs.render, *args, **kwargs)


def accordion(*args: object, **kwargs: object):
    """Return a renderable accordion."""
    return render_call(Accordion.render, *args, **kwargs)


def context_menu(*args: object, **kwargs: object):
    return render_call(ContextMenu.render, *args, **kwargs)


def hotkey_overlay(*args: object, **kwargs: object):
    return render_call(HotkeyOverlay.render, *args, **kwargs)


def input_prompt(*args: object, **kwargs: object):
    return render_call(InputPrompt.render, *args, **kwargs)


def confirm_prompt(*args: object, **kwargs: object):
    return render_call(ConfirmPrompt.render, *args, **kwargs)


def numeric_prompt(*args: object, **kwargs: object):
    return render_call(NumericPrompt.render, *args, **kwargs)


def search_box(*args: object, **kwargs: object):
    return render_call(SearchBox.render, *args, **kwargs)


def file_browser(*args: object, **kwargs: object):
    return render_call(FileBrowser.render, *args, **kwargs)


class Breadcrumbs:
    """Build breadcrumb navigation."""

    @staticmethod
    def render(
        path: list[str],
        separator: str = " › ",
        current_color: str = "gold",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render breadcrumb navigation.

        Args:
            path: List of path segments
            separator: Separator between segments
            current_color: Color for current (last) segment
            blank_lines_before: Blank lines before breadcrumbs
            blank_lines_after: Blank lines after breadcrumbs
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        if not path:
            return ""

        curr_color = Color.get_color(current_color)
        result = []

        for i, segment in enumerate(path):
            if i == len(path) - 1:
                result.append(curr_color + Color.BOLD + segment + Color.RESET)
            else:
                result.append(segment)

        out.append(separator.join(result))

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class HorizontalMenu:
    """Build horizontal menus."""

    @staticmethod
    def render(
        options: list[str],
        selected_index: int = 0,
        separator: str = " | ",
        selected_color: str = "gold",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render a horizontal menu.

        Args:
            options: List of menu options
            selected_index: Index of selected option
            separator: Separator between options
            selected_color: Color for selected option
            blank_lines_before: Blank lines before menu
            blank_lines_after: Blank lines after menu
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        sel_color = Color.get_color(selected_color)
        parts = []

        for i, option in enumerate(options):
            if i == selected_index:
                parts.append(f"{sel_color}{Color.BOLD}[ {option} ]{Color.RESET}")
            else:
                parts.append(option)

        out.append(separator.join(parts))

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Tabs:
    """Render tabs with an active tab highlighted."""

    @staticmethod
    def render(
        labels: list[str],
        active_index: int = 0,
        separator: str = "  ",
        active_color: str = "primary",
        inactive_color: str = "muted",
        underline: bool = True,
        width: int | None = None,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
    ) -> str:
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        act = Color.get_color(active_color)
        ina = Color.get_color(inactive_color)
        parts: list[str] = []
        for i, label in enumerate(labels):
            if i == active_index:
                text = f"{act}{Color.BOLD}{label}{Color.RESET}"
            else:
                text = f"{ina}{label}{Color.RESET}"
            parts.append(text)

        line = separator.join(parts)
        if width:
            pad = max(0, width - get_visible_length(line))
            line = line + " " * pad
        out.append(line)

        if underline and labels:
            # Draw underline under active tab only
            pre_len = get_visible_length(separator.join(
                [lbl if idx < active_index else "" for idx, lbl in enumerate(labels)]
            ))
            pre_sep = get_visible_length(separator) if active_index > 0 else 0
            active_label_len = get_visible_length(labels[active_index])
            out.append(" " * (pre_len + pre_sep) + Color.get_color("accent") + "─" * active_label_len + Color.RESET)

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Accordion:
    """Render an accordion with multiple sections; one expanded by index."""

    @staticmethod
    def render(
        sections: list[dict[str, str]],
        expanded_index: int | None = 0,
        width: int = 80,
        border_style: str = "rounded",
        title_color: str = "fg",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None,
    ) -> str:
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        symbols = Symbols.get_symbols(border_style, ascii_mode)
        tcol = Color.get_color(title_color)

        for idx, sec in enumerate(sections):
            title = sec.get("title", f"Section {idx + 1}")
            content = sec.get("content", "")

            # Header line
            marker = "▼" if expanded_index == idx else "▶"
            header = f" {marker} {title} "
            header_visible = get_visible_length(header)
            pad = max(0, width - 2 - header_visible)
            out.append(symbols["top_left"] + symbols["horizontal"] * (width - 2) + symbols["top_right"])
            out.append(symbols["vertical"] + tcol + header + Color.RESET + " " * pad + symbols["vertical"])

            if expanded_index == idx:
                # Content lines
                from ..utils import wrap_text  # local import to avoid cycles
                for line in wrap_text(content, width - 4):
                    visible = get_visible_length(line)
                    pad2 = max(0, width - 2 - visible)
                    out.append(symbols["vertical"] + line + " " * pad2 + symbols["vertical"])

            out.append(symbols["bottom_left"] + symbols["horizontal"] * (width - 2) + symbols["bottom_right"])

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class ContextMenu:
    @staticmethod
    def render(options: list[str], selected_index: int = 0, width: int = 24) -> None:
        print("┌" + "─" * (width - 2) + "┐")
        for i, opt in enumerate(options):
            marker = "▶" if i == selected_index else " "
            text = f" {marker} {opt}"
            pad = max(0, width - 2 - len(text))
            print("│" + text + " " * pad + "│")
        print("└" + "─" * (width - 2) + "┘")


class HotkeyOverlay:
    @staticmethod
    def render(bindings: dict[str, str]) -> None:
        print("Hotkeys:")
        for key, action in bindings.items():
            print(f"  {key:<10} {action}")


class InputPrompt:
    @staticmethod
    def render(
        prompt: str = "Input:",
        placeholder: str | None = None,
        color: str | None = None,
    ) -> str:
        col = Color.get_color(color) if color else ""
        reset = Color.RESET if color else ""
        display = f"{col}{prompt}{reset} "
        if placeholder:
            display += f"{Color.DIM}{placeholder}{Color.RESET}"
        print(display, end="")
        try:
            return input()
        except EOFError:
            return ""


class ConfirmPrompt:
    @staticmethod
    def render(
        prompt: str = "Are you sure?",
        default: bool | None = None,
        color: str | None = None,
    ) -> bool:
        suffix = " [y/n]"
        if default is True:
            suffix = " [Y/n]"
        elif default is False:
            suffix = " [y/N]"
        col = Color.get_color(color) if color else ""
        reset = Color.RESET if color else ""
        while True:
            print(f"{col}{prompt}{reset}{suffix} ", end="")
            try:
                val = input().strip().lower()
            except EOFError:
                return bool(default)
            if not val and default is not None:
                return default
            if val in ("y", "yes"):
                return True
            if val in ("n", "no"):
                return False


class NumericPrompt:
    @staticmethod
    def render(
        prompt: str = "Enter number:",
        min_value: float | None = None,
        max_value: float | None = None,
    ) -> float | None:
        while True:
            print(f"{prompt} ", end="")
            try:
                text = input().strip()
            except EOFError:
                return None
            try:
                num = float(text)
            except ValueError:
                print("Invalid number. Try again.")
                continue
            if min_value is not None and num < min_value:
                print("Value too small.")
                continue
            if max_value is not None and num > max_value:
                print("Value too large.")
                continue
            return num


class SearchBox:
    @staticmethod
    def render(
        prompt: str = "Search:",
        dataset: list[str] | None = None,
        max_results: int = 5,
    ) -> list[str]:
        print(f"{prompt} ", end="")
        try:
            query = input().strip()
        except EOFError:
            return []
        if not dataset:
            return [query]
        # Simple substring match
        results = [item for item in dataset if query.lower() in item.lower()]
        return results[:max_results]


class FileBrowser:
    @staticmethod
    def render(
        path: str = ".",
        show_hidden: bool = False,
        max_entries: int = 50,
    ) -> None:
        import os
        try:
            entries = sorted(os.listdir(path))
        except Exception as e:
            print(f"Error reading {path}: {e}")
            return
        count = 0
        for name in entries:
            if not show_hidden and name.startswith('.'):
                continue
            full = os.path.join(path, name)
            icon = "📁" if os.path.isdir(full) else "📄"
            print(f"{icon} {name}")
            count += 1
            if count >= max_entries:
                break
