"""Interactive components (forms and controls).

These components provide simple interactive prompts using stdin/stdout so they
work in any terminal. They also support static rendering to preview states in
showcases and demos.
"""

from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING, Any, Literal

from ..core import render_call
from ..style.colors import Color

if TYPE_CHECKING:
    from collections.abc import Callable

__all__ = [
    "Checkbox",
    "Radio",
    "Toggle",
    "Slider",
    "Form",
    "checkbox",
    "radio",
    "toggle",
    "slider",
    "form",
]


class Checkbox:
    @staticmethod
    def render(
        label: str,
        checked: bool = False,
        color: str = "accent",
        interactive: bool = False,
        *,
        input_func: Callable[[], str] | None = None,
    ) -> bool | None:
        box = "☑" if checked else "☐"
        col = Color.get_color(color)
        line = f"{col}{box}{Color.RESET} {label}"
        print(line)
        if not interactive:
            return None
        print("Toggle? [y/N] ", end="")
        try:
            reader = input_func or input
            ans = reader().strip().lower()
        except EOFError:
            return checked
        return checked if not ans else ans.startswith("y")


class Radio:
    @staticmethod
    def render(
        label: str,
        selected: bool = False,
        color: str = "accent",
        interactive: bool = False,
        *,
        input_func: Callable[[], str] | None = None,
    ) -> bool | None:
        dot = "◉" if selected else "◯"
        col = Color.get_color(color)
        line = f"{col}{dot}{Color.RESET} {label}"
        print(line)
        if not interactive:
            return None
        print("Select? [y/N] ", end="")
        try:
            reader = input_func or input
            ans = reader().strip().lower()
        except EOFError:
            return selected
        return selected if not ans else ans.startswith("y")


class Toggle:
    @staticmethod
    def render(
        label: str,
        on: bool = False,
        color_on: str = "green",
        color_off: str = "gray",
        interactive: bool = False,
        *,
        input_func: Callable[[], str] | None = None,
    ) -> bool | None:
        col = Color.get_color(color_on if on else color_off)
        state = "ON " if on else "OFF"
        print(f"{label}: {col}[ {state} ]{Color.RESET}")
        if not interactive:
            return None
        print("Toggle? [y/N] ", end="")
        try:
            reader = input_func or input
            ans = reader().strip().lower()
        except EOFError:
            return on
        return on if not ans else not on


class Slider:
    @staticmethod
    def render(
        label: str,
        value: float = 0,
        minimum: float = 0,
        maximum: float = 100,
        width: int = 20,
        color: str = "primary",
        interactive: bool = False,
        *,
        input_func: Callable[[], str] | None = None,
    ) -> float | None:
        pct = 0 if maximum <= minimum else (value - minimum) / (maximum - minimum)
        pct = max(0.0, min(1.0, pct))
        filled = int(width * pct)
        bar = "█" * filled + "░" * (width - filled)
        col = Color.get_color(color)
        print(f"{label}: {col}[{bar}]{Color.RESET} {value:.2f}")
        if not interactive:
            return None
        print(f"Enter value ({minimum}..{maximum}): ", end="")
        try:
            reader = input_func or input
            text = reader().strip()
        except EOFError:
            return value
        try:
            num = float(text)
        except ValueError:
            return value
        return max(minimum, min(maximum, num))


@dataclass(slots=True)
class FormField:
    name: str
    kind: Literal["text", "number", "checkbox", "radio", "toggle", "slider"]
    label: str
    options: list[str] | None = None
    default: Any | None = None


class Form:
    @staticmethod
    def render(
        title: str,
        fields: list[FormField],
        interactive: bool = False,
        width: int = 60,
        *,
        input_func: Callable[[], str] | None = None,
    ) -> dict[str, Any] | None:
        from ..components.containers import Dialog

        Dialog.render(title, "Fill the form below:", buttons=["Submit"], selected_button=0, width=width)
        results: dict[str, Any] = {}
        for f in fields:
            label = f.label
            if f.kind == "text":
                if interactive:
                    print(f"{label}: ", end="")
                    try:
                        reader = input_func or input
                        results[f.name] = reader()
                    except EOFError:
                        results[f.name] = f.default
                else:
                    print(f"{label}: {f.default if f.default is not None else ''}")
            elif f.kind == "number":
                if interactive:
                    print(f"{label}: ", end="")
                    try:
                        reader = input_func or input
                        results[f.name] = float(reader())
                    except Exception:
                        results[f.name] = f.default
                else:
                    print(f"{label}: {f.default if f.default is not None else 0}")
            elif f.kind == "checkbox":
                val = Checkbox.render(label, bool(f.default), interactive=interactive)
                results[f.name] = bool(val) if val is not None else bool(f.default)
            elif f.kind == "radio":
                selected = 0
                if isinstance(f.default, int):
                    selected = f.default
                opts = f.options or []
                for i, opt in enumerate(opts):
                    Radio.render(opt, selected == i)
                if interactive:
                    print("Select index: ", end="")
                    try:
                        reader = input_func or input
                        idx = int(reader())
                    except Exception:
                        idx = selected
                    results[f.name] = opts[idx] if 0 <= idx < len(opts) else (opts[selected] if opts else None)
                else:
                    results[f.name] = (opts[selected] if opts else None)
            elif f.kind == "toggle":
                val = Toggle.render(label, bool(f.default), interactive=interactive)
                results[f.name] = bool(val) if val is not None else bool(f.default)
            elif f.kind == "slider":
                dv = float(f.default) if isinstance(f.default, (int, float)) else 0.0
                val = Slider.render(label, dv, 0, 100, interactive=interactive)
                results[f.name] = float(val) if isinstance(val, (int, float)) else dv
        return results if interactive else None


def checkbox(*args: object, **kwargs: object):
    return render_call(Checkbox.render, *args, **kwargs)


def radio(*args: object, **kwargs: object):
    return render_call(Radio.render, *args, **kwargs)


def toggle(*args: object, **kwargs: object):
    return render_call(Toggle.render, *args, **kwargs)


def slider(*args: object, **kwargs: object):
    return render_call(Slider.render, *args, **kwargs)


def form(*args: object, **kwargs: object):
    return render_call(Form.render, *args, **kwargs)
