"""Feedback components (badges, ratings, notifications)."""
from __future__ import annotations

import time
from dataclasses import dataclass, field
from typing import TYPE_CHECKING, Literal

from ..core import render_call
from ..style.colors import Color
from ..symbols import Symbols

if TYPE_CHECKING:
    from collections.abc import Iterable

    from ..core.console import RenderableLike

__all__ = [
    "Badge",
    "Rating",
    "Toast",
    "ToastManager",
    "ConsolePanel",
    "Snackbar",
    "AlertBanner",
    "badge",
    "rating",
    "toast",
    "toast_manager",
    "console_panel",
    "snackbar",
    "alert_banner",
]


class Badge:
    """Create inline badges and tags."""

    BRACKETS: dict[str, tuple[str, str]] = {
        "square": ("[", "]"),
        "round": ("(", ")"),
        "angle": ("<", ">"),
    }

    @staticmethod
    def render(
        text: str,
        *,
        style: Literal["square", "round", "angle"] = "square",
        color: str = "cyan",
        bg_color: str | None = None,
    ) -> str:
        left, right = Badge.BRACKETS.get(style, Badge.BRACKETS["square"])
        text_color = Color.get_color(color)
        result = f"{text_color}{left}{text}{right}{Color.RESET}"
        if bg_color:
            bg = Color.BG_COLORS.get(f"bg_{bg_color}", "")
            result = bg + result + Color.RESET
        return result


class Rating:
    """Build rating displays using ASCII glyphs."""

    SYMBOLS: dict[str, dict[str, str]] = {
        "stars": {"filled": "*", "empty": "-"},
        "hearts": {"filled": "<3", "empty": "< >"},
        "circles": {"filled": "o", "empty": "()"},
        "squares": {"filled": "[]", "empty": "[]"},
    }

    @staticmethod
    def render(
        rating: float,
        max_rating: int = 5,
        *,
        style: Literal["stars", "hearts", "circles", "squares"] = "stars",
        color: str = "gold",
        show_value: bool = True,
        allow_half: bool = True,
    ) -> str:
        rating = max(0.0, min(float(rating), float(max_rating)))
        syms = Rating.SYMBOLS.get(style, Rating.SYMBOLS["stars"])
        filled_units = int(rating)
        half_unit = allow_half and (rating - filled_units) >= 0.5
        empty_units = max_rating - filled_units - (1 if half_unit else 0)

        glyphs: list[str] = []
        glyphs.extend(syms["filled"] for _ in range(filled_units))
        if half_unit:
            glyphs.append(":")  # half indicator
        glyphs.extend(syms["empty"] for _ in range(max(0, empty_units)))

        rating_color = Color.get_color(color)
        text = rating_color + " ".join(glyphs) + Color.RESET
        if show_value:
            text += f" {rating:.1f}/{max_rating}"
        return text


def badge(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable badge."""
    return render_call(Badge.render, *args, **kwargs)


def rating(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable rating."""
    return render_call(Rating.render, *args, **kwargs)


@dataclass
class ToastMessage:
    """Internal representation of a toast within the manager."""

    text: str
    tone: Literal["info", "success", "warning", "error"]
    width: int
    border_style: str
    ascii_mode: bool | None = None
    created_at: float = field(default_factory=time.monotonic)
    duration: float | None = None

    def expired(self, now: float) -> bool:
        return self.duration is not None and now - self.created_at >= self.duration


class Toast:
    """Transient toast/growl notification."""

    ICONS = {"info": "[i]", "success": "[+]", "warning": "[!]", "error": "[x]"}
    COLORS = {"info": "info", "success": "success", "warning": "warning", "error": "error"}

    @staticmethod
    def render(
        message: str,
        *,
        tone: Literal["info", "success", "warning", "error"] = "info",
        width: int = 60,
        border_style: str = "rounded",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None,
    ) -> str:
        return Toast._render_box(
            message=message,
            tone=tone,
            width=width,
            border_style=border_style,
            blank_lines_before=blank_lines_before,
            blank_lines_after=blank_lines_after,
            ascii_mode=ascii_mode,
        )

    @staticmethod
    def _render_box(
        message: str,
        tone: Literal["info", "success", "warning", "error"],
        width: int,
        border_style: str,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None,
    ) -> str:
        symbols = Symbols.get_symbols(border_style, ascii_mode)
        col = Color.get_color(Toast.COLORS.get(tone, "info"))
        icon = Toast.ICONS.get(tone, "[i]")
        inner_width = max(10, width - 2)
        text = f" {icon} {message} "
        content = text[: inner_width] if len(text) > inner_width else text
        padding = max(0, inner_width - len(content))
        lines: list[str] = []
        for _ in range(blank_lines_before):
            lines.append("")
        lines.append(symbols["top_left"] + symbols["horizontal"] * inner_width + symbols["top_right"])
        lines.append(symbols["vertical"] + col + content + Color.RESET + " " * padding + symbols["vertical"])
        lines.append(symbols["bottom_left"] + symbols["horizontal"] * inner_width + symbols["bottom_right"])
        for _ in range(blank_lines_after):
            lines.append("")
        return "\n".join(lines)


class ToastManager:
    """Small helper to manage stacked toast/growl notifications."""

    def __init__(self) -> None:
        self._messages: list[ToastMessage] = []

    def push(
        self,
        message: str,
        *,
        tone: Literal["info", "success", "warning", "error"] = "info",
        width: int = 60,
        border_style: str = "rounded",
        duration: float | None = 4.0,
        ascii_mode: bool | None = None,
    ) -> None:
        self._messages.append(
            ToastMessage(
                text=message,
                tone=tone,
                width=width,
                border_style=border_style,
                ascii_mode=ascii_mode,
                duration=duration,
            )
        )
        self.cleanup()

    def cleanup(self) -> None:
        """Remove expired notifications."""
        now = time.monotonic()
        self._messages = [msg for msg in self._messages if not msg.expired(now)]

    def render(self) -> None:
        """Render all active toasts stacked vertically."""
        self.cleanup()
        for idx, message in enumerate(self._messages):
            print(Toast._render_box(
                message=message.text,
                tone=message.tone,
                width=message.width,
                border_style=message.border_style,
                blank_lines_before=1 if idx > 0 else 0,
                ascii_mode=message.ascii_mode,
            ))

    def clear(self) -> None:
        """Clear the manager without rendering."""
        self._messages.clear()


_TOAST_MANAGER = ToastManager()


def toast(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable toast notification."""
    return render_call(Toast.render, *args, **kwargs)


def toast_manager() -> ToastManager:
    """Return the module-level toast manager instance."""
    return _TOAST_MANAGER


class ConsolePanel:
    """A simple console-like panel for streaming lines."""

    @staticmethod
    def render(
        lines: Iterable[str],
        *,
        width: int = 80,
        height: int | None = None,
        border_style: str = "box",
        color: str | None = None,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None,
    ) -> str:
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        symbols = Symbols.get_symbols(border_style, ascii_mode)
        col = Color.get_color(color) if color else ""
        reset = Color.RESET if color else ""
        inner_width = max(2, width - 2)
        out.append(symbols["top_left"] + symbols["horizontal"] * inner_width + symbols["top_right"])

        content_lines = list(lines)
        if height is not None:
            content_lines = content_lines[-height:]
        for raw in content_lines:
            visible_len = len(raw)
            pad = max(0, inner_width - visible_len)
            out.append(symbols["vertical"] + col + raw + reset + " " * pad + symbols["vertical"])

        out.append(symbols["bottom_left"] + symbols["horizontal"] * inner_width + symbols["bottom_right"])
        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Snackbar:
    """One-line notification bar."""

    ICONS = {"info": "[i]", "success": "[+]", "warning": "[!]", "error": "[x]"}

    @staticmethod
    def render(
        message: str,
        *,
        tone: Literal["info", "success", "warning", "error"] = "info",
    ) -> str:
        col = Color.get_color(Toast.COLORS.get(tone, "info"))
        icon = Snackbar.ICONS.get(tone, "[i]")
        return f"{col}{icon} {message}{Color.RESET}"


class AlertBanner:
    """Loud banner for important alerts."""

    @staticmethod
    def render(
        message: str,
        *,
        tone: Literal["info", "success", "warning", "error"] = "info",
        width: int = 80,
    ) -> str:
        col = Color.get_color(Toast.COLORS.get(tone, "info"))
        inner_width = max(2, width)
        text = f" {message} "
        if len(text) > inner_width:
            text = text[: inner_width]
        pad = max(0, inner_width - len(text))
        line = col + "=" * inner_width + Color.RESET
        return "\n".join([line, col + text + " " * pad + Color.RESET, line])


def console_panel(*args: object, **kwargs: object) -> RenderableLike:
    """Return a renderable console panel."""
    return render_call(ConsolePanel.render, *args, **kwargs)


def snackbar(*args: object, **kwargs: object) -> RenderableLike:
    """Render a snackbar notification."""
    return render_call(Snackbar.render, *args, **kwargs)


def alert_banner(*args: object, **kwargs: object) -> RenderableLike:
    """Render an alert banner."""
    return render_call(AlertBanner.render, *args, **kwargs)
