"""Decorative components such as dividers and banners."""
from __future__ import annotations

from typing import Literal

from ..core import render_call
from ..style.colors import Color
from ..symbols import Symbols

__all__ = [
    "Divider",
    "Banner",
    "Timeline",
    "Tooltip",
    "AsciiArt",
    "divider",
    "banner",
    "timeline",
    "tooltip",
    "ascii_art",
]


class Divider:
    """Build horizontal dividers and separators."""

    @staticmethod
    def render(
        width: int = 80,
        style: Literal["solid", "double", "dashed", "dotted", "heavy", "wave", "decorative"] = "solid",
        text: str = "",
        text_position: Literal["left", "center", "right"] = "center",
        color: str | None = None,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0
    ) -> str:
        """
        Render a horizontal divider.

        Args:
            width: Total width of divider
            style: Style of divider line
            text: Optional text in the divider
            text_position: Position of text
            color: Divider color
            blank_lines_before: Blank lines before divider
            blank_lines_after: Blank lines after divider
        """
        lines: list[str] = []
        for _ in range(blank_lines_before):
            lines.append("")

        # Define line characters
        chars = {
            "solid": "─",
            "double": "═",
            "dashed": "┄",
            "dotted": "·",
            "heavy": "━",
            "wave": "~",
            "decorative": "═══"
        }
        char = chars.get(style, "─")

        # Build divider
        if text:
            text_with_spaces = f" {text} "
            text_len = len(text_with_spaces)

            if text_position == "left":
                line = text_with_spaces + char * (width - text_len)
            elif text_position == "right":
                line = char * (width - text_len) + text_with_spaces
            else:  # center
                left_width = (width - text_len) // 2
                right_width = width - text_len - left_width
                line = char * left_width + text_with_spaces + char * right_width
        else:
            line = char * width

        # Apply color
        if color:
            line_color = Color.get_color(color)
            line = line_color + line + Color.RESET

        lines.append(line)

        for _ in range(blank_lines_after):
            lines.append("")
        return "\n".join(lines)


def divider(*args: object, **kwargs: object):
    """Return a renderable divider component."""
    return render_call(Divider.render, *args, **kwargs)


def banner(*args: object, **kwargs: object):
    """Return a renderable banner component."""
    return render_call(Banner.render, *args, **kwargs)


def tooltip(*args: object, **kwargs: object):
    """Return a renderable tooltip."""
    return render_call(Tooltip.render, *args, **kwargs)


def timeline(*args: object, **kwargs: object):
    """Return a renderable timeline."""
    return render_call(Timeline.render, *args, **kwargs)


def ascii_art(*args: object, **kwargs: object):
    """Return a renderable ASCII art block."""
    return render_call(AsciiArt.render, *args, **kwargs)


class Banner:
    """Create large decorative text banners."""

    @staticmethod
    def render(
        text: str,
        style: Literal["simple", "block", "double"] = "simple",
        width: int = 80,
        color: str | None = None,
        blank_lines_before: int = 1,
        blank_lines_after: int = 1
    ) -> str:
        """
        Render a decorative banner.

        Args:
            text: Banner text
            style: Banner style
            width: Banner width
            color: Text color
            blank_lines_before: Blank lines before banner
            blank_lines_after: Blank lines after banner
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        text_color = Color.get_color(color) if color else ""

        if style == "block":
            # Block style with filled background
            border = "█" * width
            padding = "█" + " " * (width - 2) + "█"
            text_centered = text.center(width - 2)
            text_line = "█" + text_color + text_centered + Color.RESET + "█"

            out.append(border)
            out.append(padding)
            out.append(text_line)
            out.append(padding)
            out.append(border)

        elif style == "double":
            # Double line border
            top = "╔" + "═" * (width - 2) + "╗"
            padding = "║" + " " * (width - 2) + "║"
            text_centered = text.center(width - 2)
            text_line = "║" + text_color + text_centered + Color.RESET + "║"
            bottom = "╚" + "═" * (width - 2) + "╝"

            out.append(top)
            out.append(padding)
            out.append(text_line)
            out.append(padding)
            out.append(bottom)

        else:  # simple
            # Simple with decorative characters
            border = "=" * width
            text_centered = text.center(width)
            if color:
                text_centered = text_color + text_centered + Color.RESET

            out.append(border)
            out.append(text_centered)
            out.append(border)

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Tooltip:
    """Render a tooltip-like box with small arrow indicator."""

    @staticmethod
    def render(
        text: str,
        width: int = 40,
        color: str | None = None,
        border_style: str = "rounded",
        arrow: Literal["up", "down", "left", "right"] = "up",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None,
    ) -> str:
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        _ = Symbols.get_symbols(border_style, ascii_mode)
        col = Color.get_color(color) if color else ""
        reset = Color.RESET if color else ""

        # Arrow line
        arrow_map = {"up": "▲", "down": "▼", "left": "◀", "right": "▶"}
        arrow_char = arrow_map.get(arrow, "▲")
        out.append(" " * ((width // 2) - 1) + arrow_char)

        # Boxed text
        out.extend(Divider.render(width=width, style="dashed").splitlines())
        lines = [text]
        for line in lines:
            visible = len(line)
            pad = max(0, width - visible)
            out.append(col + line + reset + " " * pad)
        out.extend(Divider.render(width=width, style="dashed").splitlines())

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Timeline:
    """Render a vertical timeline of events with markers and timestamps."""

    @staticmethod
    def render(
        events: list[dict[str, str]],
        width: int = 80,
        marker_color: str = "accent",
        text_color: str | None = None,
    ) -> str:
        out: list[str] = []
        mcol = Color.get_color(marker_color)
        tcol = Color.get_color(text_color) if text_color else ""
        reset = Color.RESET if text_color else ""
        for i, ev in enumerate(events):
            ts = ev.get("time", "")
            title = ev.get("title", "")
            desc = ev.get("desc", "")
            connector = "│" if i < len(events) - 1 else " "
            marker = f"{mcol}●{Color.RESET}"
            out.append(f"{connector} {marker} {tcol}{title}{reset} {Color.DIM}{ts}{Color.RESET}")
            if desc:
                out.append(f"{connector}    {desc}")
        return "\n".join(out)


class AsciiArt:
    """Render ASCII art banners from text using simple fonts.

    For now, supports two built-in simple fonts without external deps.
    """

    FONTS = {
        "tiny": {
            "A": [" /\\ ", "/--\\", "\\__/"],
            "B": ["|__]", "|__]"],
        },
        "block": {},
    }

    @staticmethod
    def render(text: str, font: str = "tiny", color: str | None = None) -> str:
        col = Color.get_color(color) if color else ""
        reset = Color.RESET if color else ""
        # Extremely small demo font: fallback to plain text if missing
        lines: list[str] = ["", "", ""]
        font_map = AsciiArt.FONTS.get(font, {})
        for ch in text.upper():
            glyph = font_map.get(ch)
            if not glyph:
                # Fallback: append the char to the middle line
                lines[1] += ch + " "
                continue
            for i, row in enumerate(glyph):
                if i >= len(lines):
                    lines.append("")
                lines[i] += row + "  "
        return "\n".join(col + line.rstrip() + reset for line in lines)
