"""Container-style components (boxes, cards, panels, dialogs)."""
from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING, Any, Literal

from ..core import Console, Measure
from ..style.colors import Color
from ..symbols import Symbols
from ..text_engine import visible_width as _vw
from ..utils import get_visible_length, wrap_text
from ..utils.input import read_key

if TYPE_CHECKING:
    from collections.abc import Sequence

__all__ = [
    "Box",
    "Panel",
    "ScrollablePanel",
    "Card",
    "Dialog",
    "box",
    "panel",
    "scrollable_panel",
    "card",
    "dialog",
]


class _BorderMixin:
    """Shared helpers for drawing framed boxes."""

    @staticmethod
    def top_line(symbols: dict[str, str], width: int) -> str:
        return symbols["top_left"] + symbols["horizontal"] * (width - 2) + symbols["top_right"]

    @staticmethod
    def bottom_line(symbols: dict[str, str], width: int) -> str:
        return symbols["bottom_left"] + symbols["horizontal"] * (width - 2) + symbols["bottom_right"]

    @staticmethod
    def empty_line(symbols: dict[str, str], width: int) -> str:
        return symbols["vertical"] + " " * (width - 2) + symbols["vertical"]


class Box(_BorderMixin):
    """Build formatted box elements with optional border and background."""

    @staticmethod
    def render(
        text: str,
        width: int = 80,
        color: str | None = None,
        symbol_style: str = "box",
        border_thickness: int = 1,
        padding: int = 1,
        margin_left: int = 0,
        bg_color: str | None = None,
        shadow: bool = False,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        align: Literal["left", "center", "right"] = "center",
        ascii_mode: bool | None = None
    ) -> str:
        """
        Render a simple boxed text element.

        Args:
            text: Text content (supports inline markup)
            width: Box width
            color: Text color
            symbol_style: Border style
            padding: Internal padding
            blank_lines_before: Blank lines before box
            blank_lines_after: Blank lines after box
            align: Text alignment
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)

        Returns:
            The rendered box as a string.
        """
        text_color = Color.get_color(color)
        bg = Color.get_bg_color(bg_color) if bg_color else ""
        symbols = Symbols.get_symbols(symbol_style, ascii_mode)

        # Collect output lines
        output_lines: list[str] = []

        # Responsive width detection
        if width <= 0:
            max_line = 0
            for ln in wrap_text(text, 10**6):
                max_line = max(max_line, _vw(Color.apply_inline_markup(ln)))
            inner = max_line + padding * 2
            width = max(4, inner + 2)

        for _ in range(blank_lines_before):
            output_lines.append("")

        # Top border (support thickness)
        for _ in range(max(1, border_thickness)):
            output_lines.append((" " * margin_left) + _BorderMixin.top_line(symbols, width))

        # Padding lines
        for _ in range(padding):
            line = symbols["vertical"] + (bg + " " * (width - 2) + (Color.RESET if bg else "")) + symbols["vertical"]
            output_lines.append((" " * margin_left) + line)

        # Content
        inner_width = width - 2  # exclude borders
        content_width = inner_width - (padding * 2)
        text_lines = wrap_text(text, content_width)
        for line in text_lines:
            colored_line = Color.apply_inline_markup(line)
            if color:
                colored_line = text_color + colored_line + Color.RESET
            visible_len = get_visible_length(colored_line)

            if align == "left":
                left_spaces = 0
            elif align == "right":
                left_spaces = max(0, content_width - visible_len)
            else:  # center
                left_spaces = max(0, (content_width - visible_len) // 2)
            right_spaces = max(0, content_width - visible_len - left_spaces)

            inner = (
                symbols["vertical"]
                + (" " * padding)
                + (bg or "")
                + (" " * left_spaces)
                + colored_line
                + (" " * right_spaces)
                + (Color.RESET if bg else "")
                + (" " * padding)
                + symbols["vertical"]
            )
            output_lines.append((" " * margin_left) + inner)

        # Padding lines
        for _ in range(padding):
            line = symbols["vertical"] + (bg + " " * (width - 2) + (Color.RESET if bg else "")) + symbols["vertical"]
            output_lines.append((" " * margin_left) + line)

        # Bottom border (support thickness)
        for _ in range(max(1, border_thickness)):
            output_lines.append((" " * margin_left) + _BorderMixin.bottom_line(symbols, width))

        # Shadow effect: a simple offset dark fill to the right and bottom
        if shadow:
            shade_char = "░"
            # Right shadow strip
            output_lines.append((" " * (margin_left + width)) + shade_char)
            # Bottom shadow line
            output_lines.append((" " * (margin_left + 1)) + shade_char * (width))

        for _ in range(blank_lines_after):
            output_lines.append("")

        return "\n".join(output_lines)


@dataclass(slots=True)
class _BoxRenderable:
    """Renderable wrapper for Box.render that returns strings."""

    text: str
    width: int = 80
    color: str | None = None
    symbol_style: str = "box"
    border_thickness: int = 1
    padding: int = 1
    margin_left: int = 0
    bg_color: str | None = None
    shadow: bool = False
    blank_lines_before: int = 0
    blank_lines_after: int = 0
    align: Literal["left", "center", "right"] = "center"
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return Box.render(
            self.text,
            self.width,
            self.color,
            self.symbol_style,
            self.border_thickness,
            self.padding,
            self.margin_left,
            self.bg_color,
            self.shadow,
            self.blank_lines_before,
            self.blank_lines_after,
            self.align,
            self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        rendered = self.render(console)
        return Measure.from_text(rendered)


def box(
    text: str,
    width: int = 80,
    color: str | None = None,
    symbol_style: str = "box",
    border_thickness: int = 1,
    padding: int = 1,
    margin_left: int = 0,
    bg_color: str | None = None,
    shadow: bool = False,
    blank_lines_before: int = 0,
    blank_lines_after: int = 0,
    align: Literal["left", "center", "right"] = "center",
    ascii_mode: bool | None = None,
) -> _BoxRenderable:
    """Return a renderable box component."""
    return _BoxRenderable(
        text=text,
        width=width,
        color=color,
        symbol_style=symbol_style,
        border_thickness=border_thickness,
        padding=padding,
        margin_left=margin_left,
        bg_color=bg_color,
        shadow=shadow,
        blank_lines_before=blank_lines_before,
        blank_lines_after=blank_lines_after,
        align=align,
        ascii_mode=ascii_mode,
    )


@dataclass(slots=True)
class _PanelRenderable:
    """Renderable wrapper for Panel.render that captures output as a string."""

    sections: list[dict[str, Any]]
    width: int = 80
    border_style: str = "box"
    blank_lines_before: int = 0
    blank_lines_after: int = 0
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return Panel.render(
            self.sections,
            width=self.width,
            border_style=self.border_style,
            blank_lines_before=self.blank_lines_before,
            blank_lines_after=self.blank_lines_after,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def panel(
    sections: list[dict[str, Any]],
    width: int = 80,
    border_style: str = "box",
    blank_lines_before: int = 0,
    blank_lines_after: int = 0,
    ascii_mode: bool | None = None,
) -> _PanelRenderable:
    """Return a renderable panel component."""
    return _PanelRenderable(
        sections=sections,
        width=width,
        border_style=border_style,
        blank_lines_before=blank_lines_before,
        blank_lines_after=blank_lines_after,
        ascii_mode=ascii_mode,
    )


@dataclass(slots=True)
class _ScrollablePanelRenderable:
    """Renderable wrapper for ScrollablePanel.render that captures output as a string."""

    content: Sequence[str] | str
    width: int = 80
    height: int = 10
    offset: int = 0
    title: str | None = None
    footer: bool = True
    border_style: str = "box"
    blank_lines_before: int = 0
    blank_lines_after: int = 0
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return ScrollablePanel.render(
            self.content,
            width=self.width,
            height=self.height,
            offset=self.offset,
            title=self.title,
            footer=self.footer,
            border_style=self.border_style,
            blank_lines_before=self.blank_lines_before,
            blank_lines_after=self.blank_lines_after,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def scrollable_panel(
    content: Sequence[str] | str,
    width: int = 80,
    height: int = 10,
    offset: int = 0,
    title: str | None = None,
    footer: bool = True,
    border_style: str = "box",
    blank_lines_before: int = 0,
    blank_lines_after: int = 0,
    ascii_mode: bool | None = None,
) -> _ScrollablePanelRenderable:
    """Return a renderable scrollable panel."""
    return _ScrollablePanelRenderable(
        content=content,
        width=width,
        height=height,
        offset=offset,
        title=title,
        footer=footer,
        border_style=border_style,
        blank_lines_before=blank_lines_before,
        blank_lines_after=blank_lines_after,
        ascii_mode=ascii_mode,
    )


@dataclass(slots=True)
class _CardRenderable:
    """Renderable wrapper for Card.render that captures output as a string."""

    title: str
    content: dict[str, Any] | list[str]
    width: int = 60
    title_color: str = "gold"
    border_style: str = "double"
    padding: int = 1
    blank_lines_before: int = 0
    blank_lines_after: int = 0
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return Card.render(
            self.title,
            self.content,
            width=self.width,
            title_color=self.title_color,
            border_style=self.border_style,
            padding=self.padding,
            blank_lines_before=self.blank_lines_before,
            blank_lines_after=self.blank_lines_after,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def card(
    title: str,
    content: dict[str, Any] | list[str],
    width: int = 60,
    title_color: str = "gold",
    border_style: str = "double",
    padding: int = 1,
    blank_lines_before: int = 0,
    blank_lines_after: int = 0,
    ascii_mode: bool | None = None,
) -> _CardRenderable:
    """Return a renderable card component."""
    return _CardRenderable(
        title=title,
        content=content,
        width=width,
        title_color=title_color,
        border_style=border_style,
        padding=padding,
        blank_lines_before=blank_lines_before,
        blank_lines_after=blank_lines_after,
        ascii_mode=ascii_mode,
    )


@dataclass(slots=True)
class _DialogRenderable:
    """Renderable wrapper for Dialog.render that captures output as a string."""

    title: str
    message: str
    buttons: list[str] | None = None
    selected_button: int = 0
    width: int = 60
    dialog_type: Literal["info", "warning", "error", "success", "question"] = "info"
    blank_lines_before: int = 1
    blank_lines_after: int = 1
    ascii_mode: bool | None = None

    def render(self, console: Console) -> str:
        return console.capture(
            Dialog.render,
            self.title,
            self.message,
            buttons=self.buttons,
            selected_button=self.selected_button,
            width=self.width,
            dialog_type=self.dialog_type,
            blank_lines_before=self.blank_lines_before,
            blank_lines_after=self.blank_lines_after,
            ascii_mode=self.ascii_mode,
        )

    def measure(self, console: Console) -> Measure:
        return Measure.from_text(self.render(console))


def dialog(
    title: str,
    message: str,
    buttons: list[str] | None = None,
    selected_button: int = 0,
    width: int = 60,
    dialog_type: Literal["info", "warning", "error", "success", "question"] = "info",
    blank_lines_before: int = 1,
    blank_lines_after: int = 1,
    ascii_mode: bool | None = None,
) -> _DialogRenderable:
    """Return a renderable dialog component."""
    return _DialogRenderable(
        title=title,
        message=message,
        buttons=buttons,
        selected_button=selected_button,
        width=width,
        dialog_type=dialog_type,
        blank_lines_before=blank_lines_before,
        blank_lines_after=blank_lines_after,
        ascii_mode=ascii_mode,
    )


class Card:
    """Build info cards for items, characters, etc."""

    @staticmethod
    def render(
        title: str,
        content: dict[str, Any] | list[str],
        width: int = 60,
        title_color: str = "gold",
        border_style: str = "double",
        padding: int = 1,
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None
    ) -> str:
        """
        Render an information card.

        Args:
            title: Card title
            content: Dictionary of key-value pairs or list of lines
            width: Card width
            title_color: Title text color
            border_style: Border style
            padding: Internal padding
            blank_lines_before: Blank lines before card
            blank_lines_after: Blank lines after card
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        symbols = Symbols.get_symbols(border_style, ascii_mode)
        tcolor = Color.get_color(title_color)

        # Top border
        top = symbols["top_left"] + symbols["horizontal"] * (width - 2) + symbols["top_right"]
        out.append(top)

        # Padding
        for _ in range(padding):
            out.append(symbols["vertical"] + " " * (width - 2) + symbols["vertical"])

        # Title
        title_text = title.center(width - 4)
        title_line = symbols["vertical"] + " " + tcolor + title_text + Color.RESET + " " + symbols["vertical"]
        out.append(title_line)

        # Separator
        separator = symbols["t_right"] + symbols["horizontal"] * (width - 2) + symbols["t_left"]
        out.append(separator)

        # Content
        if isinstance(content, dict):
            for key, value in content.items():
                key_colored = Color.apply_inline_markup(str(key))
                value_colored = Color.apply_inline_markup(str(value))
                line = f"{key_colored}: {value_colored}"
                visible_len = get_visible_length(line)
                padding_needed = width - 4 - visible_len
                content_line = symbols["vertical"] + " " + line + " " * padding_needed + " " + symbols["vertical"]
                out.append(content_line)
        else:
            for line in content:
                line_colored = Color.apply_inline_markup(str(line))
                visible_len = get_visible_length(line_colored)
                padding_needed = width - 4 - visible_len
                content_line = symbols["vertical"] + " " + line_colored + " " * padding_needed + " " + symbols["vertical"]
                out.append(content_line)

        # Padding
        for _ in range(padding):
            out.append(symbols["vertical"] + " " * (width - 2) + symbols["vertical"])

        # Bottom border
        bottom = symbols["bottom_left"] + symbols["horizontal"] * (width - 2) + symbols["bottom_right"]
        out.append(bottom)

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class Panel(_BorderMixin):
    """Build multi-section panels."""

    @staticmethod
    def render(
        sections: list[dict[str, Any]],
        width: int = 80,
        border_style: str = "box",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None
    ) -> str:
        """
        Render a panel with multiple sections.

        Args:
            sections: List of section dicts with 'title' and 'content'
            width: Panel width
            border_style: Border style
            blank_lines_before: Blank lines before panel
            blank_lines_after: Blank lines after panel
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)

        Example:
            Panel.render([
                {"title": "Player", "content": ["HP: 100", "MP: 50"]},
                {"title": "Enemy", "content": ["HP: 75", "MP: 30"]}
            ])
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        symbols = Symbols.get_symbols(border_style, ascii_mode)

        # Top border
        out.append(_BorderMixin.top_line(symbols, width))

        for i, section in enumerate(sections):
            section_title = section.get("title", "")
            section_content = section.get("content", [])

            # Section title
            if section_title:
                title_colored = Color.apply_inline_markup(section_title)
                visible_len = get_visible_length(title_colored)
                padding_needed = width - 4 - visible_len
                title_line = symbols["vertical"] + " " + Color.BOLD + title_colored + Color.RESET + " " * padding_needed + " " + symbols["vertical"]
                out.append(title_line)

            # Section content
            if isinstance(section_content, list):
                for line in section_content:
                    line_colored = Color.apply_inline_markup(str(line))
                    visible_len = get_visible_length(line_colored)
                    padding_needed = width - 4 - visible_len
                    content_line = symbols["vertical"] + " " + line_colored + " " * padding_needed + " " + symbols["vertical"]
                    out.append(content_line)
            else:
                line_colored = Color.apply_inline_markup(str(section_content))
                visible_len = get_visible_length(line_colored)
                padding_needed = width - 4 - visible_len
                content_line = symbols["vertical"] + " " + line_colored + " " * padding_needed + " " + symbols["vertical"]
                out.append(content_line)

            # Separator between sections
            if i < len(sections) - 1:
                separator = symbols["t_right"] + symbols["horizontal"] * (width - 2) + symbols["t_left"]
                out.append(separator)

        # Bottom border
        out.append(_BorderMixin.bottom_line(symbols, width))

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)


class ScrollablePanel:
    """Scrollable panel with optional interactive navigation."""

    @staticmethod
    def render(
        content: Sequence[str] | str,
        width: int = 80,
        height: int = 10,
        offset: int = 0,
        title: str | None = None,
        footer: bool = True,
        border_style: str = "box",
        blank_lines_before: int = 0,
        blank_lines_after: int = 0,
        ascii_mode: bool | None = None
    ) -> str:
        """
        Render a scrolling window over multi-line content.

        Args:
            content: Iterable of lines or a single string.
            width: Panel width.
            height: Number of content rows to display.
            offset: Starting scroll offset.
            title: Optional title shown in header.
            footer: Show a footer with viewport information.
            border_style: Border style key from Symbols.
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        if isinstance(content, str):
            lines = content.splitlines()
        else:
            lines = [str(line) for line in content]

        total = len(lines)
        if total == 0:
            lines = [""]
            total = 1

        max_offset = max(0, total - height)
        offset = max(0, min(offset, max_offset))

        symbols = Symbols.get_symbols(border_style, ascii_mode)
        inner_width = max(4, width - 2)

        # Header
        top_border = symbols["top_left"] + symbols["horizontal"] * inner_width + symbols["top_right"]
        out.append(top_border)
        if title:
            title_text = Color.apply_inline_markup(title)
            visible_len = get_visible_length(title_text)
            padding = max(0, inner_width - visible_len)
            line = symbols["vertical"] + title_text + " " * padding + symbols["vertical"]
            out.append(line)
            out.append(symbols["t_right"] + symbols["horizontal"] * inner_width + symbols["t_left"])

        # Content window
        visible = lines[offset : offset + height]
        for i in range(height):
            line = visible[i] if i < len(visible) else ""
            colored = Color.apply_inline_markup(line)
            visible_len = get_visible_length(colored)
            padding = max(0, inner_width - visible_len)
            out.append(symbols["vertical"] + colored + " " * padding + symbols["vertical"])

        if footer:
            info = f"{offset + 1}-{min(offset + height, total)} / {total}"
            info_colored = Color.DIM + info + Color.RESET
            padding = max(0, inner_width - get_visible_length(info))
            out.append(symbols["t_right"] + symbols["horizontal"] * inner_width + symbols["t_left"])
            out.append(symbols["vertical"] + info_colored + " " * padding + symbols["vertical"])

        bottom_border = symbols["bottom_left"] + symbols["horizontal"] * inner_width + symbols["bottom_right"]
        out.append(bottom_border)

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)

    @staticmethod
    def interactive(
        content: Sequence[str] | str,
        width: int = 80,
        height: int = 10,
        title: str | None = None,
        border_style: str = "box",
        instructions: bool = True,
        *,
        timeout: float = 0.2,
    ) -> None:
        """
        Interactive scrolling session using arrow keys / PgUp / PgDn / home / end.

        Press q or ESC to exit the session.
        """
        if isinstance(content, str):
            lines = content.splitlines()
        else:
            lines = [str(line) for line in content]

        offset = 0
        total = len(lines)
        max_offset = max(0, total - height)

        def _clamp(val: int) -> int:
            return max(0, min(max_offset, val))

        try:
            while True:
                print("\x1b[2J\x1b[H", end="")
                ScrollablePanel.render(
                    lines,
                    width=width,
                    height=height,
                    offset=offset,
                    title=title,
                    border_style=border_style,
                )
                if instructions:
                    print("Use ↑/↓, PgUp/PgDn, Home/End to scroll. q to quit.")
                key = read_key(timeout=timeout)
                if not key:
                    continue
                if key in {"q", "escape"}:
                    break
                if key in {"up", "k"}:
                    offset = _clamp(offset - 1)
                elif key in {"down", "j"}:
                    offset = _clamp(offset + 1)
                elif key == "page_up":
                    offset = _clamp(offset - height)
                elif key == "page_down":
                    offset = _clamp(offset + height)
                elif key == "home":
                    offset = 0
                elif key == "end":
                    offset = max_offset
        finally:
            print("\x1b[2J\x1b[H", end="")


class Dialog(_BorderMixin):
    """Build dialog boxes and modals."""

    @staticmethod
    def render(
        title: str,
        message: str,
        buttons: list[str] | None = None,
        selected_button: int = 0,
        width: int = 60,
        dialog_type: Literal["info", "warning", "error", "success", "question"] = "info",
        blank_lines_before: int = 1,
        blank_lines_after: int = 1,
        ascii_mode: bool | None = None
    ) -> str:
        """
        Render a dialog box.

        Args:
            title: Dialog title
            message: Dialog message
            buttons: List of button labels
            selected_button: Index of selected button
            width: Dialog width
            dialog_type: Type of dialog (affects icon/color)
            blank_lines_before: Blank lines before dialog
            blank_lines_after: Blank lines after dialog
            ascii_mode: Force ASCII symbols (True), Unicode (False), or auto-detect (None)
        """
        if buttons is None:
            buttons = ["OK"]

        out: list[str] = []
        for _ in range(blank_lines_before):
            out.append("")

        # Type-specific settings
        type_settings = {
            "info": {"icon": "ℹ", "color": "cyan"},
            "warning": {"icon": "⚠", "color": "yellow"},
            "error": {"icon": "✗", "color": "red"},
            "success": {"icon": "✓", "color": "green"},
            "question": {"icon": "?", "color": "blue"}
        }

        settings = type_settings.get(dialog_type, type_settings["info"])
        icon = settings["icon"]
        title_color = Color.get_color(settings["color"])

        symbols = Symbols.get_symbols("double", ascii_mode)

        # Top border
        out.append(_BorderMixin.top_line(symbols, width))

        # Title with icon
        title_text = f"{icon} {title}"
        title_centered = title_text.center(width - 4)
        title_line = symbols["vertical"] + " " + title_color + title_centered + Color.RESET + " " + symbols["vertical"]
        out.append(title_line)

        # Separator
        separator = symbols["t_right"] + symbols["horizontal"] * (width - 2) + symbols["t_left"]
        out.append(separator)

        # Message
        message_lines = wrap_text(message, width - 6)
        for line in message_lines:
            colored = Color.apply_inline_markup(line)
            visible_len = get_visible_length(colored)
            padding_needed = max(0, width - 4 - visible_len)
            message_line = symbols["vertical"] + " " + colored + " " * padding_needed + " " + symbols["vertical"]
            out.append(message_line)

        # Spacing before buttons
        out.append(symbols["vertical"] + " " * (width - 2) + symbols["vertical"])

        # Buttons
        button_str = "  ".join(
            f"{Color.BOLD}[{btn}]{Color.RESET}" if i == selected_button else f"[{btn}]"
            for i, btn in enumerate(buttons)
        )
        button_visible = sum(len(btn) + 2 for btn in buttons) + (len(buttons) - 1) * 2
        button_padding = (width - 4 - button_visible) // 2
        button_line = symbols["vertical"] + " " + " " * button_padding + button_str + " " * (width - 4 - button_visible - button_padding) + " " + symbols["vertical"]
        out.append(button_line)

        # Bottom border
        out.append(_BorderMixin.bottom_line(symbols, width))

        for _ in range(blank_lines_after):
            out.append("")
        return "\n".join(out)
