"""Textforge command-line entry point (modular)."""

from __future__ import annotations

import argparse
import sys
from typing import TYPE_CHECKING

from .errors import CLIError, ExitCode

# Ensure valid Windows stdin as early as possible when this module is imported
try:
    from ..utils.io import ensure_windows_valid_stdin
    from ..utils.logging import get_logger

    ensure_windows_valid_stdin()
except Exception as e:
    get_logger().debug(f"Failed to ensure valid Windows stdin during import: {e}")

if TYPE_CHECKING:
    from collections.abc import Sequence


class _Parser(argparse.ArgumentParser):
    def error(self, message: str) -> None:
        self.print_usage(sys.stderr)
        print(f"error: {message}", file=sys.stderr)
        raise SystemExit(ExitCode.USAGE)


def build_parser() -> argparse.ArgumentParser:
    parser = _Parser(prog="textforge", description="Textforge CLI utilities")
    parser.add_argument("--debug", action="store_true", help=argparse.SUPPRESS)
    subparsers = parser.add_subparsers(dest="command", required=True)

    # Register commands from submodules
    from .commands import (
        bench_cmd,
        demo,
        dsl_cmd,
        export_cmd,
        list_cmd,
        live_cmd,
        markup_cmd,
        new,
        plugins_cmd,
        preview,
        theme,
        typewriter_cmd,
    )

    sp = subparsers
    list_cmd.register(sp)
    demo.register(sp)
    typewriter_cmd.register(sp)
    preview.register(sp)
    export_cmd.register(sp)
    live_cmd.register(sp)
    theme.register(sp)
    markup_cmd.register(sp)
    new.register(sp)
    plugins_cmd.register(sp)
    dsl_cmd.register(sp)
    bench_cmd.register(sp)

    return parser


def main(argv: Sequence[str] | None = None) -> int:
    # On Windows under captured subprocesses, stdin handle duplication can fail.
    # Attempt to ensure stdin is backed by a valid handle to prevent WinError 6.
    try:
        from ..utils.io import ensure_windows_valid_stdin
        from ..utils.logging import get_logger
        ensure_windows_valid_stdin()
    except Exception as e:
        get_logger().debug(f"Failed to ensure valid Windows stdin in main: {e}")
    parser = build_parser()
    try:
        args = parser.parse_args(argv)
        try:
            args.func(args)
        except CLIError as e:
            print(str(e), file=sys.stderr)
            return e.code
        except OSError as e:
            print(f"I/O error: {e}", file=sys.stderr)
            return ExitCode.IO_ERROR
        except KeyboardInterrupt:
            print("Interrupted.", file=sys.stderr)
            return ExitCode.RUNTIME_ERROR
        except Exception as e:
            if getattr(args, "debug", False):
                raise
            print(f"Error: {e}", file=sys.stderr)
            return ExitCode.RUNTIME_ERROR
        return ExitCode.OK
    except SystemExit as se:
        # Normalize Windows subprocess handle errors into non-zero exit rather than raising
        try:
            code = int(se.code) if isinstance(se.code, int) else ExitCode.RUNTIME_ERROR
        except Exception:
            code = ExitCode.RUNTIME_ERROR
        return code


if __name__ == "__main__":
    raise SystemExit(main())
