from __future__ import annotations

import os
from typing import TYPE_CHECKING

from ..errors import CLIError

if TYPE_CHECKING:
    import argparse


def _scaffold_main_py() -> str:
    # Use helpers and tfprint rather than immediate printing in component renderers
    return (
        "from textforge import tfprint, components as C\n\n"
        "def main():\n"
        "    banner = C.title(text='Hello Textforge', width=60, color='primary')\n"
        "    para = C.paragraph(text='Project scaffold created.', width=60)\n"
        "    tfprint(banner)\n"
        "    tfprint(para)\n\n"
        "if __name__ == '__main__':\n"
        "    main()\n"
    )


def _run(args: argparse.Namespace) -> None:
    target = os.path.join(args.dir, args.name)
    try:
        os.makedirs(target, exist_ok=True)
    except OSError as e:
        raise CLIError(f"Failed to create directory '{target}': {e}") from e
    main_py = os.path.join(target, "main.py")
    if not os.path.exists(main_py):
        try:
            with open(main_py, "w", encoding="utf-8") as f:
                f.write(_scaffold_main_py())
        except OSError as e:
            raise CLIError(f"Failed to write scaffold file: {e}") from e
    print(f"Scaffold created at {target}")


def register(subparsers: argparse._SubParsersAction) -> None:
    parser = subparsers.add_parser("new", help="Scaffold a new Textforge demo project")
    parser.add_argument("name", help="Project name")
    parser.add_argument("--dir", default=".", help="Target directory")
    parser.set_defaults(func=_run)
