from __future__ import annotations

import sys
from typing import TYPE_CHECKING

from ...export import ansi_to_html, ansi_to_pdf, ansi_to_svg
from ...utils.io import restore_stdin
from ..errors import CLIError

if TYPE_CHECKING:
    import argparse


def _run(args: argparse.Namespace) -> None:
    """Execute the export command with parsed arguments."""
    fmt = args.format
    if fmt == "pdf":
        # PDF is binary; require explicit --output to avoid misdirected text-mode stdout
        if not args.output:
            raise CLIError("PDF export requires --output to a file.")

    # Read from stdin after validation
    # If stdin was reassigned by ensure_windows_valid_stdin, restore the original
    restore_stdin()
    try:
        data = sys.stdin.read()
    except OSError as e:
        raise CLIError(f"Failed to read from stdin: {e}") from e

    if fmt == "pdf":
        pdf_bytes = ansi_to_pdf(data)
        try:
            with open(args.output, "wb") as f:
                f.write(pdf_bytes)
        except OSError as e:
            raise CLIError(f"Failed to write PDF: {e}") from e
        return

    # Text formats
    if fmt == "html":
        out = ansi_to_html(data)
    elif fmt == "svg":
        out = ansi_to_svg(data)
    elif fmt == "text":
        from ...export import ansi_to_text
        out = ansi_to_text(data)
    else:
        raise CLIError(f"Unknown format: {fmt}")

    if args.output:
        try:
            with open(args.output, "w", encoding="utf-8") as f:
                f.write(out)
        except OSError as e:
            raise CLIError(f"Failed to write output: {e}") from e
    else:
        sys.stdout.write(out)


def register(subparsers: argparse._SubParsersAction) -> None:
    """Register the export command with the CLI argument parser."""
    parser = subparsers.add_parser("export", help="Export ANSI text from stdin to HTML/SVG/TEXT/PDF")
    parser.add_argument("--output", "-o", help="Output file (default: stdout)")
    parser.add_argument("--format", "-f", choices=["html", "svg", "text", "pdf"], default="html", help="Export format")
    parser.set_defaults(func=_run)
