from __future__ import annotations

import sys
from typing import TYPE_CHECKING

from ...cli.errors import CLIError
from ...dsl import run as dsl_run
from ...utils.io import restore_stdin

if TYPE_CHECKING:
    import argparse


def _run(args: argparse.Namespace) -> None:
    """Execute the DSL command with parsed arguments."""
    if args.file:
        try:
            with open(args.file, encoding="utf-8") as fh:
                src = fh.read()
        except OSError as e:
            raise CLIError(f"Failed to read file '{args.file}': {e}") from e
    else:
        # If stdin was reassigned by ensure_windows_valid_stdin, restore the original
        restore_stdin()
        try:
            src = sys.stdin.read()
        except OSError as e:
            raise CLIError(f"Failed to read from stdin: {e}") from e
    dsl_run(src)


def register(subparsers: argparse._SubParsersAction) -> None:
    """Register the DSL command with the CLI argument parser."""
    parser = subparsers.add_parser("dsl", help="Run the tiny Textforge DSL")
    parser.add_argument("--file", "-f", help="Source file; omit to read from stdin")
    parser.set_defaults(func=_run)
