from __future__ import annotations

from typing import TYPE_CHECKING

from ...core import tfprint
from ...utils.bench import (
    measure_first_render_ms,
    measure_import_time_ms,
    measure_live_update_ms,
    run_micro_suite,
    run_suite,
)

if TYPE_CHECKING:
    import argparse


def _run(args: argparse.Namespace) -> None:
    if getattr(args, "metrics", False):
        tfprint("Metrics:")
        imp = measure_import_time_ms(args.python)
        fr = measure_first_render_ms()
        live = measure_live_update_ms()
        tfprint(f"  - import-time: {imp:.2f} ms")
        tfprint(f"  - first-render: {fr:.2f} ms")
        tfprint(f"  - live-update-steady: {live:.2f} ms/frame")
        # Note: fall through to also run the selected bench suite unless --no-bench
        if getattr(args, "no_bench", False):
            return

    suite = run_micro_suite if getattr(args, "micro", False) else run_suite
    results = suite()
    tfprint("Benchmark results:")
    for r in results:
        per = r.ms / max(1, r.iters)
        tfprint(f"  - {r.name}: {r.ms:.1f} ms total ({per:.2f} ms/iter)")


def register(subparsers: argparse._SubParsersAction) -> None:
    parser = subparsers.add_parser("bench", help="Run benchmark suites and metrics")
    parser.add_argument("--micro", action="store_true", help="Run micro-benchmarks (markup/layout/diff)")
    parser.add_argument("--metrics", action="store_true", help="Print import/first-render/live metrics")
    parser.add_argument("--python", metavar="EXE", help="Python executable for import timing", nargs="?")
    parser.add_argument("--no-bench", action="store_true", help="Only print metrics without running benches")
    parser.set_defaults(func=_run)
