"""Simplified bidi reordering utilities.

This is a pragmatic, lightweight approach to handling mixed LTR/RTL runs
based on Unicode bidirectional categories. It is not a full UAX#9 implementation
but suffices for typical console rendering use cases.
"""

from __future__ import annotations

import unicodedata
from typing import TYPE_CHECKING

from .graphemes import iter_graphemes

if TYPE_CHECKING:
    BIDI_SENTINEL: int

__all__ = ["bidi_reorder"]


def _cluster_direction(grapheme: str) -> str:
    for ch in grapheme:
        direction = unicodedata.bidirectional(ch)
        if direction in {"R", "AL", "AN"}:
            return "R"
        if direction == "L":
            return "L"
    return "N"


def bidi_reorder(text: str) -> str:
    """Perform a simple bidi reordering for mixed LTR/RTL text."""
    clusters = list(iter_graphemes(text))
    if not clusters:
        return text

    runs: list[tuple[str, list[str]]] = []
    current_dir: str | None = None
    current_run: list[str] = []

    for cluster in clusters:
        direction = _cluster_direction(cluster)
        if direction == "N":
            if current_dir is None:
                current_dir = "L"
            current_run.append(cluster)
            continue

        if current_dir is None:
            current_dir = direction
            current_run.append(cluster)
            continue

        if direction == current_dir:
            current_run.append(cluster)
            continue

        runs.append((current_dir, current_run))
        current_run = [cluster]
        current_dir = direction

    if current_run:
        runs.append((current_dir or "L", current_run))

    reordered: list[str] = []
    for direction, run in runs:
        if direction == "R":
            reordered.extend(reversed(run))
        else:
            reordered.extend(run)
    return "".join(reordered)
