"""Minimal animation timeline with keyframes."""

from __future__ import annotations

import time
from collections import deque
from dataclasses import dataclass
from typing import TYPE_CHECKING

from .core.render_tree import diff_changes

if TYPE_CHECKING:
    from collections.abc import Callable


@dataclass(slots=True)
class Keyframe:
    t: float
    action: Callable[[], None]


class Timeline:
    def __init__(self) -> None:
        self._frames: list[Keyframe] = []

    def add(self, t: float, action: Callable[[], None]) -> None:
        self._frames.append(Keyframe(t, action))
        self._frames.sort(key=lambda k: k.t)

    def play(self) -> None:
        start = time.perf_counter()
        idx = 0
        while idx < len(self._frames):
            now = time.perf_counter() - start
            next_t = self._frames[idx].t
            if now >= next_t:
                try:
                    self._frames[idx].action()
                except Exception:
                    pass
                idx += 1
            else:
                time.sleep(min(0.01, next_t - now))


class FrameCache:
    """Cache frames and provide diff information for live rendering."""

    def __init__(self, max_history: int = 120) -> None:
        self.max_history = max_history
        self._history: deque[list[str]] = deque(maxlen=max_history)
        self._current: list[str] | None = None
        self._previous: list[str] | None = None

    def update(self, frame: str) -> bool:
        """Store a frame and return True if it differs from the previous one."""
        lines = frame.splitlines()
        if self._current == lines:
            return False
        self._previous = self._current
        self._current = lines
        self._history.append(lines)
        return True

    def diff(self) -> list[tuple[int, str]]:
        """Return line-level changes relative to the prior cached frame."""
        if self._current is None:
            return []
        if self._previous is None:
            return [(idx, line) for idx, line in enumerate(self._current)]
        changes = diff_changes(self._previous, self._current)
        return [(change.index, change.text) for change in changes]

    def diff_structured(self):
        """Return structured line changes using the VDOM API.

        Consumers can use this to apply partial updates in GUI backends.
        """
        if self._current is None:
            return []
        prev = self._previous or []
        return diff_changes(prev, self._current)

    def last(self) -> str | None:
        if self._current is None:
            return None
        return "\n".join(self._current)

    def history(self) -> list[str]:
        return ["\n".join(lines) for lines in self._history]

    def clear(self) -> None:
        self._history.clear()
        self._current = None
        self._previous = None
