import sys
from unittest.mock import Mock, patch

import pytest

from textforge.utils.io import ensure_windows_valid_stdin, restore_stdin


def test_restore_stdin_no_original_stdin():
    """Test restore_stdin when no original stdin was stored."""
    # Ensure _original_stdin is None initially
    from textforge.utils.io import _original_stdin
    original_value = _original_stdin
    try:
        # Reset to None
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        # Should not raise any errors
        restore_stdin()
        assert sys.stdin is sys.stdin  # Should remain unchanged
    finally:
        # Restore original value
        textforge.utils.io._original_stdin = original_value


def test_restore_stdin_with_original_stdin():
    """Test restore_stdin when original stdin was stored."""
    # Mock original stdin
    original_stdin = Mock()
    new_stdin = Mock()

    from textforge.utils.io import _original_stdin
    original_value = _original_stdin
    try:
        # Set up the mock scenario
        import textforge.utils.io
        textforge.utils.io._original_stdin = original_stdin
        sys.stdin = new_stdin

        # Restore should set sys.stdin back to original
        restore_stdin()
        assert sys.stdin is original_stdin
        assert textforge.utils.io._original_stdin is None
    finally:
        # Restore original values
        sys.stdin = sys.__stdin__
        textforge.utils.io._original_stdin = original_value


@pytest.mark.skipif(sys.platform != "win32", reason="Windows-specific stdin handling")
def test_ensure_windows_valid_stdin_no_reassignment_needed():
    """Test ensure_windows_valid_stdin when stdin already has a valid handle."""
    original_stdin = sys.stdin
    from textforge.utils.io import _original_stdin
    original_stored = _original_stdin

    try:
        # Reset stored original
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        # Mock msvcrt.get_osfhandle to succeed (no exception)
        with patch("msvcrt.get_osfhandle", return_value=123):
            ensure_windows_valid_stdin()

        # Should not change sys.stdin
        assert sys.stdin is original_stdin
        # Should not store original stdin
        assert textforge.utils.io._original_stdin is None
    finally:
        sys.stdin = original_stdin
        textforge.utils.io._original_stdin = original_stored


@pytest.mark.skipif(sys.platform != "win32", reason="Windows-specific stdin handling")
def test_ensure_windows_valid_stdin_conin_fallback():
    """Test ensure_windows_valid_stdin reassigns stdin to CONIN$ when handle invalid."""
    original_stdin = sys.stdin
    from textforge.utils.io import _original_stdin
    original_stored = _original_stdin

    try:
        # Reset stored original
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        # Mock msvcrt.get_osfhandle to raise exception (invalid handle)
        # Mock open to succeed for CONIN$
        mock_conin = Mock()
        mock_conin.encoding = "utf-8"

        with patch("msvcrt.get_osfhandle", side_effect=OSError("Invalid handle")), \
             patch("builtins.open", return_value=mock_conin) as mock_open:
            ensure_windows_valid_stdin()

        # Should store original stdin
        assert textforge.utils.io._original_stdin is original_stdin
        # Should reassign sys.stdin to CONIN$
        assert sys.stdin is mock_conin
        # Should have tried to open CONIN$
        mock_open.assert_called_with("CONIN$", encoding=getattr(original_stdin, "encoding", "utf-8"), errors="ignore")
    finally:
        sys.stdin = original_stdin
        textforge.utils.io._original_stdin = original_stored


@pytest.mark.skipif(sys.platform != "win32", reason="Windows-specific stdin handling")
def test_ensure_windows_valid_stdin_nul_fallback():
    """Test ensure_windows_valid_stdin falls back to NUL when CONIN$ fails."""
    original_stdin = sys.stdin
    from textforge.utils.io import _original_stdin
    original_stored = _original_stdin

    try:
        # Reset stored original
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        # Mock failures: get_osfhandle fails, CONIN$ open fails, NUL open succeeds
        mock_nul = Mock()

        with patch("msvcrt.get_osfhandle", side_effect=OSError("Invalid handle")), \
             patch("builtins.open", side_effect=[OSError("CONIN$ failed"), mock_nul]) as mock_open:
            ensure_windows_valid_stdin()

        # Should store original stdin
        assert textforge.utils.io._original_stdin is original_stdin
        # Should reassign sys.stdin to NUL
        assert sys.stdin is mock_nul
        # Should have tried CONIN$ then NUL
        assert mock_open.call_count == 2
        mock_open.assert_any_call("CONIN$", encoding=getattr(original_stdin, "encoding", "utf-8"), errors="ignore")
        mock_open.assert_any_call("NUL")
    finally:
        sys.stdin = original_stdin
        textforge.utils.io._original_stdin = original_stored


@pytest.mark.skipif(sys.platform != "win32", reason="Windows-specific stdin handling")
def test_ensure_windows_valid_stdin_no_fallback_available():
    """Test ensure_windows_valid_stdin when no fallback handles are available."""
    original_stdin = sys.stdin
    from textforge.utils.io import _original_stdin
    original_stored = _original_stdin

    try:
        # Reset stored original
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        # Mock all failures: get_osfhandle fails, both open calls fail
        with patch("msvcrt.get_osfhandle", side_effect=OSError("Invalid handle")), \
             patch("builtins.open", side_effect=OSError("All handles failed")):
            ensure_windows_valid_stdin()

        # Should store original stdin
        assert textforge.utils.io._original_stdin is original_stdin
        # Should not change sys.stdin (no valid fallback available)
        assert sys.stdin is original_stdin
    finally:
        sys.stdin = original_stdin
        textforge.utils.io._original_stdin = original_stored


@pytest.mark.skipif(sys.platform == "win32", reason="Non-Windows platforms should not modify stdin")
def test_ensure_windows_valid_stdin_non_windows_noop():
    """Test ensure_windows_valid_stdin does nothing on non-Windows platforms."""
    original_stdin = sys.stdin
    from textforge.utils.io import _original_stdin
    original_stored = _original_stdin

    try:
        # Reset stored original
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        ensure_windows_valid_stdin()

        # Should not change anything
        assert sys.stdin is original_stdin
        assert textforge.utils.io._original_stdin is None
    finally:
        sys.stdin = original_stdin
        textforge.utils.io._original_stdin = original_stored


def test_stdin_roundtrip_integration():
    """Test the complete round-trip: ensure -> restore -> ensure again."""
    original_stdin = sys.stdin
    from textforge.utils.io import _original_stdin
    original_stored = _original_stdin

    try:
        # Reset stored original
        import textforge.utils.io
        textforge.utils.io._original_stdin = None

        # First ensure (should store original if reassigned)
        ensure_windows_valid_stdin()
        stored_after_first = textforge.utils.io._original_stdin

        # Restore
        restore_stdin()
        assert sys.stdin is (stored_after_first if stored_after_first else original_stdin)
        assert textforge.utils.io._original_stdin is None

        # Second ensure should work again
        ensure_windows_valid_stdin()
        stored_after_second = textforge.utils.io._original_stdin

        # Should store the same original stdin again
        if stored_after_first:
            assert stored_after_second is stored_after_first
        else:
            assert stored_after_second is None or stored_after_second is original_stdin

        # Final restore
        restore_stdin()
        assert sys.stdin is (stored_after_second if stored_after_second else original_stdin)
        assert textforge.utils.io._original_stdin is None

    finally:
        sys.stdin = original_stdin
        textforge.utils.io._original_stdin = original_stored
