from __future__ import annotations

import os
import sys

import pytest

from textforge.utils.input import read_key


@pytest.mark.skipif(os.name != "posix", reason="pty-based tests require POSIX")
def test_read_key_posix_arrows_pty(monkeypatch):
    import os as _os
    import pty

    master_fd, slave_fd = pty.openpty()
    try:
        # Replace stdin with the slave end of the pty
        with _os.fdopen(slave_fd, "r", buffering=1) as slave_file:
            orig_stdin = sys.stdin
            sys.stdin = slave_file
            try:
                _os.write(master_fd, b"\x1b[A")  # ESC [ A => up
                token = read_key(timeout=0.2)
                assert token == "up"
            finally:
                sys.stdin = orig_stdin
    finally:
        try:
            _os.close(master_fd)
        except OSError:
            pass


@pytest.mark.skipif(os.name != "posix", reason="pty-based tests require POSIX")
def test_read_key_posix_page_down_pty(monkeypatch):
    import os as _os
    import pty

    master_fd, slave_fd = pty.openpty()
    try:
        with _os.fdopen(slave_fd, "r", buffering=1) as slave_file:
            orig_stdin = sys.stdin
            sys.stdin = slave_file
            try:
                _os.write(master_fd, b"\x1b[6~")  # ESC [ 6 ~ => page_down
                token = read_key(timeout=0.2)
                assert token == "page_down"
            finally:
                sys.stdin = orig_stdin
    finally:
        try:
            _os.close(master_fd)
        except OSError:
            pass


@pytest.mark.skipif(os.name != "posix", reason="pty-based tests require POSIX")
def test_read_key_timeout_returns_empty(monkeypatch):
    import os as _os
    import pty

    master_fd, slave_fd = pty.openpty()
    try:
        with _os.fdopen(slave_fd, "r", buffering=1) as slave_file:
            orig_stdin = sys.stdin
            sys.stdin = slave_file
            try:
                token = read_key(timeout=0.05)
                assert token == ""
            finally:
                sys.stdin = orig_stdin
    finally:
        try:
            _os.close(master_fd)
        except OSError:
            pass
