from __future__ import annotations

import io

from textforge.core.console import Console


def _reset_runtime() -> None:
    from textforge.core.rendering_gui.runtime import get_runtime

    rt = get_runtime(create=False)
    if rt is not None:
        rt.shutdown()


def test_gui_stream_updates_surface_in_headless_mode(monkeypatch):
    monkeypatch.setenv("TEXTFORGE_HEADLESS", "1")
    _reset_runtime()
    # Ensure we start without an active runtime.
    from textforge.core.rendering_gui.runtime import get_runtime

    assert get_runtime(create=False) is None

    from textforge.core.backends import Backends

    stream = Backends.create_stream("gui")
    try:
        stream.write("[bold]Hello[/]")
        stream.flush()
        runtime = get_runtime(create=False)
        assert runtime is not None
        assert runtime.surface.get_frame() == "[bold]Hello[/]"

        stream.present_frame("[blue]Frame[/]")
        assert runtime.surface.get_frame() == "[blue]Frame[/]"
    finally:
        stream.close()

    # Closing the stream should tear down the runtime.
    assert get_runtime(create=False) is None


def test_auto_backend_uses_terminal_when_configured(monkeypatch, tmp_path):
    _reset_runtime()
    config = tmp_path / "tf.config"
    config.write_text("renderer=terminal\n", encoding="utf-8")
    monkeypatch.chdir(tmp_path)

    console = Console.for_backend("auto")
    try:
        assert console.stream.__class__.__name__ == "TerminalStream"
    finally:
        # TerminalStream does not own resources, but close defensively if provided.
        close = getattr(console.stream, "close", None)
        if callable(close):
            close()


def test_auto_backend_uses_gui_when_configured(monkeypatch, tmp_path):
    monkeypatch.setenv("TEXTFORGE_HEADLESS", "1")
    _reset_runtime()
    config = tmp_path / "tf.config"
    config.write_text("renderer=gui\n", encoding="utf-8")
    monkeypatch.chdir(tmp_path)

    console = Console.for_backend("auto")
    try:
        from textforge.core.rendering_gui.runtime import get_runtime

        assert console.stream.__class__.__name__ == "GuiStream"
        with console.live(renderer="gui") as live:
            live.update("sample")
            runtime = get_runtime(create=False)
            assert runtime is not None
            assert runtime.surface.get_frame() == "sample"
    finally:
        close = getattr(console.stream, "close", None)
        if callable(close):
            close()


def test_gui_runtime_uses_custom_window(monkeypatch):
    _reset_runtime()
    monkeypatch.delenv("TEXTFORGE_HEADLESS", raising=False)
    monkeypatch.delenv("TEXTFORGE_GUI_STDOUT_FALLBACK", raising=False)

    from textforge.core.rendering_gui import runtime as gui_runtime

    updates: list[str] = []
    key_tokens: list[str] = []

    class DummyWindow:
        def __init__(self, title, width, height, on_key):
            self._on_key = on_key
            updates.append(f"init:{title}:{width}x{height}")

        def wait_until_ready(self, timeout=None):
            return True

        def update_text(self, text: str) -> None:
            updates.append(text)
            self._on_key("escape")

        def stop(self) -> None:
            updates.append("stop")

    from textforge.core.rendering_gui import win32_window

    monkeypatch.setattr(win32_window, "Win32Window", DummyWindow)
    monkeypatch.setitem(gui_runtime.__dict__, "Win32Window", DummyWindow)

    console = Console.for_backend("gui")
    try:
        console.print("demo", markup=False, end="")
        runtime = gui_runtime.get_runtime(create=False)
        assert runtime is not None
        q = runtime.get_key_queue()
        assert q is not None
        key_tokens.append(q.get(timeout=0.1))
        assert updates[-1] == "demo"
        assert key_tokens == ["escape"]
    finally:
        console.stream.close()

    assert gui_runtime.get_runtime(create=False) is None


def test_gui_runtime_falls_back_to_stdout(monkeypatch):
    _reset_runtime()
    monkeypatch.delenv("TEXTFORGE_HEADLESS", raising=False)
    monkeypatch.setenv("TEXTFORGE_GUI_STDOUT_FALLBACK", "1")

    from textforge.core.rendering_gui import runtime as gui_runtime

    class FailingWindow:
        def __init__(self, *args, **kwargs):
            pass

        def wait_until_ready(self, timeout=None):
            return False

        def stop(self) -> None:
            pass

    from textforge.core.rendering_gui import win32_window

    monkeypatch.setattr(win32_window, "Win32Window", FailingWindow)
    monkeypatch.setitem(gui_runtime.__dict__, "Win32Window", FailingWindow)
    fake_stdout = io.StringIO()
    monkeypatch.setattr(gui_runtime.sys, "stdout", fake_stdout)

    console = Console.for_backend("gui")
    try:
        console.print("fallback", markup=False)
        assert "fallback" in fake_stdout.getvalue()
    finally:
        console.stream.close()

    assert gui_runtime.get_runtime(create=False) is None


def test_terminal_stream_close_calls_stop():
    from textforge.core.rendering_cli.window import TerminalStream, WindowLike

    class DummyWindow(WindowLike):
        def __init__(self) -> None:
            self.stopped = False

        def write(self, text: str) -> int:
            return len(text)

        def flush(self) -> None:
            pass

        def stop(self) -> None:
            self.stopped = True

    window = DummyWindow()
    stream = TerminalStream(window)
    stream.write("hi")
    assert window.stopped is False
    stream.close()
    assert window.stopped is True


def test_win32_helpers_color_and_keys():
    from textforge.core.rendering_gui.win32_window import _parse_css_color, _translate_virtual_key

    assert _parse_css_color("#ff00aa") == (255, 0, 170)
    assert _parse_css_color("rgb(10,20,30)") == (10, 20, 30)
    assert _parse_css_color("invalid") == (0, 0, 0)
    assert _translate_virtual_key(0x41) == "a"
    assert _translate_virtual_key(0x0D) == "enter"
    assert _translate_virtual_key(0x9999) == ""
