import random
import string

import pytest

from textforge.export import ansi_to_html, ansi_to_svg, ansi_to_text
from textforge.markup import MarkupEngine


def random_text(n: int = 100) -> str:
    chars = string.ascii_letters + string.digits + " []#,:=_-"
    s = []
    for _ in range(n):
        if random.random() < 0.05:
            # insert a random tag-like chunk
            token = random.choice(["red", "bold", "bg=#112233", "fg=primary", "reset"])  # subset
            s.append("[" + token + "]")
        else:
            s.append(random.choice(chars))
    return "".join(s)


def test_markup_fuzz_valid_rendering_and_idempotent():
    me = MarkupEngine()
    for _ in range(50):
        txt = random_text(200)
        out = me.render(txt)
        # No tokenization sentinels should leak
        assert "__LBR__" not in out and "__RBR__" not in out
        # Rendering should be idempotent with respect to color markup
        out2 = me.render(out)
        assert out2 == out
        # Exporters should handle the ANSI output without errors
        html = ansi_to_html(out)
        assert "<pre" in html
        plain = ansi_to_text(out)
        assert isinstance(plain, str)


def test_exporter_fuzz_malformed_sgr_sequences():
    """Test exporters handle malformed SGR sequences without crashing."""
    malformed_sequences = [
        # Truncated CSI sequences
        "\x1b[",
        "\x1b[3",
        "\x1b[38",
        "\x1b[38;",
        "\x1b[38;2",
        "\x1b[38;2;",
        "\x1b[38;2;1",
        "\x1b[38;2;1;",
        "\x1b[38;2;1;2",
        "\x1b[38;2;1;2;",
        # Incomplete 24-bit sequences
        "\x1b[38;2;255m",  # Missing g,b
        "\x1b[38;2;255;0m",  # Missing b
        "\x1b[48;2;255m",  # Background missing g,b
        "\x1b[48;2;255;0m",  # Background missing b
        # Invalid parameters
        "\x1b[999m",
        "\x1b[38;999;0;0m",
        "\x1b[48;2;999;0;0m",
        # Nested brackets
        "\x1b[\x1b[31m",
        "\x1b[31\x1b[m",
        # Extreme values
        "\x1b[38;2;999999;999999;999999m",
        "\x1b[48;2;-1;-1;-1m",
        # Empty parameters
        "\x1b[;m",
        "\x1b[;;m",
        "\x1b[38;;2m",
        # Non-numeric parameters
        "\x1b[38;abc;0;0m",
        "\x1b[48;2;255;xyz;0m",
    ]

    for seq in malformed_sequences:
        test_text = f"Hello {seq} World"
        # All exporters should handle malformed input gracefully
        try:
            html = ansi_to_html(test_text)
            assert isinstance(html, str)
            assert "<pre" in html
        except Exception as e:
            pytest.fail(f"HTML export failed on malformed sequence {seq!r}: {e}")

        try:
            svg = ansi_to_svg(test_text)
            assert isinstance(svg, str)
            assert "<svg" in svg
        except Exception as e:
            pytest.fail(f"SVG export failed on malformed sequence {seq!r}: {e}")

        try:
            plain = ansi_to_text(test_text)
            assert isinstance(plain, str)
        except Exception as e:
            pytest.fail(f"Text export failed on malformed sequence {seq!r}: {e}")


def test_exporter_fuzz_extreme_24bit_sequences():
    """Test exporters handle extreme 24-bit color values."""
    extreme_sequences = [
        # Maximum values
        "\x1b[38;2;255;255;255m",
        "\x1b[48;2;255;255;255m",
        # Zero values
        "\x1b[38;2;0;0;0m",
        "\x1b[48;2;0;0;0m",
        # Very large values (should be clamped)
        "\x1b[38;2;1000;1000;1000m",
        "\x1b[48;2;1000;1000;1000m",
        # Negative values (should be clamped)
        "\x1b[38;2;-1;-1;-1m",
        "\x1b[48;2;-1;-1;-1m",
        # Mixed extreme values
        "\x1b[38;2;0;255;0m",
        "\x1b[48;2;255;0;255m",
    ]

    for seq in extreme_sequences:
        test_text = f"Test {seq} text"
        # All exporters should handle extreme values gracefully
        html = ansi_to_html(test_text)
        assert isinstance(html, str)

        svg = ansi_to_svg(test_text)
        assert isinstance(svg, str)

        plain = ansi_to_text(test_text)
        assert isinstance(plain, str)


def test_exporter_fuzz_nested_sequences():
    """Test exporters handle deeply nested ANSI sequences."""
    # Create deeply nested sequences
    nested = "Normal"
    for i in range(10):
        nested = f"\x1b[3{i % 8}m{nested}\x1b[0m"

    # Add some 24-bit colors
    nested += "\x1b[38;2;255;0;0m\x1b[48;2;0;0;255mNested\x1b[49m\x1b[39m"

    # All exporters should handle nested sequences
    html = ansi_to_html(nested)
    assert isinstance(html, str)
    assert "<pre" in html

    svg = ansi_to_svg(nested)
    assert isinstance(svg, str)
    assert "<svg" in svg

    plain = ansi_to_text(nested)
    assert isinstance(plain, str)
