﻿import io
from contextlib import redirect_stdout

from textforge.components.feedback import ToastManager
from textforge.components.status import Compass, Thermometer, WeatherIndicator


def _capture(callable_obj, *args, **kwargs):
    buf = io.StringIO()
    result = None
    with redirect_stdout(buf):
        try:
            result = callable_obj(*args, **kwargs)
        except SystemExit:
            pass
    output = buf.getvalue()
    if isinstance(result, str):
        output += result
    return output


def _assert_ascii_only(text: str) -> None:
    assert all(ord(ch) < 128 for ch in text), f"Non-ASCII output: {repr(text)}"


def test_toast_manager_renders_ascii():
    mgr = ToastManager()
    mgr.clear()
    mgr.push('Build finished', tone='success', width=24, duration=None, ascii_mode=True)
    output = _capture(mgr.render)
    assert '[+]' in output
    _assert_ascii_only(output)


def test_toast_manager_cleanup():
    mgr = ToastManager()
    mgr.clear()
    mgr.push('Short lived', tone='info', width=20, duration=0.0)
    # Force expiry
    for message in mgr._messages:
        message.created_at -= 10.0
    mgr.cleanup()
    assert not mgr._messages


def test_compass_ascii_output():
    output = _capture(Compass.render, 'NE', color='info')
    _assert_ascii_only(output)
    assert '^' in output or '>' in output


def test_thermometer_ascii_output():
    output = _capture(Thermometer.render, 5, min_val=0, max_val=10, height=4, label='Temp')
    _assert_ascii_only(output)
    assert 'Temp' in output


def test_weather_indicator_ascii_output():
    output = _capture(WeatherIndicator.render, 'storm', temperature_c=12.5)
    _assert_ascii_only(output)
    assert '(storm)' in output
