from __future__ import annotations

import re

from textforge.export import render_to_ansi, render_to_html, render_to_svg
from textforge.utils.testing import snapshot_assert


def _normalize_html(html: str) -> str:
    # Remove insignificant whitespace for stable snapshots
    html = re.sub(r"\s+", " ", html)
    # Newlines preserved as <br>; keep them
    return html.strip()


def _normalize_svg(svg: str) -> str:
    svg = re.sub(r"\s+", " ", svg)
    return svg.strip()


def test_html_handles_nested_and_resets_snapshot():
    ansi = "".join([
        "\x1b[1mBold ",
        "\x1b[3mItalic ",
        "\x1b[38;2;255;0;0mRed ",
        "\x1b[48;2;0;0;255mOnBlue\x1b[49m ",
        "\x1b[0mReset",
        "\n",
        "Plain",
    ])
    html = render_to_html(ansi, markup=False)
    snapshot_assert("export_html_nested", _normalize_html(html))


def test_svg_handles_nested_and_background_snapshot():
    ansi = "".join([
        "\x1b[1mB\x1b[0m ",
        "\x1b[38;2;10;200;10mG\x1b[0m ",
        "\x1b[48;2;20;20;200mBG\x1b[0m",
    ])
    svg = render_to_svg(ansi, markup=False)
    snapshot_assert("export_svg_nested", _normalize_svg(svg))


def test_round_trip_render_to_ansi_html_svg_agree_on_basic_markup():
    # Use markup input to exercise MarkupEngine too
    src = "[bold][cyan]Hello[/reset] [#ff8800]World[/reset]"
    ansi = render_to_ansi(src, markup=True)
    html = render_to_html(src, markup=True)
    svg = render_to_svg(src, markup=True)

    assert "Hello" in ansi and "World" in ansi
    assert "Hello" in html and "World" in html
    assert "Hello" in svg and "World" in svg
