from __future__ import annotations

import io
import sys

from textforge.core import Console


class _MemStream(io.StringIO):
    @property
    def encoding(self) -> str:
        return "utf-8"

    @property
    def buffer(self):
        return None


def test_capture_includes_stderr():
    c = Console(stream=_MemStream())

    def fn():
        sys.stdout.write("out\n")
        sys.stderr.write("err\n")

    captured = c.capture(fn)
    assert "out" in captured and "err" in captured


def test_print_async_handle():
    s = _MemStream()
    c = Console(stream=s)
    h = c.print_async("hello", style=None)
    h.join(1.0)
    assert h.exception is None
    assert "hello" in s.getvalue()


def test_scheduler_exception_capture():
    sched = Console.Scheduler(fps=1000)
    called = {"n": 0}

    def cb(_dt: float) -> None:
        called["n"] += 1
        raise RuntimeError("boom")

    h = sched.start(cb)
    h.join(1.0)
    assert h.exception is not None
    assert called["n"] >= 1


def test_live_session_shrink_and_wipe():
    s = _MemStream()
    live = Console(stream=s).live()
    live.update("line1\nline2")
    before = s.getvalue()
    assert "line1" in before and "line2" in before
    # Shrink to single line and ensure a wipe occurred
    live.update("x")
    after = s.getvalue()
    assert "x\n" in after
    assert "\x1b[2K" in after  # at least one line clear during shrink


def test_for_backend_tty():
    c = Console.for_backend("tty")
    c.print("ok")
