import ast
import os
from pathlib import Path
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from collections.abc import Iterator


def get_textforge_files() -> Iterator[Path]:
    """Get all Python files in the textforge package."""
    textforge_dir = Path(__file__).parent.parent / "textforge"
    for root, dirs, files in os.walk(textforge_dir):
        # Skip __pycache__ directories
        dirs[:] = [d for d in dirs if not d.startswith("__")]
        for file in files:
            if file.endswith(".py"):
                yield Path(root) / file


def extract_method_definitions(content: str) -> dict[tuple[str, str | None], list[int]]:
    """Extract function/method definitions grouped by enclosing class name.

    Keys are (name, class_name) where class_name is None for module-level functions.
    """
    tree = ast.parse(content)
    methods: dict[tuple[str, str | None], list[int]] = {}

    class _Visitor(ast.NodeVisitor):
        def __init__(self) -> None:
            self.class_stack: list[str] = []

        def visit_ClassDef(self, node: ast.ClassDef) -> None:
            self.class_stack.append(node.name)
            self.generic_visit(node)
            self.class_stack.pop()

        def visit_FunctionDef(self, node: ast.FunctionDef) -> None:
            cls = self.class_stack[-1] if self.class_stack else None
            key = (node.name, cls)
            methods.setdefault(key, []).append(node.lineno)

    _Visitor().visit(tree)
    return methods


def test_no_type_ignore_comments():
    """Ensure no '# type: ignore' comments remain in shipped code."""
    violations = []

    for file_path in get_textforge_files():
        try:
            content = file_path.read_text(encoding="utf-8")
            lines = content.splitlines()

            for line_num, line in enumerate(lines, 1):
                # Check for type: ignore comments (but allow the special case in jupyter.py)
                if "# type: ignore" in line:
                    if "jupyter.py" in str(file_path) and "IPython" in line:
                        continue  # Allowed special case
                    violations.append(f"{file_path}:{line_num}: {line.strip()}")

        except Exception as e:
            violations.append(f"Error reading {file_path}: {e}")

    assert not violations, "Found # type: ignore comments in shipped code:\n" + "\n".join(violations)


def test_no_pragma_no_cover_comments():
    """Ensure no '# pragma: no cover' comments remain in shipped code."""
    violations = []

    for file_path in get_textforge_files():
        try:
            content = file_path.read_text(encoding="utf-8")
            lines = content.splitlines()

            for line_num, line in enumerate(lines, 1):
                if "# pragma: no cover" in line:
                    violations.append(f"{file_path}:{line_num}: {line.strip()}")

        except Exception as e:
            violations.append(f"Error reading {file_path}: {e}")

    assert not violations, "Found # pragma: no cover comments in shipped code:\n" + "\n".join(violations)


def test_no_duplicated_method_definitions():
    """Ensure no duplicated method definitions in shipped code."""
    violations = []

    for file_path in get_textforge_files():
        try:
            content = file_path.read_text(encoding="utf-8")
            methods = extract_method_definitions(content)

            for (method_name, class_name), line_numbers in methods.items():
                if len(line_numbers) > 1:
                    violations.append(
                        f"{file_path}: Method '{method_name}' defined multiple times "
                        f"in {class_name or 'module'} at lines {line_numbers}"
                    )

        except Exception as e:
            violations.append(f"Error parsing {file_path}: {e}")

    assert not violations, "Found duplicated method definitions:\n" + "\n".join(violations)
