import io
import subprocess
import sys
from contextlib import redirect_stderr, redirect_stdout

from textforge.cli.main import main as cli_main


def _run_and_capture(*argv: str) -> str:
    buf = io.StringIO()
    with redirect_stdout(buf):
        try:
            cli_main(list(argv))
        except SystemExit:
            pass
    return buf.getvalue()


def _run_with_stdin(*argv: str, stdin_text: str = "") -> tuple[int, str, str]:
    out = io.StringIO()
    err = io.StringIO()
    code = 0
    old_stdin = sys.stdin
    sys.stdin = io.StringIO(stdin_text)
    try:
        with redirect_stdout(out), redirect_stderr(err):
            try:
                code = cli_main(list(argv))
            except SystemExit as se:
                try:
                    code = int(se.code)
                except Exception:
                    code = 1
    finally:
        sys.stdin = old_stdin
    return code, out.getvalue(), err.getvalue()


def test_cli_list_outputs_components():
    out = _run_and_capture("list")
    assert "Available components:" in out


def test_cli_dsl_runs_simple(tmp_path):
    src = tmp_path / "sample.dsl"
    src.write_text("Title text='Hello' width=40\n", encoding="utf-8")
    out = subprocess.run(
        [sys.executable, "-m", "textforge.cli", "dsl", "--file", str(src)],
        text=True,
        capture_output=True,
        check=False,
    )
    assert out.returncode == 0
    combined = (out.stdout or "") + (out.stderr or "")
    assert "Hello" in combined


def test_cli_preview_handles_missing_file():
    res = subprocess.run([sys.executable, "-m", "textforge.cli", "preview", "__nope__.md"], capture_output=True, text=True)
    assert res.returncode != 0
    assert "Failed to read" in res.stderr


def test_cli_export_pdf_binary_requires_output(tmp_path):
    code, _out, err = _run_with_stdin("export", "-f", "pdf", stdin_text="hello")
    assert code != 0
    assert "requires --output" in err.lower() or code != 0

    out_file = tmp_path / "out.pdf"
    code2, _out2, err2 = _run_with_stdin("export", "-f", "pdf", "-o", str(out_file), stdin_text="hello")
    assert code2 == 0
    assert out_file.exists() and out_file.stat().st_size > 0


def test_cli_plugins_list_runs():
    out = _run_and_capture("plugins", "list")
    assert "Registered plugins:" in out


def test_cli_bench_runs_and_prints():
    out = _run_and_capture("bench")
    assert "Benchmark results:" in out


def test_cli_new_scaffolds(tmp_path):
    proj = tmp_path / "demoapp"
    res = subprocess.run([sys.executable, "-m", "textforge.cli", "new", "demoapp", "--dir", str(tmp_path)], capture_output=True, text=True)
    assert res.returncode == 0
    main_py = proj / "main.py"
    assert main_py.exists()
    code = main_py.read_text(encoding="utf-8")
    assert "tfprint" in code and "components as C" in code
