﻿# Versioning

## Current Version Status

- **Current Version**: Textforge 0.1.0 (pre-1.0, development/pre-release)
- **Target Version**: Textforge 1.0 (stable API)
- **Version Location**: Defined in `pyproject.toml` (not exposed in package `__init__.py`)

Textforge follows semantic versioning (`MAJOR.MINOR.PATCH`) starting with version 1.0. The public 1.x API surface is defined by `textforge.api` and package root exports noted in `docs/API_REFERENCE.md`.

## Version Numbering

- **Major Version (X.y.z)**: Breaking API changes that require migration
- **Minor Version (x.Y.z)**: Backwards-compatible features and enhancements
- **Patch Version (x.y.Z)**: Bug fixes, documentation updates, and internal improvements

## Pre-1.0 Versioning (Current)

Versions 0.x.y are development releases with no API stability guarantees:

- APIs may change between minor versions
- Breaking changes can occur in any release
- Migration guides provided in `docs/MIGRATION.md`
- Use namespace imports for better future compatibility: `from textforge import components as C`

## 1.x API Stability

When Textforge reaches version 1.0, the following APIs will be guaranteed stable:

- **Stable Surface**: Import from `textforge.api` for guaranteed 1.x compatibility
- **Package Root**: Convenience re-exports may change; prefer namespaced imports
- **Semantic Versioning**: Breaking changes only in major version bumps
- **Deprecation Policy**: Functions marked with `@deprecated` emit warnings before removal

```python
# For maximum future compatibility (recommended):
from textforge import api
console = api.Console()

# For convenience (may change):
from textforge import Console, components
```

## Release Checklist

1. **Code Quality**: Ensure tests pass (`pytest`) and linting passes (`python -m mypy`, `ruff`).
2. **Documentation**: Update docs, examples, and API reference to cover new behavior.
3. **Version Bump**: Update version in `pyproject.toml` (and `textforge/__init__.py` if `__version__` is added).
4. **Snapshots**: Regenerate test snapshots (`TEXTFORGE_SNAPSHOT_UPDATE=1 pytest`).
5. **Changelog**: Update version history and migration notes.
6. **Release**: Tag the release (`git tag vX.Y.Z`) and publish to PyPI.
7. **Post-Release**: Update development version to next patch/minor version.

## 1.0 Goals

The following criteria must be met before Textforge can reach version 1.0:

### Core Stability
- **Stable Public APIs**: All APIs in `textforge.api` must be stable and well-tested
- **Semantic Versioning**: Full adherence to semver starting with 1.0.0
- **Deprecation System**: Working deprecation warnings for API changes

### Component Completeness
- **Component Catalogue**: All components listed in `docs/API_REFERENCE.md` must be implemented and tested
- **Test Coverage**: Comprehensive unit tests for all components with snapshot testing
- **API Consistency**: Uniform API design across all components

### Export System
- **Format Support**: Complete HTML, SVG, PDF, and plain text export implementations
- **Documented Behavior**: Clear documentation of export capabilities and limitations
- **Safety**: Proper input sanitization and safe mode operation

### Quality Assurance
- **Type Safety**: Strict mypy checking with zero `# type: ignore` comments in shipped code
- **Code Quality**: Zero ruff linting violations
- **Performance**: Established performance baselines and regression testing
- **Cross-Platform**: Verified compatibility across Windows, macOS, and Linux

### Documentation
- **API Reference**: Complete and accurate API documentation
- **Usage Examples**: Comprehensive examples for all major use cases
- **Migration Guide**: Clear upgrade paths from pre-1.0 versions

## Version History

### 0.1.0 (Current)
- Initial modular architecture with core rendering engine
- Component library with typography, containers, data visualization, and interactive elements
- Export system supporting HTML, SVG, PDF, and plain text
- Theme system with color tokens and inheritance
- CLI tool with multiple commands
- Basic performance benchmarking suite

## Deprecation Policy

### Pre-1.0 Deprecations (Current)
During pre-1.0 development, breaking changes may occur without deprecation warnings. Migration guides are provided in `docs/MIGRATION.md` for major changes.

### 1.x Deprecation Policy (Future)
Once Textforge reaches 1.0, the following deprecation policy will apply:

- **Deprecation Decorator**: Use `textforge.utils.deprecation.deprecated` (available as `textforge.deprecated`)
- **Warning Period**: Deprecations emit `DeprecationWarning` and remain available for at least one minor release
- **Removal Timeline**: Removals happen in major version bumps unless otherwise stated
- **Migration Guidance**: Each deprecation includes migration path and target removal version

Example deprecation (for 1.x+ versions):

```python
from textforge import deprecated

@deprecated("Use new_api() instead.", since="1.2", remove_in="2.0")
def old_api(...):
    ...
```

### Breaking Changes
- **Pre-1.0**: Breaking changes may occur in any release with migration documentation
- **1.x+**: Breaking changes only in major versions with proper deprecation cycle
- **Security**: Critical security fixes may introduce breaking changes with shorter notice
