﻿# Usage Guide

## Installation

Textforge requires Python 3.14 or later.

```bash
pip install textforge
```

### Optional Dependencies

For Windows compatibility:
```bash
pip install textforge[legacy_windows]
```

For Jupyter notebook support:
```bash
pip install textforge[jupyter]
```

## Rendering Basics
```
from textforge import tfprint

tfprint('[primary]Hello Textforge![reset]')
```

## Building Layouts

```python
from textforge import tfprint, components as C

# Simple component usage
tfprint(C.title('Dashboard', width=60))
tfprint(C.progress_bar(45, 100, label='Upload', color='green'))

# Table with data
data = [['Name', 'Score'], ['Alice', 91], ['Bob', 77]]
tfprint(C.table(data))

# Composing multiple components
layout = C.columns([
    C.panel(C.title('System Status')),
    C.panel(C.progress_bar(75, 100, label='CPU Usage'))
])
tfprint(layout)
```

### Using Templates

```python
from textforge.templates import Dashboard, Template

# Pre-built dashboard
Dashboard.system_overview()

# Game UI elements
Template.game_header("My Game", "Epic Adventure")
Template.inventory_screen([
    {"name": "Magic Sword", "quantity": 1, "rarity": "rare"},
    {"name": "Health Potion", "quantity": 5, "rarity": "common"}
])
```

## Accessibility & Semantic Metadata

```python
from textforge.utils import AccessibilityAPI

# Clear existing accessibility registry
AccessibilityAPI.clear()

# Annotate progress indicators for screen readers
AccessibilityAPI.annotate_progress(
    'upload_job',
    label='File Upload',
    current=8,
    maximum=10
)

# Register custom semantic elements
node = AccessibilityAPI.register_node(
    'main_menu',
    role='navigation',
    name='Main Menu',
    description='Primary navigation menu'
)

# Export accessibility data
accessibility_data = AccessibilityAPI.export_registry()
print("Accessibility nodes:", list(accessibility_data.keys()))

# Check color contrast
is_accessible = AccessibilityAPI.is_contrast_ok('#ffffff', '#000000')
print(f"Good contrast: {is_accessible}")
```

## Exporting to Other Formats

```python
from textforge import tfprint, components as C
from textforge.export import render_to_html, render_to_svg, ansi_to_pdf

# Create a component
status = C.panel(C.title("Deployment Status", color="success"))

# Export to HTML
html_output = render_to_html(status)
print("HTML:", html_output[:100] + "...")

# Export to SVG
svg_output = render_to_svg(status)

# Export to PDF (returns bytes)
pdf_bytes = ansi_to_pdf(render_to_ansi(status))

# Safe mode for security (disables markup interpretation)
safe_html = render_to_html("[user_input]Hello[/user_input]", safe_mode=True)
```

### CLI Export

```bash
# Export ANSI from stdin to HTML
echo "[bold]Hello World[/bold]" | textforge export -f html > output.html

# Export to PDF
textforge export -f pdf -o document.pdf < input.ansi
```

## Text Measurement & Layout

Textforge uses grapheme-aware measurement following Unicode standards:

```python
from textforge.utils import visible_width, measure_text

# Count visible width (ignores ANSI codes and markup)
text = "[bold red]Hello[reset] World"
width = visible_width(text)  # Returns 11 (not counting markup)

# Measure multi-line text
multiline = "Line 1\nLonger Line 2\nLine 3"
max_width, num_lines = measure_text(multiline)
# max_width = 12, num_lines = 3

# Tabs advance to next tab stop (default tab_size=4)
tabbed = "col1\tcol2\tcol3"
width = visible_width(tabbed)  # Accounts for tab expansion
```

## Live Rendering & Animation

```python
from textforge import Console, tfprint, components as C
import time

# Simple live updates
console = Console()
with console.live() as live:
    for i in range(10):
        progress = C.progress_bar(i * 10, 100, label=f"Processing {i}/10")
        live.update(progress)
        time.sleep(0.1)

tfprint(C.toast("Process complete!", tone="success"))
```

## Interactive Components

```python
from textforge import components as C, tfprint

# Menu selection
choice = C.Menu.run(
    "Select an option:",
    ["Start Game", "Settings", "Quit"],
    timeout=10.0  # Auto-select after 10 seconds
)
tfprint(f"You selected: {choice}")

# Form input (if available)
# form_data = C.Form.run([
#     {"name": "username", "type": "text", "label": "Username"},
#     {"name": "age", "type": "number", "label": "Age"}
# ])
```

## Theming & Styling

```python
from textforge import tfprint, components as C
from textforge.style import themes

# List available themes
print("Available themes:", list(themes.available().keys()))

# Use theme colors
tfprint(C.title("Themed Title", color="primary"))
tfprint(C.panel(C.paragraph("This uses theme styling", color="secondary")))

# Custom styling with markup
tfprint("[bold yellow on_blue]Custom styled text[/bold yellow on_blue]")
```

## Command Line Interface

Textforge provides a comprehensive CLI for development and testing:

```bash
# List all components
textforge list

# Run interactive demos
textforge demo --preset full

# Preview markdown files
textforge preview README.md

# Benchmark performance
textforge bench --metrics

# Export formats
textforge export -f html < input.ansi > output.html

# Theme management
textforge theme list

# Plugin system
textforge plugins list
```

## Advanced Usage

### Custom Components

```python
from textforge.core import Renderable
from textforge import tfprint

class CustomCard(Renderable):
    def __init__(self, title: str, content: str):
        self.title = title
        self.content = content

    def render(self, console, width: int) -> str:
        from textforge import components as C
        return console.render(C.panel([
            C.title(self.title, width=width),
            C.paragraph(self.content, width=width)
        ]), width=width)

tfprint(CustomCard("My Card", "Custom component content"))
```

### Layout Composition

```python
from textforge import tfprint, components as C

# Complex layout
dashboard = C.columns([
    C.panel([
        C.title("System Status"),
        C.progress_bar(75, 100, label="CPU"),
        C.progress_bar(60, 100, label="Memory")
    ]),
    C.panel([
        C.title("Recent Activity"),
        C.list_element(["User login", "File upload", "Backup complete"])
    ])
])

tfprint(dashboard)
```
