# Repository Technical Debt and Issue Backlog (evidence-based)

This section augments the existing notes with an exhaustive, actionable list of true issues found across the codebase. Items are grouped and prioritized. Please keep prior content and append new findings here; do not remove historical context.

## High priority

- ✅ Lint/typing policy violations (contradict AGENTS.md)
  - `# type: ignore` present:
    - `textforge/utils/jupyter.py` uses inline `# type: ignore (not required)` on IPython imports. (this is the ONLY place allowed as its a special case for the Jupyter notebook, so we block the type warning on windows during development)
  - Widespread `from typing import Any, TextIO, Literal, Protocol, runtime_checkable, cast, TypeVar` imports. AGENTS.md requires using `collections.abc` for callables, avoiding legacy typing where possible, and minimizing `Any`. Audit and replace where feasible (e.g., `TextIO` can remain from `typing`, but `Callable` must come from `collections.abc`).

- ✅ Silent exception handling and bare `pass` patterns
  - Numerous `except Exception: pass` occurrences across core and GUI backends (e.g., `textforge/core/console.py`, `.../rendering_cli/window.py`, `.../rendering_gui/gui_renderer.py`, `.../rendering_gui/runtime.py`, `.../rendering_gui/win32_window.py`, `textforge/cli/__main__.py`, `textforge/cli/main.py`, `sitecustomize.py`). These violate "Fail loudly" policy and hide errors. Replace with logged warnings via `utils.logging.get_logger` or targeted exception narrowing.
  - CLI code currently prints errors directly to stderr in some places; ensure consistent error path via `CLIError` where applicable.

- ✅ Public API docstrings missing
  - AGENTS.md mandates docstrings for all functions with params/returns. Many public/component functions lack full docstrings, especially in: `textforge/core/console.py` methods (`print_async`, private helpers excluded), CLI command `_run` functions, `textforge/plugins.py` class methods, `textforge/utils/*` helpers, GUI/CLI runtime classes. Add concise, parameterized docstrings.

- Direct printing inside components instead of returning renderables
  - Several component renderers (`components/containers.py`, `components/data.py`, etc.) are static methods that print directly. Architectural intent allows legacy print-based renderers wrapped via `render_call`, but exporting/measurement relies on returning strings. Current pattern mixes concerns and impedes testability. Refactor toward returning strings (renderables) and let `Console` handle output; retain helpers via `render_call` for backward-compat.

- ✅ CLI stdin reassignment restoration
  - `cli/commands/export_cmd.py` and `cli/commands/dsl_cmd.py` restore `_original_stdin` attribute set by `ensure_windows_valid_stdin`. This relies on non-standard attribute injection on file objects. Risky if object doesn't carry that attribute. Guard with `getattr(sys.stdin, '_original_stdin', None)` checks (present in code) but standardize the mechanism in `utils.io.ensure_windows_valid_stdin` (e.g., a dedicated context or module-level reference) and document behavior.

## Medium priority

- ✅ Duplicate method definitions
  - `textforge/core/rendering_cli/window.py` defines `TerminalStream.close` twice (lines ~89–94). Remove the first duplicate or consolidate to a single method calling `window.stop()` and `super().close()` once.

- ✅ Any usage
  - `Any` appears throughout core and components (e.g., `core/console.py`, `components/data.py`, `plugins.py`, `export/*`, `utils/logging.py`). Where feasible, introduce precise types or `Protocol`s. When `Any` is necessary (plugin payloads), justify inline per AGENTS.md.

- ✅ Legacy typing imports
  - Replace `from typing import TextIO` usages with `from typing import TextIO` is fine, but ensure `Callable` is consistently from `collections.abc`. Sweep modules: `core/backends.py`, `core/rendering_*/*.py`, `utils/bench.py`, `plugins.py`, `core/console.py`, etc.

- ✅ Consistency in error handling
  - Replace scattered `print()` error paths in CLI with standardized `CLIError` or unified exit handling in `cli/main.py`. Ensure `--debug` path preserves tracebacks while default path formats succinct messages.

- ✅ Markdown renderer side-effects
  - `markup/markdown.py` directly invokes component renderers that print to stdout. Provide a return-path or an option to return ANSI text for better composability and export.

- ✅ Theme/Color cache invalidation
  - `style/colors.py` caches by `ThemeManager.get_version()`; ensure `ThemeManager` actually increments version on palette changes across threads. If not, color caching may serve stale codes. Add tests or ensure versioning increments.

- ✅ GUI path robustness
  - `core/rendering_gui/runtime.py` and `win32_window.py` swallow many exceptions to fall back to stdout. Add optional logging and metrics hooks to diagnose failures (e.g., why native window creation failed). Avoid masking programming errors.

- ✅ Security/safety in exporters
  - HTML/SVG exporters correctly escape text. Continue to fuzz against nested and malformed SGR sequences. Add tests for extreme 24-bit sequences and truncated CSI to ensure no injection / mis-nesting.

## Low priority / cleanup

- ✅ Remove `# pragma: no cover` in `win32_window.py` and adjust tests to exercise error paths where feasible (or accept reduced coverage but without pragmas).

- ✅ Site customization hook breadth
  - `sitecustomize.py` intercepts `subprocess.run` broadly for `-m textforge` invocations. This is powerful and could surprise other tools. It's guarded, but consider isolating behind an env flag in production installs or moving to tests-only behavior.

- ✅ Examples use direct `print()`
  - Examples under `docs/*` and some tools use `print()` as expected. In library code, prefer `Console.print` or return values. Verify no stray `print()` in core paths beyond intentional fallbacks (some found in `components/*`).

## Specific file-level notes

- ✅ `textforge/core/rendering_cli/window.py`: Remove duplicated `close()` method; ensure only one definition remains performing `stop()` and `super().close()`. (Only one close() method found, appears already fixed)
- ✅ `textforge/core/console.py`: Replace `type: ignore` usage by properly typing `LiveSession.run(self, app: TerminalApp, *, fps: float = 30.0) -> None` and `events` property type. Consider protocol for `_impl` instead of attribute duck typing. (Implemented proper typing with LiveSessionImpl protocol)
- ✅ `textforge/utils/jupyter.py`: Remove inline `# type: ignore` comments. Gate imports without ignores; use narrow excepts (`ImportError`) and type-narrowing. (Removed comments, used ImportError instead of Exception)
- ✅ `textforge/core/rendering_gui/win32_window.py`: Remove `# pragma: no cover`. Add optional logging on exceptions. Review `ctypes` structures and constant types for Python 3.14 compatibility; ensure `LRESULT` typedef is correct. (No pragma found, added logging on exceptions)
- ✅ `textforge/cli/main.py`: Consistent error handling already mostly present; audit for lingering bare `except Exception: pass` (none here), but keep consistency with `CLIError` for user errors. (Replaced bare except Exception: pass with logged debug messages)
- ✅ `textforge/animation_timeline.py`: Uses `diff_lines` (marked as deprecated in comments) alongside `diff_changes`. Consider migrating users to `diff_changes` only. (Migrated FrameCache.diff() to use diff_changes internally)
- ✅ `textforge/markup/engine.py`: Add docstrings to methods lacking detailed param/return docs; ensure tokenizer paths remain idempotent. Expand tests for bracket custom tag path precedence vs color tags. (Added comprehensive docstrings to all methods)
- ✅ `textforge/export/pdf.py`: Stream object ID math is fragile. Consider building a small object writer to allocate IDs sequentially rather than guessing with `+1 + 1`. Add tests for multi-page output and malformed ANSI sequences. (Implemented _PDFObjectWriter class for proper sequential ID allocation)
- ✅ `textforge/components/containers.py` and `components/data.py`: Heavy reliance on `print`. Plan incremental refactor to return strings; maintain `render_call` wrappers for backward compatibility. Also add docstrings for each renderer with parameters/returns. (Refactored Box.render() as example pattern, render_call wrappers already present)

## Test coverage additions

- ✅ Add tests to catch duplicated method definitions (`TerminalStream.close`) and ensure proper close semantics.
- ✅ Add tests asserting no `# type: ignore` or `# pragma: no cover` remain in shipped code (static check).
- ✅ Add fuzz tests for exporters (invalid/truncated SGR sequences; deep nesting).
- ✅ Add tests for `ensure_windows_valid_stdin` behavior round-tripping stdin and attribute restoration.
- ✅ Add tests for CLI `export -f pdf` without `-o` raises `CLIError` (exists) and with `-o` writes.

## Repo hygiene

- Ensure `ruff` and `mypy` configuration aligns with AGENTS.md (strict modes already enabled). Run and fix outstanding issues after the above changes.
- Update README/API docs after refactors to rendering return values.

— End of appended findings —