# Theming

Textforge provides a comprehensive token-based theming system with thread-local overrides, theme inheritance, and multiple color specification formats.

## Concepts

### Theme Tokens

Themes define semantic color tokens that map to color specifications:

- **Core Colors**: `primary`, `secondary`, `accent`, `info`, `success`, `warning`, `error`
- **Neutral Colors**: `muted`, `fg` (foreground), `bg` (background)

### Color Specifications

Tokens can resolve to various color formats:
- **Named Colors**: `"red"`, `"blue"`, `"green"`, etc.
- **Hex Colors**: `"#ff0000"`, `"#3366cc"`
- **RGB Values**: `"255,0,0"`, `"51,102,204"`
- **Raw ANSI Codes**: `"\033[91m"` (direct ANSI escape sequences)

### Theme Management

- **Global Themes**: Registered themes available application-wide
- **Current Theme**: Active theme for token resolution
- **Thread-Local Overrides**: Per-thread token overrides without affecting global state
- **Console Overrides**: Per-console theme customizations
- **Theme Inheritance**: Create derived themes from existing ones

## Default Theme

Textforge includes a default theme with these tokens:

```python
{
    "primary": "#4ea1ff",    # Blue - main actions, links
    "secondary": "#9aa5b1",  # Gray - secondary elements
    "accent": "#ffb020",     # Orange - highlights, accents
    "info": "#2dc6ff",       # Light blue - informational messages
    "success": "#22c55e",    # Green - success states, positive feedback
    "warning": "#f59e0b",    # Yellow - warnings, cautions
    "error": "#ef4444",      # Red - errors, destructive actions
    "muted": "#94a3b8",      # Muted gray - disabled states, subtle text
    "fg": "#e5e7eb",         # Light gray - primary text color
    "bg": "#111827",         # Dark blue-gray - background color
}
```

## API

### Basic Theme Management

```python
from textforge.style.themes import Theme, ThemeManager
from textforge import tfprint

# Register a custom theme
ThemeManager.register(Theme(name='solarized', palette={
    'primary': '#268bd2',
    'secondary': '#586e75',
    'accent': '#b58900',
    'info': '#2aa198',
    'success': '#859900',
    'warning': '#cb4b16',
    'error': '#dc322f',
    'muted': '#93a1a1',
    'fg': '#eee8d5',
    'bg': '#002b36',
}))

# Switch to a theme
ThemeManager.set_current('solarized')

# List available themes
themes = ThemeManager.available()
print("Available themes:", list(themes.keys()))
```

### Theme Inheritance

Create derived themes that inherit from existing ones:

```python
# Create a high-contrast variant of the default theme
ThemeManager.inherit(
    name='high_contrast',
    base='default',
    overrides={
        'fg': '#ffffff',
        'bg': '#000000',
        'muted': '#666666',
        'primary': '#00ffff',
        'accent': '#ffff00',
    }
)

ThemeManager.set_current('high_contrast')
```

### Thread-Local Overrides

Temporarily override theme tokens without affecting other threads:

```python
# Context manager for temporary overrides
with ThemeManager.override({'accent': '#ff00ff', 'error': '#ff4444'}):
    tfprint('[accent]This text uses magenta accent[reset]')
    tfprint('[error]This error is bright red[reset]')

# Outside the context, original theme values are restored
tfprint('[accent]Back to normal accent color[reset]')
```

### Console-Level Theme Overrides

Apply theme customizations per console instance:

```python
from textforge.core import Console

# Create console with custom theme overrides
console = Console(theme_overrides={'primary': '#00ff00', 'bg': '#000000'})
console.print('[primary]Green text on black background[reset]')
```

### Manual Override Management

For advanced use cases, manage overrides manually:

```python
# Push overrides
ThemeManager.push_overrides({'accent': '#ff69b4'})

# Use the overridden theme
tfprint('[accent]Hot pink accent text[reset]')

# Pop overrides to restore previous state
ThemeManager.pop_overrides()
```

## Usage in Components

Components automatically resolve theme tokens:

```python
from textforge.components import ProgressBar, AlertBanner

# Components use theme tokens for colors
ProgressBar.render(current=75, maximum=100, color='primary')
AlertBanner.render("Success!", tone='success')
```

## Usage in Markup

Markup tags resolve theme tokens:

```python
# These all work with theme tokens
tfprint('[fg=primary]Primary colored text[reset]')
tfprint('[bg=accent]Accent background[reset]')
tfprint('[success]Success colored text[reset]')
```

## CLI Integration

Manage themes from the command line:

```bash
# List available themes
textforge theme list

# Output:
# Available themes:
#   - default
#   - solarized
#   - high_contrast
```

## Advanced Features

### Version Tracking

Themes include version tracking for cache invalidation:

```python
version = ThemeManager.get_version()
# Version increments whenever themes change, enabling cache invalidation
```

### Color Resolution

Themes support flexible color specifications:

```python
ThemeManager.register(Theme(name='mixed', palette={
    'primary': 'blue',           # Named color
    'accent': '#ff6600',         # Hex color
    'success': '0,255,0',        # RGB values
    'warning': '\033[93m',       # Raw ANSI code
}))
```

### Thread Safety

All theme operations are thread-safe:
- Global theme registry is shared across threads
- Thread-local overrides isolate per-thread customizations
- Current theme setting affects all threads (use overrides for thread isolation)

## Complete Example

```python
from textforge.style.themes import Theme, ThemeManager
from textforge.core import Console, tfprint
from textforge.components import AlertBanner, ProgressBar

# Register a custom theme
ThemeManager.register(Theme(name='ocean', palette={
    'primary': '#0066cc',
    'secondary': '#4d94ff',
    'accent': '#00aaff',
    'info': '#87ceeb',
    'success': '#20b2aa',
    'warning': '#ffa500',
    'error': '#dc143c',
    'muted': '#778899',
    'fg': '#f0f8ff',
    'bg': '#191970',
}))

# Set as current theme
ThemeManager.set_current('ocean')

# Create console with additional overrides
console = Console(theme_overrides={'primary': '#00ffff'})

# Use themed components
AlertBanner.render("System ready", tone='success')
ProgressBar.render(current=85, maximum=100, color='primary')

# Use themed markup
tfprint('[primary]Primary action[reset] | [accent]Accent highlight[reset]')
tfprint('[bg=success][fg=bg]Success banner[reset]')

# Temporary theme override
with ThemeManager.override({'accent': '#ff1493'}):
    tfprint('[accent]Temporarily hot pink accent[reset]')

# Back to normal
tfprint('[accent]Normal accent color restored[reset]')
```

## Best Practices

1. **Consistent Token Usage**: Use semantic tokens (`primary`, `success`) rather than specific colors
2. **Thread-Local Overrides**: Use context managers for temporary theme changes
3. **Theme Inheritance**: Build theme variants using inheritance rather than duplicating palettes
4. **Console Overrides**: Use per-console overrides for isolated customizations
5. **Named Colors**: Prefer named colors and hex values over raw ANSI codes for portability
