# Plugin API

Textforge provides a comprehensive plugin system with runtime registration and automatic entry point discovery. Third-party packages can extend Textforge functionality without import-time coupling.

## Runtime registration

```python
from textforge.plugins import PluginRegistry

# Register a function
PluginRegistry.register('hello', lambda name: f'Hello, {name}!')

# List and call
print(PluginRegistry.list())        # ['hello']
func = PluginRegistry.get('hello')
print(func('World'))                # Hello, World!
```

## Entry points

Plugins can be automatically discovered and registered via Python entry points. This allows third-party packages to extend Textforge without requiring import-time coupling.

### Package Configuration

Add entry points to your `pyproject.toml`:

```toml
[project.entry-points."textforge.plugins"]
my-plugin = "my_package.plugins:my_function"
custom-renderer = "my_package.plugins:CustomRenderer"
```

Or in `setup.py`:

```python
setup(
    # ... other config
    entry_points={
        "textforge.plugins": [
            "my-plugin = my_package.plugins:my_function",
            "custom-renderer = my_package.plugins:CustomRenderer",
        ]
    }
)
```

### Plugin Registration Methods

Entry points support two registration patterns:

1. **Direct callable**: The entry point object must be callable and will be registered under the entry point name.

```python
# my_package/plugins.py
def my_function(text: str) -> str:
    """A simple text transformation plugin."""
    return f"[PLUGIN] {text}"

class CustomRenderer:
    """A class-based plugin."""
    def __call__(self, content: str) -> str:
        return f"<custom>{content}</custom>"
```

2. **Factory pattern**: The object may expose a `register` method that returns a `(name, func)` tuple.

```python
# my_package/plugins.py
class PluginFactory:
    @staticmethod
    def register():
        """Return (name, callable) tuple for registration."""
        def advanced_formatter(text: str, style: str = "bold") -> str:
            return f"[{style}]{text}[/{style}]"

        return ("advanced-format", advanced_formatter)
```

### Automatic Discovery

Plugins are automatically discovered when needed:

```python
from textforge.plugins import PluginRegistry

# Load all entry point plugins
count = PluginRegistry.load_entry_points()
print(f"Loaded {count} plugins")

# Now available for use
plugins = PluginRegistry.list()
print(f"Available plugins: {plugins}")
```

## API Reference

### PluginRegistry

The `PluginRegistry` class provides the core plugin management functionality:

#### `register(name: str, func: Callable[..., Any]) -> None`
Register a plugin function with the given name.

```python
PluginRegistry.register('markdown', lambda text: render_markdown(text))
```

#### `get(name: str) -> Callable[..., Any] | None`
Retrieve a registered plugin by name. Returns `None` if not found.

```python
plugin = PluginRegistry.get('markdown')
if plugin:
    result = plugin("# Hello World")
```

#### `list() -> list[str]`
Return a sorted list of all registered plugin names.

```python
names = PluginRegistry.list()  # ['markdown', 'html', 'json']
```

#### `load_entry_points(group: str = "textforge.plugins") -> int`
Discover and register plugins from Python entry points. Returns the number of plugins loaded.

```python
count = PluginRegistry.load_entry_points()
print(f"Loaded {count} entry point plugins")
```

This method is safe to call multiple times and handles missing dependencies gracefully.

## CLI integration

The CLI automatically discovers entry point plugins before listing:

```bash
# List all registered plugins (runtime + entry points)
textforge plugins list

# Output:
# Registered plugins:
#   - hello
#   - my-plugin
#   - custom-renderer
```

The CLI command handles plugin discovery gracefully and will work even if entry point discovery fails due to missing dependencies or configuration issues.

## Plugin Development Guide

### Best Practices

1. **Use descriptive names**: Plugin names should clearly indicate their purpose (e.g., `markdown-renderer`, `json-formatter`)

2. **Handle errors gracefully**: Plugins should not crash the application. Use try/except blocks and return sensible defaults.

3. **Document your plugins**: Include docstrings and type hints for better developer experience.

4. **Follow naming conventions**: Use lowercase with hyphens for entry point names, matching typical Python package naming.

### Complete Example

Here's a complete plugin package example:

```python
# my_textforge_plugins/__init__.py
"""Textforge plugins for enhanced text processing."""

__version__ = "1.0.0"

# my_textforge_plugins/markdown.py
from typing import Any
import markdown

def render_markdown(text: str, **options: Any) -> str:
    """Render markdown text to ANSI-formatted output.

    Args:
        text: Markdown content to render
        **options: Additional options passed to markdown parser

    Returns:
        ANSI-formatted string
    """
    html = markdown.markdown(text, **options)
    # Convert HTML to ANSI (implementation depends on your needs)
    return html_to_ansi(html)

# my_textforge_plugins/json_formatter.py
import json
from typing import Any

class JSONFormatter:
    """JSON formatting and validation plugin."""

    def __call__(self, data: Any, indent: int = 2, sort_keys: bool = False) -> str:
        """Format data as pretty-printed JSON."""
        try:
            return json.dumps(data, indent=indent, sort_keys=sort_keys)
        except (TypeError, ValueError) as e:
            return f"[ERROR] Invalid JSON data: {e}"

    @staticmethod
    def register():
        """Register the JSON formatter plugin."""
        return ("json-format", JSONFormatter())
```

```toml
# pyproject.toml
[project]
name = "my-textforge-plugins"
version = "1.0.0"
dependencies = ["markdown", "textforge"]

[project.entry-points."textforge.plugins"]
markdown-renderer = "my_textforge_plugins.markdown:render_markdown"
json-formatter = "my_textforge_plugins.json_formatter:JSONFormatter"
```

### Testing Plugins

Test your plugins thoroughly:

```python
# tests/test_plugins.py
from textforge.plugins import PluginRegistry

def test_markdown_plugin():
    # Register plugin
    from my_textforge_plugins.markdown import render_markdown
    PluginRegistry.register('test-markdown', render_markdown)

    # Test functionality
    result = PluginRegistry.get('test-markdown')("# Hello")
    assert "[bold]" in result or "<h1>" in result  # Depending on implementation

def test_entry_points():
    count = PluginRegistry.load_entry_points()
    assert count >= 0  # Should not crash

    plugins = PluginRegistry.list()
    assert 'markdown-renderer' in plugins
```

### Plugin Types

Consider what type of plugin you're building:

- **Transformers**: Convert input formats (markdown, JSON, etc.)
- **Renderers**: Generate specific output formats (HTML, PDF, etc.)
- **Utilities**: Helper functions for common tasks
- **Extensions**: Add new component types or behaviors

## Current Status

✅ **Implemented:**
- Runtime plugin registration API
- Entry point discovery via `importlib.metadata`
- CLI integration with automatic plugin discovery
- Support for both direct callable and factory registration patterns
- Graceful error handling for missing dependencies

## Future Enhancements

- Plugin signature validation and type checking
- Plugin metadata and versioning support
- Plugin dependency management
- Hot-reload capabilities for development
- Plugin marketplace/registry integration
