# Migration Guide

This guide covers migration paths for Textforge users. Currently at version 0.1.0 (pre-1.0), this document provides guidance for users upgrading from pre-modular versions and preparing for the stable 1.x API.

## Version Status

- **Current**: Textforge 0.1.0 (development/pre-release)
- **Target**: Textforge 1.0 (stable API)
- **Python Requirement**: Python 3.14+ (required for all versions)

## Migration from Pre-Modular Versions

The following changes apply when upgrading from early prototype versions to the current modular architecture.

## Imports and Public API

### Current Recommendations (0.1.0)

```python
# Recommended: Use namespace imports for clarity
from textforge import tfprint, components as C, templates as T
from textforge.core import Console

# Alternative: Direct imports from package root (convenience)
from textforge import Console, components, templates

# For stable 1.x compatibility (when released), use:
from textforge import api  # Stable 1.x surface
```

### Migration from Pre-Modular Versions

- **Old**: `from textforge import Console, Title, ProgressBar` (global namespace)
- **New**: Prefer `from textforge import tfprint, components as C` and `from textforge.core import Console`
- **Available**: Package root re-exports common symbols for convenience, but prefer namespaced imports for future compatibility

## Component Rendering

### Migration from Pre-Modular Versions

- **Old**: Components printed directly to stdout via `.render()` methods
- **New**: Components return renderable objects used via `tfprint` or `Console` helpers

```python
from textforge import tfprint, components as C

# Old approach (pre-modular):
# Title.render('Hello', width=60)  # Printed directly

# New approach (current):
tfprint(C.title('Hello', width=60))  # Returns renderable, prints via tfprint

# Alternative: Use Console directly for more control
from textforge.core import Console
console = Console()
console.print(C.title('Hello', width=60))
```

### Current Best Practices

```python
# Use tfprint for simple output
tfprint(C.title('Welcome', color='primary'))

# Use Console for live updates or complex layouts
console = Console()
with console.live() as live:
    live.update(C.progress_bar(50, 100, label="Loading..."))

# Compose multiple components
layout = C.columns([
    C.panel(C.title('Left Panel')),
    C.panel(C.title('Right Panel'))
])
tfprint(layout)
```

## Markup System

### Migration from Pre-Modular Versions

- **Old**: Ad-hoc regex-based parser with limited features
- **New**: `MarkupEngine` with proper escaping, custom tag registration, and nested tag support

```python
from textforge import MarkupEngine

# Old approach (limited):
# "[bold]Text[/bold]"  # Basic regex parsing

# New approach (current):
engine = MarkupEngine()
result = engine.render("[bold red]Styled text[/bold red] with [italic]markup[/italic]")

# Custom tags supported
@engine.tag("highlight")
def highlight_tag(content, **attrs):
    return f"[yellow on_blue]{content}[reset]"

tfprint(engine.render("Regular text with [highlight]highlighted content[/highlight]"))
```

## Themes and Styling

### Current Features (0.1.0)

- **ThemeManager**: Centralized theme management with inheritance and overrides
- **Tokenized Colors**: Named color tokens with theme-aware resolution
- **Dynamic Styling**: Color blending and transitions

```python
from textforge.style import themes

# List available themes
print(themes.available())

# Use themes in components
tfprint(C.title('Themed Title', color='primary'))  # Uses theme colors
```

## Export System

### Current Capabilities (0.1.0)

- **Multiple Formats**: HTML, SVG, PDF, and plain text export
- **Pipeline**: `render_to_ansi()` → format-specific converters
- **Safe Mode**: Markup sanitization for security

```python
from textforge import export

# Export to various formats
html_output = export.render_to_html(my_component)
svg_output = export.render_to_svg(my_component)
pdf_bytes = export.ansi_to_pdf(ansi_text)
```

## CLI and Tools

### Current CLI (0.1.0)

- **Modular Design**: Commands in `textforge.cli.commands.*`
- **Entry Point**: `textforge` command via `textforge.cli:main`
- **Available Commands**: demo, export, list, live, markup, new, plugins, preview, theme, typewriter, bench, dsl

```bash
# Current CLI usage
textforge demo --preset full
textforge export -f html < input.ansi > output.html
textforge theme list
```

## Future 1.x API Stability

When Textforge reaches version 1.0, the following will be guaranteed stable:

- **Stable Surface**: Import from `textforge.api` for guaranteed 1.x compatibility
- **Deprecation Policy**: Functions marked with `@deprecated` decorator emit warnings before removal
- **Semantic Versioning**: Breaking changes only in major version bumps

```python
# For maximum future compatibility:
from textforge import api

# Access stable APIs through the api module
console = api.Console()
console.print(api.components.title('Hello'))
```

## Where to Find Help

- **Current APIs**: See `docs/API_REFERENCE.md` for available functionality
- **Stable 1.x APIs**: Use `textforge.api` when version 1.0 is released
- **Examples**: Check `examples/` directory for usage patterns
- **CLI Help**: Run `textforge --help` or `textforge <command> --help`
