# Markup Language

Textforge provides a lightweight inline markup system with square brackets for built-in effects and colors, and custom tag hooks using both square brackets and curly braces.

## Syntax

### Built-in Color and Effect Tags

Use square brackets for built-in colors, effects, and advanced color specifications:

- **Named Colors**: `[red]`, `[green]`, `[blue]`, `[yellow]`, `[cyan]`, `[magenta]`, `[white]`, `[black]`, `[gray]`, `[orange]`, `[purple]`, `[brown]`, `[pink]`, `[lime]`, `[navy]`, `[teal]`, `[gold]`, `[silver]`, `[maroon]`, `[olive]`
- **Light Variants**: `[light_red]`, `[light_green]`, `[light_blue]`, `[light_yellow]`, `[light_cyan]`, `[light_magenta]`, `[light_white]`, `[light_gray]`, `[light_purple]`, `[light_orange]`, `[light_brown]`
- **Text Effects**: `[bold]`, `[dim]`, `[italic]`, `[underline]`, `[double_underline]`, `[blink]`, `[reverse]`, `[hidden]`, `[strikethrough]`, `[overline]`, `[framed]`, `[encircled]`, `[reset]`
- **Background Colors**: `[bg_red]`, `[bg_green]`, `[bg_blue]`, `[bg_yellow]`, `[bg_cyan]`, `[bg_magenta]`, `[bg_white]`, `[bg_black]`, `[bg_gray]`, `[bg_orange]`, `[bg_purple]`, `[bg_brown]`, `[bg_pink]`, `[bg_lime]`, `[bg_navy]`, `[bg_teal]`, `[bg_gold]`

### Advanced Color Specifications

- **Hex Colors**: `[#rrggbb]` or `[#rgb]` for foreground, `[bg=#rrggbb]` for background
- **RGB Colors**: `[r,g,b]` for foreground, `[bg=r,g,b]` for background
- **Theme Tokens**: `[fg=token]` or `[bg=token]` where `token` is a theme palette value (e.g., `[fg=primary]`, `[bg=accent]`)

### Custom Tags

Register custom tags that can be invoked with either curly brace or square bracket syntax:

- **Curly Brace Syntax**: `{name:payload}`
- **Square Bracket Syntax**: `[name:payload]` (only for registered custom tags to avoid conflicts)

### Escaping

- Escape `[` and `]` by doubling: `[[` → `[`, `]]` → `]`
- Escape `{` and `}` by doubling: `{{` → `{`, `}}` → `}`
- Or prefix with backslash: `\[` → `[`, `\]` → `]`, `\{` → `{`, `\}` → `}`

### Safe Mode

Enable safe mode to treat all markup as literal text:

```python
engine = MarkupEngine(safe_mode=True)
engine.render("[red]This is literal text[/red]")  # Renders as-is without processing
```

## Custom Tags

Use `MarkupEngine` to register custom tags that transform content:

```python
from textforge.markup.engine import MarkupEngine

engine = MarkupEngine()
engine.register_tag('link', lambda url: f"[underline][fg=#60a5fa]{url}[reset]")

print(engine.render('See {link:https://example.com}'))
# Output: See [underline][fg=#60a5fa]https://example.com[reset]
```

### Global Registration

Register tags globally for use across all markup engines:

```python
from textforge.markup.engine import MarkupEngine

MarkupEngine.register_global_tag('badge', lambda text: f"[bg=#374151][fg=#e5e7eb] {text} [reset]")
```

### Built-in Global Tags

Textforge automatically registers these global tags:

- **Time Tag**: `{time:format}` - Insert current time with strftime format
  ```python
  print(engine.render('Current time: {time:%H:%M:%S}'))
  ```
- **Internationalization Tag**: `[t:key]` - Look up translated strings (requires i18n setup)

## Markdown Support

Textforge includes basic Markdown-to-component rendering:

```python
from textforge.markup.markdown import render_markdown

render_markdown("""
# My Document

This is a paragraph with **bold** and *italic* text.

- Bullet item 1
- Bullet item 2

1. Numbered item
2. Another item

| Header 1 | Header 2 |
|----------|----------|
| Cell 1   | Cell 2   |

```python
print("Hello, World!")
```
""")
```

Supported Markdown features:
- Headers (`#`, `##`, etc.) → `Title` components
- Paragraphs → `Paragraph` components
- Bullet lists (`-`, `*`) → `ListElement` components
- Numbered lists (`1.`, `2.`) → `ListElement` components
- Code blocks (````) → `CodeBlock` components
- Simple tables (pipe-separated) → `Table` components

## CLI Integration

List registered custom tags:

```bash
textforge markup list-tags
```

## Complete Example

```python
from textforge.markup.engine import MarkupEngine
from textforge.core import tfprint

# Create markup engine
engine = MarkupEngine()

# Register custom tags
engine.register_tag('warning', lambda text: f"[bg=#fbbf24][fg=#92400e]⚠ {text}[reset]")
engine.register_tag('code', lambda snippet: f"[bg=#1f2937][fg=#e5e7eb]`{snippet}`[reset]")

# Render complex markup
text = """
[bold][fg=#3b82f6]System Status[reset]

{warning:High CPU usage detected}

Current time: {time:%Y-%m-%d %H:%M:%S}

{code:cpu_usage = get_system_load()}

[green]✓ All systems operational[reset]
"""

tfprint(engine.render(text))
```
