﻿# Textforge Examples

## Styled output

```python
from textforge import tfprint

tfprint('[info]Build succeeded[reset]')
```

## Components

```python
from textforge import components as C

# Typography
tfprint(C.title('System Status', width=60, symbol_style='rounded'))
tfprint(C.paragraph('This is a sample paragraph with [bold]bold text[reset] and [accent]accent colors[reset].', width=60))
tfprint(C.list_element(['First item', '[primary]Second item[reset]', 'Third item'], list_type='numbered'))

# Status indicators
tfprint(C.progress_bar(72, 100, label='Upload', color='green'))
tfprint(C.meter(64.3, label='CPU Usage', thresholds={0.0: 'green', 0.6: 'yellow', 0.8: 'red'}))
tfprint(C.spinner('Processing', frame=5))

# Data visualization
tfprint(C.table(
    headers=['Name', 'Score', 'Status'],
    rows=[['Alice', 95, '[green]Pass[reset]'], ['Bob', 87, '[yellow]Warn[reset]'], ['Carol', 92, '[green]Pass[reset]']]
))
tfprint(C.chart.bar_chart({'Jan': 10, 'Feb': 15, 'Mar': 12, 'Apr': 18}))
tfprint(C.code_block('def hello():\n    return "world"', language='python'))

# Containers and layout
tfprint(C.box('Important message', width=40, symbol_style='double'))
tfprint(C.card('User Profile', {'Name': 'John Doe', 'Role': 'Admin', 'Status': 'Active'}))
tfprint(C.columns([['Left column content'], ['Right column content']]))
tfprint(C.grid([['A1', 'B1'], ['A2', 'B2']], cell_width=10))

# Navigation
tfprint(C.breadcrumbs(['Home', 'Settings', 'Profile']))
tfprint(C.tabs(['Overview', 'Details', 'Settings'], active_index=1))

# Feedback
tfprint(C.toast('Operation completed', tone='success'))
tfprint(C.alert_banner('System maintenance tonight', tone='warning'))
tfprint(C.console_panel(['INFO: Process started', '[warning]WARN: High memory[reset]', 'INFO: Process completed']))

# Interactive elements (static preview)
tfprint(C.checkbox('Enable notifications', checked=True))
tfprint(C.radio('Option A', selected=True))
tfprint(C.slider('Volume', value=75, width=20))
```

## Dashboards and templates

```python
from textforge.templates.dashboard import Dashboard

Dashboard.system_overview()
```

## Business templates

```python
from textforge.templates.business import Business

# Report card with key-value pairs
Business.report_card(
    title="Monthly Report",
    items={"Revenue": "$45,230", "Users": "1,247", "Conversion": "3.2%"}
)

# Invoice with line items
Business.invoice(
    company="Acme Corp",
    invoice_no="INV-2024-001",
    rows=[
        ("Web Development", 40, 125.00),
        ("Design Services", 20, 150.00),
        ("Hosting Setup", 1, 299.99)
    ]
)

# KPI summary
Business.kpi_summary({
    "Revenue": 45230.50,
    "Growth": 12.5,
    "Satisfaction": 4.7
})
```

## Game templates

```python
from textforge.templates.game import Template

# Game header with banner
Template.game_header(
    title="Epic Quest",
    subtitle="A terminal-based RPG adventure",
    width=70
)

# Combat status display
Template.combat_status(
    player_hp=85, player_max_hp=100,
    enemy_name="Dragon", enemy_hp=45, enemy_max_hp=200,
    width=70
)

# Inventory screen
Template.inventory_screen([
    {"name": "Iron Sword", "quantity": 1, "rarity": "common"},
    {"name": "Health Potion", "quantity": 5, "rarity": "uncommon"},
    {"name": "Dragon Scale", "quantity": 1, "rarity": "legendary"}
])

# Character sheet
Template.character_sheet(
    name="Hero",
    level=15,
    stats={"STR": 18, "DEX": 14, "INT": 12}
)

# Loading screen
Template.loading_screen(
    message="Loading game world...",
    progress=75,
    total=100
)
```

## CLI app templates

```python
from textforge.templates.cli_apps import CliApps

# Setup wizard
CliApps.wizard_setup(width=80)

# Deployment dashboard
CliApps.deploy_dashboard(width=80)
```

## Exporters

```python
from textforge.export import render_to_html, render_to_ansi, render_to_svg, render_to_pdf

# HTML export with styling
html = render_to_html('[accent]Hello[reset]')
print(html)  # <span style="color: #ffaa32">Hello</span>

# ANSI escape sequences
ansi = render_to_ansi('[warning]Caution[reset]')

# SVG vector graphics
svg = render_to_svg('[bold]Vector Text[reset]')

# PDF document generation
pdf_bytes = render_to_pdf('[primary]PDF Content[reset]')
```

## Live and interactive features

```python
import time
from textforge import Console, tfprint
from textforge import components as C

# Live updating display
def create_monitor_frame(cpu_usage: int, memory_usage: int):
    return [
        C.title("SYSTEM MONITOR", width=60),
        C.progress_bar(cpu_usage, 100, label="CPU", color="warning"),
        C.progress_bar(memory_usage, 100, label="Memory", color="info"),
    ]

console = Console.for_stdout()
with console.live() as live:
    for i in range(10):
        cpu = 20 + i * 5  # Simulate increasing CPU usage
        mem = 30 + i * 3  # Simulate increasing memory usage
        live.update(create_monitor_frame(cpu, mem))
        time.sleep(0.5)

# Interactive menu
choice = C.Menu.run(
    title="Main Menu",
    options=["Start Game", "Settings", "High Scores", "Quit"],
    timeout=0.1  # Auto-select after timeout
)
tfprint(f"Selected: {choice}")
```

## Accessibility metadata

```python
from textforge.utils.accessibility import AccessibilityAPI

AccessibilityAPI.clear()
AccessibilityAPI.annotate_progress('deploy', label='Deployment', current=4, maximum=5, unit='steps')
print(AccessibilityAPI.describe_registry())
```

## CLI usage

```bash
# Create a new Textforge project
textforge new my-app

# Run demo presets
textforge demo --preset full          # Complete component showcase
textforge demo --preset dashboard     # System monitoring dashboard
textforge demo --preset game-header   # Game UI header
textforge demo --preset inventory     # Game inventory screen

# Export content to different formats
echo "[bold]Hello World[reset]" | textforge export -f html > output.html
echo "[primary]Terminal content[reset]" | textforge export -f svg > output.svg
echo "[accent]Content[reset]" | textforge export -f pdf -o output.pdf

# Use the DSL for quick prototyping
textforge dsl "Title text='My App' width=50"
textforge dsl "ProgressBar value=75 total=100 label='Progress'"

# Preview markdown files
textforge preview README.md

# List available components
textforge list

# Benchmark performance
textforge bench
```

## Complete showcase

```python
from textforge.templates.showcase import Showcase

# Display all components in a comprehensive demo
Showcase.full(width=100)
```
