"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('create HTTPS redirect', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        recordNames: ['foo.example.com', 'baz.example.com'],
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: ['foo.example.com', 'baz.example.com'],
            DefaultRootObject: '',
        },
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Name: 'foo.example.com.',
        HostedZoneId: 'ID',
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Name: 'baz.example.com.',
        HostedZoneId: 'ID',
    });
});
test('create HTTPS redirect for apex', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https',
            },
        },
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Name: 'example.com.',
    });
});
test('create HTTPS redirect with existing cert', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        recordNames: ['foo.example.com'],
        certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        }),
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            ViewerCertificate: {
                AcmCertificateArn: 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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