#!/usr/bin/env python
'''
Diffraction functions require for fitting and analyzing data.

mkak 2017.02.06 (originally written spring 2016)
'''

##########################################################################
# IMPORT PYTHON PACKAGES

import time
import numpy as np
import os
import re
import math
from io import StringIO
from collections import namedtuple

from xraydb import f0, f1_chantler, f2_chantler

from .xrd_tools import (generate_hkl, qv_from_hkl, d_from_hkl, q_from_d,
                        twth_from_d, d_from_q, twth_from_q, E_from_lambda,
                        lambda_from_E)

from ..utils.physical_constants import PI
from ..math import index_nearest

HAS_CifFile = False
try:
    import CifFile
    HAS_CifFile = True
except ImportError:
    pass


StructureFactor = namedtuple('StructureFactor', ('q', 'intensity', 'hkl',
                                                 'twotheta', 'd',
                                                 'wavelength', 'energy',
                                                 'f2hkl', 'degen', 'lorentz'))


##########################################################################
# GLOBAL CONSTANTS

elem_symbol = ['H', 'He', 'Li', 'Be', 'B', 'C', 'N', 'O', 'F', 'Ne', 'Na', 'Mg', 'Al',
               'Si', 'P', 'S', 'Cl', 'Ar', 'K', 'Ca', 'Sc', 'Ti', 'V', 'Cr', 'Mn', 'Fe',
               'Co', 'Ni', 'Cu', 'Zn', 'Ga', 'Ge', 'As', 'Se', 'Br', 'Kr', 'Rb', 'Sr',
               'Y', 'Zr', 'Nb', 'Mo', 'Tc', 'Ru', 'Rh', 'Pd', 'Ag', 'Cd', 'In', 'Sn',
               'Sb', 'Te', 'I', 'Xe', 'Cs', 'Ba', 'La', 'Ce', 'Pr', 'Nd', 'Pm', 'Sm',
               'Eu', 'Gd', 'Tb', 'Dy', 'Ho', 'Er', 'Tm', 'Yb', 'Lu', 'Hf', 'Ta', 'W',
               'Re', 'Os', 'Ir', 'Pt', 'Au', 'Hg', 'Tl', 'Pb', 'Bi', 'Po', 'At', 'Rn',
               'Fr', 'Ra', 'Ac', 'Th', 'Pa', 'U', 'Np', 'Pu', 'Am', 'Cm', 'Bk', 'Cf',
               'Es', 'Fm', 'Md', 'No', 'Lr', 'Rf', 'Db', 'Sg', 'Bh', 'Hs', 'Mt', 'Ds',
               'Rg', 'Uub', 'Uut', 'Uuq', 'Uup', 'Uuh', 'Uus', 'Uuo']

##   1 -   2 : Triclinic
##   3 -  15 : Monoclinic
##  16 -  74 : Orthorhombic
##  75 - 142 : Tetragonal
## 143 - 167 : Trigonal
## 168 - 194 : Hexagonal
## 195 - 230 : Cubic

## data reference: http://www.cryst.ehu.es/cgi-bin/cryst/programs/nph-wp-list
SPGRP_SYMM = {'1': {'1a': [['x', 'y', 'z']]},
              '2': {'1a': [[0., 0., 0]],
                    '1b': [[0., 0., 1./2]],
                    '1c': [[0., 1./2, 0]],
                    '1d': [[1./2, 0., 0]],
                    '1e': [[1./2, 1./2, 0]],
                    '1f': [[1./2, 0., 1./2]],
                    '1g': [[0., 1./2, 1./2]],
                    '1h': [[1./2, 1./2, 1./2]],
                    '2i': [['x', 'y', 'z'],['-x', '-y', '-z']]},
              '3:b': {'1a': [[0., 'y', 0]],
                      '1b': [[0., 'y', 1./2]],
                      '1c': [[1./2, 'y', 0]],
                      '1d': [[1./2, 'y', 1./2]],
                      '2e': [['x', 'y', 'z'],['-x', 'y', '-z']]},
              '3:c': {'1a': [[0., 0., 'z']],
                      '1b': [[1./2, 0., 'z']],
                      '1c': [[0., 1./2, 'z']],
                      '1d': [[1./2, 1./2, 'z']],
                      '2e': [['x', 'y', 'z'],['-x', '-y', 'z']]},
              '4:b': {'2a': [['x', 'y', 'z'],['-x', 'y+1./2', '-z']]},
              '4:c': {'2a': [['x', 'y', 'z'],['-x', '-y', 'z+1./2']]},
              '5:b': {'2a': [[0., 'y', 0],[1./2, 'y+1./2', 0]],
                      '2b': [[0., 'y', 1./2],[1./2, 'y+1./2', 1./2]],
                      '4c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z']]},
              '5:c': {'2a': [[0., 0., 'z'],[0., 1./2, 'z+1./2']],
                      '2b': [[1./2, 0., 'z'],[1./2, 1./2, 'z+1./2']],
                      '4c': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2']]},
              '6:b': {'1a': [['x', 0., 'z']],
                      '1b': [['x', 1./2, 'z']],
                      '2c': [['x', 'y', 'z'],['x', '-y', 'z']]},
              '6:c': {'1a': [['x', 'y', 0]],
                      '1b': [['x', 'y', 1./2]],
                      '2c': [['x', 'y', 'z'],['x', 'y', '-z']]},
              '7:b': {'2a': [['x', 'y', 'z'],['x', '-y', 'z+1./2']]},
              '7:c': {'2a': [['x', 'y', 'z'],['x+1./2', 'y', '-z']]},
              '8:b': {'2a': [['x', 0., 'z'],['x+1./2', 1./2, 'z']],
                      '4b': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z']]},
              '8:c': {'2a': [['x', 'y', 0],['x', 'y+1./2', 1./2]],
                      '4b': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x', 'y', '-z'],['x', 'y+1./2', '-z+1./2']]},
              '9:b': {'4a': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2']]},
              '9:c': {'4a': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2']]},
              '10:b': {'1a': [[0., 0., 0]],
                       '1b': [[0., 1./2, 0]],
                       '1c': [[0., 0., 1./2]],
                       '1d': [[1./2, 0., 0]],
                       '1e': [[1./2, 1./2, 0]],
                       '1f': [[0., 1./2, 1./2]],
                       '1g': [[1./2, 0., 1./2]],
                       '1h': [[1./2, 1./2, 1./2]],
                       '2i': [[0., 'y', 0],[0., '-y', 0]],
                       '2j': [[1./2, 'y', 0],[1./2, '-y', 0]],
                       '2k': [[0., 'y', 1./2],[0., '-y', 1./2]],
                       '2l': [[1./2, 'y', 1./2],[1./2, '-y', 1./2]],
                       '2m': [['x', 0., 'z'],['-x', 0., '-z']],
                       '2n': [['x', 1./2, 'z'],['-x', 1./2, '-z']],
                       '4o': [['x', 'y', 'z'],['-x', 'y', '-z'],['-x', '-y', '-z'],['x', '-y', 'z']]},
              '10:c': {'1a': [[0., 0., 0]],
                       '1b': [[0., 0., 1./2]],
                       '1c': [[1./2, 0., 0]],
                       '1d': [[0., 1./2, 0]],
                       '1e': [[0., 1./2, 1./2]],
                       '1f': [[1./2, 0., 1./2]],
                       '1g': [[1./2, 1./2, 0]],
                       '1h': [[1./2, 1./2, 1./2]],
                       '2i': [[0., 0., 'z'],[0., 0., '-z']],
                       '2j': [[0., 1./2, 'z'],[0., 1./2, '-z']],
                       '2k': [[1./2, 0., 'z'],[1./2, 0., '-z']],
                       '2l': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                       '2m': [['x', 'y', 0],['-x', '-y', 0]],
                       '2n': [['x', 'y', 1./2],['-x', '-y', 1./2]],
                       '4o': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', '-y', '-z'],['x', 'y', '-z']]},
              '11:b': {'2a': [[0., 0., 0],[0., 1./2, 0]],
                       '2b': [[1./2, 0., 0],[1./2, 1./2, 0]],
                       '2c': [[0., 0., 1./2],[0., 1./2, 1./2]],
                       '2d': [[1./2, 0., 1./2],[1./2, 1./2, 1./2]],
                       '2e': [['x', 1./4, 'z'],['-x', 3./4, '-z']],
                       '4f': [['x', 'y', 'z'],['-x', 'y+1./2', '-z'],['-x', '-y', '-z'],['x', '-y+1./2', 'z']]},
              '11:c': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                       '2b': [[0., 1./2, 0],[0., 1./2, 1./2]],
                       '2c': [[1./2, 0., 0],[1./2, 0., 1./2]],
                       '2d': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                       '2e': [['x', 'y', 1./4],['-x', '-y', 3./4]],
                       '4f': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z+1./2']]},
              '12:b': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                       '2b': [[0., 1./2, 0],[1./2, 0., 0]],
                       '2c': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                       '2d': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                       '4e': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 0]],
                       '4f': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                       '4g': [[0., 'y', 0],[1./2, 'y+1./2', 0],[0., '-y', 0],[1./2, '-y+1./2', 0]],
                       '4h': [[0., 'y', 1./2],[1./2, 'y+1./2', 1./2],[0., '-y', 1./2],[1./2, '-y+1./2', 1./2]],
                       '4i': [['x', 0., 'z'],['x+1./2', 1./2, 'z'],['-x', 0., '-z'],['-x+1./2', 1./2, '-z']],
                       '8j': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z']]},
              '12:c': {'2a': [[0., 0., 0],[0., 1./2, 1./2]],
                       '2b': [[0., 0., 1./2],[0., 1./2, 0]],
                       '2c': [[1./2, 0., 0],[1./2, 1./2, 1./2]],
                       '2d': [[1./2, 0., 1./2],[1./2, 1./2, 0]],
                       '4e': [[0., 1./4, 1./4],[0., 3./4, 3./4],[0., 3./4, 1./4],[0., 1./4, 3./4]],
                       '4f': [[1./2, 1./4, 1./4],[1./2, 3./4, 3./4],[1./2, 3./4, 1./4],[1./2, 1./4, 3./4]],
                       '4g': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[0., 0., '-z'],[0., 1./2, '-z+1./2']],
                       '4h': [[1./2, 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 0., '-z'],[1./2, 1./2, '-z+1./2']],
                       '4i': [['x', 'y', 0],['x', 'y+1./2', 1./2],['-x', '-y', 0],['-x', '-y+1./2', 1./2]],
                       '8j': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x', 'y+1./2', '-z+1./2']]},
              '13:b': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                       '2b': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                       '2c': [[0., 1./2, 0],[0., 1./2, 1./2]],
                       '2d': [[1./2, 0., 0],[1./2, 0., 1./2]],
                       '2e': [[0., 'y', 1./4],[0., '-y', 3./4]],
                       '2f': [[1./2, 'y', 1./4],[1./2, '-y', 3./4]],
                       '4g': [['x', 'y', 'z'],['-x', 'y', '-z+1./2'],['-x', '-y', '-z'],['x', '-y', 'z+1./2']]},
              '13:c': {'2a': [[0., 0., 0],[1./2, 0., 0]],
                       '2b': [[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                       '2c': [[0., 0., 1./2],[1./2, 0., 1./2]],
                       '2d': [[0., 1./2, 0],[1./2, 1./2, 0]],
                       '2e': [[1./4, 0., 'z'],[3./4, 0., '-z']],
                       '2f': [[1./4, 1./2, 'z'],[3./4, 1./2, '-z']],
                       '4g': [['x', 'y', 'z'],['-x+1./2', '-y', 'z'],['-x', '-y', '-z'],['x+1./2', 'y', '-z']]},
              '14:b': {'2a': [[0., 0., 0],[0., 1./2, 1./2]],
                       '2b': [[1./2, 0., 0],[1./2, 1./2, 1./2]],
                       '2c': [[0., 0., 1./2],[0., 1./2, 0]],
                       '2d': [[1./2, 0., 1./2],[1./2, 1./2, 0]],
                       '4e': [['x', 'y', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x', '-y', '-z'],['x', '-y+1./2', 'z+1./2']]},
              '14:c': {'2a': [[0., 0., 0],[1./2, 0., 1./2]],
                       '2b': [[0., 1./2, 0],[1./2, 1./2, 1./2]],
                       '2c': [[1./2, 0., 0],[0., 0., 1./2]],
                       '2d': [[1./2, 1./2, 0],[0., 1./2, 1./2]],
                       '4e': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y', '-z+1./2']]},
              '15:b': {'4a': [[0., 0., 0],[1./2, 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 1./2]],
                       '4b': [[0., 1./2, 0],[1./2, 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                       '4c': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                       '4d': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 0],[1./4, 3./4, 0]],
                       '4e': [[0., 'y', 1./4],[1./2, 'y+1./2', 1./4],[0., '-y', 3./4],[1./2, '-y+1./2', 3./4]],
                       '8f': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2']]},
              '15:c': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 1./2, 1./2]],
                       '4b': [[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                       '4c': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 3./4, 1./4],[1./2, 1./4, 3./4]],
                       '4d': [[1./2, 1./4, 1./4],[1./2, 3./4, 3./4],[0., 3./4, 1./4],[0., 1./4, 3./4]],
                       '4e': [[1./4, 0., 'z'],[1./4, 1./2, 'z+1./2'],[3./4, 0., '-z'],[3./4, 1./2, '-z+1./2']],
                       '8f': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2']]},
              '16': {'1a': [[0., 0., 0]],
                     '1b': [[1./2, 0., 0]],
                     '1c': [[0., 1./2, 0]],
                     '1d': [[0., 0., 1./2]],
                     '1e': [[1./2, 1./2, 0]],
                     '1f': [[1./2, 0., 1./2]],
                     '1g': [[0., 1./2, 1./2]],
                     '1h': [[1./2, 1./2, 1./2]],
                     '2i': [['x', 0., 0],['-x', 0., 0]],
                     '2j': [['x', 0., 1./2],['-x', 0., 1./2]],
                     '2k': [['x', 1./2, 0],['-x', 1./2, 0]],
                     '2l': [['x', 1./2, 1./2],['-x', 1./2, 1./2]],
                     '2m': [[0., 'y', 0],[0., '-y', 0]],
                     '2n': [[0., 'y', 1./2],[0., '-y', 1./2]],
                     '2o': [[1./2, 'y', 0],[1./2, '-y', 0]],
                     '2p': [[1./2, 'y', 1./2],[1./2, '-y', 1./2]],
                     '2q': [[0., 0., 'z'],[0., 0., '-z']],
                     '2r': [[1./2, 0., 'z'],[1./2, 0., '-z']],
                     '2s': [[0., 1./2, 'z'],[0., 1./2, '-z']],
                     '2t': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                     '4u': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z']]},
              '17': {'2a': [['x', 0., 0],['-x', 0., 1./2]],
                     '2b': [['x', 1./2, 0],['-x', 1./2, 1./2]],
                     '2c': [[0., 'y', 1./4],[0., '-y', 3./4]],
                     '2d': [[1./2, 'y', 1./4],[1./2, '-y', 3./4]],
                     '4e': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-x', 'y', '-z+1./2'],['x', '-y', '-z']]},
              '18': {'2a': [[0., 0., 'z'],[1./2, 1./2, '-z']],
                     '2b': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                     '4c': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z']]},
              '19': {'4a': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z']]},
              '20': {'4a': [['x', 0., 0],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2]],
                     '4b': [[0., 'y', 1./4],[1./2, 'y+1./2', 1./4],[0., '-y', 3./4],[1./2, '-y+1./2', 3./4]],
                     '8c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z']]},
              '21': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                     '2b': [[0., 1./2, 0],[1./2, 0., 0]],
                     '2c': [[1./2, 0., 1./2],[0., 1./2, 1./2]],
                     '2d': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '4e': [['x', 0., 0],['x+1./2', 1./2, 0],['-x', 0., 0],['-x+1./2', 1./2, 0]],
                     '4f': [['x', 0., 1./2],['x+1./2', 1./2, 1./2],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2]],
                     '4g': [[0., 'y', 0],[1./2, 'y+1./2', 0],[0., '-y', 0],[1./2, '-y+1./2', 0]],
                     '4h': [[0., 'y', 1./2],[1./2, 'y+1./2', 1./2],[0., '-y', 1./2],[1./2, '-y+1./2', 1./2]],
                     '4i': [[0., 0., 'z'],[1./2, 1./2, 'z'],[0., 0., '-z'],[1./2, 1./2, '-z']],
                     '4j': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z'],[1./2, 0., '-z']],
                     '4k': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z']],
                     '8l': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z']]},
              '22': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                     '4b': [[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 0],[1./2, 1./2, 1./2]],
                     '4c': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                     '4d': [[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                     '8e': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0]],
                     '8f': [[0., 'y', 0],[0., 'y+1./2', 1./2],[1./2, 'y', 1./2],[1./2, 'y+1./2', 0],[0., '-y', 0],[0., '-y+1./2', 1./2],[1./2, '-y', 1./2],[1./2, '-y+1./2', 0]],
                     '8g': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 1./2, 'z'],[0., 0., '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 1./2, '-z']],
                     '8h': [[1./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z+1./2'],[3./4, 3./4, 'z'],[3./4, 1./4, '-z'],[3./4, 3./4, '-z+1./2'],[1./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z']],
                     '8i': [[1./4, 'y', 1./4],[1./4, 'y+1./2', 3./4],[3./4, 'y', 3./4],[3./4, 'y+1./2', 1./4],[3./4, '-y', 1./4],[3./4, '-y+1./2', 3./4],[1./4, '-y', 3./4],[1./4, '-y+1./2', 1./4]],
                     '8j': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4]],
                     '16k': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z']]},
              '23': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[1./2, 0., 0],[0., 1./2, 1./2]],
                     '2c': [[0., 0., 1./2],[1./2, 1./2, 0]],
                     '2d': [[0., 1./2, 0],[1./2, 0., 1./2]],
                     '4e': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2]],
                     '4f': [['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0]],
                     '4g': [[0., 'y', 0],[1./2, 'y+1./2', 1./2],[0., '-y', 0],[1./2, '-y+1./2', 1./2]],
                     '4h': [[1./2, 'y', 0],[0., 'y+1./2', 1./2],[1./2, '-y', 0],[0., '-y+1./2', 1./2]],
                     '4i': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                     '4j': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2']],
                     '8k': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2']]},
              '24': {'4a': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4]],
                     '4b': [[1./4, 'y', 0],[3./4, 'y+1./2', 1./2],[1./4, '-y', 1./2],[3./4, '-y+1./2', 0]],
                     '4c': [[0., 1./4, 'z'],[1./2, 3./4, 'z+1./2'],[0., 3./4, '-z+1./2'],[1./2, 1./4, '-z']],
                     '8d': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2']]},
              '25': {'1a': [[0., 0., 'z']],
                     '1b': [[0., 1./2, 'z']],
                     '1c': [[1./2, 0., 'z']],
                     '1d': [[1./2, 1./2, 'z']],
                     '2e': [['x', 0., 'z'],['-x', 0., 'z']],
                     '2f': [['x', 1./2, 'z'],['-x', 1./2, 'z']],
                     '2g': [[0., 'y', 'z'],[0., '-y', 'z']],
                     '2h': [[1./2, 'y', 'z'],[1./2, '-y', 'z']],
                     '4i': [['x', 'y', 'z'],['-x', '-y', 'z'],['x', '-y', 'z'],['-x', 'y', 'z']]},
              '26': {'2a': [[0., 'y', 'z'],[0., '-y', 'z+1./2']],
                     '2b': [[1./2, 'y', 'z'],[1./2, '-y', 'z+1./2']],
                     '4c': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['x', '-y', 'z+1./2'],['-x', 'y', 'z']]},
              '27': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                     '2b': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2']],
                     '2c': [[1./2, 0., 'z'],[1./2, 0., 'z+1./2']],
                     '2d': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                     '4e': [['x', 'y', 'z'],['-x', '-y', 'z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2']]},
              '28': {'2a': [[0., 0., 'z'],[1./2, 0., 'z']],
                     '2b': [[0., 1./2, 'z'],[1./2, 1./2, 'z']],
                     '2c': [[1./4, 'y', 'z'],[3./4, '-y', 'z']],
                     '4d': [['x', 'y', 'z'],['-x', '-y', 'z'],['x+1./2', '-y', 'z'],['-x+1./2', 'y', 'z']]},
              '29': {'4a': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['x+1./2', '-y', 'z'],['-x+1./2', 'y', 'z+1./2']]},
              '30': {'2a': [[0., 0., 'z'],[0., 1./2, 'z+1./2']],
                     '2b': [[1./2, 0., 'z'],[1./2, 1./2, 'z+1./2']],
                     '4c': [['x', 'y', 'z'],['-x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['-x', 'y+1./2', 'z+1./2']]},
              '31': {'2a': [[0., 'y', 'z'],[1./2, '-y', 'z+1./2']],
                     '4b': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['-x', 'y', 'z']]},
              '32': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z']],
                     '2b': [[0., 1./2, 'z'],[1./2, 0., 'z']],
                     '4c': [['x', 'y', 'z'],['-x', '-y', 'z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '33': {'4a': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '34': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2']],
                     '2b': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2']],
                     '4c': [['x', 'y', 'z'],['-x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '35': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z']],
                     '2b': [[0., 1./2, 'z'],[1./2, 0., 'z']],
                     '4c': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z'],[1./4, 3./4, 'z'],[3./4, 1./4, 'z']],
                     '4d': [['x', 0., 'z'],['x+1./2', 1./2, 'z'],['-x', 0., 'z'],['-x+1./2', 1./2, 'z']],
                     '4e': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z']],
                     '8f': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '36': {'4a': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z+1./2'],[1./2, '-y+1./2', 'z+1./2']],
                     '8b': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '37': {'4a': [[0., 0., 'z'],[1./2, 1./2, 'z'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z+1./2']],
                     '4b': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2']],
                     '4c': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z+1./2']],
                     '8d': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '38': {'2a': [[0., 0., 'z'],[0., 1./2, 'z+1./2']],
                     '2b': [[1./2, 0., 'z'],[1./2, 1./2, 'z+1./2']],
                     '4c': [['x', 0., 'z'],['x', 1./2, 'z+1./2'],['-x', 0., 'z'],['-x', 1./2, 'z+1./2']],
                     '4d': [[0., 'y', 'z'],[0., 'y+1./2', 'z+1./2'],[0., '-y', 'z'],[0., '-y+1./2', 'z+1./2']],
                     '4e': [[1./2, 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[1./2, '-y', 'z'],[1./2, '-y+1./2', 'z+1./2']],
                     '8f': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x', 'y+1./2', 'z+1./2']]},
              '39': {'4a': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, 'z'],[0., 0., 'z+1./2']],
                     '4b': [[1./2, 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, 'z'],[1./2, 0., 'z+1./2']],
                     '4c': [['x', 1./4, 'z'],['x', 3./4, 'z+1./2'],['-x', 3./4, 'z'],['-x', 1./4, 'z+1./2']],
                     '8d': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['x', '-y+1./2', 'z'],['x', '-y', 'z+1./2'],['-x', 'y+1./2', 'z'],['-x', 'y', 'z+1./2']]},
              '40': {'4a': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z'],[1./2, 1./2, 'z+1./2']],
                     '4b': [[1./4, 'y', 'z'],[1./4, 'y+1./2', 'z+1./2'],[3./4, '-y', 'z'],[3./4, '-y+1./2', 'z+1./2']],
                     '8c': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '41': {'4a': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 1./2, 'z'],[1./2, 0., 'z+1./2']],
                     '8b': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['x+1./2', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-x+1./2', 'y', 'z+1./2']]},
              '42': {'4a': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 1./2, 'z']],
                     '8b': [[1./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z+1./2'],[3./4, 3./4, 'z'],[1./4, 3./4, 'z'],[1./4, 1./4, 'z+1./2'],[3./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z']],
                     '8c': [[0., 'y', 'z'],[0., 'y+1./2', 'z+1./2'],[1./2, 'y', 'z+1./2'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[0., '-y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[1./2, '-y+1./2', 'z']],
                     '8d': [['x', 0., 'z'],['x', 1./2, 'z+1./2'],['x+1./2', 0., 'z+1./2'],['x+1./2', 1./2, 'z'],['-x', 0., 'z'],['-x', 1./2, 'z+1./2'],['-x+1./2', 0., 'z+1./2'],['-x+1./2', 1./2, 'z']],
                     '16e': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x', 'y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z']]},
              '43': {'8a': [[0., 0., 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, 'z+1./2'],[1./2, 1./2, 'z'],[1./4, 1./4, 'z+1./4'],[3./4, 1./4, 'z+3./4'],[1./4, 3./4, 'z+3./4'],[3./4, 3./4, 'z+1./4']],
                     '16b': [['x', 'y', 'z'],['x+1./2', 'y', 'z+1./2'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['x+1./4', '-y+1./4', 'z+1./4'],['x+3./4', '-y+1./4', 'z+3./4'],['x+1./4', '-y+3./4', 'z+3./4'],['x+3./4', '-y+3./4', 'z+1./4'],['-x+1./4', 'y+1./4', 'z+1./4'],['-x+3./4', 'y+1./4', 'z+3./4'],['-x+1./4', 'y+3./4', 'z+3./4'],['-x+3./4', 'y+3./4', 'z+1./4']]},
              '44': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2']],
                     '2b': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2']],
                     '4c': [['x', 0., 'z'],['x+1./2', 1./2, 'z+1./2'],['-x', 0., 'z'],['-x+1./2', 1./2, 'z+1./2']],
                     '4d': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z+1./2']],
                     '8e': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '45': {'4a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, 'z'],[0., 0., 'z+1./2']],
                     '4b': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                     '8c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['x', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-x', 'y', 'z+1./2']]},
              '46': {'4a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                     '4b': [[1./4, 'y', 'z'],[3./4, 'y+1./2', 'z+1./2'],[3./4, '-y', 'z'],[1./4, '-y+1./2', 'z+1./2']],
                     '8c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z'],['-x', 'y+1./2', 'z+1./2']]},
              '47': {'1a': [[0., 0., 0]],
                     '1b': [[1./2, 0., 0]],
                     '1c': [[0., 0., 1./2]],
                     '1d': [[1./2, 0., 1./2]],
                     '1e': [[0., 1./2, 0]],
                     '1f': [[1./2, 1./2, 0]],
                     '1g': [[0., 1./2, 1./2]],
                     '1h': [[1./2, 1./2, 1./2]],
                     '2i': [['x', 0., 0],['-x', 0., 0]],
                     '2j': [['x', 0., 1./2],['-x', 0., 1./2]],
                     '2k': [['x', 1./2, 0],['-x', 1./2, 0]],
                     '2l': [['x', 1./2, 1./2],['-x', 1./2, 1./2]],
                     '2m': [[0., 'y', 0],[0., '-y', 0]],
                     '2n': [[0., 'y', 1./2],[0., '-y', 1./2]],
                     '2o': [[1./2, 'y', 0],[1./2, '-y', 0]],
                     '2p': [[1./2, 'y', 1./2],[1./2, '-y', 1./2]],
                     '2q': [[0., 0., 'z'],[0., 0., '-z']],
                     '2r': [[0., 1./2, 'z'],[0., 1./2, '-z']],
                     '2s': [[1./2, 0., 'z'],[1./2, 0., '-z']],
                     '2t': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                     '4u': [[0., 'y', 'z'],[0., '-y', 'z'],[0., 'y', '-z'],[0., '-y', '-z']],
                     '4v': [[1./2, 'y', 'z'],[1./2, '-y', 'z'],[1./2, 'y', '-z'],[1./2, '-y', '-z']],
                     '4w': [['x', 0., 'z'],['-x', 0., 'z'],['-x', 0., '-z'],['x', 0., '-z']],
                     '4x': [['x', 1./2, 'z'],['-x', 1./2, 'z'],['-x', 1./2, '-z'],['x', 1./2, '-z']],
                     '4y': [['x', 'y', 0],['-x', '-y', 0],['-x', 'y', 0],['x', '-y', 0]],
                     '4z': [['x', 'y', 1./2],['-x', '-y', 1./2],['-x', 'y', 1./2],['x', '-y', 1./2]],
                     '8a': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['-x', '-y', '-z'],['x', 'y', '-z'],['x', '-y', 'z'],['-x', 'y', 'z']]},
              '48:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                       '2b': [[1./2, 0., 0],[0., 1./2, 1./2]],
                       '2c': [[0., 0., 1./2],[1./2, 1./2, 0]],
                       '2d': [[0., 1./2, 0],[1./2, 0., 1./2]],
                       '4e': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4]],
                       '4f': [[3./4, 3./4, 3./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                       '4g': [['x', 0., 0],['-x', 0., 0],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2]],
                       '4h': [['x', 0., 1./2],['-x', 0., 1./2],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0]],
                       '4i': [[0., 'y', 0],[0., '-y', 0],[1./2, '-y+1./2', 1./2],[1./2, 'y+1./2', 1./2]],
                       '4j': [[1./2, 'y', 0],[1./2, '-y', 0],[0., '-y+1./2', 1./2],[0., 'y+1./2', 1./2]],
                       '4k': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, 'z+1./2']],
                       '4l': [[0., 1./2, 'z'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., 'z+1./2']],
                       '8m': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '48:2': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                       '2b': [[3./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                       '2c': [[1./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                       '2d': [[1./4, 3./4, 1./4],[3./4, 1./4, 3./4]],
                       '4e': [[1./2, 1./2, 1./2],[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                       '4f': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                       '4g': [['x', 1./4, 1./4],['-x+1./2', 1./4, 1./4],['-x', 3./4, 3./4],['x+1./2', 3./4, 3./4]],
                       '4h': [['x', 1./4, 3./4],['-x+1./2', 1./4, 3./4],['-x', 3./4, 1./4],['x+1./2', 3./4, 1./4]],
                       '4i': [[1./4, 'y', 1./4],[1./4, '-y+1./2', 1./4],[3./4, '-y', 3./4],[3./4, 'y+1./2', 3./4]],
                       '4j': [[3./4, 'y', 1./4],[3./4, '-y+1./2', 1./4],[1./4, '-y', 3./4],[1./4, 'y+1./2', 3./4]],
                       '4k': [[1./4, 1./4, 'z'],[1./4, 1./4, '-z+1./2'],[3./4, 3./4, '-z'],[3./4, 3./4, 'z+1./2']],
                       '4l': [[1./4, 3./4, 'z'],[1./4, 3./4, '-z+1./2'],[3./4, 1./4, '-z'],[3./4, 1./4, 'z+1./2']],
                       '8m': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2']]},
              '49': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                     '2b': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                     '2c': [[0., 1./2, 0],[0., 1./2, 1./2]],
                     '2d': [[1./2, 0., 0],[1./2, 0., 1./2]],
                     '2e': [[0., 0., 1./4],[0., 0., 3./4]],
                     '2f': [[1./2, 0., 1./4],[1./2, 0., 3./4]],
                     '2g': [[0., 1./2, 1./4],[0., 1./2, 3./4]],
                     '2h': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                     '4i': [['x', 0., 1./4],['-x', 0., 1./4],['-x', 0., 3./4],['x', 0., 3./4]],
                     '4j': [['x', 1./2, 1./4],['-x', 1./2, 1./4],['-x', 1./2, 3./4],['x', 1./2, 3./4]],
                     '4k': [[0., 'y', 1./4],[0., '-y', 1./4],[0., '-y', 3./4],[0., 'y', 3./4]],
                     '4l': [[1./2, 'y', 1./4],[1./2, '-y', 1./4],[1./2, '-y', 3./4],[1./2, 'y', 3./4]],
                     '4m': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2']],
                     '4n': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, 'z+1./2']],
                     '4o': [[0., 1./2, 'z'],[0., 1./2, '-z+1./2'],[0., 1./2, '-z'],[0., 1./2, 'z+1./2']],
                     '4p': [[1./2, 0., 'z'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[1./2, 0., 'z+1./2']],
                     '4q': [['x', 'y', 0],['-x', '-y', 0],['-x', 'y', 1./2],['x', '-y', 1./2]],
                     '8r': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z+1./2'],['x', '-y', '-z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2']]},
              '50:1': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                       '2b': [[1./2, 0., 0],[0., 1./2, 0]],
                       '2c': [[1./2, 0., 1./2],[0., 1./2, 1./2]],
                       '2d': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                       '4e': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 0]],
                       '4f': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                       '4g': [['x', 0., 0],['-x', 0., 0],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0]],
                       '4h': [['x', 0., 1./2],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2]],
                       '4i': [[0., 'y', 0],[0., '-y', 0],[1./2, '-y+1./2', 0],[1./2, 'y+1./2', 0]],
                       '4j': [[0., 'y', 1./2],[0., '-y', 1./2],[1./2, '-y+1./2', 1./2],[1./2, 'y+1./2', 1./2]],
                       '4k': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, '-z'],[1./2, 1./2, 'z']],
                       '4l': [[0., 1./2, 'z'],[0., 1./2, '-z'],[1./2, 0., '-z'],[1./2, 0., 'z']],
                       '8m': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '50:2': {'2a': [[1./4, 1./4, 0],[3./4, 3./4, 0]],
                       '2b': [[3./4, 1./4, 0],[1./4, 3./4, 0]],
                       '2c': [[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                       '2d': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2]],
                       '4e': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0]],
                       '4f': [[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                       '4g': [['x', 1./4, 0],['-x+1./2', 1./4, 0],['-x', 3./4, 0],['x+1./2', 3./4, 0]],
                       '4h': [['x', 1./4, 1./2],['-x+1./2', 1./4, 1./2],['-x', 3./4, 1./2],['x+1./2', 3./4, 1./2]],
                       '4i': [[1./4, 'y', 0],[1./4, '-y+1./2', 0],[3./4, '-y', 0],[3./4, 'y+1./2', 0]],
                       '4j': [[1./4, 'y', 1./2],[1./4, '-y+1./2', 1./2],[3./4, '-y', 1./2],[3./4, 'y+1./2', 1./2]],
                       '4k': [[1./4, 1./4, 'z'],[1./4, 1./4, '-z'],[3./4, 3./4, '-z'],[3./4, 3./4, 'z']],
                       '4l': [[1./4, 3./4, 'z'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z'],[3./4, 1./4, 'z']],
                       '8m': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y', '-z'],['x', '-y+1./2', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x+1./2', '-y', 'z'],['-x', 'y+1./2', 'z']]},
              '51': {'2a': [[0., 0., 0],[1./2, 0., 0]],
                     '2b': [[0., 1./2, 0],[1./2, 1./2, 0]],
                     '2c': [[0., 0., 1./2],[1./2, 0., 1./2]],
                     '2d': [[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                     '2e': [[1./4, 0., 'z'],[3./4, 0., '-z']],
                     '2f': [[1./4, 1./2, 'z'],[3./4, 1./2, '-z']],
                     '4g': [[0., 'y', 0],[1./2, '-y', 0],[0., '-y', 0],[1./2, 'y', 0]],
                     '4h': [[0., 'y', 1./2],[1./2, '-y', 1./2],[0., '-y', 1./2],[1./2, 'y', 1./2]],
                     '4i': [['x', 0., 'z'],['-x+1./2', 0., 'z'],['-x', 0., '-z'],['x+1./2', 0., '-z']],
                     '4j': [['x', 1./2, 'z'],['-x+1./2', 1./2, 'z'],['-x', 1./2, '-z'],['x+1./2', 1./2, '-z']],
                     '4k': [[1./4, 'y', 'z'],[1./4, '-y', 'z'],[3./4, 'y', '-z'],[3./4, '-y', '-z']],
                     '8l': [['x', 'y', 'z'],['-x+1./2', '-y', 'z'],['-x', 'y', '-z'],['x+1./2', '-y', '-z'],['-x', '-y', '-z'],['x+1./2', 'y', '-z'],['x', '-y', 'z'],['-x+1./2', 'y', 'z']]},
              '52': {'4a': [[0., 0., 0],[1./2, 0., 0],[1./2, 1./2, 1./2],[0., 1./2, 1./2]],
                     '4b': [[0., 0., 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 0]],
                     '4c': [[1./4, 0., 'z'],[1./4, 1./2, '-z+1./2'],[3./4, 0., '-z'],[3./4, 1./2, 'z+1./2']],
                     '4d': [['x', 1./4, 1./4],['-x+1./2', 3./4, 1./4],['-x', 3./4, 3./4],['x+1./2', 1./4, 3./4]],
                     '8e': [['x', 'y', 'z'],['-x+1./2', '-y', 'z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y+1./2', 'z+1./2']]},
              '53': {'2a': [[0., 0., 0],[1./2, 0., 1./2]],
                     '2b': [[1./2, 0., 0],[0., 0., 1./2]],
                     '2c': [[1./2, 1./2, 0],[0., 1./2, 1./2]],
                     '2d': [[0., 1./2, 0],[1./2, 1./2, 1./2]],
                     '4e': [['x', 0., 0],['-x+1./2', 0., 1./2],['-x', 0., 0],['x+1./2', 0., 1./2]],
                     '4f': [['x', 1./2, 0],['-x+1./2', 1./2, 1./2],['-x', 1./2, 0],['x+1./2', 1./2, 1./2]],
                     '4g': [[1./4, 'y', 1./4],[1./4, '-y', 3./4],[3./4, '-y', 3./4],[3./4, 'y', 1./4]],
                     '4h': [[0., 'y', 'z'],[1./2, '-y', 'z+1./2'],[1./2, 'y', '-z+1./2'],[0., '-y', '-z']],
                     '8i': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', 'y', '-z+1./2'],['x', '-y', '-z'],['-x', '-y', '-z'],['x+1./2', 'y', '-z+1./2'],['x+1./2', '-y', 'z+1./2'],['-x', 'y', 'z']]},
              '54': {'4a': [[0., 0., 0],[1./2, 0., 0],[0., 0., 1./2],[1./2, 0., 1./2]],
                     '4b': [[0., 1./2, 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                     '4c': [[0., 'y', 1./4],[1./2, '-y', 1./4],[0., '-y', 3./4],[1./2, 'y', 3./4]],
                     '4d': [[1./4, 0., 'z'],[3./4, 0., '-z+1./2'],[3./4, 0., '-z'],[1./4, 0., 'z+1./2']],
                     '4e': [[1./4, 1./2, 'z'],[3./4, 1./2, '-z+1./2'],[3./4, 1./2, '-z'],[1./4, 1./2, 'z+1./2']],
                     '8f': [['x', 'y', 'z'],['-x+1./2', '-y', 'z'],['-x', 'y', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y', '-z'],['x', '-y', 'z+1./2'],['-x+1./2', 'y', 'z+1./2']]},
              '55': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '2c': [[0., 1./2, 0],[1./2, 0., 0]],
                     '2d': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '4e': [[0., 0., 'z'],[1./2, 1./2, '-z'],[0., 0., '-z'],[1./2, 1./2, 'z']],
                     '4f': [[0., 1./2, 'z'],[1./2, 0., '-z'],[0., 1./2, '-z'],[1./2, 0., 'z']],
                     '4g': [['x', 'y', 0],['-x', '-y', 0],['-x+1./2', 'y+1./2', 0],['x+1./2', '-y+1./2', 0]],
                     '4h': [['x', 'y', 1./2],['-x', '-y', 1./2],['-x+1./2', 'y+1./2', 1./2],['x+1./2', '-y+1./2', 1./2]],
                     '8i': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['x', 'y', '-z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '56': {'4a': [[0., 0., 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '4b': [[0., 0., 1./2],[1./2, 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                     '4c': [[1./4, 1./4, 'z'],[3./4, 3./4, '-z+1./2'],[3./4, 3./4, '-z'],[1./4, 1./4, 'z+1./2']],
                     '4d': [[1./4, 3./4, 'z'],[3./4, 1./4, '-z+1./2'],[3./4, 1./4, '-z'],[1./4, 3./4, 'z+1./2']],
                     '8e': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x', '-y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2']]},
              '57': {'4a': [[0., 0., 0],[0., 0., 1./2],[0., 1./2, 1./2],[0., 1./2, 0]],
                     '4b': [[1./2, 0., 0],[1./2, 0., 1./2],[1./2, 1./2, 1./2],[1./2, 1./2, 0]],
                     '4c': [['x', 1./4, 0],['-x', 3./4, 1./2],['-x', 3./4, 0],['x', 1./4, 1./2]],
                     '4d': [['x', 'y', 1./4],['-x', '-y', 3./4],['-x', 'y+1./2', 1./4],['x', '-y+1./2', 3./4]],
                     '8e': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x', '-y+1./2', '-z'],['-x', '-y', '-z'],['x', 'y', '-z+1./2'],['x', '-y+1./2', 'z+1./2'],['-x', 'y+1./2', 'z']]},
              '58': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                     '2c': [[0., 1./2, 0],[1./2, 0., 1./2]],
                     '2d': [[0., 1./2, 1./2],[1./2, 0., 0]],
                     '4e': [[0., 0., 'z'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z'],[1./2, 1./2, 'z+1./2']],
                     '4f': [[0., 1./2, 'z'],[1./2, 0., '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., 'z+1./2']],
                     '4g': [['x', 'y', 0],['-x', '-y', 0],['-x+1./2', 'y+1./2', 1./2],['x+1./2', '-y+1./2', 1./2]],
                     '8h': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '59:1': {'2a': [[0., 0., 'z'],[1./2, 1./2, '-z']],
                       '2b': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                       '4c': [[1./4, 1./4, 0],[3./4, 3./4, 0],[1./4, 3./4, 0],[3./4, 1./4, 0]],
                       '4d': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2]],
                       '4e': [[0., 'y', 'z'],[0., '-y', 'z'],[1./2, 'y+1./2', '-z'],[1./2, '-y+1./2', '-z']],
                       '4f': [['x', 0., 'z'],['-x', 0., 'z'],['-x+1./2', 1./2, '-z'],['x+1./2', 1./2, '-z']],
                       '8g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z'],['-x', 'y', 'z']]},
              '59:2': {'2a': [[1./4, 1./4, 'z'],[3./4, 3./4, '-z']],
                       '2b': [[1./4, 3./4, 'z'],[3./4, 1./4, '-z']],
                       '4c': [[0., 0., 0],[1./2, 1./2, 0],[0., 1./2, 0],[1./2, 0., 0]],
                       '4d': [[0., 0., 1./2],[1./2, 1./2, 1./2],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                       '4e': [[1./4, 'y', 'z'],[1./4, '-y+1./2', 'z'],[3./4, 'y+1./2', '-z'],[3./4, '-y', '-z']],
                       '4f': [['x', 1./4, 'z'],['-x+1./2', 1./4, 'z'],['-x', 3./4, '-z'],['x+1./2', 3./4, '-z']],
                       '8g': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y+1./2', '-z'],['x+1./2', '-y', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x', '-y+1./2', 'z'],['-x+1./2', 'y', 'z']]},
              '60': {'4a': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 0]],
                     '4b': [[0., 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 0., 0]],
                     '4c': [[0., 'y', 1./4],[1./2, '-y+1./2', 3./4],[0., '-y', 3./4],[1./2, 'y+1./2', 1./4]],
                     '8d': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z']]},
              '61': {'4a': [[0., 0., 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 0]],
                     '4b': [[0., 0., 1./2],[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 1./2]],
                     '8c': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z']]},
              '62': {'4a': [[0., 0., 0],[1./2, 0., 1./2],[0., 1./2, 0],[1./2, 1./2, 1./2]],
                     '4b': [[0., 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2],[1./2, 1./2, 0]],
                     '4c': [['x', 1./4, 'z'],['-x+1./2', 3./4, 'z+1./2'],['-x', 3./4, '-z'],['x+1./2', 1./4, '-z+1./2']],
                     '8d': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '63': {'4a': [[0., 0., 0],[1./2, 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '4b': [[0., 1./2, 0],[1./2, 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '4c': [[0., 'y', 1./4],[1./2, 'y+1./2', 1./4],[0., '-y', 3./4],[1./2, '-y+1./2', 3./4]],
                     '8d': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2],[1./4, 3./4, 0],[3./4, 1./4, 0]],
                     '8e': [['x', 0., 0],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 0],['x', 0., 1./2],['x+1./2', 1./2, 1./2]],
                     '8f': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z+1./2'],[1./2, '-y+1./2', 'z+1./2'],[0., 'y', '-z+1./2'],[1./2, 'y+1./2', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z']],
                     '8g': [['x', 'y', 1./4],['x+1./2', 'y+1./2', 1./4],['-x', '-y', 3./4],['-x+1./2', '-y+1./2', 3./4],['-x', 'y', 1./4],['-x+1./2', 'y+1./2', 1./4],['x', '-y', 3./4],['x+1./2', '-y+1./2', 3./4]],
                     '16h': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '64': {'4a': [[0., 0., 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '4b': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 1./2],[0., 0., 1./2]],
                     '8c': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0],[3./4, 1./4, 0]],
                     '8d': [['x', 0., 0],['x+1./2', 1./2, 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x', 0., 0],['-x+1./2', 1./2, 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2]],
                     '8e': [[1./4, 'y', 1./4],[3./4, 'y+1./2', 1./4],[3./4, '-y+1./2', 3./4],[1./4, '-y', 3./4],[3./4, '-y', 3./4],[1./4, '-y+1./2', 3./4],[1./4, 'y+1./2', 1./4],[3./4, 'y', 1./4]],
                     '8f': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z'],[0., '-y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[0., 'y+1./2', '-z+1./2'],[1./2, 'y', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z']],
                     '16g': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', 'y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '65': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                     '2b': [[1./2, 0., 0],[0., 1./2, 0]],
                     '2c': [[1./2, 0., 1./2],[0., 1./2, 1./2]],
                     '2d': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '4e': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 0]],
                     '4f': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                     '4g': [['x', 0., 0],['x+1./2', 1./2, 0],['-x', 0., 0],['-x+1./2', 1./2, 0]],
                     '4h': [['x', 0., 1./2],['x+1./2', 1./2, 1./2],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2]],
                     '4i': [[0., 'y', 0],[1./2, 'y+1./2', 0],[0., '-y', 0],[1./2, '-y+1./2', 0]],
                     '4j': [[0., 'y', 1./2],[1./2, 'y+1./2', 1./2],[0., '-y', 1./2],[1./2, '-y+1./2', 1./2]],
                     '4k': [[0., 0., 'z'],[1./2, 1./2, 'z'],[0., 0., '-z'],[1./2, 1./2, '-z']],
                     '4l': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z'],[1./2, 0., '-z']],
                     '8m': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z'],[3./4, 3./4, '-z'],[1./4, 1./4, '-z'],[1./4, 3./4, 'z'],[3./4, 1./4, 'z']],
                     '8n': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z'],[0., 'y', '-z'],[1./2, 'y+1./2', '-z'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z']],
                     '8o': [['x', 0., 'z'],['x+1./2', 1./2, 'z'],['-x', 0., 'z'],['-x+1./2', 1./2, 'z'],['-x', 0., '-z'],['-x+1./2', 1./2, '-z'],['x', 0., '-z'],['x+1./2', 1./2, '-z']],
                     '8p': [['x', 'y', 0],['x+1./2', 'y+1./2', 0],['-x', '-y', 0],['-x+1./2', '-y+1./2', 0],['-x', 'y', 0],['-x+1./2', 'y+1./2', 0],['x', '-y', 0],['x+1./2', '-y+1./2', 0]],
                     '8q': [['x', 'y', 1./2],['x+1./2', 'y+1./2', 1./2],['-x', '-y', 1./2],['-x+1./2', '-y+1./2', 1./2],['-x', 'y', 1./2],['-x+1./2', 'y+1./2', 1./2],['x', '-y', 1./2],['x+1./2', '-y+1./2', 1./2]],
                     '16r': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '66': {'4a': [[0., 0., 1./4],[1./2, 1./2, 1./4],[0., 0., 3./4],[1./2, 1./2, 3./4]],
                     '4b': [[0., 1./2, 1./4],[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4]],
                     '4c': [[0., 0., 0],[1./2, 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '4d': [[0., 1./2, 0],[1./2, 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '4e': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                     '4f': [[1./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2]],
                     '8g': [['x', 0., 1./4],['x+1./2', 1./2, 1./4],['-x', 0., 1./4],['-x+1./2', 1./2, 1./4],['-x', 0., 3./4],['-x+1./2', 1./2, 3./4],['x', 0., 3./4],['x+1./2', 1./2, 3./4]],
                     '8h': [[0., 'y', 1./4],[1./2, 'y+1./2', 1./4],[0., '-y', 1./4],[1./2, '-y+1./2', 1./4],[0., '-y', 3./4],[1./2, '-y+1./2', 3./4],[0., 'y', 3./4],[1./2, 'y+1./2', 3./4]],
                     '8i': [[0., 0., 'z'],[1./2, 1./2, 'z'],[0., 0., '-z+1./2'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z+1./2']],
                     '8j': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2']],
                     '8k': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z'],[3./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z+1./2'],[3./4, 3./4, '-z'],[1./4, 1./4, '-z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z+1./2']],
                     '8l': [['x', 'y', 0],['x+1./2', 'y+1./2', 0],['-x', '-y', 0],['-x+1./2', '-y+1./2', 0],['-x', 'y', 1./2],['-x+1./2', 'y+1./2', 1./2],['x', '-y', 1./2],['x+1./2', '-y+1./2', 1./2]],
                     '16m': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '67': {'4a': [[1./4, 0., 0],[3./4, 1./2, 0],[3./4, 0., 0],[1./4, 1./2, 0]],
                     '4b': [[1./4, 0., 1./2],[3./4, 1./2, 1./2],[3./4, 0., 1./2],[1./4, 1./2, 1./2]],
                     '4c': [[0., 0., 0],[1./2, 1./2, 0],[0., 1./2, 0],[1./2, 0., 0]],
                     '4d': [[0., 0., 1./2],[1./2, 1./2, 1./2],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '4e': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 0]],
                     '4f': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                     '4g': [[0., 1./4, 'z'],[1./2, 3./4, 'z'],[0., 3./4, '-z'],[1./2, 1./4, '-z']],
                     '8h': [['x', 0., 0],['x+1./2', 1./2, 0],['-x', 1./2, 0],['-x+1./2', 0., 0],['-x', 0., 0],['-x+1./2', 1./2, 0],['x', 1./2, 0],['x+1./2', 0., 0]],
                     '8i': [['x', 0., 1./2],['x+1./2', 1./2, 1./2],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2],['x', 1./2, 1./2],['x+1./2', 0., 1./2]],
                     '8j': [[1./4, 'y', 0],[3./4, 'y+1./2', 0],[3./4, '-y+1./2', 0],[1./4, '-y', 0],[3./4, '-y', 0],[1./4, '-y+1./2', 0],[1./4, 'y+1./2', 0],[3./4, 'y', 0]],
                     '8k': [[1./4, 'y', 1./2],[3./4, 'y+1./2', 1./2],[3./4, '-y+1./2', 1./2],[1./4, '-y', 1./2],[3./4, '-y', 1./2],[1./4, '-y+1./2', 1./2],[1./4, 'y+1./2', 1./2],[3./4, 'y', 1./2]],
                     '8l': [[1./4, 0., 'z'],[3./4, 1./2, 'z'],[3./4, 1./2, '-z'],[1./4, 0., '-z'],[3./4, 0., '-z'],[1./4, 1./2, '-z'],[1./4, 1./2, 'z'],[3./4, 0., 'z']],
                     '8m': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z'],[0., '-y+1./2', 'z'],[1./2, '-y', 'z'],[0., 'y+1./2', '-z'],[1./2, 'y', '-z'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z']],
                     '8n': [['x', 1./4, 'z'],['x+1./2', 3./4, 'z'],['-x', 1./4, 'z'],['-x+1./2', 3./4, 'z'],['-x', 3./4, '-z'],['-x+1./2', 1./4, '-z'],['x', 3./4, '-z'],['x+1./2', 1./4, '-z']],
                     '16o': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x', '-y+1./2', 'z'],['-x+1./2', '-y', 'z'],['-x', 'y+1./2', '-z'],['-x+1./2', 'y', '-z'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x', 'y+1./2', '-z'],['x+1./2', 'y', '-z'],['x', '-y+1./2', 'z'],['x+1./2', '-y', 'z'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z']]},
              '68:1': {'4a': [[0., 0., 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                       '4b': [[0., 0., 1./2],[1./2, 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                       '8c': [[1./4, 0., 1./4],[3./4, 1./2, 1./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 0., 3./4],[1./4, 1./2, 3./4],[3./4, 1./2, 3./4],[1./4, 0., 3./4]],
                       '8d': [[0., 1./4, 1./4],[1./2, 3./4, 1./4],[1./2, 1./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 3./4],[0., 3./4, 3./4]],
                       '8e': [['x', 0., 0],['x+1./2', 1./2, 0],['-x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['x+1./2', 0., 1./2],['x', 1./2, 1./2]],
                       '8f': [[0., 'y', 0],[1./2, 'y+1./2', 0],[1./2, '-y+1./2', 0],[0., '-y', 0],[0., '-y+1./2', 1./2],[1./2, '-y', 1./2],[1./2, 'y', 1./2],[0., 'y+1./2', 1./2]],
                       '8g': [[0., 0., 'z'],[1./2, 1./2, 'z'],[0., 0., '-z'],[1./2, 1./2, '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2']],
                       '8h': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z+1./2'],[1./4, 1./4, 'z+1./2'],[3./4, 3./4, 'z+1./2']],
                       '16i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z+1./2'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2']]},
              '68:2': {'4a': [[0., 1./4, 1./4],[1./2, 3./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4]],
                       '4b': [[0., 1./4, 3./4],[1./2, 3./4, 3./4],[0., 3./4, 1./4],[1./2, 1./4, 1./4]],
                       '8c': [[1./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                       '8d': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                       '8e': [['x', 1./4, 1./4],['x+1./2', 3./4, 1./4],['-x+1./2', 3./4, 1./4],['-x', 1./4, 1./4],['-x', 3./4, 3./4],['-x+1./2', 1./4, 3./4],['x+1./2', 1./4, 3./4],['x', 3./4, 3./4]],
                       '8f': [[0., 'y', 1./4],[1./2, 'y+1./2', 1./4],[1./2, '-y', 1./4],[0., '-y+1./2', 1./4],[0., '-y', 3./4],[1./2, '-y+1./2', 3./4],[1./2, 'y', 3./4],[0., 'y+1./2', 3./4]],
                       '8g': [[0., 1./4, 'z'],[1./2, 3./4, 'z'],[0., 1./4, '-z+1./2'],[1./2, 3./4, '-z+1./2'],[0., 3./4, '-z'],[1./2, 1./4, '-z'],[0., 3./4, 'z+1./2'],[1./2, 1./4, 'z+1./2']],
                       '8h': [[1./4, 0., 'z'],[3./4, 1./2, 'z'],[3./4, 0., '-z+1./2'],[1./4, 1./2, '-z+1./2'],[3./4, 0., '-z'],[1./4, 1./2, '-z'],[1./4, 0., 'z+1./2'],[3./4, 1./2, 'z+1./2']],
                       '16i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z'],['-x+1./2', '-y', 'z'],['-x', '-y+1./2', 'z'],['-x', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y', '-z'],['x', 'y+1./2', '-z'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2']]},
              '69': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                     '4b': [[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 0],[1./2, 1./2, 1./2]],
                     '8c': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4]],
                     '8d': [[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4]],
                     '8e': [[1./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0]],
                     '8f': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                     '8g': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0]],
                     '8h': [[0., 'y', 0],[0., 'y+1./2', 1./2],[1./2, 'y', 1./2],[1./2, 'y+1./2', 0],[0., '-y', 0],[0., '-y+1./2', 1./2],[1./2, '-y', 1./2],[1./2, '-y+1./2', 0]],
                     '8i': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 1./2, 'z'],[0., 0., '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 1./2, '-z']],
                     '16j': [[1./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z+1./2'],[3./4, 3./4, 'z'],[3./4, 1./4, '-z'],[3./4, 3./4, '-z+1./2'],[1./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z'],[3./4, 3./4, '-z'],[3./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z+1./2'],[1./4, 1./4, '-z'],[1./4, 3./4, 'z'],[1./4, 1./4, 'z+1./2'],[3./4, 3./4, 'z+1./2'],[3./4, 1./4, 'z']],
                     '16k': [[1./4, 'y', 1./4],[1./4, 'y+1./2', 3./4],[3./4, 'y', 3./4],[3./4, 'y+1./2', 1./4],[3./4, '-y', 1./4],[3./4, '-y+1./2', 3./4],[1./4, '-y', 3./4],[1./4, '-y+1./2', 1./4],[3./4, '-y', 3./4],[3./4, '-y+1./2', 1./4],[1./4, '-y', 1./4],[1./4, '-y+1./2', 3./4],[1./4, 'y', 3./4],[1./4, 'y+1./2', 1./4],[3./4, 'y', 1./4],[3./4, 'y+1./2', 3./4]],
                     '16l': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],['-x', 3./4, 3./4],['-x', 1./4, 1./4],['-x+1./2', 3./4, 1./4],['-x+1./2', 1./4, 3./4],['x', 1./4, 3./4],['x', 3./4, 1./4],['x+1./2', 1./4, 1./4],['x+1./2', 3./4, 3./4]],
                     '16m': [[0., 'y', 'z'],[0., 'y+1./2', 'z+1./2'],[1./2, 'y', 'z+1./2'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[0., '-y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[1./2, '-y+1./2', 'z'],[0., 'y', '-z'],[0., 'y+1./2', '-z+1./2'],[1./2, 'y', '-z+1./2'],[1./2, 'y+1./2', '-z'],[0., '-y', '-z'],[0., '-y+1./2', '-z+1./2'],[1./2, '-y', '-z+1./2'],[1./2, '-y+1./2', '-z']],
                     '16n': [['x', 0., 'z'],['x', 1./2, 'z+1./2'],['x+1./2', 0., 'z+1./2'],['x+1./2', 1./2, 'z'],['-x', 0., 'z'],['-x', 1./2, 'z+1./2'],['-x+1./2', 0., 'z+1./2'],['-x+1./2', 1./2, 'z'],['-x', 0., '-z'],['-x', 1./2, '-z+1./2'],['-x+1./2', 0., '-z+1./2'],['-x+1./2', 1./2, '-z'],['x', 0., '-z'],['x', 1./2, '-z+1./2'],['x+1./2', 0., '-z+1./2'],['x+1./2', 1./2, '-z']],
                     '16o': [['x', 'y', 0],['x', 'y+1./2', 1./2],['x+1./2', 'y', 1./2],['x+1./2', 'y+1./2', 0],['-x', '-y', 0],['-x', '-y+1./2', 1./2],['-x+1./2', '-y', 1./2],['-x+1./2', '-y+1./2', 0],['-x', 'y', 0],['-x', 'y+1./2', 1./2],['-x+1./2', 'y', 1./2],['-x+1./2', 'y+1./2', 0],['x', '-y', 0],['x', '-y+1./2', 1./2],['x+1./2', '-y', 1./2],['x+1./2', '-y+1./2', 0]],
                     '32p': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z'],['x', 'y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x', 'y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z']]},
              '70:1': {'8a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                       '8b': [[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 0],[1./2, 1./2, 1./2],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                       '16c': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 7/8, 1/8],[7/8, 3/8, 5/8],[3/8, 7/8, 5/8],[3/8, 3/8, 1/8],[7/8, 1/8, 7/8],[7/8, 5/8, 3/8],[3/8, 1/8, 3/8],[3/8, 5/8, 7/8],[1/8, 7/8, 7/8],[1/8, 3/8, 3/8],[5/8, 7/8, 3/8],[5/8, 3/8, 7/8]],
                       '16d': [[5/8, 5/8, 5/8],[5/8, 1/8, 1/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[3/8, 3/8, 5/8],[3/8, 7/8, 1/8],[7/8, 3/8, 1/8],[7/8, 7/8, 5/8],[3/8, 5/8, 3/8],[3/8, 1/8, 7/8],[7/8, 5/8, 7/8],[7/8, 1/8, 3/8],[5/8, 3/8, 3/8],[5/8, 7/8, 7/8],[1/8, 3/8, 7/8],[1/8, 7/8, 3/8]],
                       '16e': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],['-x+1./4', 1./4, 1./4],['-x+1./4', 3./4, 3./4],['-x+3./4', 1./4, 3./4],['-x+3./4', 3./4, 1./4],['x+1./4', 1./4, 1./4],['x+1./4', 3./4, 3./4],['x+3./4', 1./4, 3./4],['x+3./4', 3./4, 1./4]],
                       '16f': [[0., 'y', 0],[0., 'y+1./2', 1./2],[1./2, 'y', 1./2],[1./2, 'y+1./2', 0],[0., '-y', 0],[0., '-y+1./2', 1./2],[1./2, '-y', 1./2],[1./2, '-y+1./2', 0],[1./4, '-y+1./4', 1./4],[1./4, '-y+3./4', 3./4],[3./4, '-y+1./4', 3./4],[3./4, '-y+3./4', 1./4],[1./4, 'y+1./4', 1./4],[1./4, 'y+3./4', 3./4],[3./4, 'y+1./4', 3./4],[3./4, 'y+3./4', 1./4]],
                       '16g': [[0., 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 1./2, 'z'],[0., 0., '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 1./2, '-z'],[1./4, 1./4, '-z+1./4'],[1./4, 3./4, '-z+3./4'],[3./4, 1./4, '-z+3./4'],[3./4, 3./4, '-z+1./4'],[1./4, 1./4, 'z+1./4'],[1./4, 3./4, 'z+3./4'],[3./4, 1./4, 'z+3./4'],[3./4, 3./4, 'z+1./4']],
                       '32h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['-x+1./4', '-y+1./4', '-z+1./4'],['-x+1./4', '-y+3./4', '-z+3./4'],['-x+3./4', '-y+1./4', '-z+3./4'],['-x+3./4', '-y+3./4', '-z+1./4'],['x+1./4', 'y+1./4', '-z+1./4'],['x+1./4', 'y+3./4', '-z+3./4'],['x+3./4', 'y+1./4', '-z+3./4'],['x+3./4', 'y+3./4', '-z+1./4'],['x+1./4', '-y+1./4', 'z+1./4'],['x+1./4', '-y+3./4', 'z+3./4'],['x+3./4', '-y+1./4', 'z+3./4'],['x+3./4', '-y+3./4', 'z+1./4'],['-x+1./4', 'y+1./4', 'z+1./4'],['-x+1./4', 'y+3./4', 'z+3./4'],['-x+3./4', 'y+1./4', 'z+3./4'],['-x+3./4', 'y+3./4', 'z+1./4']]},
              '70:2': {'8a': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 7/8, 7/8],[7/8, 3/8, 3/8],[3/8, 7/8, 3/8],[3/8, 3/8, 7/8]],
                       '8b': [[1/8, 1/8, 5/8],[1/8, 5/8, 1/8],[5/8, 1/8, 1/8],[5/8, 5/8, 5/8],[7/8, 7/8, 3/8],[7/8, 3/8, 7/8],[3/8, 7/8, 7/8],[3/8, 3/8, 3/8]],
                       '16c': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[3./4, 3./4, 0],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2],[1./4, 1./4, 0],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[0., 3./4, 3./4],[0., 1./4, 1./4],[1./2, 3./4, 1./4],[1./2, 1./4, 3./4]],
                       '16d': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./2, 1./4],[1./4, 0., 3./4],[3./4, 1./2, 3./4],[3./4, 0., 1./4],[1./2, 1./4, 1./4],[1./2, 3./4, 3./4],[0., 1./4, 3./4],[0., 3./4, 1./4]],
                       '16e': [['x', 1/8, 1/8],['x', 5/8, 5/8],['x+1./2', 1/8, 5/8],['x+1./2', 5/8, 1/8],['-x+3./4', 5/8, 1/8],['-x+3./4', 1/8, 5/8],['-x+1./4', 5/8, 5/8],['-x+1./4', 1/8, 1/8],['-x', 7/8, 7/8],['-x', 3/8, 3/8],['-x+1./2', 7/8, 3/8],['-x+1./2', 3/8, 7/8],['x+1./4', 3/8, 7/8],['x+1./4', 7/8, 3/8],['x+3./4', 3/8, 3/8],['x+3./4', 7/8, 7/8]],
                       '16f': [[1/8, 'y', 1/8],[1/8, 'y+1./2', 5/8],[5/8, 'y', 5/8],[5/8, 'y+1./2', 1/8],[5/8, '-y+3./4', 1/8],[5/8, '-y+1./4', 5/8],[1/8, '-y+3./4', 5/8],[1/8, '-y+1./4', 1/8],[7/8, '-y', 7/8],[7/8, '-y+1./2', 3/8],[3/8, '-y', 3/8],[3/8, '-y+1./2', 7/8],[3/8, 'y+1./4', 7/8],[3/8, 'y+3./4', 3/8],[7/8, 'y+1./4', 3/8],[7/8, 'y+3./4', 7/8]],
                       '16g': [[1/8, 1/8, 'z'],[1/8, 5/8, 'z+1./2'],[5/8, 1/8, 'z+1./2'],[5/8, 5/8, 'z'],[5/8, 1/8, '-z+3./4'],[5/8, 5/8, '-z+1./4'],[1/8, 1/8, '-z+1./4'],[1/8, 5/8, '-z+3./4'],[7/8, 7/8, '-z'],[7/8, 3/8, '-z+1./2'],[3/8, 7/8, '-z+1./2'],[3/8, 3/8, '-z'],[3/8, 7/8, 'z+1./4'],[3/8, 3/8, 'z+3./4'],[7/8, 7/8, 'z+3./4'],[7/8, 3/8, 'z+1./4']],
                       '32h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x+3./4', '-y+3./4', 'z'],['-x+3./4', '-y+1./4', 'z+1./2'],['-x+1./4', '-y+3./4', 'z+1./2'],['-x+1./4', '-y+1./4', 'z'],['-x+3./4', 'y', '-z+3./4'],['-x+3./4', 'y+1./2', '-z+1./4'],['-x+1./4', 'y', '-z+1./4'],['-x+1./4', 'y+1./2', '-z+3./4'],['x', '-y+3./4', '-z+3./4'],['x', '-y+1./4', '-z+1./4'],['x+1./2', '-y+3./4', '-z+1./4'],['x+1./2', '-y+1./4', '-z+3./4'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x+1./4', 'y+1./4', '-z'],['x+1./4', 'y+3./4', '-z+1./2'],['x+3./4', 'y+1./4', '-z+1./2'],['x+3./4', 'y+3./4', '-z'],['x+1./4', '-y', 'z+1./4'],['x+1./4', '-y+1./2', 'z+3./4'],['x+3./4', '-y', 'z+3./4'],['x+3./4', '-y+1./2', 'z+1./4'],['-x', 'y+1./4', 'z+1./4'],['-x', 'y+3./4', 'z+3./4'],['-x+1./2', 'y+1./4', 'z+3./4'],['-x+1./2', 'y+3./4', 'z+1./4']]},
              '71': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[0., 1./2, 1./2],[1./2, 0., 0]],
                     '2c': [[1./2, 1./2, 0],[0., 0., 1./2]],
                     '2d': [[1./2, 0., 1./2],[0., 1./2, 0]],
                     '4e': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2]],
                     '4f': [['x', 1./2, 0],['x+1./2', 0., 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2]],
                     '4g': [[0., 'y', 0],[1./2, 'y+1./2', 1./2],[0., '-y', 0],[1./2, '-y+1./2', 1./2]],
                     '4h': [[0., 'y', 1./2],[1./2, 'y+1./2', 0],[0., '-y', 1./2],[1./2, '-y+1./2', 0]],
                     '4i': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                     '4j': [[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2']],
                     '8k': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                     '8l': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z+1./2'],[0., 'y', '-z'],[1./2, 'y+1./2', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z+1./2']],
                     '8m': [['x', 0., 'z'],['x+1./2', 1./2, 'z+1./2'],['-x', 0., 'z'],['-x+1./2', 1./2, 'z+1./2'],['-x', 0., '-z'],['-x+1./2', 1./2, '-z+1./2'],['x', 0., '-z'],['x+1./2', 1./2, '-z+1./2']],
                     '8n': [['x', 'y', 0],['x+1./2', 'y+1./2', 1./2],['-x', '-y', 0],['-x+1./2', '-y+1./2', 1./2],['-x', 'y', 0],['-x+1./2', 'y+1./2', 1./2],['x', '-y', 0],['x+1./2', '-y+1./2', 1./2]],
                     '16o': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '72': {'4a': [[0., 0., 1./4],[1./2, 1./2, 3./4],[0., 0., 3./4],[1./2, 1./2, 1./4]],
                     '4b': [[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4],[0., 1./2, 1./4]],
                     '4c': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 1./2, 0],[0., 0., 1./2]],
                     '4d': [[1./2, 0., 0],[0., 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 1./2]],
                     '8e': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4]],
                     '8f': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x', 0., 1./4],['-x+1./2', 1./2, 3./4],['-x', 0., 3./4],['-x+1./2', 1./2, 1./4],['x', 0., 3./4],['x+1./2', 1./2, 1./4]],
                     '8g': [[0., 'y', 1./4],[1./2, 'y+1./2', 3./4],[0., '-y', 1./4],[1./2, '-y+1./2', 3./4],[0., '-y', 3./4],[1./2, '-y+1./2', 1./4],[0., 'y', 3./4],[1./2, 'y+1./2', 1./4]],
                     '8h': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, 'z'],[0., 0., 'z+1./2']],
                     '8i': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                     '8j': [['x', 'y', 0],['x+1./2', 'y+1./2', 1./2],['-x', '-y', 0],['-x+1./2', '-y+1./2', 1./2],['-x+1./2', 'y+1./2', 0],['-x', 'y', 1./2],['x+1./2', '-y+1./2', 0],['x', '-y', 1./2]],
                     '16k': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', '-z'],['-x', 'y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z'],['x', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-x', 'y', 'z+1./2']]},
              '73': {'8a': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                     '8b': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4]],
                     '8c': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],['-x', 0., 3./4],['-x+1./2', 1./2, 1./4],['x+1./2', 0., 1./4],['x', 1./2, 3./4]],
                     '8d': [[1./4, 'y', 0],[3./4, 'y+1./2', 1./2],[1./4, '-y', 1./2],[3./4, '-y+1./2', 0],[3./4, '-y', 0],[1./4, '-y+1./2', 1./2],[3./4, 'y', 1./2],[1./4, 'y+1./2', 0]],
                     '8e': [[0., 1./4, 'z'],[1./2, 3./4, 'z+1./2'],[0., 3./4, '-z+1./2'],[1./2, 1./4, '-z'],[0., 3./4, '-z'],[1./2, 1./4, '-z+1./2'],[0., 1./4, 'z+1./2'],[1./2, 3./4, 'z']],
                     '16f': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z'],['-x+1./2', 'y+1./2', 'z'],['-x', 'y', 'z+1./2']]},
              '74': {'4a': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 1./2]],
                     '4b': [[0., 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 0]],
                     '4c': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                     '4d': [[1./4, 1./4, 3./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4]],
                     '4e': [[0., 1./4, 'z'],[1./2, 3./4, 'z+1./2'],[0., 3./4, '-z'],[1./2, 1./4, '-z+1./2']],
                     '8f': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],['x', 1./2, 0],['x+1./2', 0., 1./2]],
                     '8g': [[1./4, 'y', 1./4],[3./4, 'y+1./2', 3./4],[3./4, '-y+1./2', 1./4],[1./4, '-y', 3./4],[3./4, '-y', 3./4],[1./4, '-y+1./2', 1./4],[1./4, 'y+1./2', 3./4],[3./4, 'y', 1./4]],
                     '8h': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[0., '-y+1./2', 'z'],[1./2, '-y', 'z+1./2'],[0., 'y+1./2', '-z'],[1./2, 'y', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z+1./2']],
                     '8i': [['x', 1./4, 'z'],['x+1./2', 3./4, 'z+1./2'],['-x', 1./4, 'z'],['-x+1./2', 3./4, 'z+1./2'],['-x', 3./4, '-z'],['-x+1./2', 1./4, '-z+1./2'],['x', 3./4, '-z'],['x+1./2', 1./4, '-z+1./2']],
                     '16j': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z'],['-x+1./2', 'y', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y+1./2', '-z'],['x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', 'z'],['x+1./2', '-y', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2']]},
              '75': {'1a': [[0., 0., 'z']],
                     '1b': [[1./2, 1./2, 'z']],
                     '2c': [[0., 1./2, 'z'],[1./2, 0., 'z']],
                     '4d': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z']]},
              '76': {'4a': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-y', 'x', 'z+1./4'],['y', '-x', 'z+3./4']]},
              '77': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                     '2b': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                     '2c': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2']],
                     '4d': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2']]},
              '78': {'4a': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-y', 'x', 'z+3./4'],['y', '-x', 'z+1./4']]},
              '79': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2']],
                     '4b': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                     '8c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2']]},
              '80': {'4a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4']],
                     '8b': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4']]},
              '81': {'1a': [[0., 0., 0]],
                     '1b': [[0., 0., 1./2]],
                     '1c': [[1./2, 1./2, 0]],
                     '1d': [[1./2, 1./2, 1./2]],
                     '2e': [[0., 0., 'z'],[0., 0., '-z']],
                     '2f': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                     '2g': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                     '4h': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z']]},
              '82': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                     '2c': [[0., 1./2, 1./4],[1./2, 0., 3./4]],
                     '2d': [[0., 1./2, 3./4],[1./2, 0., 1./4]],
                     '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                     '4f': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2']],
                     '8g': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2']]},
              '83': {'1a': [[0., 0., 0]],
                     '1b': [[0., 0., 1./2]],
                     '1c': [[1./2, 1./2, 0]],
                     '1d': [[1./2, 1./2, 1./2]],
                     '2e': [[0., 1./2, 0],[1./2, 0., 0]],
                     '2f': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                     '2g': [[0., 0., 'z'],[0., 0., '-z']],
                     '2h': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                     '4i': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z'],[1./2, 0., '-z']],
                     '4j': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 0],['y', '-x', 0]],
                     '4k': [['x', 'y', 1./2],['-x', '-y', 1./2],['-y', 'x', 1./2],['y', '-x', 1./2]],
                     '8l': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z'],['-y', 'x', '-z']]},
              '84': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                     '2b': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                     '2c': [[0., 1./2, 0],[1./2, 0., 1./2]],
                     '2d': [[0., 1./2, 1./2],[1./2, 0., 0]],
                     '2e': [[0., 0., 1./4],[0., 0., 3./4]],
                     '2f': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                     '4g': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2']],
                     '4h': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, '-z+1./2']],
                     '4i': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2']],
                     '4j': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 1./2],['y', '-x', 1./2]],
                     '8k': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z+1./2'],['-y', 'x', '-z+1./2']]},
              '85:1': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '2c': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                     '4d': [[1./4, 1./4, 0],[3./4, 3./4, 0],[1./4, 3./4, 0],[3./4, 1./4, 0]],
                     '4e': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2]],
                     '4f': [[0., 0., 'z'],[1./2, 1./2, 'z'],[1./2, 1./2, '-z'],[0., 0., '-z']],
                     '8g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z'],['y+1./2', '-x+1./2', 'z'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y+1./2', '-z'],['y', '-x', '-z'],['-y', 'x', '-z']]},
              '85:2': {'2a': [[1./4, 3./4, 0],[3./4, 1./4, 0]],
                       '2b': [[1./4, 3./4, 1./2],[3./4, 1./4, 1./2]],
                       '2c': [[1./4, 1./4, 'z'],[3./4, 3./4, '-z']],
                       '4d': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0]],
                       '4e': [[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                       '4f': [[1./4, 3./4, 'z'],[3./4, 1./4, 'z'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z']],
                       '8g': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z'],['y', '-x+1./2', 'z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z'],['-y', 'x+1./2', '-z']]},
              '86:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                       '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                       '4c': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4]],
                       '4d': [[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                       '4e': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z']],
                       '4f': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z']],
                       '8g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['-y', 'x', '-z']]},
              '86:2': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                       '2b': [[1./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                       '4c': [[0., 0., 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                       '4d': [[0., 0., 1./2],[1./2, 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                       '4e': [[3./4, 1./4, 'z'],[3./4, 1./4, 'z+1./2'],[1./4, 3./4, '-z'],[1./4, 3./4, '-z+1./2']],
                       '4f': [[1./4, 1./4, 'z'],[3./4, 3./4, 'z+1./2'],[3./4, 3./4, '-z'],[1./4, 1./4, '-z+1./2']],
                       '8g': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y', 'x+1./2', 'z+1./2'],['y+1./2', '-x', 'z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y', '-x+1./2', '-z+1./2'],['-y+1./2', 'x', '-z+1./2']]},
              '87': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                     '4c': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                     '4d': [[0., 1./2, 1./4],[1./2, 0., 3./4],[1./2, 0., 1./4],[0., 1./2, 3./4]],
                     '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                     '8f': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4]],
                     '8g': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2']],
                     '8h': [['x', 'y', 0],['x+1./2', 'y+1./2', 1./2],['-x', '-y', 0],['-x+1./2', '-y+1./2', 1./2],['-y', 'x', 0],['-y+1./2', 'x+1./2', 1./2],['y', '-x', 0],['y+1./2', '-x+1./2', 1./2]],
                     '16i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2']]},
              '88:1': {'4a': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 1./2, 1./4],[1./2, 0., 3./4]],
                       '4b': [[0., 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                       '8c': [[0., 1./4, 1/8],[1./2, 3./4, 5/8],[1./2, 1./4, 5/8],[0., 3./4, 1/8],[3./4, 1./2, 3/8],[1./4, 0., 7/8],[3./4, 0., 7/8],[1./4, 1./2, 3/8]],
                       '8d': [[0., 1./4, 5/8],[1./2, 3./4, 1/8],[1./2, 1./4, 1/8],[0., 3./4, 5/8],[3./4, 1./2, 7/8],[1./4, 0., 3/8],[3./4, 0., 3/8],[1./4, 1./2, 7/8]],
                       '8e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4'],[0., 1./2, '-z+1./4'],[1./2, 0., '-z+3./4'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                       '16f': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4'],['-x', '-y+1./2', '-z+1./4'],['-x+1./2', '-y', '-z+3./4'],['x+1./2', 'y', '-z+3./4'],['x', 'y+1./2', '-z+1./4'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['-y', 'x', '-z']]},
              '88:2': {'4a': [[0., 1./4, 1/8],[1./2, 3./4, 5/8],[1./2, 1./4, 3/8],[0., 3./4, 7/8]],
                       '4b': [[0., 1./4, 5/8],[1./2, 3./4, 1/8],[1./2, 1./4, 7/8],[0., 3./4, 3/8]],
                       '8c': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 0],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 3./4, 3./4],[1./4, 1./4, 1./4]],
                       '8d': [[0., 0., 1./2],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 1./2],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                       '8e': [[0., 1./4, 'z'],[1./2, 3./4, 'z+1./2'],[1./2, 1./4, 'z+1./4'],[0., 3./4, 'z+3./4'],[0., 3./4, '-z'],[1./2, 1./4, '-z+1./2'],[1./2, 3./4, '-z+3./4'],[0., 1./4, '-z+1./4']],
                       '16f': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-y+3./4', 'x+1./4', 'z+1./4'],['-y+1./4', 'x+3./4', 'z+3./4'],['y+3./4', '-x+3./4', 'z+3./4'],['y+1./4', '-x+1./4', 'z+1./4'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z'],['y+1./4', '-x+3./4', '-z+3./4'],['y+3./4', '-x+1./4', '-z+1./4'],['-y+1./4', 'x+1./4', '-z+1./4'],['-y+3./4', 'x+3./4', '-z+3./4']]},
              '89': {'1a': [[0., 0., 0]],
                     '1b': [[0., 0., 1./2]],
                     '1c': [[1./2, 1./2, 0]],
                     '1d': [[1./2, 1./2, 1./2]],
                     '2e': [[1./2, 0., 0],[0., 1./2, 0]],
                     '2f': [[1./2, 0., 1./2],[0., 1./2, 1./2]],
                     '2g': [[0., 0., 'z'],[0., 0., '-z']],
                     '2h': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                     '4i': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z'],[1./2, 0., '-z']],
                     '4j': [['x', 'x', 0],['-x', '-x', 0],['-x', 'x', 0],['x', '-x', 0]],
                     '4k': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x', 'x', 1./2],['x', '-x', 1./2]],
                     '4l': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0]],
                     '4m': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2]],
                     '4n': [['x', 0., 1./2],['-x', 0., 1./2],[0., 'x', 1./2],[0., '-x', 1./2]],
                     '4o': [['x', 1./2, 0],['-x', 1./2, 0],[1./2, 'x', 0],[1./2, '-x', 0]],
                     '8p': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['y', 'x', '-z'],['-y', '-x', '-z']]},
              '90': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                     '2c': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                     '4d': [[0., 0., 'z'],[1./2, 1./2, 'z'],[1./2, 1./2, '-z'],[0., 0., '-z']],
                     '4e': [['x', 'x', 0],['-x', '-x', 0],['-x+1./2', 'x+1./2', 0],['x+1./2', '-x+1./2', 0]],
                     '4f': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2]],
                     '8g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z'],['y+1./2', '-x+1./2', 'z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['y', 'x', '-z'],['-y', '-x', '-z']]},
              '91': {'4a': [[0., 'y', 0],[0., '-y', 1./2],['-y', 0., 1./4],['y', 0., 3./4]],
                     '4b': [[1./2, 'y', 0],[1./2, '-y', 1./2],['-y', 1./2, 1./4],['y', 1./2, 3./4]],
                     '4c': [['x', 'x', 3/8],['-x', '-x', 7/8],['-x', 'x', 5/8],['x', '-x', 1/8]],
                     '8d': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-y', 'x', 'z+1./4'],['y', '-x', 'z+3./4'],['-x', 'y', '-z'],['x', '-y', '-z+1./2'],['y', 'x', '-z+3./4'],['-y', '-x', '-z+1./4']]},
              '92': {'4a': [['x', 'x', 0],['-x', '-x', 1./2],['-x+1./2', 'x+1./2', 1./4],['x+1./2', '-x+1./2', 3./4]],
                     '8b': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-y+1./2', 'x+1./2', 'z+1./4'],['y+1./2', '-x+1./2', 'z+3./4'],['-x+1./2', 'y+1./2', '-z+1./4'],['x+1./2', '-y+1./2', '-z+3./4'],['y', 'x', '-z'],['-y', '-x', '-z+1./2']]},
              '93': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                     '2b': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                     '2c': [[0., 1./2, 0],[1./2, 0., 1./2]],
                     '2d': [[0., 1./2, 1./2],[1./2, 0., 0]],
                     '2e': [[0., 0., 1./4],[0., 0., 3./4]],
                     '2f': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                     '4g': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2']],
                     '4h': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, '-z+1./2']],
                     '4i': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2']],
                     '4j': [['x', 0., 0],['-x', 0., 0],[0., 'x', 1./2],[0., '-x', 1./2]],
                     '4k': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 0],[1./2, '-x', 0]],
                     '4l': [['x', 0., 1./2],['-x', 0., 1./2],[0., 'x', 0],[0., '-x', 0]],
                     '4m': [['x', 1./2, 0],['-x', 1./2, 0],[1./2, 'x', 1./2],[1./2, '-x', 1./2]],
                     '4n': [['x', 'x', 1./4],['-x', '-x', 1./4],['-x', 'x', 3./4],['x', '-x', 3./4]],
                     '4o': [['x', 'x', 3./4],['-x', '-x', 3./4],['-x', 'x', 1./4],['x', '-x', 1./4]],
                     '8p': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['-x', 'y', '-z'],['x', '-y', '-z'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2']]},
              '94': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                     '4c': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z']],
                     '4d': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z']],
                     '4e': [['x', 'x', 0],['-x', '-x', 0],['-x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2]],
                     '4f': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x+1./2', 'x+1./2', 0],['x+1./2', '-x+1./2', 0]],
                     '8g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['y', 'x', '-z'],['-y', '-x', '-z']]},
              '95': {'4a': [[0., 'y', 0],[0., '-y', 1./2],['-y', 0., 3./4],['y', 0., 1./4]],
                     '4b': [[1./2, 'y', 0],[1./2, '-y', 1./2],['-y', 1./2, 3./4],['y', 1./2, 1./4]],
                     '4c': [['x', 'x', 5/8],['-x', '-x', 1/8],['-x', 'x', 3/8],['x', '-x', 7/8]],
                     '8d': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-y', 'x', 'z+3./4'],['y', '-x', 'z+1./4'],['-x', 'y', '-z'],['x', '-y', '-z+1./2'],['y', 'x', '-z+1./4'],['-y', '-x', '-z+3./4']]},
              '96': {'4a': [['x', 'x', 0],['-x', '-x', 1./2],['-x+1./2', 'x+1./2', 3./4],['x+1./2', '-x+1./2', 1./4]],
                     '8b': [['x', 'y', 'z'],['-x', '-y', 'z+1./2'],['-y+1./2', 'x+1./2', 'z+3./4'],['y+1./2', '-x+1./2', 'z+1./4'],['-x+1./2', 'y+1./2', '-z+3./4'],['x+1./2', '-y+1./2', '-z+1./4'],['y', 'x', '-z'],['-y', '-x', '-z+1./2']]},
              '97': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                     '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                     '4c': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                     '4d': [[0., 1./2, 1./4],[1./2, 0., 3./4],[1./2, 0., 1./4],[0., 1./2, 3./4]],
                     '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                     '8f': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2']],
                     '8g': [['x', 'x', 0],['x+1./2', 'x+1./2', 1./2],['-x', '-x', 0],['-x+1./2', '-x+1./2', 1./2],['-x', 'x', 0],['-x+1./2', 'x+1./2', 1./2],['x', '-x', 0],['x+1./2', '-x+1./2', 1./2]],
                     '8h': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2]],
                     '8i': [['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 1./2],[1./2, '-x+1./2', 0]],
                     '8j': [['x', 'x+1./2', 1./4],['x+1./2', 'x', 3./4],['-x', '-x+1./2', 1./4],['-x+1./2', '-x', 3./4],['-x+1./2', 'x', 1./4],['-x', 'x+1./2', 3./4],['x+1./2', '-x', 1./4],['x', '-x+1./2', 3./4]],
                     '16k': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['y', 'x', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2']]},
              '98': {'4a': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 1./2, 1./4],[1./2, 0., 3./4]],
                     '4b': [[0., 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                     '8c': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4'],[1./2, 0., '-z+3./4'],[0., 1./2, '-z+1./4'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z']],
                     '8d': [['x', 'x', 0],['x+1./2', 'x+1./2', 1./2],['-x+1./2', '-x+1./2', 1./2],['-x', '-x', 0],['-x', 'x+1./2', 1./4],['-x+1./2', 'x', 3./4],['x+1./2', '-x', 3./4],['x', '-x+1./2', 1./4]],
                     '8e': [['-x', 'x', 0],['-x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2],['x', '-x', 0],['-x', '-x+1./2', 1./4],['-x+1./2', '-x', 3./4],['x+1./2', 'x', 3./4],['x', 'x+1./2', 1./4]],
                     '8f': [['x', 1./4, 1/8],['x+1./2', 3./4, 5/8],['-x+1./2', 1./4, 5/8],['-x', 3./4, 1/8],[3./4, 'x+1./2', 3/8],[1./4, 'x', 7/8],[3./4, '-x', 7/8],[1./4, '-x+1./2', 3/8]],
                     '16g': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4'],['-x+1./2', 'y', '-z+3./4'],['-x', 'y+1./2', '-z+1./4'],['x', '-y+1./2', '-z+1./4'],['x+1./2', '-y', '-z+3./4'],['y+1./2', 'x+1./2', '-z+1./2'],['y', 'x', '-z'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2']]},
              '99': {'1a': [[0., 0., 'z']],
                     '1b': [[1./2, 1./2, 'z']],
                     '2c': [[1./2, 0., 'z'],[0., 1./2, 'z']],
                     '4d': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', 'z'],['x', '-x', 'z']],
                     '4e': [['x', 0., 'z'],['-x', 0., 'z'],[0., 'x', 'z'],[0., '-x', 'z']],
                     '4f': [['x', 1./2, 'z'],['-x', 1./2, 'z'],[1./2, 'x', 'z'],[1./2, '-x', 'z']],
                     '8g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '100': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z']],
                      '2b': [[1./2, 0., 'z'],[0., 1./2, 'z']],
                      '4c': [['x', 'x+1./2', 'z'],['-x', '-x+1./2', 'z'],['-x+1./2', 'x', 'z'],['x+1./2', '-x', 'z']],
                      '8d': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-y+1./2', '-x+1./2', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '101': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                      '4c': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z']],
                      '4d': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', 'z+1./2'],['x', '-x', 'z+1./2']],
                      '8e': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '102': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2']],
                      '4b': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z']],
                      '4c': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x+1./2', 'x+1./2', 'z+1./2'],['x+1./2', '-x+1./2', 'z+1./2']],
                      '8d': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '103': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                      '4c': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2']],
                      '8d': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-y', '-x', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '104': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2']],
                      '4b': [[0., 1./2, 'z'],[1./2, 0., 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, 'z+1./2']],
                      '8c': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '105': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                      '2c': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2']],
                      '4d': [['x', 0., 'z'],['-x', 0., 'z'],[0., 'x', 'z+1./2'],[0., '-x', 'z+1./2']],
                      '4e': [['x', 1./2, 'z'],['-x', 1./2, 'z'],[1./2, 'x', 'z+1./2'],[1./2, '-x', 'z+1./2']],
                      '8f': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['x', '-y', 'z'],['-x', 'y', 'z'],['-y', '-x', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '106': {'4a': [[0., 0., 'z'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                      '4b': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                      '8c': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '107': {'2a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2']],
                      '4b': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                      '8c': [['x', 'x', 'z'],['x+1./2', 'x+1./2', 'z+1./2'],['-x', '-x', 'z'],['-x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', 'z'],['-x+1./2', 'x+1./2', 'z+1./2'],['x', '-x', 'z'],['x+1./2', '-x+1./2', 'z+1./2']],
                      '8d': [['x', 0., 'z'],['x+1./2', 1./2, 'z+1./2'],['-x', 0., 'z'],['-x+1./2', 1./2, 'z+1./2'],[0., 'x', 'z'],[1./2, 'x+1./2', 'z+1./2'],[0., '-x', 'z'],[1./2, '-x+1./2', 'z+1./2']],
                      '16e': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y', '-x', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', 'x', 'z'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '108': {'4a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z']],
                      '4b': [[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, 'z'],[1./2, 0., 'z+1./2']],
                      '8c': [['x', 'x+1./2', 'z'],['x+1./2', 'x', 'z+1./2'],['-x', '-x+1./2', 'z'],['-x+1./2', '-x', 'z+1./2'],['-x+1./2', 'x', 'z'],['-x', 'x+1./2', 'z+1./2'],['x+1./2', '-x', 'z'],['x', '-x+1./2', 'z+1./2']],
                      '16d': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-y', '-x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['y', 'x', 'z+1./2'],['y+1./2', 'x+1./2', 'z']]},
              '109': {'4a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4']],
                      '8b': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[1./2, '-y+1./2', 'z+1./2'],[0., '-y', 'z'],['-y', 1./2, 'z+1./4'],['-y+1./2', 0., 'z+3./4'],['y+1./2', 0., 'z+3./4'],['y', 1./2, 'z+1./4']],
                      '16c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-y', '-x+1./2', 'z+1./4'],['-y+1./2', '-x', 'z+3./4'],['y+1./2', 'x', 'z+3./4'],['y', 'x+1./2', 'z+1./4']]},
              '110': {'8a': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z'],[0., 1./2, 'z+3./4'],[1./2, 0., 'z+1./4']],
                      '16b': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-x', 'y', 'z+1./2'],['-y', '-x+1./2', 'z+3./4'],['-y+1./2', '-x', 'z+1./4'],['y+1./2', 'x', 'z+1./4'],['y', 'x+1./2', 'z+3./4']]},
              '111': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 1./2]],
                      '1c': [[0., 0., 1./2]],
                      '1d': [[1./2, 1./2, 0]],
                      '2e': [[1./2, 0., 0],[0., 1./2, 0]],
                      '2f': [[1./2, 0., 1./2],[0., 1./2, 1./2]],
                      '2g': [[0., 0., 'z'],[0., 0., '-z']],
                      '2h': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                      '4i': [['x', 0., 0],['-x', 0., 0],[0., '-x', 0],[0., 'x', 0]],
                      '4j': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, '-x', 1./2],[1./2, 'x', 1./2]],
                      '4k': [['x', 0., 1./2],['-x', 0., 1./2],[0., '-x', 1./2],[0., 'x', 1./2]],
                      '4l': [['x', 1./2, 0],['-x', 1./2, 0],[1./2, '-x', 0],[1./2, 'x', 0]],
                      '4m': [[0., 1./2, 'z'],[1./2, 0., '-z'],[0., 1./2, '-z'],[1./2, 0., 'z']],
                      '4n': [['x', 'x', 'z'],['-x', '-x', 'z'],['x', '-x', '-z'],['-x', 'x', '-z']],
                      '8o': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['-x', 'y', '-z'],['x', '-y', '-z'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '112': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[1./2, 0., 1./4],[0., 1./2, 3./4]],
                      '2c': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                      '2d': [[0., 1./2, 1./4],[1./2, 0., 3./4]],
                      '2e': [[0., 0., 0],[0., 0., 1./2]],
                      '2f': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                      '4g': [['x', 0., 1./4],['-x', 0., 1./4],[0., '-x', 3./4],[0., 'x', 3./4]],
                      '4h': [[1./2, 'y', 1./4],[1./2, '-y', 1./4],['y', 1./2, 3./4],['-y', 1./2, 3./4]],
                      '4i': [['x', 1./2, 1./4],['-x', 1./2, 1./4],[1./2, '-x', 3./4],[1./2, 'x', 3./4]],
                      '4j': [[0., 'y', 1./4],[0., '-y', 1./4],['y', 0., 3./4],['-y', 0., 3./4]],
                      '4k': [[0., 0., 'z'],[0., 0., '-z'],[0., 0., '-z+1./2'],[0., 0., 'z+1./2']],
                      '4l': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, 'z+1./2']],
                      '4m': [[0., 1./2, 'z'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., 'z+1./2']],
                      '8n': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['-x', 'y', '-z+1./2'],['x', '-y', '-z+1./2'],['-y', '-x', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '113': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                      '2c': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                      '4d': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, '-z'],[1./2, 1./2, 'z']],
                      '4e': [['x', 'x+1./2', 'z'],['-x', '-x+1./2', 'z'],['x+1./2', '-x', '-z'],['-x+1./2', 'x', '-z']],
                      '8f': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['-y+1./2', '-x+1./2', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '114': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4c': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, 'z+1./2']],
                      '4d': [[0., 1./2, 'z'],[1./2, 0., '-z'],[1./2, 0., '-z+1./2'],[0., 1./2, 'z+1./2']],
                      '8e': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '115': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 0]],
                      '1c': [[1./2, 1./2, 1./2]],
                      '1d': [[0., 0., 1./2]],
                      '2e': [[0., 0., 'z'],[0., 0., '-z']],
                      '2f': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                      '2g': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                      '4h': [['x', 'x', 0],['-x', '-x', 0],['x', '-x', 0],['-x', 'x', 0]],
                      '4i': [['x', 'x', 1./2],['-x', '-x', 1./2],['x', '-x', 1./2],['-x', 'x', 1./2]],
                      '4j': [['x', 0., 'z'],['-x', 0., 'z'],[0., '-x', '-z'],[0., 'x', '-z']],
                      '4k': [['x', 1./2, 'z'],['-x', 1./2, 'z'],[1./2, '-x', '-z'],[1./2, 'x', '-z']],
                      '8l': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['y', 'x', '-z'],['-y', '-x', '-z']]},
              '116': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                      '2c': [[0., 0., 0],[0., 0., 1./2]],
                      '2d': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                      '4e': [['x', 'x', 1./4],['-x', '-x', 1./4],['x', '-x', 3./4],['-x', 'x', 3./4]],
                      '4f': [['x', 'x', 3./4],['-x', '-x', 3./4],['x', '-x', 1./4],['-x', 'x', 1./4]],
                      '4g': [[0., 0., 'z'],[0., 0., '-z'],[0., 0., 'z+1./2'],[0., 0., '-z+1./2']],
                      '4h': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2']],
                      '4i': [[0., 1./2, 'z'],[1./2, 0., '-z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z+1./2']],
                      '8j': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2']]},
              '117': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                      '2c': [[0., 1./2, 0],[1./2, 0., 0]],
                      '2d': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                      '4e': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                      '4f': [[0., 1./2, 'z'],[1./2, 0., '-z'],[1./2, 0., 'z'],[0., 1./2, '-z']],
                      '4g': [['x', 'x+1./2', 0],['-x', '-x+1./2', 0],['x+1./2', '-x', 0],['-x+1./2', 'x', 0]],
                      '4h': [['x', 'x+1./2', 1./2],['-x', '-x+1./2', 1./2],['x+1./2', '-x', 1./2],['-x+1./2', 'x', 1./2]],
                      '8i': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['y+1./2', 'x+1./2', '-z'],['-y+1./2', '-x+1./2', '-z']]},
              '118': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '2c': [[0., 1./2, 1./4],[1./2, 0., 3./4]],
                      '2d': [[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '4e': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2']],
                      '4f': [['x', '-x+1./2', 1./4],['-x', 'x+1./2', 1./4],['-x+1./2', '-x', 3./4],['x+1./2', 'x', 3./4]],
                      '4g': [['x', 'x+1./2', 1./4],['-x', '-x+1./2', 1./4],['x+1./2', '-x', 3./4],['-x+1./2', 'x', 3./4]],
                      '4h': [[0., 1./2, 'z'],[1./2, 0., '-z'],[1./2, 0., 'z+1./2'],[0., 1./2, '-z+1./2']],
                      '8i': [['x', 'y', 'z'],['-x', '-y', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2']]},
              '119': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '2c': [[0., 1./2, 1./4],[1./2, 0., 3./4]],
                      '2d': [[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                      '4f': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2']],
                      '8g': [['x', 'x', 0],['x+1./2', 'x+1./2', 1./2],['-x', '-x', 0],['-x+1./2', '-x+1./2', 1./2],['x', '-x', 0],['x+1./2', '-x+1./2', 1./2],['-x', 'x', 0],['-x+1./2', 'x+1./2', 1./2]],
                      '8h': [['x', 'x+1./2', 1./4],['x+1./2', 'x', 3./4],['-x', '-x+1./2', 1./4],['-x+1./2', '-x', 3./4],['x+1./2', '-x', 3./4],['x', '-x+1./2', 1./4],['-x+1./2', 'x', 3./4],['-x', 'x+1./2', 1./4]],
                      '8i': [['x', 0., 'z'],['x+1./2', 1./2, 'z+1./2'],['-x', 0., 'z'],['-x+1./2', 1./2, 'z+1./2'],[0., '-x', '-z'],[1./2, '-x+1./2', '-z+1./2'],[0., 'x', '-z'],[1./2, 'x+1./2', '-z+1./2']],
                      '16j': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['y', 'x', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2']]},
              '120': {'4a': [[0., 0., 1./4],[1./2, 1./2, 3./4],[0., 0., 3./4],[1./2, 1./2, 1./4]],
                      '4b': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4c': [[0., 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '4d': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                      '8e': [['x', 'x', 1./4],['x+1./2', 'x+1./2', 3./4],['-x', '-x', 1./4],['-x+1./2', '-x+1./2', 3./4],['x', '-x', 3./4],['x+1./2', '-x+1./2', 1./4],['-x', 'x', 3./4],['-x+1./2', 'x+1./2', 1./4]],
                      '8f': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z'],[0., 0., '-z+1./2'],[1./2, 1./2, '-z']],
                      '8g': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z'],[1./2, 0., '-z+1./2'],[0., 1./2, '-z']],
                      '8h': [['x', 'x+1./2', 0],['x+1./2', 'x', 1./2],['-x', '-x+1./2', 0],['-x+1./2', '-x', 1./2],['x+1./2', '-x', 0],['x', '-x+1./2', 1./2],['-x+1./2', 'x', 0],['-x', 'x+1./2', 1./2]],
                      '16i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['y', 'x', '-z+1./2'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z']]},
              '121': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4c': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                      '4d': [[0., 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                      '8f': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2]],
                      '8g': [['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0],[0., '-x', 1./2],[1./2, '-x+1./2', 0],[0., 'x', 1./2],[1./2, 'x+1./2', 0]],
                      '8h': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                      '8i': [['x', 'x', 'z'],['x+1./2', 'x+1./2', 'z+1./2'],['-x', '-x', 'z'],['-x+1./2', '-x+1./2', 'z+1./2'],['x', '-x', '-z'],['x+1./2', '-x+1./2', '-z+1./2'],['-x', 'x', '-z'],['-x+1./2', 'x+1./2', '-z+1./2']],
                      '16j': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['-y', '-x', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', 'x', 'z'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '122': {'4a': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 3./4],[0., 1./2, 1./4]],
                      '4b': [[0., 0., 1./2],[1./2, 1./2, 0],[1./2, 0., 1./4],[0., 1./2, 3./4]],
                      '8c': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 0., '-z+3./4'],[0., 1./2, '-z+1./4'],[1./2, 0., 'z+3./4'],[0., 1./2, 'z+1./4']],
                      '8d': [['x', 1./4, 1/8],['x+1./2', 3./4, 5/8],['-x', 3./4, 1/8],['-x+1./2', 1./4, 5/8],[1./4, '-x', 7/8],[3./4, '-x+1./2', 3/8],[3./4, 'x', 7/8],[1./4, 'x+1./2', 3/8]],
                      '16e': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+3./4'],['-x', 'y+1./2', '-z+1./4'],['x+1./2', '-y', '-z+3./4'],['x', '-y+1./2', '-z+1./4'],['-y+1./2', '-x', 'z+3./4'],['-y', '-x+1./2', 'z+1./4'],['y+1./2', 'x', 'z+3./4'],['y', 'x+1./2', 'z+1./4']]},
              '123': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '1c': [[1./2, 1./2, 0]],
                      '1d': [[1./2, 1./2, 1./2]],
                      '2e': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                      '2f': [[0., 1./2, 0],[1./2, 0., 0]],
                      '2g': [[0., 0., 'z'],[0., 0., '-z']],
                      '2h': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z']],
                      '4i': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z'],[1./2, 0., '-z']],
                      '4j': [['x', 'x', 0],['-x', '-x', 0],['-x', 'x', 0],['x', '-x', 0]],
                      '4k': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x', 'x', 1./2],['x', '-x', 1./2]],
                      '4l': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0]],
                      '4m': [['x', 0., 1./2],['-x', 0., 1./2],[0., 'x', 1./2],[0., '-x', 1./2]],
                      '4n': [['x', 1./2, 0],['-x', 1./2, 0],[1./2, 'x', 0],[1./2, '-x', 0]],
                      '4o': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2]],
                      '8p': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 0],['y', '-x', 0],['-x', 'y', 0],['x', '-y', 0],['y', 'x', 0],['-y', '-x', 0]],
                      '8q': [['x', 'y', 1./2],['-x', '-y', 1./2],['-y', 'x', 1./2],['y', '-x', 1./2],['-x', 'y', 1./2],['x', '-y', 1./2],['y', 'x', 1./2],['-y', '-x', 1./2]],
                      '8r': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', 'z'],['x', '-x', 'z'],['-x', 'x', '-z'],['x', '-x', '-z'],['x', 'x', '-z'],['-x', '-x', '-z']],
                      '8s': [['x', 0., 'z'],['-x', 0., 'z'],[0., 'x', 'z'],[0., '-x', 'z'],['-x', 0., '-z'],['x', 0., '-z'],[0., 'x', '-z'],[0., '-x', '-z']],
                      '8t': [['x', 1./2, 'z'],['-x', 1./2, 'z'],[1./2, 'x', 'z'],[1./2, '-x', 'z'],['-x', 1./2, '-z'],['x', 1./2, '-z'],[1./2, 'x', '-z'],[1./2, '-x', '-z']],
                      '16u': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '124': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[0., 0., 0],[0., 0., 1./2]],
                      '2c': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                      '2d': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                      '4e': [[0., 1./2, 0],[1./2, 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                      '4f': [[0., 1./2, 1./4],[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4]],
                      '4g': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2']],
                      '4h': [[1./2, 1./2, 'z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, 'z+1./2']],
                      '8i': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2']],
                      '8j': [['x', 'x', 1./4],['-x', '-x', 1./4],['-x', 'x', 1./4],['x', '-x', 1./4],['-x', '-x', 3./4],['x', 'x', 3./4],['x', '-x', 3./4],['-x', 'x', 3./4]],
                      '8k': [['x', 0., 1./4],['-x', 0., 1./4],[0., 'x', 1./4],[0., '-x', 1./4],['-x', 0., 3./4],['x', 0., 3./4],[0., '-x', 3./4],[0., 'x', 3./4]],
                      '8l': [['x', 1./2, 1./4],['-x', 1./2, 1./4],[1./2, 'x', 1./4],[1./2, '-x', 1./4],['-x', 1./2, 3./4],['x', 1./2, 3./4],[1./2, '-x', 3./4],[1./2, 'x', 3./4]],
                      '8m': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 0],['y', '-x', 0],['-x', 'y', 1./2],['x', '-y', 1./2],['y', 'x', 1./2],['-y', '-x', 1./2]],
                      '16n': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x', 'y', '-z+1./2'],['x', '-y', '-z+1./2'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-y', '-x', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '125:1': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                        '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                        '2c': [[0., 1./2, 0],[1./2, 0., 0]],
                        '2d': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                        '4e': [[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 0]],
                        '4f': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                        '4g': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, '-z'],[1./2, 1./2, 'z']],
                        '4h': [[0., 1./2, 'z'],[1./2, 0., 'z'],[0., 1./2, '-z'],[1./2, 0., '-z']],
                        '8i': [['x', 'x', 0],['-x', '-x', 0],['-x', 'x', 0],['x', '-x', 0],['-x+1./2', '-x+1./2', 0],['x+1./2', 'x+1./2', 0],['x+1./2', '-x+1./2', 0],['-x+1./2', 'x+1./2', 0]],
                        '8j': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x', 'x', 1./2],['x', '-x', 1./2],['-x+1./2', '-x+1./2', 1./2],['x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2],['-x+1./2', 'x+1./2', 1./2]],
                        '8k': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0],[1./2, '-x+1./2', 0],[1./2, 'x+1./2', 0]],
                        '8l': [['x', 0., 1./2],['-x', 0., 1./2],[0., 'x', 1./2],[0., '-x', 1./2],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2],[1./2, '-x+1./2', 1./2],[1./2, 'x+1./2', 1./2]],
                        '8m': [['x', 'x+1./2', 'z'],['-x', '-x+1./2', 'z'],['-x+1./2', 'x', 'z'],['x+1./2', '-x', 'z'],['-x', 'x+1./2', '-z'],['x', '-x+1./2', '-z'],['x+1./2', 'x', '-z'],['-x+1./2', '-x', '-z']],
                        '16n': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x+1./2', '-z'],['-y+1./2', 'x+1./2', '-z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-y+1./2', '-x+1./2', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '125:2': {'2a': [[1./4, 1./4, 0],[3./4, 3./4, 0]],
                        '2b': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2]],
                        '2c': [[3./4, 1./4, 0],[1./4, 3./4, 0]],
                        '2d': [[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                        '4e': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0]],
                        '4f': [[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '4g': [[1./4, 1./4, 'z'],[1./4, 1./4, '-z'],[3./4, 3./4, '-z'],[3./4, 3./4, 'z']],
                        '4h': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z']],
                        '8i': [['x', 'x', 0],['-x+1./2', '-x+1./2', 0],['-x+1./2', 'x', 0],['x', '-x+1./2', 0],['-x', '-x', 0],['x+1./2', 'x+1./2', 0],['x+1./2', '-x', 0],['-x', 'x+1./2', 0]],
                        '8j': [['x', 'x', 1./2],['-x+1./2', '-x+1./2', 1./2],['-x+1./2', 'x', 1./2],['x', '-x+1./2', 1./2],['-x', '-x', 1./2],['x+1./2', 'x+1./2', 1./2],['x+1./2', '-x', 1./2],['-x', 'x+1./2', 1./2]],
                        '8k': [['x', 1./4, 0],['-x+1./2', 1./4, 0],[1./4, 'x', 0],[1./4, '-x+1./2', 0],['-x', 3./4, 0],['x+1./2', 3./4, 0],[3./4, '-x', 0],[3./4, 'x+1./2', 0]],
                        '8l': [['x', 1./4, 1./2],['-x+1./2', 1./4, 1./2],[1./4, 'x', 1./2],[1./4, '-x+1./2', 1./2],['-x', 3./4, 1./2],['x+1./2', 3./4, 1./2],[3./4, '-x', 1./2],[3./4, 'x+1./2', 1./2]],
                        '8m': [['x', '-x', 'z'],['-x+1./2', 'x+1./2', 'z'],['x+1./2', 'x', 'z'],['-x', '-x+1./2', 'z'],['-x+1./2', '-x', '-z'],['x', 'x+1./2', '-z'],['-x', 'x', '-z'],['x+1./2', '-x+1./2', '-z']],
                        '16n': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z'],['y', '-x+1./2', 'z'],['-x+1./2', 'y', '-z'],['x', '-y+1./2', '-z'],['y', 'x', '-z'],['-y+1./2', '-x+1./2', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z'],['-y', 'x+1./2', '-z'],['x+1./2', '-y', 'z'],['-x', 'y+1./2', 'z'],['-y', '-x', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '126:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                        '4c': [[1./2, 0., 0],[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2]],
                        '4d': [[1./2, 0., 1./4],[0., 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4]],
                        '4e': [[0., 0., 'z'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, 'z+1./2']],
                        '8f': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4]],
                        '8g': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 0., '-z'],[0., 1./2, '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2']],
                        '8h': [['x', 'x', 0],['-x', '-x', 0],['-x', 'x', 0],['x', '-x', 0],['-x+1./2', '-x+1./2', 1./2],['x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2],['-x+1./2', 'x+1./2', 1./2]],
                        '8i': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2],[1./2, '-x+1./2', 1./2],[1./2, 'x+1./2', 1./2]],
                        '8j': [['x', 0., 1./2],['-x', 0., 1./2],[0., 'x', 1./2],[0., '-x', 1./2],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0],[1./2, '-x+1./2', 0],[1./2, 'x+1./2', 0]],
                        '16k': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '126:2': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '2b': [[1./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                        '4c': [[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4]],
                        '4d': [[1./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2]],
                        '4e': [[1./4, 1./4, 'z'],[1./4, 1./4, '-z+1./2'],[3./4, 3./4, '-z'],[3./4, 3./4, 'z+1./2']],
                        '8f': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 1./2]],
                        '8g': [[1./4, 3./4, 'z'],[3./4, 1./4, 'z'],[1./4, 3./4, '-z+1./2'],[3./4, 1./4, '-z+1./2'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z'],[3./4, 1./4, 'z+1./2'],[1./4, 3./4, 'z+1./2']],
                        '8h': [['x', 'x', 1./4],['-x+1./2', '-x+1./2', 1./4],['-x+1./2', 'x', 1./4],['x', '-x+1./2', 1./4],['-x', '-x', 3./4],['x+1./2', 'x+1./2', 3./4],['x+1./2', '-x', 3./4],['-x', 'x+1./2', 3./4]],
                        '8i': [['x', 1./4, 1./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, '-x+1./2', 1./4],['-x', 3./4, 3./4],['x+1./2', 3./4, 3./4],[3./4, '-x', 3./4],[3./4, 'x+1./2', 3./4]],
                        '8j': [['x', 3./4, 1./4],['-x+1./2', 3./4, 1./4],[3./4, 'x', 1./4],[3./4, '-x+1./2', 1./4],['-x', 1./4, 3./4],['x+1./2', 1./4, 3./4],[1./4, '-x', 3./4],[1./4, 'x+1./2', 3./4]],
                        '16k': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z'],['y', '-x+1./2', 'z'],['-x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['y', 'x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z'],['-y', 'x+1./2', '-z'],['x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2'],['-y', '-x', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '127': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                      '2c': [[0., 1./2, 1./2],[1./2, 0., 1./2]],
                      '2d': [[0., 1./2, 0],[1./2, 0., 0]],
                      '4e': [[0., 0., 'z'],[1./2, 1./2, '-z'],[0., 0., '-z'],[1./2, 1./2, 'z']],
                      '4f': [[0., 1./2, 'z'],[1./2, 0., 'z'],[1./2, 0., '-z'],[0., 1./2, '-z']],
                      '4g': [['x', 'x+1./2', 0],['-x', '-x+1./2', 0],['-x+1./2', 'x', 0],['x+1./2', '-x', 0]],
                      '4h': [['x', 'x+1./2', 1./2],['-x', '-x+1./2', 1./2],['-x+1./2', 'x', 1./2],['x+1./2', '-x', 1./2]],
                      '8i': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 0],['y', '-x', 0],['-x+1./2', 'y+1./2', 0],['x+1./2', '-y+1./2', 0],['y+1./2', 'x+1./2', 0],['-y+1./2', '-x+1./2', 0]],
                      '8j': [['x', 'y', 1./2],['-x', '-y', 1./2],['-y', 'x', 1./2],['y', '-x', 1./2],['-x+1./2', 'y+1./2', 1./2],['x+1./2', '-y+1./2', 1./2],['y+1./2', 'x+1./2', 1./2],['-y+1./2', '-x+1./2', 1./2]],
                      '8k': [['x', 'x+1./2', 'z'],['-x', '-x+1./2', 'z'],['-x+1./2', 'x', 'z'],['x+1./2', '-x', 'z'],['-x+1./2', 'x', '-z'],['x+1./2', '-x', '-z'],['x', 'x+1./2', '-z'],['-x', '-x+1./2', '-z']],
                      '16l': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['y+1./2', 'x+1./2', '-z'],['-y+1./2', '-x+1./2', '-z'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-y+1./2', '-x+1./2', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '128': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4c': [[0., 1./2, 0],[1./2, 0., 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                      '4d': [[0., 1./2, 1./4],[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4]],
                      '4e': [[0., 0., 'z'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z'],[1./2, 1./2, 'z+1./2']],
                      '8f': [[0., 1./2, 'z'],[1./2, 0., 'z'],[1./2, 0., '-z+1./2'],[0., 1./2, '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z'],[1./2, 0., 'z+1./2'],[0., 1./2, 'z+1./2']],
                      '8g': [['x', 'x+1./2', 1./4],['-x', '-x+1./2', 1./4],['-x+1./2', 'x', 1./4],['x+1./2', '-x', 1./4],['-x', '-x+1./2', 3./4],['x', 'x+1./2', 3./4],['x+1./2', '-x', 3./4],['-x+1./2', 'x', 3./4]],
                      '8h': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 0],['y', '-x', 0],['-x+1./2', 'y+1./2', 1./2],['x+1./2', '-y+1./2', 1./2],['y+1./2', 'x+1./2', 1./2],['-y+1./2', '-x+1./2', 1./2]],
                      '16i': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z'],['y', '-x', 'z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '129:1': {'2a': [[0., 0., 0],[1./2, 1./2, 0]],
                        '2b': [[0., 0., 1./2],[1./2, 1./2, 1./2]],
                        '2c': [[0., 1./2, 'z'],[1./2, 0., '-z']],
                        '4d': [[1./4, 1./4, 0],[3./4, 3./4, 0],[1./4, 3./4, 0],[3./4, 1./4, 0]],
                        '4e': [[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2]],
                        '4f': [[0., 0., 'z'],[1./2, 1./2, 'z'],[1./2, 1./2, '-z'],[0., 0., '-z']],
                        '8g': [['x', 'x', 0],['-x', '-x', 0],['-x+1./2', 'x+1./2', 0],['x+1./2', '-x+1./2', 0],['-x+1./2', '-x+1./2', 0],['x+1./2', 'x+1./2', 0],['x', '-x', 0],['-x', 'x', 0]],
                        '8h': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2],['-x+1./2', '-x+1./2', 1./2],['x+1./2', 'x+1./2', 1./2],['x', '-x', 1./2],['-x', 'x', 1./2]],
                        '8i': [[0., 'y', 'z'],[0., '-y', 'z'],['-y+1./2', 1./2, 'z'],['y+1./2', 1./2, 'z'],[1./2, 'y+1./2', '-z'],[1./2, '-y+1./2', '-z'],['y', 0., '-z'],['-y', 0., '-z']],
                        '8j': [['x', 'x+1./2', 'z'],['-x', '-x+1./2', 'z'],['-x', 'x+1./2', 'z'],['x', '-x+1./2', 'z'],['-x+1./2', 'x', '-z'],['x+1./2', '-x', '-z'],['x+1./2', 'x', '-z'],['-x+1./2', '-x', '-z']],
                        '16k': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z'],['y+1./2', '-x+1./2', 'z'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y+1./2', '-z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-y+1./2', '-x+1./2', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '129:2': {'2a': [[3./4, 1./4, 0],[1./4, 3./4, 0]],
                        '2b': [[3./4, 1./4, 1./2],[1./4, 3./4, 1./2]],
                        '2c': [[1./4, 1./4, 'z'],[3./4, 3./4, '-z']],
                        '4d': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0]],
                        '4e': [[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '4f': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z']],
                        '8g': [['x', '-x', 0],['-x+1./2', 'x+1./2', 0],['x+1./2', 'x', 0],['-x', '-x+1./2', 0],['-x', 'x', 0],['x+1./2', '-x+1./2', 0],['-x+1./2', '-x', 0],['x', 'x+1./2', 0]],
                        '8h': [['x', '-x', 1./2],['-x+1./2', 'x+1./2', 1./2],['x+1./2', 'x', 1./2],['-x', '-x+1./2', 1./2],['-x', 'x', 1./2],['x+1./2', '-x+1./2', 1./2],['-x+1./2', '-x', 1./2],['x', 'x+1./2', 1./2]],
                        '8i': [[1./4, 'y', 'z'],[1./4, '-y+1./2', 'z'],['-y+1./2', 1./4, 'z'],['y', 1./4, 'z'],[3./4, 'y+1./2', '-z'],[3./4, '-y', '-z'],['y+1./2', 3./4, '-z'],['-y', 3./4, '-z']],
                        '8j': [['x', 'x', 'z'],['-x+1./2', '-x+1./2', 'z'],['-x+1./2', 'x', 'z'],['x', '-x+1./2', 'z'],['-x', 'x+1./2', '-z'],['x+1./2', '-x', '-z'],['x+1./2', 'x+1./2', '-z'],['-x', '-x', '-z']],
                        '16k': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z'],['y', '-x+1./2', 'z'],['-x', 'y+1./2', '-z'],['x+1./2', '-y', '-z'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z'],['-y', 'x+1./2', '-z'],['x', '-y+1./2', 'z'],['-x+1./2', 'y', 'z'],['-y+1./2', '-x+1./2', 'z'],['y', 'x', 'z']]},
              '130:1': {'4a': [[0., 0., 1./4],[1./2, 1./2, 1./4],[1./2, 1./2, 3./4],[0., 0., 3./4]],
                        '4b': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 1./2, 1./2],[0., 0., 1./2]],
                        '4c': [[0., 1./2, 'z'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, 'z+1./2']],
                        '8d': [[1./4, 1./4, 0],[3./4, 3./4, 0],[1./4, 3./4, 0],[3./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[1./4, 1./4, 1./2],[3./4, 3./4, 1./2]],
                        '8e': [[0., 0., 'z'],[1./2, 1./2, 'z'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z+1./2'],[1./2, 1./2, '-z'],[0., 0., '-z'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z+1./2']],
                        '8f': [['x', 'x', 1./4],['-x', '-x', 1./4],['-x+1./2', 'x+1./2', 1./4],['x+1./2', '-x+1./2', 1./4],['-x+1./2', '-x+1./2', 3./4],['x+1./2', 'x+1./2', 3./4],['x', '-x', 3./4],['-x', 'x', 3./4]],
                        '16g': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z'],['y+1./2', '-x+1./2', 'z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x+1./2', 'y+1./2', '-z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '130:2': {'4a': [[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4]],
                        '4b': [[3./4, 1./4, 0],[1./4, 3./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2]],
                        '4c': [[1./4, 1./4, 'z'],[3./4, 3./4, '-z+1./2'],[3./4, 3./4, '-z'],[1./4, 1./4, 'z+1./2']],
                        '8d': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 1./2],[0., 0., 1./2]],
                        '8e': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z'],[1./4, 3./4, '-z+1./2'],[3./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z'],[3./4, 1./4, 'z+1./2'],[1./4, 3./4, 'z+1./2']],
                        '8f': [['x', '-x', 1./4],['-x+1./2', 'x+1./2', 1./4],['x+1./2', 'x', 1./4],['-x', '-x+1./2', 1./4],['-x', 'x', 3./4],['x+1./2', '-x+1./2', 3./4],['-x+1./2', '-x', 3./4],['x', 'x+1./2', 3./4]],
                        '16g': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z'],['y', '-x+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z'],['-y', 'x+1./2', '-z'],['x', '-y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '131': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                      '2b': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                      '2c': [[0., 1./2, 0],[1./2, 0., 1./2]],
                      '2d': [[0., 1./2, 1./2],[1./2, 0., 0]],
                      '2e': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2f': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                      '4g': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2']],
                      '4h': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, '-z+1./2']],
                      '4i': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2']],
                      '4j': [['x', 0., 0],['-x', 0., 0],[0., 'x', 1./2],[0., '-x', 1./2]],
                      '4k': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 0],[1./2, '-x', 0]],
                      '4l': [['x', 0., 1./2],['-x', 0., 1./2],[0., 'x', 0],[0., '-x', 0]],
                      '4m': [['x', 1./2, 0],['-x', 1./2, 0],[1./2, 'x', 1./2],[1./2, '-x', 1./2]],
                      '8n': [['x', 'x', 1./4],['-x', '-x', 1./4],['-x', 'x', 3./4],['x', '-x', 3./4],['-x', '-x', 3./4],['x', 'x', 3./4],['x', '-x', 1./4],['-x', 'x', 1./4]],
                      '8o': [[0., 'y', 'z'],[0., '-y', 'z'],['-y', 0., 'z+1./2'],['y', 0., 'z+1./2'],[0., 'y', '-z'],[0., '-y', '-z'],['y', 0., '-z+1./2'],['-y', 0., '-z+1./2']],
                      '8p': [[1./2, 'y', 'z'],[1./2, '-y', 'z'],['-y', 1./2, 'z+1./2'],['y', 1./2, 'z+1./2'],[1./2, 'y', '-z'],[1./2, '-y', '-z'],['y', 1./2, '-z+1./2'],['-y', 1./2, '-z+1./2']],
                      '8q': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 1./2],['y', '-x', 1./2],['-x', 'y', 0],['x', '-y', 0],['y', 'x', 1./2],['-y', '-x', 1./2]],
                      '16r': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['-x', 'y', '-z'],['x', '-y', '-z'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z+1./2'],['-y', 'x', '-z+1./2'],['x', '-y', 'z'],['-x', 'y', 'z'],['-y', '-x', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '132': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                      '2b': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2c': [[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                      '2d': [[1./2, 1./2, 1./4],[1./2, 1./2, 3./4]],
                      '4e': [[0., 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '4f': [[0., 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 0., 0]],
                      '4g': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z+1./2'],[0., 0., '-z']],
                      '4h': [[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, '-z']],
                      '4i': [['x', 'x', 0],['-x', '-x', 0],['-x', 'x', 1./2],['x', '-x', 1./2]],
                      '4j': [['x', 'x', 1./2],['-x', '-x', 1./2],['-x', 'x', 0],['x', '-x', 0]],
                      '8k': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z']],
                      '8l': [['x', 0., 1./4],['-x', 0., 1./4],[0., 'x', 3./4],[0., '-x', 3./4],['-x', 0., 3./4],['x', 0., 3./4],[0., '-x', 1./4],[0., 'x', 1./4]],
                      '8m': [['x', 1./2, 1./4],['-x', 1./2, 1./4],[1./2, 'x', 3./4],[1./2, '-x', 3./4],['-x', 1./2, 3./4],['x', 1./2, 3./4],[1./2, '-x', 1./4],[1./2, 'x', 1./4]],
                      '8n': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 1./2],['y', '-x', 1./2],['-x', 'y', 1./2],['x', '-y', 1./2],['y', 'x', 0],['-y', '-x', 0]],
                      '8o': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', 'z+1./2'],['x', '-x', 'z+1./2'],['-x', 'x', '-z+1./2'],['x', '-x', '-z+1./2'],['x', 'x', '-z'],['-x', '-x', '-z']],
                      '16p': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['-x', 'y', '-z+1./2'],['x', '-y', '-z+1./2'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z+1./2'],['-y', 'x', '-z+1./2'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '133:1': {'4a': [[0., 1./2, 1./4],[0., 1./2, 3./4],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                        '4b': [[0., 0., 1./4],[1./2, 1./2, 3./4],[1./2, 1./2, 1./4],[0., 0., 3./4]],
                        '4c': [[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 0., 0]],
                        '4d': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 0]],
                        '8e': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[3./4, 3./4, 3./4],[1./4, 1./4, 3./4]],
                        '8f': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[1./2, 0., 'z'],[1./2, 0., 'z+1./2']],
                        '8g': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z'],[1./2, 1./2, 'z'],[0., 0., 'z+1./2']],
                        '8h': [['x', 0., 1./4],['-x', 0., 1./4],[1./2, 'x+1./2', 3./4],[1./2, '-x+1./2', 3./4],['-x+1./2', 1./2, 1./4],['x+1./2', 1./2, 1./4],[0., '-x', 3./4],[0., 'x', 3./4]],
                        '8i': [['x', 0., 3./4],['-x', 0., 3./4],[1./2, 'x+1./2', 1./4],[1./2, '-x+1./2', 1./4],['-x+1./2', 1./2, 3./4],['x+1./2', 1./2, 3./4],[0., '-x', 1./4],[0., 'x', 1./4]],
                        '8j': [['x', 'x+1./2', 0],['-x', '-x+1./2', 0],['-x', 'x+1./2', 1./2],['x', '-x+1./2', 1./2],['-x+1./2', '-x', 1./2],['x+1./2', 'x', 1./2],['x+1./2', '-x', 0],['-x+1./2', 'x', 0]],
                        '16k': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x', 'y', '-z+1./2'],['x', '-y', '-z+1./2'],['y+1./2', 'x+1./2', '-z'],['-y+1./2', '-x+1./2', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-y', '-x', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '133:2': {'4a': [[1./4, 1./4, 0],[1./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 3./4, 1./2]],
                        '4b': [[3./4, 1./4, 0],[1./4, 3./4, 1./2],[1./4, 3./4, 0],[3./4, 1./4, 1./2]],
                        '4c': [[1./4, 1./4, 1./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4]],
                        '4d': [[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                        '8e': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 1./2]],
                        '8f': [[1./4, 1./4, 'z'],[1./4, 1./4, 'z+1./2'],[1./4, 1./4, '-z'],[1./4, 1./4, '-z+1./2'],[3./4, 3./4, '-z'],[3./4, 3./4, '-z+1./2'],[3./4, 3./4, 'z'],[3./4, 3./4, 'z+1./2']],
                        '8g': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z+1./2'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z+1./2'],[1./4, 3./4, 'z'],[3./4, 1./4, 'z+1./2']],
                        '8h': [['x', 1./4, 0],['-x+1./2', 1./4, 0],[1./4, 'x', 1./2],[1./4, '-x+1./2', 1./2],['-x', 3./4, 0],['x+1./2', 3./4, 0],[3./4, '-x', 1./2],[3./4, 'x+1./2', 1./2]],
                        '8i': [['x', 1./4, 1./2],['-x+1./2', 1./4, 1./2],[1./4, 'x', 0],[1./4, '-x+1./2', 0],['-x', 3./4, 1./2],['x+1./2', 3./4, 1./2],[3./4, '-x', 0],[3./4, 'x+1./2', 0]],
                        '8j': [['x', 'x', 1./4],['-x+1./2', '-x+1./2', 1./4],['-x+1./2', 'x', 3./4],['x', '-x+1./2', 3./4],['-x', '-x', 3./4],['x+1./2', 'x+1./2', 3./4],['x+1./2', '-x', 1./4],['-x', 'x+1./2', 1./4]],
                        '16k': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z+1./2'],['y', '-x+1./2', 'z+1./2'],['-x+1./2', 'y', '-z'],['x', '-y+1./2', '-z'],['y', 'x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z+1./2'],['-y', 'x+1./2', '-z+1./2'],['x+1./2', '-y', 'z'],['-x', 'y+1./2', 'z'],['-y', '-x', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '134:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                        '4c': [[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 0., 0]],
                        '4d': [[0., 1./2, 1./4],[0., 1./2, 3./4],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                        '4e': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4]],
                        '4f': [[3./4, 3./4, 3./4],[1./4, 1./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4]],
                        '4g': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                        '8h': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z']],
                        '8i': [['x', 0., 0],['-x', 0., 0],[1./2, 'x+1./2', 1./2],[1./2, '-x+1./2', 1./2],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2],[0., '-x', 0],[0., 'x', 0]],
                        '8j': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x+1./2', 0],[1./2, '-x+1./2', 0],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0],[0., '-x', 1./2],[0., 'x', 1./2]],
                        '8k': [['x', 'x+1./2', 1./4],['-x', '-x+1./2', 1./4],['-x', 'x+1./2', 3./4],['x', '-x+1./2', 3./4],['-x+1./2', '-x', 1./4],['x+1./2', 'x', 1./4],['x+1./2', '-x', 3./4],['-x+1./2', 'x', 3./4]],
                        '8l': [['x', 'x+1./2', 3./4],['-x', '-x+1./2', 3./4],['-x', 'x+1./2', 1./4],['x', '-x+1./2', 1./4],['-x+1./2', '-x', 3./4],['x+1./2', 'x', 3./4],['x+1./2', '-x', 1./4],['-x+1./2', 'x', 1./4]],
                        '8m': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x+1./2', 'x+1./2', 'z+1./2'],['x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', '-z'],['x', '-x', '-z'],['x+1./2', 'x+1./2', '-z+1./2'],['-x+1./2', '-x+1./2', '-z+1./2']],
                        '16n': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x', 'y', '-z'],['x', '-y', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '134:2': {'2a': [[1./4, 3./4, 1./4],[3./4, 1./4, 3./4]],
                        '2b': [[3./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                        '4c': [[1./4, 1./4, 1./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4]],
                        '4d': [[1./4, 1./4, 0],[1./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 3./4, 1./2]],
                        '4e': [[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0]],
                        '4f': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '4g': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[3./4, 1./4, '-z+1./2'],[1./4, 3./4, '-z']],
                        '8h': [[1./4, 1./4, 'z'],[1./4, 1./4, 'z+1./2'],[1./4, 1./4, '-z+1./2'],[1./4, 1./4, '-z'],[3./4, 3./4, '-z'],[3./4, 3./4, '-z+1./2'],[3./4, 3./4, 'z+1./2'],[3./4, 3./4, 'z']],
                        '8i': [['x', 1./4, 3./4],['-x+1./2', 1./4, 3./4],[1./4, 'x', 1./4],[1./4, '-x+1./2', 1./4],['-x', 3./4, 1./4],['x+1./2', 3./4, 1./4],[3./4, '-x', 3./4],[3./4, 'x+1./2', 3./4]],
                        '8j': [['x', 1./4, 1./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 3./4],[1./4, '-x+1./2', 3./4],['-x', 3./4, 3./4],['x+1./2', 3./4, 3./4],[3./4, '-x', 1./4],[3./4, 'x+1./2', 1./4]],
                        '8k': [['x', 'x', 0],['-x+1./2', '-x+1./2', 0],['-x+1./2', 'x', 1./2],['x', '-x+1./2', 1./2],['-x', '-x', 0],['x+1./2', 'x+1./2', 0],['x+1./2', '-x', 1./2],['-x', 'x+1./2', 1./2]],
                        '8l': [['x', 'x', 1./2],['-x+1./2', '-x+1./2', 1./2],['-x+1./2', 'x', 0],['x', '-x+1./2', 0],['-x', '-x', 1./2],['x+1./2', 'x+1./2', 1./2],['x+1./2', '-x', 0],['-x', 'x+1./2', 0]],
                        '8m': [['x', '-x', 'z'],['-x+1./2', 'x+1./2', 'z'],['x+1./2', 'x', 'z+1./2'],['-x', '-x+1./2', 'z+1./2'],['-x+1./2', '-x', '-z+1./2'],['x', 'x+1./2', '-z+1./2'],['-x', 'x', '-z'],['x+1./2', '-x+1./2', '-z']],
                        '16n': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z+1./2'],['y', '-x+1./2', 'z+1./2'],['-x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['y', 'x', '-z'],['-y+1./2', '-x+1./2', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z+1./2'],['-y', 'x+1./2', '-z+1./2'],['x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2'],['-y', '-x', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '135': {'4a': [[0., 0., 0],[0., 0., 1./2],[1./2, 1./2, 0],[1./2, 1./2, 1./2]],
                      '4b': [[0., 0., 1./4],[0., 0., 3./4],[1./2, 1./2, 3./4],[1./2, 1./2, 1./4]],
                      '4c': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                      '4d': [[0., 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '8e': [[0., 0., 'z'],[0., 0., 'z+1./2'],[1./2, 1./2, '-z'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2'],[1./2, 1./2, 'z'],[1./2, 1./2, 'z+1./2']],
                      '8f': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2']],
                      '8g': [['x', 'x+1./2', 1./4],['-x', '-x+1./2', 1./4],['-x+1./2', 'x', 3./4],['x+1./2', '-x', 3./4],['-x', '-x+1./2', 3./4],['x', 'x+1./2', 3./4],['x+1./2', '-x', 1./4],['-x+1./2', 'x', 1./4]],
                      '8h': [['x', 'y', 0],['-x', '-y', 0],['-y', 'x', 1./2],['y', '-x', 1./2],['-x+1./2', 'y+1./2', 0],['x+1./2', '-y+1./2', 0],['y+1./2', 'x+1./2', 1./2],['-y+1./2', '-x+1./2', 1./2]],
                      '16i': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y', 'x', 'z+1./2'],['y', '-x', 'z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['y', '-x', '-z+1./2'],['-y', 'x', '-z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y+1./2', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '136': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4c': [[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 0., 0]],
                      '4d': [[0., 1./2, 1./4],[0., 1./2, 3./4],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                      '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z']],
                      '4f': [['x', 'x', 0],['-x', '-x', 0],['-x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2]],
                      '4g': [['x', '-x', 0],['-x', 'x', 0],['x+1./2', 'x+1./2', 1./2],['-x+1./2', '-x+1./2', 1./2]],
                      '8h': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z'],[0., 1./2, '-z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z']],
                      '8i': [['x', 'y', 0],['-x', '-y', 0],['-y+1./2', 'x+1./2', 1./2],['y+1./2', '-x+1./2', 1./2],['-x+1./2', 'y+1./2', 1./2],['x+1./2', '-y+1./2', 1./2],['y', 'x', 0],['-y', '-x', 0]],
                      '8j': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x+1./2', 'x+1./2', 'z+1./2'],['x+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', 'x+1./2', '-z+1./2'],['x+1./2', '-x+1./2', '-z+1./2'],['x', 'x', '-z'],['-x', '-x', '-z']],
                      '16k': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x', '-y', '-z'],['x', 'y', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y', '-x', 'z'],['y', 'x', 'z']]},
              '137:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                        '4c': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z']],
                        '4d': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z']],
                        '8e': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4]],
                        '8f': [['x', 'x', 0],['-x', '-x', 0],['-x+1./2', 'x+1./2', 1./2],['x+1./2', '-x+1./2', 1./2],['-x+1./2', '-x+1./2', 1./2],['x+1./2', 'x+1./2', 1./2],['x', '-x', 0],['-x', 'x', 0]],
                        '8g': [[0., 'y', 'z'],[0., '-y', 'z'],['-y+1./2', 1./2, 'z+1./2'],['y+1./2', 1./2, 'z+1./2'],[1./2, 'y+1./2', '-z+1./2'],[1./2, '-y+1./2', '-z+1./2'],['y', 0., '-z'],['-y', 0., '-z']],
                        '16h': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z+1./2'],['y', 'x', '-z'],['-y', '-x', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '137:2': {'2a': [[3./4, 1./4, 3./4],[1./4, 3./4, 1./4]],
                        '2b': [[3./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                        '4c': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z+1./2']],
                        '4d': [[1./4, 1./4, 'z'],[1./4, 1./4, 'z+1./2'],[3./4, 3./4, '-z'],[3./4, 3./4, '-z+1./2']],
                        '8e': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 0],[1./2, 1./2, 1./2],[0., 0., 1./2]],
                        '8f': [['x', '-x', 1./4],['-x+1./2', 'x+1./2', 1./4],['x+1./2', 'x', 3./4],['-x', '-x+1./2', 3./4],['-x', 'x', 3./4],['x+1./2', '-x+1./2', 3./4],['-x+1./2', '-x', 1./4],['x', 'x+1./2', 1./4]],
                        '8g': [[1./4, 'y', 'z'],[1./4, '-y+1./2', 'z'],['-y+1./2', 1./4, 'z+1./2'],['y', 1./4, 'z+1./2'],[3./4, 'y+1./2', '-z'],[3./4, '-y', '-z'],['y+1./2', 3./4, '-z+1./2'],['-y', 3./4, '-z+1./2']],
                        '16h': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z+1./2'],['y', '-x+1./2', 'z+1./2'],['-x', 'y+1./2', '-z'],['x+1./2', '-y', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z+1./2'],['-y', 'x+1./2', '-z+1./2'],['x', '-y+1./2', 'z'],['-x+1./2', 'y', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', 'x', 'z+1./2']]},
              '138:1': {'4a': [[0., 0., 1./4],[1./2, 1./2, 3./4],[1./2, 1./2, 1./4],[0., 0., 3./4]],
                        '4b': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 1./2, 0],[0., 0., 1./2]],
                        '4c': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4]],
                        '4d': [[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                        '4e': [[0., 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., '-z'],[1./2, 0., '-z+1./2']],
                        '8f': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[1./2, 1./2, '-z'],[0., 0., '-z+1./2'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z']],
                        '8g': [['x', 'x', 1./4],['-x', '-x', 1./4],['-x+1./2', 'x+1./2', 3./4],['x+1./2', '-x+1./2', 3./4],['-x+1./2', '-x+1./2', 1./4],['x+1./2', 'x+1./2', 1./4],['x', '-x', 3./4],['-x', 'x', 3./4]],
                        '8h': [['x', 'x', 3./4],['-x', '-x', 3./4],['-x+1./2', 'x+1./2', 1./4],['x+1./2', '-x+1./2', 1./4],['-x+1./2', '-x+1./2', 3./4],['x+1./2', 'x+1./2', 3./4],['x', '-x', 1./4],['-x', 'x', 1./4]],
                        '8i': [['x', 'x+1./2', 'z'],['-x', '-x+1./2', 'z'],['-x', 'x+1./2', 'z+1./2'],['x', '-x+1./2', 'z+1./2'],['-x+1./2', 'x', '-z'],['x+1./2', '-x', '-z'],['x+1./2', 'x', '-z+1./2'],['-x+1./2', '-x', '-z+1./2']],
                        '16j': [['x', 'y', 'z'],['-x', '-y', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x+1./2', '-y+1./2', '-z'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['y+1./2', 'x+1./2', 'z']]},
              '138:2': {'4a': [[3./4, 1./4, 0],[1./4, 3./4, 1./2],[1./4, 3./4, 0],[3./4, 1./4, 1./2]],
                        '4b': [[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                        '4c': [[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0]],
                        '4d': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '4e': [[1./4, 1./4, 'z'],[1./4, 1./4, 'z+1./2'],[3./4, 3./4, '-z+1./2'],[3./4, 3./4, '-z']],
                        '8f': [[3./4, 1./4, 'z'],[1./4, 3./4, 'z+1./2'],[1./4, 3./4, '-z+1./2'],[3./4, 1./4, '-z'],[1./4, 3./4, '-z'],[3./4, 1./4, '-z+1./2'],[3./4, 1./4, 'z+1./2'],[1./4, 3./4, 'z']],
                        '8g': [['x', '-x', 1./2],['-x+1./2', 'x+1./2', 1./2],['x+1./2', 'x', 0],['-x', '-x+1./2', 0],['-x', 'x', 1./2],['x+1./2', '-x+1./2', 1./2],['-x+1./2', '-x', 0],['x', 'x+1./2', 0]],
                        '8h': [['x', '-x', 0],['-x+1./2', 'x+1./2', 0],['x+1./2', 'x', 1./2],['-x', '-x+1./2', 1./2],['-x', 'x', 0],['x+1./2', '-x+1./2', 0],['-x+1./2', '-x', 1./2],['x', 'x+1./2', 1./2]],
                        '8i': [['x', 'x', 'z'],['-x+1./2', '-x+1./2', 'z'],['-x+1./2', 'x', 'z+1./2'],['x', '-x+1./2', 'z+1./2'],['-x', 'x+1./2', '-z+1./2'],['x+1./2', '-x', '-z+1./2'],['x+1./2', 'x+1./2', '-z'],['-x', '-x', '-z']],
                        '16j': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-y+1./2', 'x', 'z+1./2'],['y', '-x+1./2', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['y+1./2', '-x', '-z+1./2'],['-y', 'x+1./2', '-z+1./2'],['x', '-y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['y', 'x', 'z']]},
              '139': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '2b': [[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4c': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                      '4d': [[0., 1./2, 1./4],[1./2, 0., 3./4],[1./2, 0., 1./4],[0., 1./2, 3./4]],
                      '4e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2']],
                      '8f': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4]],
                      '8g': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, '-z'],[1./2, 0., '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z+1./2']],
                      '8h': [['x', 'x', 0],['x+1./2', 'x+1./2', 1./2],['-x', '-x', 0],['-x+1./2', '-x+1./2', 1./2],['-x', 'x', 0],['-x+1./2', 'x+1./2', 1./2],['x', '-x', 0],['x+1./2', '-x+1./2', 1./2]],
                      '8i': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2]],
                      '8j': [['x', 1./2, 0],['x+1./2', 0., 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2],[1./2, 'x', 0],[0., 'x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 1./2]],
                      '16k': [['x', 'x+1./2', 1./4],['x+1./2', 'x', 3./4],['-x', '-x+1./2', 1./4],['-x+1./2', '-x', 3./4],['-x+1./2', 'x', 1./4],['-x', 'x+1./2', 3./4],['x+1./2', '-x', 1./4],['x', '-x+1./2', 3./4],['-x', '-x+1./2', 3./4],['-x+1./2', '-x', 1./4],['x', 'x+1./2', 3./4],['x+1./2', 'x', 1./4],['x+1./2', '-x', 3./4],['x', '-x+1./2', 1./4],['-x+1./2', 'x', 3./4],['-x', 'x+1./2', 1./4]],
                      '16l': [['x', 'y', 0],['x+1./2', 'y+1./2', 1./2],['-x', '-y', 0],['-x+1./2', '-y+1./2', 1./2],['-y', 'x', 0],['-y+1./2', 'x+1./2', 1./2],['y', '-x', 0],['y+1./2', '-x+1./2', 1./2],['-x', 'y', 0],['-x+1./2', 'y+1./2', 1./2],['x', '-y', 0],['x+1./2', '-y+1./2', 1./2],['y', 'x', 0],['y+1./2', 'x+1./2', 1./2],['-y', '-x', 0],['-y+1./2', '-x+1./2', 1./2]],
                      '16m': [['x', 'x', 'z'],['x+1./2', 'x+1./2', 'z+1./2'],['-x', '-x', 'z'],['-x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', 'z'],['-x+1./2', 'x+1./2', 'z+1./2'],['x', '-x', 'z'],['x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', '-z'],['-x+1./2', 'x+1./2', '-z+1./2'],['x', '-x', '-z'],['x+1./2', '-x+1./2', '-z+1./2'],['x', 'x', '-z'],['x+1./2', 'x+1./2', '-z+1./2'],['-x', '-x', '-z'],['-x+1./2', '-x+1./2', '-z+1./2']],
                      '16n': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z+1./2'],['-y', 0., 'z'],['-y+1./2', 1./2, 'z+1./2'],['y', 0., 'z'],['y+1./2', 1./2, 'z+1./2'],[0., 'y', '-z'],[1./2, 'y+1./2', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z+1./2'],['y', 0., '-z'],['y+1./2', 1./2, '-z+1./2'],['-y', 0., '-z'],['-y+1./2', 1./2, '-z+1./2']],
                      '32o': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['y', 'x', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y', '-x', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', 'x', 'z'],['y+1./2', 'x+1./2', 'z+1./2']]},
              '140': {'4a': [[0., 0., 1./4],[1./2, 1./2, 3./4],[0., 0., 3./4],[1./2, 1./2, 1./4]],
                      '4b': [[0., 1./2, 1./4],[1./2, 0., 3./4],[1./2, 0., 1./4],[0., 1./2, 3./4]],
                      '4c': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 0]],
                      '4d': [[0., 1./2, 0],[1./2, 0., 1./2],[1./2, 0., 0],[0., 1./2, 1./2]],
                      '8e': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4]],
                      '8f': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 0., '-z+1./2'],[1./2, 1./2, '-z'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[0., 0., 'z+1./2'],[1./2, 1./2, 'z']],
                      '8g': [[0., 1./2, 'z'],[1./2, 0., 'z+1./2'],[1./2, 0., 'z'],[0., 1./2, 'z+1./2'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z'],[1./2, 0., '-z+1./2'],[0., 1./2, '-z']],
                      '8h': [['x', 'x+1./2', 0],['x+1./2', 'x', 1./2],['-x', '-x+1./2', 0],['-x+1./2', '-x', 1./2],['-x+1./2', 'x', 0],['-x', 'x+1./2', 1./2],['x+1./2', '-x', 0],['x', '-x+1./2', 1./2]],
                      '16i': [['x', 'x', 1./4],['x+1./2', 'x+1./2', 3./4],['-x', '-x', 1./4],['-x+1./2', '-x+1./2', 3./4],['-x', 'x', 1./4],['-x+1./2', 'x+1./2', 3./4],['x', '-x', 1./4],['x+1./2', '-x+1./2', 3./4],['-x', '-x', 3./4],['-x+1./2', '-x+1./2', 1./4],['x', 'x', 3./4],['x+1./2', 'x+1./2', 1./4],['x', '-x', 3./4],['x+1./2', '-x+1./2', 1./4],['-x', 'x', 3./4],['-x+1./2', 'x+1./2', 1./4]],
                      '16j': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x', 0., 1./4],['-x+1./2', 1./2, 3./4],[0., 'x', 1./4],[1./2, 'x+1./2', 3./4],[0., '-x', 1./4],[1./2, '-x+1./2', 3./4],['-x', 0., 3./4],['-x+1./2', 1./2, 1./4],['x', 0., 3./4],['x+1./2', 1./2, 1./4],[0., '-x', 3./4],[1./2, '-x+1./2', 1./4],[0., 'x', 3./4],[1./2, 'x+1./2', 1./4]],
                      '16k': [['x', 'y', 0],['x+1./2', 'y+1./2', 1./2],['-x', '-y', 0],['-x+1./2', '-y+1./2', 1./2],['-y', 'x', 0],['-y+1./2', 'x+1./2', 1./2],['y', '-x', 0],['y+1./2', '-x+1./2', 1./2],['-x', 'y', 1./2],['-x+1./2', 'y+1./2', 0],['x', '-y', 1./2],['x+1./2', '-y+1./2', 0],['y', 'x', 1./2],['y+1./2', 'x+1./2', 0],['-y', '-x', 1./2],['-y+1./2', '-x+1./2', 0]],
                      '16l': [['x', 'x+1./2', 'z'],['x+1./2', 'x', 'z+1./2'],['-x', '-x+1./2', 'z'],['-x+1./2', '-x', 'z+1./2'],['-x+1./2', 'x', 'z'],['-x', 'x+1./2', 'z+1./2'],['x+1./2', '-x', 'z'],['x', '-x+1./2', 'z+1./2'],['-x', 'x+1./2', '-z+1./2'],['-x+1./2', 'x', '-z'],['x', '-x+1./2', '-z+1./2'],['x+1./2', '-x', '-z'],['x+1./2', 'x', '-z+1./2'],['x', 'x+1./2', '-z'],['-x+1./2', '-x', '-z+1./2'],['-x', '-x+1./2', '-z']],
                      '32m': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['-x', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['y', 'x', '-z+1./2'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['x', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-y', '-x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['y', 'x', 'z+1./2'],['y+1./2', 'x+1./2', 'z']]},
              '141:1': {'4a': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 1./2, 1./4],[1./2, 0., 3./4]],
                        '4b': [[0., 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                        '8c': [[0., 1./4, 1/8],[1./2, 3./4, 5/8],[1./2, 1./4, 5/8],[0., 3./4, 1/8],[3./4, 1./2, 3/8],[1./4, 0., 7/8],[3./4, 0., 7/8],[1./4, 1./2, 3/8]],
                        '8d': [[0., 1./4, 5/8],[1./2, 3./4, 1/8],[1./2, 1./4, 1/8],[0., 3./4, 5/8],[3./4, 1./2, 7/8],[1./4, 0., 3/8],[3./4, 0., 3/8],[1./4, 1./2, 7/8]],
                        '8e': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4'],[1./2, 0., '-z+3./4'],[0., 1./2, '-z+1./4'],[1./2, 1./2, '-z+1./2'],[0., 0., '-z']],
                        '16f': [['x', 1./4, 1/8],['x+1./2', 3./4, 5/8],['-x+1./2', 1./4, 5/8],['-x', 3./4, 1/8],[3./4, 'x+1./2', 3/8],[1./4, 'x', 7/8],[3./4, '-x', 7/8],[1./4, '-x+1./2', 3/8],['-x', 1./4, 1/8],['-x+1./2', 3./4, 5/8],['x+1./2', 1./4, 5/8],['x', 3./4, 1/8],[1./4, '-x', 7/8],[3./4, '-x+1./2', 3/8],[1./4, 'x+1./2', 3/8],[3./4, 'x', 7/8]],
                        '16g': [['x', 'x', 0],['x+1./2', 'x+1./2', 1./2],['-x+1./2', '-x+1./2', 1./2],['-x', '-x', 0],['-x', 'x+1./2', 1./4],['-x+1./2', 'x', 3./4],['x+1./2', '-x', 3./4],['x', '-x+1./2', 1./4],['-x', '-x+1./2', 1./4],['-x+1./2', '-x', 3./4],['x+1./2', 'x', 3./4],['x', 'x+1./2', 1./4],['x', '-x', 0],['x+1./2', '-x+1./2', 1./2],['-x+1./2', 'x+1./2', 1./2],['-x', 'x', 0]],
                        '16h': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[1./2, '-y+1./2', 'z+1./2'],[0., '-y', 'z'],['-y', 1./2, 'z+1./4'],['-y+1./2', 0., 'z+3./4'],['y+1./2', 0., 'z+3./4'],['y', 1./2, 'z+1./4'],[1./2, 'y', '-z+3./4'],[0., 'y+1./2', '-z+1./4'],[0., '-y+1./2', '-z+1./4'],[1./2, '-y', '-z+3./4'],['y+1./2', 1./2, '-z+1./2'],['y', 0., '-z'],['-y', 0., '-z'],['-y+1./2', 1./2, '-z+1./2']],
                        '32i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4'],['-x+1./2', 'y', '-z+3./4'],['-x', 'y+1./2', '-z+1./4'],['x', '-y+1./2', '-z+1./4'],['x+1./2', '-y', '-z+3./4'],['y+1./2', 'x+1./2', '-z+1./2'],['y', 'x', '-z'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x', '-y+1./2', '-z+1./4'],['-x+1./2', '-y', '-z+3./4'],['x+1./2', 'y', '-z+3./4'],['x', 'y+1./2', '-z+1./4'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z+1./2'],['x', '-y', 'z'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y+1./2', '-x', 'z+3./4'],['-y', '-x+1./2', 'z+1./4'],['y', 'x+1./2', 'z+1./4'],['y+1./2', 'x', 'z+3./4']]},
              '141:2': {'4a': [[0., 3./4, 1/8],[1./2, 1./4, 5/8],[1./2, 3./4, 3/8],[0., 1./4, 7/8]],
                        '4b': [[0., 1./4, 3/8],[1./2, 3./4, 7/8],[0., 3./4, 5/8],[1./2, 1./4, 1/8]],
                        '8c': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 0],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                        '8d': [[0., 0., 1./2],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 1./2],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '8e': [[0., 1./4, 'z'],[1./2, 3./4, 'z+1./2'],[0., 3./4, 'z+1./4'],[1./2, 1./4, 'z+3./4'],[1./2, 1./4, '-z+1./2'],[0., 3./4, '-z'],[1./2, 3./4, '-z+1./4'],[0., 1./4, '-z+3./4']],
                        '16f': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x', 1./2, 0],[1./4, 'x+3./4', 1./4],[3./4, 'x+1./4', 3./4],[1./4, '-x+1./4', 3./4],[3./4, '-x+3./4', 1./4],['-x', 0., 0],['-x+1./2', 1./2, 1./2],['x+1./2', 0., 1./2],['x', 1./2, 0],[3./4, '-x+1./4', 3./4],[1./4, '-x+3./4', 1./4],[3./4, 'x+3./4', 1./4],[1./4, 'x+1./4', 3./4]],
                        '16g': [['x', 'x+1./4', 7/8],['x+1./2', 'x+3./4', 3/8],['-x+1./2', '-x+3./4', 3/8],['-x', '-x+1./4', 7/8],['-x', 'x+3./4', 1/8],['-x+1./2', 'x+1./4', 5/8],['x+1./2', '-x+1./4', 5/8],['x', '-x+3./4', 1/8],['-x', '-x+3./4', 1/8],['-x+1./2', '-x+1./4', 5/8],['x+1./2', 'x+1./4', 5/8],['x', 'x+3./4', 1/8],['x', '-x+1./4', 7/8],['x+1./2', '-x+3./4', 3/8],['-x+1./2', 'x+3./4', 3/8],['-x', 'x+1./4', 7/8]],
                        '16h': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[0., '-y+1./2', 'z'],['-y+1./4', 3./4, 'z+1./4'],['-y+3./4', 1./4, 'z+3./4'],['y+1./4', 1./4, 'z+3./4'],['y+3./4', 3./4, 'z+1./4'],[1./2, 'y', '-z+1./2'],[0., 'y+1./2', '-z'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z+1./2'],['y+1./4', 3./4, '-z+1./4'],['y+3./4', 1./4, '-z+3./4'],['-y+1./4', 1./4, '-z+3./4'],['-y+3./4', 3./4, '-z+1./4']],
                        '32i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-y+1./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['y+1./4', '-x+1./4', 'z+3./4'],['y+3./4', '-x+3./4', 'z+1./4'],['-x+1./2', 'y', '-z+1./2'],['-x', 'y+1./2', '-z'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['y+1./4', 'x+3./4', '-z+1./4'],['y+3./4', 'x+1./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+3./4'],['-y+3./4', '-x+3./4', '-z+1./4'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z'],['y+3./4', '-x+1./4', '-z+3./4'],['y+1./4', '-x+3./4', '-z+1./4'],['-y+3./4', 'x+3./4', '-z+1./4'],['-y+1./4', 'x+1./4', '-z+3./4'],['x+1./2', '-y', 'z+1./2'],['x', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['-y+3./4', '-x+1./4', 'z+3./4'],['-y+1./4', '-x+3./4', 'z+1./4'],['y+3./4', 'x+3./4', 'z+1./4'],['y+1./4', 'x+1./4', 'z+3./4']]},
              '142:1': {'8a': [[0., 0., 0],[1./2, 1./2, 1./2],[0., 1./2, 1./4],[1./2, 0., 3./4],[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 1./2, 0],[0., 0., 1./2]],
                        '8b': [[0., 0., 1./4],[1./2, 1./2, 3./4],[0., 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[1./2, 0., 1./2],[0., 0., 3./4],[1./2, 1./2, 1./4]],
                        '16c': [[0., 1./4, 1/8],[1./2, 3./4, 5/8],[1./2, 1./4, 5/8],[0., 3./4, 1/8],[3./4, 1./2, 3/8],[1./4, 0., 7/8],[3./4, 0., 7/8],[1./4, 1./2, 3/8],[1./2, 1./4, 1/8],[0., 3./4, 5/8],[0., 1./4, 5/8],[1./2, 3./4, 1/8],[3./4, 1./2, 7/8],[1./4, 0., 3/8],[3./4, 0., 3/8],[1./4, 1./2, 7/8]],
                        '16d': [[0., 0., 'z'],[1./2, 1./2, 'z+1./2'],[0., 1./2, 'z+1./4'],[1./2, 0., 'z+3./4'],[1./2, 0., '-z+1./4'],[0., 1./2, '-z+3./4'],[1./2, 1./2, '-z'],[0., 0., '-z+1./2'],[0., 1./2, '-z+1./4'],[1./2, 0., '-z+3./4'],[0., 0., '-z'],[1./2, 1./2, '-z+1./2'],[1./2, 1./2, 'z'],[0., 0., 'z+1./2'],[1./2, 0., 'z+1./4'],[0., 1./2, 'z+3./4']],
                        '16e': [[1./4, 'y', 1/8],[3./4, 'y+1./2', 5/8],[1./4, '-y+1./2', 5/8],[3./4, '-y', 1/8],['-y', 3./4, 3/8],['-y+1./2', 1./4, 7/8],['y+1./2', 3./4, 7/8],['y', 1./4, 3/8],[3./4, '-y+1./2', 1/8],[1./4, '-y', 5/8],[3./4, 'y', 5/8],[1./4, 'y+1./2', 1/8],['y', 3./4, 7/8],['y+1./2', 1./4, 3/8],['-y+1./2', 3./4, 3/8],['-y', 1./4, 7/8]],
                        '16f': [['x', 'x', 1./4],['x+1./2', 'x+1./2', 3./4],['-x+1./2', '-x+1./2', 3./4],['-x', '-x', 1./4],['-x', 'x+1./2', 1./2],['-x+1./2', 'x', 0],['x+1./2', '-x', 0],['x', '-x+1./2', 1./2],['-x', '-x+1./2', 0],['-x+1./2', '-x', 1./2],['x+1./2', 'x', 1./2],['x', 'x+1./2', 0],['x', '-x', 3./4],['x+1./2', '-x+1./2', 1./4],['-x+1./2', 'x+1./2', 1./4],['-x', 'x', 3./4]],
                        '32g': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-y', 'x+1./2', 'z+1./4'],['-y+1./2', 'x', 'z+3./4'],['y+1./2', '-x', 'z+3./4'],['y', '-x+1./2', 'z+1./4'],['-x+1./2', 'y', '-z+1./4'],['-x', 'y+1./2', '-z+3./4'],['x', '-y+1./2', '-z+3./4'],['x+1./2', '-y', '-z+1./4'],['y+1./2', 'x+1./2', '-z'],['y', 'x', '-z+1./2'],['-y', '-x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z'],['-x', '-y+1./2', '-z+1./4'],['-x+1./2', '-y', '-z+3./4'],['x+1./2', 'y', '-z+3./4'],['x', 'y+1./2', '-z+1./4'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['-y', 'x', '-z'],['x+1./2', '-y+1./2', 'z'],['x', '-y', 'z+1./2'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-y+1./2', '-x', 'z+1./4'],['-y', '-x+1./2', 'z+3./4'],['y', 'x+1./2', 'z+3./4'],['y+1./2', 'x', 'z+1./4']]},
              '142:2': {'8a': [[0., 1./4, 3/8],[1./2, 3./4, 7/8],[0., 3./4, 5/8],[1./2, 1./4, 1/8],[1./2, 1./4, 5/8],[0., 3./4, 1/8],[1./2, 3./4, 3/8],[0., 1./4, 7/8]],
                        '8b': [[0., 1./4, 1/8],[1./2, 3./4, 5/8],[0., 3./4, 3/8],[1./2, 1./4, 7/8],[0., 3./4, 7/8],[1./2, 1./4, 3/8],[0., 1./4, 5/8],[1./2, 3./4, 1/8]],
                        '16c': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 0],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./2, 0., 0],[0., 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 0],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '16d': [[0., 1./4, 'z'],[1./2, 3./4, 'z+1./2'],[0., 3./4, 'z+1./4'],[1./2, 1./4, 'z+3./4'],[1./2, 1./4, '-z'],[0., 3./4, '-z+1./2'],[1./2, 3./4, '-z+3./4'],[0., 1./4, '-z+1./4'],[0., 3./4, '-z'],[1./2, 1./4, '-z+1./2'],[0., 1./4, '-z+3./4'],[1./2, 3./4, '-z+1./4'],[1./2, 3./4, 'z'],[0., 1./4, 'z+1./2'],[1./2, 1./4, 'z+1./4'],[0., 3./4, 'z+3./4']],
                        '16e': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],[1./4, 'x+3./4', 1./2],[3./4, 'x+1./4', 0],[1./4, '-x+1./4', 0],[3./4, '-x+3./4', 1./2],['-x', 0., 3./4],['-x+1./2', 1./2, 1./4],['x+1./2', 0., 1./4],['x', 1./2, 3./4],[3./4, '-x+1./4', 1./2],[1./4, '-x+3./4', 0],[3./4, 'x+3./4', 0],[1./4, 'x+1./4', 1./2]],
                        '16f': [['x', 'x+1./4', 1/8],['x+1./2', 'x+3./4', 5/8],['-x+1./2', '-x+3./4', 5/8],['-x', '-x+1./4', 1/8],['-x', 'x+3./4', 3/8],['-x+1./2', 'x+1./4', 7/8],['x+1./2', '-x+1./4', 7/8],['x', '-x+3./4', 3/8],['-x', '-x+3./4', 7/8],['-x+1./2', '-x+1./4', 3/8],['x+1./2', 'x+1./4', 3/8],['x', 'x+3./4', 7/8],['x', '-x+1./4', 5/8],['x+1./2', '-x+3./4', 1/8],['-x+1./2', 'x+3./4', 1/8],['-x', 'x+1./4', 5/8]],
                        '32g': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-y+1./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['y+1./4', '-x+1./4', 'z+3./4'],['y+3./4', '-x+3./4', 'z+1./4'],['-x+1./2', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['x', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['y+1./4', 'x+3./4', '-z+3./4'],['y+3./4', 'x+1./4', '-z+1./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['-y+3./4', '-x+3./4', '-z+3./4'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z'],['y+3./4', '-x+1./4', '-z+3./4'],['y+1./4', '-x+3./4', '-z+1./4'],['-y+3./4', 'x+3./4', '-z+1./4'],['-y+1./4', 'x+1./4', '-z+3./4'],['x+1./2', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['-x', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-y+3./4', '-x+1./4', 'z+1./4'],['-y+1./4', '-x+3./4', 'z+3./4'],['y+3./4', 'x+3./4', 'z+3./4'],['y+1./4', 'x+1./4', 'z+1./4']]},
              '143': {'1a': [[0., 0., 'z']],
                      '1b': [[1./3, 2./3, 'z']],
                      '1c': [[2./3, 1./3, 'z']],
                      '3d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z']]},
              '144': {'3a': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3']]},
              '145': {'3a': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3']]},
              '146:H': {'3a': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3']],
                        '9b': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3']]},
              '146:R': {'1a': [['x', 'x', 'x']],
                        '3b': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x']]},
              '147': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[0., 0., 'z'],[0., 0., '-z']],
                      '2d': [[1./3, 2./3, 'z'],[2./3, 1./3, '-z']],
                      '3e': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0]],
                      '3f': [[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '6g': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z']]},
              '148:H': {'3a': [[0., 0., 0],[2./3, 1./3, 1./3],[1./3, 2./3, 2./3]],
                        '3b': [[0., 0., 1./2],[2./3, 1./3, 5./6],[1./3, 2./3, 1./6]],
                        '6c': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3'],[0., 0., '-z'],[2./3, 1./3, '-z+1./3'],[1./3, 2./3, '-z+2./3']],
                        '9d': [[1./2, 0., 1./2],[1./6, 1./3, 5./6],[5./6, 2./3, 1./6],[0., 1./2, 1./2],[2./3, 5./6, 5./6],[1./3, 1./6, 1./6],[1./2, 1./2, 1./2],[1./6, 5./6, 5./6],[5./6, 1./6, 1./6]],
                        '9e': [[1./2, 0., 0],[1./6, 1./3, 1./3],[5./6, 2./3, 2./3],[0., 1./2, 0],[2./3, 5./6, 1./3],[1./3, 1./6, 2./3],[1./2, 1./2, 0],[1./6, 5./6, 1./3],[5./6, 1./6, 2./3]],
                        '18f': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3'],['-x', '-y', '-z'],['-x+2./3', '-y+1./3', '-z+1./3'],['-x+1./3', '-y+2./3', '-z+2./3'],['y', '-x+y', '-z'],['y+2./3', '-x+y+1./3', '-z+1./3'],['y+1./3', '-x+y+2./3', '-z+2./3'],['x-y', 'x', '-z'],['x-y+2./3', 'x+1./3', '-z+1./3'],['x-y+1./3', 'x+2./3', '-z+2./3']]},
              '148:R': {'1a': [[0., 0., 0]],
                        '1b': [[1./2, 1./2, 1./2]],
                        '2c': [['x', 'x', 'x'],['-x', '-x', '-x']],
                        '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                        '3e': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                        '6f': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x'],['-x', '-y', '-z'],['-z', '-x', '-y'],['-y', '-z', '-x']]},
              '149': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '1c': [[1./3, 2./3, 0]],
                      '1d': [[1./3, 2./3, 1./2]],
                      '1e': [[2./3, 1./3, 0]],
                      '1f': [[2./3, 1./3, 1./2]],
                      '2g': [[0., 0., 'z'],[0., 0., '-z']],
                      '2h': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z']],
                      '2i': [[2./3, 1./3, 'z'],[2./3, 1./3, '-z']],
                      '3j': [['x', '-x', 0],['x', '2.*x', 0],['-2.*x', '-x', 0]],
                      '3k': [['x', '-x', 1./2],['x', '2.*x', 1./2],['-2.*x', '-x', 1./2]],
                      '6l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z']]},
              '150': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[0., 0., 'z'],[0., 0., '-z']],
                      '2d': [[1./3, 2./3, 'z'],[2./3, 1./3, '-z']],
                      '3e': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0]],
                      '3f': [['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2]],
                      '6g': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z']]},
              '151': {'3a': [['x', '-x', 1./3],['x', '2.*x', 2./3],['-2.*x', '-x', 0]],
                      '3b': [['x', '-x', 5./6],['x', '2.*x', 1./6],['-2.*x', '-x', 1./2]],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3'],['-y', '-x', '-z+2./3'],['-x+y', 'y', '-z+1./3'],['x', 'x-y', '-z']]},
              '152': {'3a': [['x', 0., 1./3],[0., 'x', 2./3],['-x', '-x', 0]],
                      '3b': [['x', 0., 5./6],[0., 'x', 1./6],['-x', '-x', 1./2]],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3'],['y', 'x', '-z'],['x-y', '-y', '-z+2./3'],['-x', '-x+y', '-z+1./3']]},
              '153': {'3a': [['x', '-x', 2./3],['x', '2.*x', 1./3],['-2.*x', '-x', 0]],
                      '3b': [['x', '-x', 1./6],['x', '2.*x', 5./6],['-2.*x', '-x', 1./2]],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3'],['-y', '-x', '-z+1./3'],['-x+y', 'y', '-z+2./3'],['x', 'x-y', '-z']]},
              '154': {'3a': [['x', 0., 2./3],[0., 'x', 1./3],['-x', '-x', 0]],
                      '3b': [['x', 0., 1./6],[0., 'x', 5./6],['-x', '-x', 1./2]],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3'],['y', 'x', '-z'],['x-y', '-y', '-z+1./3'],['-x', '-x+y', '-z+2./3']]},
              '155:H': {'3a': [[0., 0., 0],[2./3, 1./3, 1./3],[1./3, 2./3, 2./3]],
                        '3b': [[0., 0., 1./2],[2./3, 1./3, 5./6],[1./3, 2./3, 1./6]],
                        '6c': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3'],[0., 0., '-z'],[2./3, 1./3, '-z+1./3'],[1./3, 2./3, '-z+2./3']],
                        '9d': [['x', 0., 0],['x+2./3', 1./3, 1./3],['x+1./3', 2./3, 2./3],[0., 'x', 0],[2./3, 'x+1./3', 1./3],[1./3, 'x+2./3', 2./3],['-x', '-x', 0],['-x+2./3', '-x+1./3', 1./3],['-x+1./3', '-x+2./3', 2./3]],
                        '9e': [['x', 0., 1./2],['x+2./3', 1./3, 5./6],['x+1./3', 2./3, 1./6],[0., 'x', 1./2],[2./3, 'x+1./3', 5./6],[1./3, 'x+2./3', 1./6],['-x', '-x', 1./2],['-x+2./3', '-x+1./3', 5./6],['-x+1./3', '-x+2./3', 1./6]],
                        '18f': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3'],['y', 'x', '-z'],['y+2./3', 'x+1./3', '-z+1./3'],['y+1./3', 'x+2./3', '-z+2./3'],['x-y', '-y', '-z'],['x-y+2./3', '-y+1./3', '-z+1./3'],['x-y+1./3', '-y+2./3', '-z+2./3'],['-x', '-x+y', '-z'],['-x+2./3', '-x+y+1./3', '-z+1./3'],['-x+1./3', '-x+y+2./3', '-z+2./3']]},
              '155:R': {'1a': [[0., 0., 0]],
                        '1b': [[1./2, 1./2, 1./2]],
                        '2c': [['x', 'x', 'x'],['-x', '-x', '-x']],
                        '3d': [[0., 'y', '-y'],['-y', 0., 'y'],['y', '-y', 0]],
                        '3e': [[1./2, 'y', '-y'],['-y', 1./2, 'y'],['y', '-y', 1./2]],
                        '6f': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x'],['-z', '-y', '-x'],['-y', '-x', '-z'],['-x', '-z', '-y'],]},
              '156': {'1a': [[0., 0., 'z']],
                      '1b': [[1./3, 2./3, 'z']],
                      '1c': [[2./3, 1./3, 'z']],
                      '3d': [['x', '-x', 'z'],['x', '2.*x', 'z'],['-2.*x', '-x', 'z']],
                      '6e': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z']]},
              '157': {'1a': [[0., 0., 'z']],
                      '2b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z']],
                      '3c': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z']],
                      '6d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '158': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./3, 2./3, 'z'],[1./3, 2./3, 'z+1./2']],
                      '2c': [[2./3, 1./3, 'z'],[2./3, 1./3, 'z+1./2']],
                      '6d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2']]},
              '159': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2']],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '160:H': {'3a': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3']],
                        '9b': [['x', '-x', 'z'],['x+2./3', '-x+1./3', 'z+1./3'],['x+1./3', '-x+2./3', 'z+2./3'],['x', '2.*x', 'z'],['x+2./3', '2.*x+1./3', 'z+1./3'],['x+1./3', '2.*x+2./3', 'z+2./3'],['-2.*x', '-x', 'z'],['-2.*x+2./3', '-x+1./3', 'z+1./3'],['-2.*x+1./3', '-x+2./3', 'z+2./3']],
                        '18c': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3'],['-y', '-x', 'z'],['-y+2./3', '-x+1./3', 'z+1./3'],['-y+1./3', '-x+2./3', 'z+2./3'],['-x+y', 'y', 'z'],['-x+y+2./3', 'y+1./3', 'z+1./3'],['-x+y+1./3', 'y+2./3', 'z+2./3'],['x', 'x-y', 'z'],['x+2./3', 'x-y+1./3', 'z+1./3'],['x+1./3', 'x-y+2./3', 'z+2./3']]},
              '160:R': {'1a': [['x', 'x', 'x']],
                        '3b': [['x', 'x', 'z'],['z', 'x', 'x'],['x', 'z', 'x']],
                        '6c': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x'],['z', 'y', 'x'],['y', 'x', 'z'],['x', 'z', 'y']]},
              '161:H': {'6a': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3'],[0., 0., 'z+1./2'],[2./3, 1./3, 'z+5./6'],[1./3, 2./3, 'z+1./6']],
                        '18b': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3'],['-y', '-x', 'z+1./2'],['-y+2./3', '-x+1./3', 'z+5./6'],['-y+1./3', '-x+2./3', 'z+1./6'],['-x+y', 'y', 'z+1./2'],['-x+y+2./3', 'y+1./3', 'z+5./6'],['-x+y+1./3', 'y+2./3', 'z+1./6'],['x', 'x-y', 'z+1./2'],['x+2./3', 'x-y+1./3', 'z+5./6'],['x+1./3', 'x-y+2./3', 'z+1./6']]},
              '161:R': {'2a': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2']],
                        '6b': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x'],['z+1./2', 'y+1./2', 'x+1./2'],['y+1./2', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', 'y+1./2']]},
              '162': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 0],[2./3, 1./3, 0]],
                      '2d': [[1./3, 2./3, 1./2],[2./3, 1./3, 1./2]],
                      '2e': [[0., 0., 'z'],[0., 0., '-z']],
                      '3f': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0]],
                      '3g': [[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '4h': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z'],[2./3, 1./3, '-z'],[2./3, 1./3, 'z']],
                      '6i': [['x', '-x', 0],['x', '2.*x', 0],['-2.*x', '-x', 0],['-x', 'x', 0],['-x', '-2.*x', 0],['2.*x', 'x', 0]],
                      '6j': [['x', '-x', 1./2],['x', '2.*x', 1./2],['-2.*x', '-x', 1./2],['-x', 'x', 1./2],['-x', '-2.*x', 1./2],['2.*x', 'x', 1./2]],
                      '6k': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z'],[0., '-x', '-z'],['-x', 0., '-z'],['x', 'x', '-z']],
                      '12l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '163': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[0., 0., 0],[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 1./4],[2./3, 1./3, 3./4]],
                      '2d': [[2./3, 1./3, 1./4],[1./3, 2./3, 3./4]],
                      '4e': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2']],
                      '4f': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z+1./2'],[2./3, 1./3, '-z'],[2./3, 1./3, 'z+1./2']],
                      '6g': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 1./2]],
                      '6h': [['x', '-x', 1./4],['x', '2.*x', 1./4],['-2.*x', '-x', 1./4],['-x', 'x', 3./4],['-x', '-2.*x', 3./4],['2.*x', 'x', 3./4]],
                      '12i': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-y', '-x', '-z+1./2'],['-x+y', 'y', '-z+1./2'],['x', 'x-y', '-z+1./2'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '164': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[0., 0., 'z'],[0., 0., '-z']],
                      '2d': [[1./3, 2./3, 'z'],[2./3, 1./3, '-z']],
                      '3e': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0]],
                      '3f': [[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '6g': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0],['-x', 0., 0],[0., '-x', 0],['x', 'x', 0]],
                      '6h': [['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2],['-x', 0., 1./2],[0., '-x', 1./2],['x', 'x', 1./2]],
                      '6i': [['x', '-x', 'z'],['x', '2.*x', 'z'],['-2.*x', '-x', 'z'],['-x', 'x', '-z'],['2.*x', 'x', '-z'],['-x', '-2.*x', '-z']],
                      '12j': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z']]},
              '165': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[0., 0., 0],[0., 0., 1./2]],
                      '4c': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2']],
                      '4d': [[1./3, 2./3, 'z'],[2./3, 1./3, '-z+1./2'],[2./3, 1./3, '-z'],[1./3, 2./3, 'z+1./2']],
                      '6e': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 1./2]],
                      '6f': [['x', 0., 1./4],[0., 'x', 1./4],['-x', '-x', 1./4],['-x', 0., 3./4],[0., '-x', 3./4],['x', 'x', 3./4]],
                      '12g': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['y', 'x', '-z+1./2'],['x-y', '-y', '-z+1./2'],['-x', '-x+y', '-z+1./2'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2']]},
              '166:H': {'3a': [[0., 0., 0],[2./3, 1./3, 1./3],[1./3, 2./3, 2./3]],
                        '3b': [[0., 0., 1./2],[2./3, 1./3, 5./6],[1./3, 2./3, 1./6]],
                        '6c': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3'],[0., 0., '-z'],[2./3, 1./3, '-z+1./3'],[1./3, 2./3, '-z+2./3']],
                        '9d': [[1./2, 0., 1./2],[1./6, 1./3, 5./6],[5./6, 2./3, 1./6],[0., 1./2, 1./2],[2./3, 5./6, 5./6],[1./3, 1./6, 1./6],[1./2, 1./2, 1./2],[1./6, 5./6, 5./6],[5./6, 1./6, 1./6]],
                        '9e': [[1./2, 0., 0],[1./6, 1./3, 1./3],[5./6, 2./3, 2./3],[0., 1./2, 0],[2./3, 5./6, 1./3],[1./3, 1./6, 2./3],[1./2, 1./2, 0],[1./6, 5./6, 1./3],[5./6, 1./6, 2./3]],
                        '18f': [['x', 0., 0],['x+2./3', 1./3, 1./3],['x+1./3', 2./3, 2./3],[0., 'x', 0],[2./3, 'x+1./3', 1./3],[1./3, 'x+2./3', 2./3],['-x', '-x', 0],['-x+2./3', '-x+1./3', 1./3],['-x+1./3', '-x+2./3', 2./3],['-x', 0., 0],['-x+2./3', 1./3, 1./3],['-x+1./3', 2./3, 2./3],[0., '-x', 0],[2./3, '-x+1./3', 1./3],[1./3, '-x+2./3', 2./3],['x', 'x', 0],['x+2./3', 'x+1./3', 1./3],['x+1./3', 'x+2./3', 2./3]],
                        '18g': [['x', 0., 1./2],['x+2./3', 1./3, 5./6],['x+1./3', 2./3, 1./6],[0., 'x', 1./2],[2./3, 'x+1./3', 5./6],[1./3, 'x+2./3', 1./6],['-x', '-x', 1./2],['-x+2./3', '-x+1./3', 5./6],['-x+1./3', '-x+2./3', 1./6],['-x', 0., 1./2],['-x+2./3', 1./3, 5./6],['-x+1./3', 2./3, 1./6],[0., '-x', 1./2],[2./3, '-x+1./3', 5./6],[1./3, '-x+2./3', 1./6],['x', 'x', 1./2],['x+2./3', 'x+1./3', 5./6],['x+1./3', 'x+2./3', 1./6]],
                        '18h': [['x', '-x', 'z'],['x+2./3', '-x+1./3', 'z+1./3'],['x+1./3', '-x+2./3', 'z+2./3'],['x', '2.*x', 'z'],['x+2./3', '2.*x+1./3', 'z+1./3'],['x+1./3', '2.*x+2./3', 'z+2./3'],['-2.*x', '-x', 'z'],['-2.*x+2./3', '-x+1./3', 'z+1./3'],['-2.*x+1./3', '-x+2./3', 'z+2./3'],['-x', 'x', '-z'],['-x+2./3', 'x+1./3', '-z+1./3'],['-x+1./3', 'x+2./3', '-z+2./3'],['2.*x', 'x', '-z'],['2.*x+2./3', 'x+1./3', '-z+1./3'],['2.*x+1./3', 'x+2./3', '-z+2./3'],['-x', '-2.*x', '-z'],['-x+2./3', '-2.*x+1./3', '-z+1./3'],['-x+1./3', '-2.*x+2./3', '-z+2./3']],
                        '36i': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3'],['y', 'x', '-z'],['y+2./3', 'x+1./3', '-z+1./3'],['y+1./3', 'x+2./3', '-z+2./3'],['x-y', '-y', '-z'],['x-y+2./3', '-y+1./3', '-z+1./3'],['x-y+1./3', '-y+2./3', '-z+2./3'],['-x', '-x+y', '-z'],['-x+2./3', '-x+y+1./3', '-z+1./3'],['-x+1./3', '-x+y+2./3', '-z+2./3'],['-x', '-y', '-z'],['-x+2./3', '-y+1./3', '-z+1./3'],['-x+1./3', '-y+2./3', '-z+2./3'],['y', '-x+y', '-z'],['y+2./3', '-x+y+1./3', '-z+1./3'],['y+1./3', '-x+y+2./3', '-z+2./3'],['x-y', 'x', '-z'],['x-y+2./3', 'x+1./3', '-z+1./3'],['x-y+1./3', 'x+2./3', '-z+2./3'],['-y', '-x', 'z'],['-y+2./3', '-x+1./3', 'z+1./3'],['-y+1./3', '-x+2./3', 'z+2./3'],['-x+y', 'y', 'z'],['-x+y+2./3', 'y+1./3', 'z+1./3'],['-x+y+1./3', 'y+2./3', 'z+2./3'],['x', 'x-y', 'z'],['x+2./3', 'x-y+1./3', 'z+1./3'],['x+1./3', 'x-y+2./3', 'z+2./3']]},
              '166:R': {'1a': [[0., 0., 0]],
                        '1b': [[1./2, 1./2, 1./2]],
                        '2c': [['x', 'x', 'x'],['-x', '-x', '-x']],
                        '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                        '3e': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                        '6f': [['x', '-x', 0],[0., 'x', '-x'],['-x', 0., 'x'],['-x', 'x', 0],[0., '-x', 'x'],['x', 0., '-x']],
                        '6g': [['x', '-x', 1./2],[1./2, 'x', '-x'],['-x', 1./2, 'x'],['-x', 'x', 1./2],[1./2, '-x', 'x'],['x', 1./2, '-x']],
                        '6h': [['x', 'x', 'z'],['z', 'x', 'x'],['x', 'z', 'x'],['-z', '-x', '-x'],['-x', '-x', '-z'],['-x', '-z', '-x']],
                        '12i': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x'],['-z', '-y', '-x'],['-y', '-x', '-z'],['-x', '-z', '-y'],['-x', '-y', '-z'],['-z', '-x', '-y'],['-y', '-z', '-x'],['z', 'y', 'x'],['y', 'x', 'z'],['x', 'z', 'y']]},
              '167:H': {'6a': [[0., 0., 1./4],[2./3, 1./3, 7./12],[1./3, 2./3, 11./12],[0., 0., 3./4],[2./3, 1./3, 1./12],[1./3, 2./3, 5./12]],
                        '6b': [[0., 0., 0],[2./3, 1./3, 1./3],[1./3, 2./3, 2./3],[0., 0., 1./2],[2./3, 1./3, 5./6],[1./3, 2./3, 1./6]],
                        '12c': [[0., 0., 'z'],[2./3, 1./3, 'z+1./3'],[1./3, 2./3, 'z+2./3'],[0., 0., '-z+1./2'],[2./3, 1./3, '-z+5./6'],[1./3, 2./3, '-z+1./6'],[0., 0., '-z'],[2./3, 1./3, '-z+1./3'],[1./3, 2./3, '-z+2./3'],[0., 0., 'z+1./2'],[2./3, 1./3, 'z+5./6'],[1./3, 2./3, 'z+1./6']],
                        '18d': [[1./2, 0., 0],[1./6, 1./3, 1./3],[5./6, 2./3, 2./3],[0., 1./2, 0],[2./3, 5./6, 1./3],[1./3, 1./6, 2./3],[1./2, 1./2, 0],[1./6, 5./6, 1./3],[5./6, 1./6, 2./3],[0., 1./2, 1./2],[2./3, 5./6, 5./6],[1./3, 1./6, 1./6],[1./2, 0., 1./2],[1./6, 1./3, 5./6],[5./6, 2./3, 1./6],[1./2, 1./2, 1./2],[1./6, 5./6, 5./6],[5./6, 1./6, 1./6]],
                        '18e': [['x', 0., 1./4],['x+2./3', 1./3, 7./12],['x+1./3', 2./3, 11./12],[0., 'x', 1./4],[2./3, 'x+1./3', 7./12],[1./3, 'x+2./3', 11./12],['-x', '-x', 1./4],['-x+2./3', '-x+1./3', 7./12],['-x+1./3', '-x+2./3', 11./12],['-x', 0., 3./4],['-x+2./3', 1./3, 1./12],['-x+1./3', 2./3, 5./12],[0., '-x', 3./4],[2./3, '-x+1./3', 1./12],[1./3, '-x+2./3', 5./12],['x', 'x', 3./4],['x+2./3', 'x+1./3', 1./12],['x+1./3', 'x+2./3', 5./12]],
                        '36f': [['x', 'y', 'z'],['x+2./3', 'y+1./3', 'z+1./3'],['x+1./3', 'y+2./3', 'z+2./3'],['-y', 'x-y', 'z'],['-y+2./3', 'x-y+1./3', 'z+1./3'],['-y+1./3', 'x-y+2./3', 'z+2./3'],['-x+y', '-x', 'z'],['-x+y+2./3', '-x+1./3', 'z+1./3'],['-x+y+1./3', '-x+2./3', 'z+2./3'],['y', 'x', '-z+1./2'],['y+2./3', 'x+1./3', '-z+5./6'],['y+1./3', 'x+2./3', '-z+1./6'],['x-y', '-y', '-z+1./2'],['x-y+2./3', '-y+1./3', '-z+5./6'],['x-y+1./3', '-y+2./3', '-z+1./6'],['-x', '-x+y', '-z+1./2'],['-x+2./3', '-x+y+1./3', '-z+5./6'],['-x+1./3', '-x+y+2./3', '-z+1./6'],['-x', '-y', '-z'],['-x+2./3', '-y+1./3', '-z+1./3'],['-x+1./3', '-y+2./3', '-z+2./3'],['y', '-x+y', '-z'],['y+2./3', '-x+y+1./3', '-z+1./3'],['y+1./3', '-x+y+2./3', '-z+2./3'],['x-y', 'x', '-z'],['x-y+2./3', 'x+1./3', '-z+1./3'],['x-y+1./3', 'x+2./3', '-z+2./3'],['-y', '-x', 'z+1./2'],['-y+2./3', '-x+1./3', 'z+5./6'],['-y+1./3', '-x+2./3', 'z+1./6'],['-x+y', 'y', 'z+1./2'],['-x+y+2./3', 'y+1./3', 'z+5./6'],['-x+y+1./3', 'y+2./3', 'z+1./6'],['x', 'x-y', 'z+1./2'],['x+2./3', 'x-y+1./3', 'z+5./6'],['x+1./3', 'x-y+2./3', 'z+1./6']]},
              '167:R': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '2b': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '4c': [['x', 'x', 'x'],['-x+1./2', '-x+1./2', '-x+1./2'],['-x', '-x', '-x'],['x+1./2', 'x+1./2', 'x+1./2']],
                        '6d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '6e': [['x', '-x+1./2', 1./4],[1./4, 'x', '-x+1./2'],['-x+1./2', 1./4, 'x'],['-x', 'x+1./2', 3./4],[3./4, '-x', 'x+1./2'],['x+1./2', 3./4, '-x']],
                        '12f': [['x', 'y', 'z'],['z', 'x', 'y'],['y', 'z', 'x'],['-z+1./2', '-y+1./2', '-x+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-x+1./2', '-z+1./2', '-y+1./2'],['-x', '-y', '-z'],['-z', '-x', '-y'],['-y', '-z', '-x'],['z+1./2', 'y+1./2', 'x+1./2'],['y+1./2', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', 'y+1./2']]},
              '168': {'1a': [[0., 0., 'z']],
                      '2b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z']],
                      '3c': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z']],
                      '6d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z']]},
              '169': {'6a': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+5./6'],['x-y', 'x', 'z+1./6']]},
              '170': {'6a': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./6'],['x-y', 'x', 'z+5./6']]},
              '171': {'3a': [[0., 0., 'z'],[0., 0., 'z+2./3'],[0., 0., 'z+1./3']],
                      '3b': [[1./2, 1./2, 'z'],[1./2, 0., 'z+2./3'],[0., 1./2, 'z+1./3']],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3'],['-x', '-y', 'z'],['y', '-x+y', 'z+2./3'],['x-y', 'x', 'z+1./3']]},
              '172': {'3a': [[0., 0., 'z'],[0., 0., 'z+1./3'],[0., 0., 'z+2./3']],
                      '3b': [[1./2, 1./2, 'z'],[1./2, 0., 'z+1./3'],[0., 1./2, 'z+2./3']],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3'],['-x', '-y', 'z'],['y', '-x+y', 'z+1./3'],['x-y', 'x', 'z+2./3']]},
              '173': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2']],
                      '6c': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2']]},
              '174': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '1c': [[1./3, 2./3, 0]],
                      '1d': [[1./3, 2./3, 1./2]],
                      '1e': [[2./3, 1./3, 0]],
                      '1f': [[2./3, 1./3, 1./2]],
                      '2g': [[0., 0., 'z'],[0., 0., '-z']],
                      '2h': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z']],
                      '2i': [[2./3, 1./3, 'z'],[2./3, 1./3, '-z']],
                      '3j': [['x', 'y', 0],['-y', 'x-y', 0],['-x+y', '-x', 0]],
                      '3k': [['x', 'y', 1./2],['-y', 'x-y', 1./2],['-x+y', '-x', 1./2]],
                      '6l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['x', 'y', '-z'],['-y', 'x-y', '-z'],['-x+y', '-x', '-z']]},
              '175': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 0],[2./3, 1./3, 0]],
                      '2d': [[1./3, 2./3, 1./2],[2./3, 1./3, 1./2]],
                      '2e': [[0., 0., 'z'],[0., 0., '-z']],
                      '3f': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0]],
                      '3g': [[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '4h': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z']],
                      '6i': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z'],[1./2, 0., '-z'],[0., 1./2, '-z'],[1./2, 1./2, '-z']],
                      '6j': [['x', 'y', 0],['-y', 'x-y', 0],['-x+y', '-x', 0],['-x', '-y', 0],['y', '-x+y', 0],['x-y', 'x', 0]],
                      '6k': [['x', 'y', 1./2],['-y', 'x-y', 1./2],['-x+y', '-x', 1./2],['-x', '-y', 1./2],['y', '-x+y', 1./2],['x-y', 'x', 1./2]],
                      '12l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['x', 'y', '-z'],['-y', 'x-y', '-z'],['-x+y', '-x', '-z']]},
              '176': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[0., 0., 0],[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 1./4],[2./3, 1./3, 3./4]],
                      '2d': [[2./3, 1./3, 1./4],[1./3, 2./3, 3./4]],
                      '4e': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2']],
                      '4f': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z+1./2']],
                      '6g': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '6h': [['x', 'y', 1./4],['-y', 'x-y', 1./4],['-x+y', '-x', 1./4],['-x', '-y', 3./4],['y', '-x+y', 3./4],['x-y', 'x', 3./4]],
                      '12i': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['x', 'y', '-z+1./2'],['-y', 'x-y', '-z+1./2'],['-x+y', '-x', '-z+1./2']]},
              '177': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 0],[2./3, 1./3, 0]],
                      '2d': [[1./3, 2./3, 1./2],[2./3, 1./3, 1./2]],
                      '2e': [[0., 0., 'z'],[0., 0., '-z']],
                      '3f': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0]],
                      '3g': [[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '4h': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z']],
                      '6i': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z'],[0., 1./2, '-z'],[1./2, 0., '-z'],[1./2, 1./2, '-z']],
                      '6j': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0],['-x', 0., 0],[0., '-x', 0],['x', 'x', 0]],
                      '6k': [['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2],['-x', 0., 1./2],[0., '-x', 1./2],['x', 'x', 1./2]],
                      '6l': [['x', '-x', 0],['x', '2.*x', 0],['-2.*x', '-x', 0],['-x', 'x', 0],['-x', '-2.*x', 0],['2.*x', 'x', 0]],
                      '6m': [['x', '-x', 1./2],['x', '2.*x', 1./2],['-2.*x', '-x', 1./2],['-x', 'x', 1./2],['-x', '-2.*x', 1./2],['2.*x', 'x', 1./2]],
                      '12n': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z']]},
              '178': {'6a': [['x', 0., 0],[0., 'x', 1./3],['-x', '-x', 2./3],['-x', 0., 1./2],[0., '-x', 5./6],['x', 'x', 1./6]],
                      '6b': [['x', '2.*x', 1./4],['-2.*x', '-x', 7./12],['x', '-x', 11./12],['-x', '-2.*x', 3./4],['2.*x', 'x', 1./12],['-x', 'x', 5./12]],
                      '12c': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+5./6'],['x-y', 'x', 'z+1./6'],['y', 'x', '-z+1./3'],['x-y', '-y', '-z'],['-x', '-x+y', '-z+2./3'],['-y', '-x', '-z+5./6'],['-x+y', 'y', '-z+1./2'],['x', 'x-y', '-z+1./6']]},
              '179': {'6a': [['x', 0., 0],[0., 'x', 2./3],['-x', '-x', 1./3],['-x', 0., 1./2],[0., '-x', 1./6],['x', 'x', 5./6]],
                      '6b': [['x', '2.*x', 3./4],['-2.*x', '-x', 5./12],['x', '-x', 1./12],['-x', '-2.*x', 1./4],['2.*x', 'x', 11./12],['-x', 'x', 7./12]],
                      '12c': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./6'],['x-y', 'x', 'z+5./6'],['y', 'x', '-z+2./3'],['x-y', '-y', '-z'],['-x', '-x+y', '-z+1./3'],['-y', '-x', '-z+1./6'],['-x+y', 'y', '-z+1./2'],['x', 'x-y', '-z+5./6']]},
              '180': {'3a': [[0., 0., 0],[0., 0., 2./3],[0., 0., 1./3]],
                      '3b': [[0., 0., 1./2],[0., 0., 1./6],[0., 0., 5./6]],
                      '3c': [[1./2, 0., 0],[0., 1./2, 2./3],[1./2, 1./2, 1./3]],
                      '3d': [[1./2, 0., 1./2],[0., 1./2, 1./6],[1./2, 1./2, 5./6]],
                      '6e': [[0., 0., 'z'],[0., 0., 'z+2./3'],[0., 0., 'z+1./3'],[0., 0., '-z+2./3'],[0., 0., '-z'],[0., 0., '-z+1./3']],
                      '6f': [[1./2, 0., 'z'],[0., 1./2, 'z+2./3'],[1./2, 1./2, 'z+1./3'],[0., 1./2, '-z+2./3'],[1./2, 0., '-z'],[1./2, 1./2, '-z+1./3']],
                      '6g': [['x', 0., 0],[0., 'x', 2./3],['-x', '-x', 1./3],['-x', 0., 0],[0., '-x', 2./3],['x', 'x', 1./3]],
                      '6h': [['x', 0., 1./2],[0., 'x', 1./6],['-x', '-x', 5./6],['-x', 0., 1./2],[0., '-x', 1./6],['x', 'x', 5./6]],
                      '6i': [['x', '2.*x', 0],['-2.*x', '-x', 2./3],['x', '-x', 1./3],['-x', '-2.*x', 0],['2.*x', 'x', 2./3],['-x', 'x', 1./3]],
                      '6j': [['x', '2.*x', 1./2],['-2.*x', '-x', 1./6],['x', '-x', 5./6],['-x', '-2.*x', 1./2],['2.*x', 'x', 1./6],['-x', 'x', 5./6]],
                      '12k': [['x', 'y', 'z'],['-y', 'x-y', 'z+2./3'],['-x+y', '-x', 'z+1./3'],['-x', '-y', 'z'],['y', '-x+y', 'z+2./3'],['x-y', 'x', 'z+1./3'],['y', 'x', '-z+2./3'],['x-y', '-y', '-z'],['-x', '-x+y', '-z+1./3'],['-y', '-x', '-z+2./3'],['-x+y', 'y', '-z'],['x', 'x-y', '-z+1./3']]},
              '181': {'3a': [[0., 0., 0],[0., 0., 1./3],[0., 0., 2./3]],
                      '3b': [[0., 0., 1./2],[0., 0., 5./6],[0., 0., 1./6]],
                      '3c': [[1./2, 0., 0],[0., 1./2, 1./3],[1./2, 1./2, 2./3]],
                      '3d': [[1./2, 0., 1./2],[0., 1./2, 5./6],[1./2, 1./2, 1./6]],
                      '6e': [[0., 0., 'z'],[0., 0., 'z+1./3'],[0., 0., 'z+2./3'],[0., 0., '-z+1./3'],[0., 0., '-z'],[0., 0., '-z+2./3']],
                      '6f': [[1./2, 0., 'z'],[0., 1./2, 'z+1./3'],[1./2, 1./2, 'z+2./3'],[0., 1./2, '-z+1./3'],[1./2, 0., '-z'],[1./2, 1./2, '-z+2./3']],
                      '6g': [['x', 0., 0],[0., 'x', 1./3],['-x', '-x', 2./3],['-x', 0., 0],[0., '-x', 1./3],['x', 'x', 2./3]],
                      '6h': [['x', 0., 1./2],[0., 'x', 5./6],['-x', '-x', 1./6],['-x', 0., 1./2],[0., '-x', 5./6],['x', 'x', 1./6]],
                      '6i': [['x', '2.*x', 0],['-2.*x', '-x', 1./3],['x', '-x', 2./3],['-x', '-2.*x', 0],['2.*x', 'x', 1./3],['-x', 'x', 2./3]],
                      '6j': [['x', '2.*x', 1./2],['-2.*x', '-x', 5./6],['x', '-x', 1./6],['-x', '-2.*x', 1./2],['2.*x', 'x', 5./6],['-x', 'x', 1./6]],
                      '12k': [['x', 'y', 'z'],['-y', 'x-y', 'z+1./3'],['-x+y', '-x', 'z+2./3'],['-x', '-y', 'z'],['y', '-x+y', 'z+1./3'],['x-y', 'x', 'z+2./3'],['y', 'x', '-z+1./3'],['x-y', '-y', '-z'],['-x', '-x+y', '-z+2./3'],['-y', '-x', '-z+1./3'],['-x+y', 'y', '-z'],['x', 'x-y', '-z+2./3']]},
              '182': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                      '2b': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2c': [[1./3, 2./3, 1./4],[2./3, 1./3, 3./4]],
                      '2d': [[1./3, 2./3, 3./4],[2./3, 1./3, 1./4]],
                      '4e': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2']],
                      '4f': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z+1./2']],
                      '6g': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0],['-x', 0., 1./2],[0., '-x', 1./2],['x', 'x', 1./2]],
                      '6h': [['x', '2.*x', 1./4],['-2.*x', '-x', 1./4],['x', '-x', 1./4],['-x', '-2.*x', 3./4],['2.*x', 'x', 3./4],['-x', 'x', 3./4]],
                      '12i': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['-y', '-x', '-z+1./2'],['-x+y', 'y', '-z+1./2'],['x', 'x-y', '-z+1./2']]},
              '183': {'1a': [[0., 0., 'z']],
                      '2b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z']],
                      '3c': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z']],
                      '6d': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z'],['-x', 0., 'z'],[0., '-x', 'z'],['x', 'x', 'z']],
                      '6e': [['x', '-x', 'z'],['x', '2.*x', 'z'],['-2.*x', '-x', 'z'],['-x', 'x', 'z'],['-x', '-2.*x', 'z'],['2.*x', 'x', 'z']],
                      '12f': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '184': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '4b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z'],[1./3, 2./3, 'z+1./2'],[2./3, 1./3, 'z+1./2']],
                      '6c': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 1./2, 'z+1./2']],
                      '12d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '185': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '4b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2'],[1./3, 2./3, 'z+1./2'],[2./3, 1./3, 'z']],
                      '6c': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z'],['-x', 0., 'z+1./2'],[0., '-x', 'z+1./2'],['x', 'x', 'z+1./2']],
                      '12d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '186': {'2a': [[0., 0., 'z'],[0., 0., 'z+1./2']],
                      '2b': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2']],
                      '6c': [['x', '-x', 'z'],['x', '2.*x', 'z'],['-2.*x', '-x', 'z'],['-x', 'x', 'z+1./2'],['-x', '-2.*x', 'z+1./2'],['2.*x', 'x', 'z+1./2']],
                      '12d': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '187': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '1c': [[1./3, 2./3, 0]],
                      '1d': [[1./3, 2./3, 1./2]],
                      '1e': [[2./3, 1./3, 0]],
                      '1f': [[2./3, 1./3, 1./2]],
                      '2g': [[0., 0., 'z'],[0., 0., '-z']],
                      '2h': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z']],
                      '2i': [[2./3, 1./3, 'z'],[2./3, 1./3, '-z']],
                      '3j': [['x', '-x', 0],['x', '2.*x', 0],['-2.*x', '-x', 0]],
                      '3k': [['x', '-x', 1./2],['x', '2.*x', 1./2],['-2.*x', '-x', 1./2]],
                      '6l': [['x', 'y', 0],['-y', 'x-y', 0],['-x+y', '-x', 0],['-y', '-x', 0],['-x+y', 'y', 0],['x', 'x-y', 0]],
                      '6m': [['x', 'y', 1./2],['-y', 'x-y', 1./2],['-x+y', '-x', 1./2],['-y', '-x', 1./2],['-x+y', 'y', 1./2],['x', 'x-y', 1./2]],
                      '6n': [['x', '-x', 'z'],['x', '2.*x', 'z'],['-2.*x', '-x', 'z'],['x', '-x', '-z'],['x', '2.*x', '-z'],['-2.*x', '-x', '-z']],
                      '12o': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['x', 'y', '-z'],['-y', 'x-y', '-z'],['-x+y', '-x', '-z'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z']]},
              '188': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                      '2b': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2c': [[1./3, 2./3, 0],[1./3, 2./3, 1./2]],
                      '2d': [[1./3, 2./3, 1./4],[1./3, 2./3, 3./4]],
                      '2e': [[2./3, 1./3, 0],[2./3, 1./3, 1./2]],
                      '2f': [[2./3, 1./3, 1./4],[2./3, 1./3, 3./4]],
                      '4g': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., 'z+1./2'],[0., 0., '-z']],
                      '4h': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z+1./2'],[1./3, 2./3, 'z+1./2'],[1./3, 2./3, '-z']],
                      '4i': [[2./3, 1./3, 'z'],[2./3, 1./3, '-z+1./2'],[2./3, 1./3, 'z+1./2'],[2./3, 1./3, '-z']],
                      '6j': [['x', '-x', 0],['x', '2.*x', 0],['-2.*x', '-x', 0],['x', '-x', 1./2],['x', '2.*x', 1./2],['-2.*x', '-x', 1./2]],
                      '6k': [['x', 'y', 1./4],['-y', 'x-y', 1./4],['-x+y', '-x', 1./4],['-y', '-x', 3./4],['-x+y', 'y', 3./4],['x', 'x-y', 3./4]],
                      '12l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['x', 'y', '-z+1./2'],['-y', 'x-y', '-z+1./2'],['-x+y', '-x', '-z+1./2'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z']]},
              '189': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 0],[2./3, 1./3, 0]],
                      '2d': [[1./3, 2./3, 1./2],[2./3, 1./3, 1./2]],
                      '2e': [[0., 0., 'z'],[0., 0., '-z']],
                      '3f': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0]],
                      '3g': [['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2]],
                      '4h': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z'],[2./3, 1./3, '-z'],[2./3, 1./3, 'z']],
                      '6i': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z'],['x', 0., '-z'],[0., 'x', '-z'],['-x', '-x', '-z']],
                      '6j': [['x', 'y', 0],['-y', 'x-y', 0],['-x+y', '-x', 0],['y', 'x', 0],['x-y', '-y', 0],['-x', '-x+y', 0]],
                      '6k': [['x', 'y', 1./2],['-y', 'x-y', 1./2],['-x+y', '-x', 1./2],['y', 'x', 1./2],['x-y', '-y', 1./2],['-x', '-x+y', 1./2]],
                      '12l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['x', 'y', '-z'],['-y', 'x-y', '-z'],['-x+y', '-x', '-z'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '190': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                      '2b': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2c': [[1./3, 2./3, 1./4],[2./3, 1./3, 3./4]],
                      '2d': [[2./3, 1./3, 1./4],[1./3, 2./3, 3./4]],
                      '4e': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2']],
                      '4f': [[1./3, 2./3, 'z'],[1./3, 2./3, '-z+1./2'],[2./3, 1./3, '-z'],[2./3, 1./3, 'z+1./2']],
                      '6g': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0],['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2]],
                      '6h': [['x', 'y', 1./4],['-y', 'x-y', 1./4],['-x+y', '-x', 1./4],['y', 'x', 3./4],['x-y', '-y', 3./4],['-x', '-x+y', 3./4]],
                      '12i': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['x', 'y', '-z+1./2'],['-y', 'x-y', '-z+1./2'],['-x+y', '-x', '-z+1./2'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '191': {'1a': [[0., 0., 0]],
                      '1b': [[0., 0., 1./2]],
                      '2c': [[1./3, 2./3, 0],[2./3, 1./3, 0]],
                      '2d': [[1./3, 2./3, 1./2],[2./3, 1./3, 1./2]],
                      '2e': [[0., 0., 'z'],[0., 0., '-z']],
                      '3f': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0]],
                      '3g': [[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '4h': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z']],
                      '6i': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z'],[0., 1./2, '-z'],[1./2, 0., '-z'],[1./2, 1./2, '-z']],
                      '6j': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0],['-x', 0., 0],[0., '-x', 0],['x', 'x', 0]],
                      '6k': [['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2],['-x', 0., 1./2],[0., '-x', 1./2],['x', 'x', 1./2]],
                      '6l': [['x', '2.*x', 0],['-2.*x', '-x', 0],['x', '-x', 0],['-x', '-2.*x', 0],['2.*x', 'x', 0],['-x', 'x', 0]],
                      '6m': [['x', '2.*x', 1./2],['-2.*x', '-x', 1./2],['x', '-x', 1./2],['-x', '-2.*x', 1./2],['2.*x', 'x', 1./2],['-x', 'x', 1./2]],
                      '12n': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z'],['-x', 0., 'z'],[0., '-x', 'z'],['x', 'x', 'z'],[0., 'x', '-z'],['x', 0., '-z'],['-x', '-x', '-z'],[0., '-x', '-z'],['-x', 0., '-z'],['x', 'x', '-z']],
                      '12o': [['x', '2.*x', 'z'],['-2.*x', '-x', 'z'],['x', '-x', 'z'],['-x', '-2.*x', 'z'],['2.*x', 'x', 'z'],['-x', 'x', 'z'],['2.*x', 'x', '-z'],['-x', '-2.*x', '-z'],['-x', 'x', '-z'],['-2.*x', '-x', '-z'],['x', '2.*x', '-z'],['x', '-x', '-z']],
                      '12p': [['x', 'y', 0],['-y', 'x-y', 0],['-x+y', '-x', 0],['-x', '-y', 0],['y', '-x+y', 0],['x-y', 'x', 0],['y', 'x', 0],['x-y', '-y', 0],['-x', '-x+y', 0],['-y', '-x', 0],['-x+y', 'y', 0],['x', 'x-y', 0]],
                      '12q': [['x', 'y', 1./2],['-y', 'x-y', 1./2],['-x+y', '-x', 1./2],['-x', '-y', 1./2],['y', '-x+y', 1./2],['x-y', 'x', 1./2],['y', 'x', 1./2],['x-y', '-y', 1./2],['-x', '-x+y', 1./2],['-y', '-x', 1./2],['-x+y', 'y', 1./2],['x', 'x-y', 1./2]],
                      '24r': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['x', 'y', '-z'],['-y', 'x-y', '-z'],['-x+y', '-x', '-z'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '192': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[0., 0., 0],[0., 0., 1./2]],
                      '4c': [[1./3, 2./3, 1./4],[2./3, 1./3, 1./4],[2./3, 1./3, 3./4],[1./3, 2./3, 3./4]],
                      '4d': [[1./3, 2./3, 0],[2./3, 1./3, 0],[2./3, 1./3, 1./2],[1./3, 2./3, 1./2]],
                      '4e': [[0., 0., 'z'],[0., 0., '-z+1./2'],[0., 0., '-z'],[0., 0., 'z+1./2']],
                      '6f': [[1./2, 0., 1./4],[0., 1./2, 1./4],[1./2, 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4],[1./2, 1./2, 3./4]],
                      '6g': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 1./2]],
                      '8h': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z'],[2./3, 1./3, '-z+1./2'],[1./3, 2./3, '-z+1./2'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z'],[1./3, 2./3, 'z+1./2'],[2./3, 1./3, 'z+1./2']],
                      '12i': [[1./2, 0., 'z'],[0., 1./2, 'z'],[1./2, 1./2, 'z'],[0., 1./2, '-z+1./2'],[1./2, 0., '-z+1./2'],[1./2, 1./2, '-z+1./2'],[1./2, 0., '-z'],[0., 1./2, '-z'],[1./2, 1./2, '-z'],[0., 1./2, 'z+1./2'],[1./2, 0., 'z+1./2'],[1./2, 1./2, 'z+1./2']],
                      '12j': [['x', 0., 1./4],[0., 'x', 1./4],['-x', '-x', 1./4],['-x', 0., 1./4],[0., '-x', 1./4],['x', 'x', 1./4],['-x', 0., 3./4],[0., '-x', 3./4],['x', 'x', 3./4],['x', 0., 3./4],[0., 'x', 3./4],['-x', '-x', 3./4]],
                      '12k': [['x', '2.*x', 1./4],['-2.*x', '-x', 1./4],['x', '-x', 1./4],['-x', '-2.*x', 1./4],['2.*x', 'x', 1./4],['-x', 'x', 1./4],['-x', '-2.*x', 3./4],['2.*x', 'x', 3./4],['-x', 'x', 3./4],['x', '2.*x', 3./4],['-2.*x', '-x', 3./4],['x', '-x', 3./4]],
                      '12l': [['x', 'y', 0],['-y', 'x-y', 0],['-x+y', '-x', 0],['-x', '-y', 0],['y', '-x+y', 0],['x-y', 'x', 0],['y', 'x', 1./2],['x-y', '-y', 1./2],['-x', '-x+y', 1./2],['-y', '-x', 1./2],['-x+y', 'y', 1./2],['x', 'x-y', 1./2]],
                      '24m': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z'],['y', '-x+y', 'z'],['x-y', 'x', 'z'],['y', 'x', '-z+1./2'],['x-y', '-y', '-z+1./2'],['-x', '-x+y', '-z+1./2'],['-y', '-x', '-z+1./2'],['-x+y', 'y', '-z+1./2'],['x', 'x-y', '-z+1./2'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['x', 'y', '-z'],['-y', 'x-y', '-z'],['-x+y', '-x', '-z'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '193': {'2a': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2b': [[0., 0., 0],[0., 0., 1./2]],
                      '4c': [[1./3, 2./3, 1./4],[2./3, 1./3, 3./4],[2./3, 1./3, 1./4],[1./3, 2./3, 3./4]],
                      '4d': [[1./3, 2./3, 0],[2./3, 1./3, 1./2],[2./3, 1./3, 0],[1./3, 2./3, 1./2]],
                      '4e': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z+1./2'],[0., 0., '-z']],
                      '6f': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '6g': [['x', 0., 1./4],[0., 'x', 1./4],['-x', '-x', 1./4],['-x', 0., 3./4],[0., '-x', 3./4],['x', 'x', 3./4]],
                      '8h': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2'],[2./3, 1./3, '-z+1./2'],[1./3, 2./3, '-z'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z+1./2'],[1./3, 2./3, 'z+1./2'],[2./3, 1./3, 'z']],
                      '12i': [['x', '2.*x', 0],['-2.*x', '-x', 0],['x', '-x', 0],['-x', '-2.*x', 1./2],['2.*x', 'x', 1./2],['-x', 'x', 1./2],['-x', '-2.*x', 0],['2.*x', 'x', 0],['-x', 'x', 0],['x', '2.*x', 1./2],['-2.*x', '-x', 1./2],['x', '-x', 1./2]],
                      '12j': [['x', 'y', 1./4],['-y', 'x-y', 1./4],['-x+y', '-x', 1./4],['-x', '-y', 3./4],['y', '-x+y', 3./4],['x-y', 'x', 3./4],['y', 'x', 1./4],['x-y', '-y', 1./4],['-x', '-x+y', 1./4],['-y', '-x', 3./4],['-x+y', 'y', 3./4],['x', 'x-y', 3./4]],
                      '12k': [['x', 0., 'z'],[0., 'x', 'z'],['-x', '-x', 'z'],['-x', 0., 'z+1./2'],[0., '-x', 'z+1./2'],['x', 'x', 'z+1./2'],[0., 'x', '-z+1./2'],['x', 0., '-z+1./2'],['-x', '-x', '-z+1./2'],[0., '-x', '-z'],['-x', 0., '-z'],['x', 'x', '-z']],
                      '24l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2'],['y', 'x', '-z+1./2'],['x-y', '-y', '-z+1./2'],['-x', '-x+y', '-z+1./2'],['-y', '-x', '-z'],['-x+y', 'y', '-z'],['x', 'x-y', '-z'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['x', 'y', '-z+1./2'],['-y', 'x-y', '-z+1./2'],['-x+y', '-x', '-z+1./2'],['-y', '-x', 'z+1./2'],['-x+y', 'y', 'z+1./2'],['x', 'x-y', 'z+1./2'],['y', 'x', 'z'],['x-y', '-y', 'z'],['-x', '-x+y', 'z']]},
              '194': {'2a': [[0., 0., 0],[0., 0., 1./2]],
                      '2b': [[0., 0., 1./4],[0., 0., 3./4]],
                      '2c': [[1./3, 2./3, 1./4],[2./3, 1./3, 3./4]],
                      '2d': [[1./3, 2./3, 3./4],[2./3, 1./3, 1./4]],
                      '4e': [[0., 0., 'z'],[0., 0., 'z+1./2'],[0., 0., '-z'],[0., 0., '-z+1./2']],
                      '4f': [[1./3, 2./3, 'z'],[2./3, 1./3, 'z+1./2'],[2./3, 1./3, '-z'],[1./3, 2./3, '-z+1./2']],
                      '6g': [[1./2, 0., 0],[0., 1./2, 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 1./2]],
                      '6h': [['x', '2.*x', 1./4],['-2.*x', '-x', 1./4],['x', '-x', 1./4],['-x', '-2.*x', 3./4],['2.*x', 'x', 3./4],['-x', 'x', 3./4]],
                      '12i': [['x', 0., 0],[0., 'x', 0],['-x', '-x', 0],['-x', 0., 1./2],[0., '-x', 1./2],['x', 'x', 1./2],['-x', 0., 0],[0., '-x', 0],['x', 'x', 0],['x', 0., 1./2],[0., 'x', 1./2],['-x', '-x', 1./2]],
                      '12j': [['x', 'y', 1./4],['-y', 'x-y', 1./4],['-x+y', '-x', 1./4],['-x', '-y', 3./4],['y', '-x+y', 3./4],['x-y', 'x', 3./4],['y', 'x', 3./4],['x-y', '-y', 3./4],['-x', '-x+y', 3./4],['-y', '-x', 1./4],['-x+y', 'y', 1./4],['x', 'x-y', 1./4]],
                      '12k': [['x', '2.*x', 'z'],['-2.*x', '-x', 'z'],['x', '-x', 'z'],['-x', '-2.*x', 'z+1./2'],['2.*x', 'x', 'z+1./2'],['-x', 'x', 'z+1./2'],['2.*x', 'x', '-z'],['-x', '-2.*x', '-z'],['-x', 'x', '-z'],['-2.*x', '-x', '-z+1./2'],['x', '2.*x', '-z+1./2'],['x', '-x', '-z+1./2']],
                      '24l': [['x', 'y', 'z'],['-y', 'x-y', 'z'],['-x+y', '-x', 'z'],['-x', '-y', 'z+1./2'],['y', '-x+y', 'z+1./2'],['x-y', 'x', 'z+1./2'],['y', 'x', '-z'],['x-y', '-y', '-z'],['-x', '-x+y', '-z'],['-y', '-x', '-z+1./2'],['-x+y', 'y', '-z+1./2'],['x', 'x-y', '-z+1./2'],['-x', '-y', '-z'],['y', '-x+y', '-z'],['x-y', 'x', '-z'],['x', 'y', '-z+1./2'],['-y', 'x-y', '-z+1./2'],['-x+y', '-x', '-z+1./2'],['-y', '-x', 'z'],['-x+y', 'y', 'z'],['x', 'x-y', 'z'],['y', 'x', 'z+1./2'],['x-y', '-y', 'z+1./2'],['-x', '-x+y', 'z+1./2']]},
              '195': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 1./2]],
                      '3c': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '4e': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x']],
                      '6f': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x']],
                      '6g': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x']],
                      '6h': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x']],
                      '6i': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2],[1./2, 1./2, 'x'],[1./2, 1./2, '-x']],
                      '12j': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x']]},
              '196': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '4b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '4c': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                      '4d': [[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                      '16e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x']],
                      '24f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x']],
                      '24g': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x']],
                      '48h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x']]},
              '197': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 0., 1./2],[0., 1./2, 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                      '8c': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', '-x'],['-x+1./2', 'x+1./2', '-x+1./2'],['x', '-x', '-x'],['x+1./2', '-x+1./2', '-x+1./2']],
                      '12d': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2],[0., 0., 'x'],[1./2, 1./2, 'x+1./2'],[0., 0., '-x'],[1./2, 1./2, '-x+1./2']],
                      '12e': [['x', 1./2, 0],['x+1./2', 0., 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2],[0., 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 1./2],[1./2, '-x+1./2', 0],[1./2, 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., '-x'],[0., 1./2, '-x+1./2']],
                      '24f': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z', '-x', '-y'],['z+1./2', '-x+1./2', '-y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y+1./2'],['-z', 'x', '-y'],['-z+1./2', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z', '-x'],['-y+1./2', 'z+1./2', '-x+1./2'],['y', '-z', '-x'],['y+1./2', '-z+1./2', '-x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x+1./2']]},
              '198': {'4a': [['x', 'x', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', '-x']],
                      '12b': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z+1./2', '-x+1./2', '-y'],['-z+1./2', '-x', 'y+1./2'],['-z', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['-y', 'z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y+1./2', '-z', 'x+1./2']]},
              '199': {'8a': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x', '-x+1./2', 'x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x+1./2']],
                      '12b': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],[1./4, 'x', 0],[3./4, 'x+1./2', 1./2],[3./4, '-x+1./2', 0],[1./4, '-x', 1./2],[0., 1./4, 'x'],[1./2, 3./4, 'x+1./2'],[0., 3./4, '-x+1./2'],[1./2, 1./4, '-x']],
                      '24c': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z', '-x+1./2', 'y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y', '-z+1./2', 'x']]},
              '200': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 1./2]],
                      '3c': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '6e': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x']],
                      '6f': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x']],
                      '6g': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x']],
                      '6h': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2],[1./2, 1./2, 'x'],[1./2, 1./2, '-x']],
                      '8i': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['-x', '-x', '-x'],['x', 'x', '-x'],['x', '-x', 'x'],['-x', 'x', 'x']],
                      '12j': [[0., 'y', 'z'],[0., '-y', 'z'],[0., 'y', '-z'],[0., '-y', '-z'],['z', 0., 'y'],['z', 0., '-y'],['-z', 0., 'y'],['-z', 0., '-y'],['y', 'z', 0],['-y', 'z', 0],['y', '-z', 0],['-y', '-z', 0]],
                      '12k': [[1./2, 'y', 'z'],[1./2, '-y', 'z'],[1./2, 'y', '-z'],[1./2, '-y', '-z'],['z', 1./2, 'y'],['z', 1./2, '-y'],['-z', 1./2, 'y'],['-z', 1./2, '-y'],['y', 'z', 1./2],['-y', 'z', 1./2],['y', '-z', 1./2],['-y', '-z', 1./2]],
                      '24l': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['-x', '-y', '-z'],['x', 'y', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-z', '-x', '-y'],['-z', 'x', 'y'],['z', 'x', '-y'],['z', '-x', 'y'],['-y', '-z', '-x'],['y', '-z', 'x'],['-y', 'z', 'x'],['y', 'z', '-x']]},
              '201:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '4b': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4]],
                        '4c': [[3./4, 3./4, 3./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                        '6d': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                        '8e': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', 'x+1./2']],
                        '12f': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x'],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2],[1./2, '-x+1./2', 1./2],[1./2, 'x+1./2', 1./2],[1./2, 1./2, '-x+1./2'],[1./2, 1./2, 'x+1./2']],
                        '12g': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],['-x+1./2', 0., 1./2],['x+1./2', 0., 1./2],[1./2, '-x+1./2', 0],[1./2, 'x+1./2', 0],[0., 1./2, '-x+1./2'],[0., 1./2, 'x+1./2']],
                        '24h': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', 'x+1./2', '-y+1./2'],['z+1./2', '-x+1./2', 'y+1./2'],['-y+1./2', '-z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', 'x+1./2'],['-y+1./2', 'z+1./2', 'x+1./2'],['y+1./2', 'z+1./2', '-x+1./2']]},
              '201:2': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '4b': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '4c': [[1./2, 1./2, 1./2],[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                        '6d': [[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                        '8e': [['x', 'x', 'x'],['-x+1./2', '-x+1./2', 'x'],['-x+1./2', 'x', '-x+1./2'],['x', '-x+1./2', '-x+1./2'],['-x', '-x', '-x'],['x+1./2', 'x+1./2', '-x'],['x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', 'x+1./2']],
                        '12f': [['x', 1./4, 1./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, '-x+1./2', 1./4],[1./4, 1./4, 'x'],[1./4, 1./4, '-x+1./2'],['-x', 3./4, 3./4],['x+1./2', 3./4, 3./4],[3./4, '-x', 3./4],[3./4, 'x+1./2', 3./4],[3./4, 3./4, '-x'],[3./4, 3./4, 'x+1./2']],
                        '12g': [['x', 3./4, 1./4],['-x+1./2', 3./4, 1./4],[1./4, 'x', 3./4],[1./4, '-x+1./2', 3./4],[3./4, 1./4, 'x'],[3./4, 1./4, '-x+1./2'],['-x', 1./4, 3./4],['x+1./2', 1./4, 3./4],[3./4, '-x', 1./4],[3./4, 'x+1./2', 1./4],[1./4, 3./4, '-x'],[1./4, 3./4, 'x+1./2']],
                        '24h': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z+1./2', 'x', '-y+1./2'],['y', 'z', 'x'],['-y+1./2', 'z', '-x+1./2'],['y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z+1./2', 'x'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2'],['-z', '-x', '-y'],['-z', 'x+1./2', 'y+1./2'],['z+1./2', 'x+1./2', '-y'],['z+1./2', '-x', 'y+1./2'],['-y', '-z', '-x'],['y+1./2', '-z', 'x+1./2'],['-y', 'z+1./2', 'x+1./2'],['y+1./2', 'z+1./2', '-x']]},
              '202': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '4b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '8c': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                      '24d': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0]],
                      '24e': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x']],
                      '32f': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x', 'x', '-x'],['x', 'x+1./2', '-x+1./2'],['x+1./2', 'x', '-x+1./2'],['x+1./2', 'x+1./2', '-x'],['x', '-x', 'x'],['x', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x+1./2'],['x+1./2', '-x+1./2', 'x'],['-x', 'x', 'x'],['-x', 'x+1./2', 'x+1./2'],['-x+1./2', 'x', 'x+1./2'],['-x+1./2', 'x+1./2', 'x']],
                      '48g': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x'],['-x', 3./4, 3./4],['-x', 1./4, 1./4],['-x+1./2', 3./4, 1./4],['-x+1./2', 1./4, 3./4],['x', 1./4, 3./4],['x', 3./4, 1./4],['x+1./2', 1./4, 1./4],['x+1./2', 3./4, 3./4],[3./4, '-x', 3./4],[3./4, '-x+1./2', 1./4],[1./4, '-x', 1./4],[1./4, '-x+1./2', 3./4],[3./4, 'x', 1./4],[3./4, 'x+1./2', 3./4],[1./4, 'x', 3./4],[1./4, 'x+1./2', 1./4],[3./4, 3./4, '-x'],[3./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x'],[1./4, 3./4, 'x'],[1./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x']],
                      '48h': [[0., 'y', 'z'],[0., 'y+1./2', 'z+1./2'],[1./2, 'y', 'z+1./2'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[0., '-y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[1./2, '-y+1./2', 'z'],[0., 'y', '-z'],[0., 'y+1./2', '-z+1./2'],[1./2, 'y', '-z+1./2'],[1./2, 'y+1./2', '-z'],[0., '-y', '-z'],[0., '-y+1./2', '-z+1./2'],[1./2, '-y', '-z+1./2'],[1./2, '-y+1./2', '-z'],['z', 0., 'y'],['z', 1./2, 'y+1./2'],['z+1./2', 0., 'y+1./2'],['z+1./2', 1./2, 'y'],['z', 0., '-y'],['z', 1./2, '-y+1./2'],['z+1./2', 0., '-y+1./2'],['z+1./2', 1./2, '-y'],['-z', 0., 'y'],['-z', 1./2, 'y+1./2'],['-z+1./2', 0., 'y+1./2'],['-z+1./2', 1./2, 'y'],['-z', 0., '-y'],['-z', 1./2, '-y+1./2'],['-z+1./2', 0., '-y+1./2'],['-z+1./2', 1./2, '-y'],['y', 'z', 0],['y', 'z+1./2', 1./2],['y+1./2', 'z', 1./2],['y+1./2', 'z+1./2', 0],['-y', 'z', 0],['-y', 'z+1./2', 1./2],['-y+1./2', 'z', 1./2],['-y+1./2', 'z+1./2', 0],['y', '-z', 0],['y', '-z+1./2', 1./2],['y+1./2', '-z', 1./2],['y+1./2', '-z+1./2', 0],['-y', '-z', 0],['-y', '-z+1./2', 1./2],['-y+1./2', '-z', 1./2],['-y+1./2', '-z+1./2', 0]],
                      '96i': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z'],['x', 'y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x', 'y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-z', '-x', '-y'],['-z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x', '-y+1./2'],['-z+1./2', '-x+1./2', '-y'],['-z', 'x', 'y'],['-z', 'x+1./2', 'y+1./2'],['-z+1./2', 'x', 'y+1./2'],['-z+1./2', 'x+1./2', 'y'],['z', 'x', '-y'],['z', 'x+1./2', '-y+1./2'],['z+1./2', 'x', '-y+1./2'],['z+1./2', 'x+1./2', '-y'],['z', '-x', 'y'],['z', '-x+1./2', 'y+1./2'],['z+1./2', '-x', 'y+1./2'],['z+1./2', '-x+1./2', 'y'],['-y', '-z', '-x'],['-y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z', '-x+1./2'],['-y+1./2', '-z+1./2', '-x'],['y', '-z', 'x'],['y', '-z+1./2', 'x+1./2'],['y+1./2', '-z', 'x+1./2'],['y+1./2', '-z+1./2', 'x'],['-y', 'z', 'x'],['-y', 'z+1./2', 'x+1./2'],['-y+1./2', 'z', 'x+1./2'],['-y+1./2', 'z+1./2', 'x'],['y', 'z', '-x'],['y', 'z+1./2', '-x+1./2'],['y+1./2', 'z', '-x+1./2'],['y+1./2', 'z+1./2', '-x']]},
              '203:1': {'8a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                        '8b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                        '16c': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 7/8, 1/8],[7/8, 3/8, 5/8],[3/8, 7/8, 5/8],[3/8, 3/8, 1/8],[7/8, 1/8, 7/8],[7/8, 5/8, 3/8],[3/8, 1/8, 3/8],[3/8, 5/8, 7/8],[1/8, 7/8, 7/8],[1/8, 3/8, 3/8],[5/8, 7/8, 3/8],[5/8, 3/8, 7/8]],
                        '16d': [[5/8, 5/8, 5/8],[5/8, 1/8, 1/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[3/8, 3/8, 5/8],[3/8, 7/8, 1/8],[7/8, 3/8, 1/8],[7/8, 7/8, 5/8],[3/8, 5/8, 3/8],[3/8, 1/8, 7/8],[7/8, 5/8, 7/8],[7/8, 1/8, 3/8],[5/8, 3/8, 3/8],[5/8, 7/8, 7/8],[1/8, 3/8, 7/8],[1/8, 7/8, 3/8]],
                        '32e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['-x+1./4', '-x+1./4', '-x+1./4'],['-x+1./4', '-x+3./4', '-x+3./4'],['-x+3./4', '-x+1./4', '-x+3./4'],['-x+3./4', '-x+3./4', '-x+1./4'],['x+1./4', 'x+1./4', '-x+1./4'],['x+1./4', 'x+3./4', '-x+3./4'],['x+3./4', 'x+1./4', '-x+3./4'],['x+3./4', 'x+3./4', '-x+1./4'],['x+1./4', '-x+1./4', 'x+1./4'],['x+1./4', '-x+3./4', 'x+3./4'],['x+3./4', '-x+1./4', 'x+3./4'],['x+3./4', '-x+3./4', 'x+1./4'],['-x+1./4', 'x+1./4', 'x+1./4'],['-x+1./4', 'x+3./4', 'x+3./4'],['-x+3./4', 'x+1./4', 'x+3./4'],['-x+3./4', 'x+3./4', 'x+1./4']],
                        '48f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x'],['-x+1./4', 1./4, 1./4],['-x+1./4', 3./4, 3./4],['-x+3./4', 1./4, 3./4],['-x+3./4', 3./4, 1./4],['x+1./4', 1./4, 1./4],['x+1./4', 3./4, 3./4],['x+3./4', 1./4, 3./4],['x+3./4', 3./4, 1./4],[1./4, '-x+1./4', 1./4],[1./4, '-x+3./4', 3./4],[3./4, '-x+1./4', 3./4],[3./4, '-x+3./4', 1./4],[1./4, 'x+1./4', 1./4],[1./4, 'x+3./4', 3./4],[3./4, 'x+1./4', 3./4],[3./4, 'x+3./4', 1./4],[1./4, 1./4, '-x+1./4'],[1./4, 3./4, '-x+3./4'],[3./4, 1./4, '-x+3./4'],[3./4, 3./4, '-x+1./4'],[1./4, 1./4, 'x+1./4'],[1./4, 3./4, 'x+3./4'],[3./4, 1./4, 'x+3./4'],[3./4, 3./4, 'x+1./4']],
                        '96g': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['-x+1./4', '-y+1./4', '-z+1./4'],['-x+1./4', '-y+3./4', '-z+3./4'],['-x+3./4', '-y+1./4', '-z+3./4'],['-x+3./4', '-y+3./4', '-z+1./4'],['x+1./4', 'y+1./4', '-z+1./4'],['x+1./4', 'y+3./4', '-z+3./4'],['x+3./4', 'y+1./4', '-z+3./4'],['x+3./4', 'y+3./4', '-z+1./4'],['x+1./4', '-y+1./4', 'z+1./4'],['x+1./4', '-y+3./4', 'z+3./4'],['x+3./4', '-y+1./4', 'z+3./4'],['x+3./4', '-y+3./4', 'z+1./4'],['-x+1./4', 'y+1./4', 'z+1./4'],['-x+1./4', 'y+3./4', 'z+3./4'],['-x+3./4', 'y+1./4', 'z+3./4'],['-x+3./4', 'y+3./4', 'z+1./4'],['-z+1./4', '-x+1./4', '-y+1./4'],['-z+1./4', '-x+3./4', '-y+3./4'],['-z+3./4', '-x+1./4', '-y+3./4'],['-z+3./4', '-x+3./4', '-y+1./4'],['-z+1./4', 'x+1./4', 'y+1./4'],['-z+1./4', 'x+3./4', 'y+3./4'],['-z+3./4', 'x+1./4', 'y+3./4'],['-z+3./4', 'x+3./4', 'y+1./4'],['z+1./4', 'x+1./4', '-y+1./4'],['z+1./4', 'x+3./4', '-y+3./4'],['z+3./4', 'x+1./4', '-y+3./4'],['z+3./4', 'x+3./4', '-y+1./4'],['z+1./4', '-x+1./4', 'y+1./4'],['z+1./4', '-x+3./4', 'y+3./4'],['z+3./4', '-x+1./4', 'y+3./4'],['z+3./4', '-x+3./4', 'y+1./4'],['-y+1./4', '-z+1./4', '-x+1./4'],['-y+1./4', '-z+3./4', '-x+3./4'],['-y+3./4', '-z+1./4', '-x+3./4'],['-y+3./4', '-z+3./4', '-x+1./4'],['y+1./4', '-z+1./4', 'x+1./4'],['y+1./4', '-z+3./4', 'x+3./4'],['y+3./4', '-z+1./4', 'x+3./4'],['y+3./4', '-z+3./4', 'x+1./4'],['-y+1./4', 'z+1./4', 'x+1./4'],['-y+1./4', 'z+3./4', 'x+3./4'],['-y+3./4', 'z+1./4', 'x+3./4'],['-y+3./4', 'z+3./4', 'x+1./4'],['y+1./4', 'z+1./4', '-x+1./4'],['y+1./4', 'z+3./4', '-x+3./4'],['y+3./4', 'z+1./4', '-x+3./4'],['y+3./4', 'z+3./4', '-x+1./4']]},
              '203:2': {'8a': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 7/8, 7/8],[7/8, 3/8, 3/8],[3/8, 7/8, 3/8],[3/8, 3/8, 7/8]],
                        '8b': [[5/8, 5/8, 5/8],[5/8, 1/8, 1/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[3/8, 3/8, 3/8],[3/8, 7/8, 7/8],[7/8, 3/8, 7/8],[7/8, 7/8, 3/8]],
                        '16c': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[3./4, 3./4, 0],[3./4, 1./4, 1./2],[1./4, 3./4, 1./2],[1./4, 1./4, 0],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[0., 3./4, 3./4],[0., 1./4, 1./4],[1./2, 3./4, 1./4],[1./2, 1./4, 3./4]],
                        '16d': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./2, 1./4],[1./4, 0., 3./4],[3./4, 1./2, 3./4],[3./4, 0., 1./4],[1./2, 1./4, 1./4],[1./2, 3./4, 3./4],[0., 1./4, 3./4],[0., 3./4, 1./4]],
                        '32e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x+3./4', '-x+3./4', 'x'],['-x+3./4', '-x+1./4', 'x+1./2'],['-x+1./4', '-x+3./4', 'x+1./2'],['-x+1./4', '-x+1./4', 'x'],['-x+3./4', 'x', '-x+3./4'],['-x+3./4', 'x+1./2', '-x+1./4'],['-x+1./4', 'x', '-x+1./4'],['-x+1./4', 'x+1./2', '-x+3./4'],['x', '-x+3./4', '-x+3./4'],['x', '-x+1./4', '-x+1./4'],['x+1./2', '-x+3./4', '-x+1./4'],['x+1./2', '-x+1./4', '-x+3./4'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x+1./4', 'x+1./4', '-x'],['x+1./4', 'x+3./4', '-x+1./2'],['x+3./4', 'x+1./4', '-x+1./2'],['x+3./4', 'x+3./4', '-x'],['x+1./4', '-x', 'x+1./4'],['x+1./4', '-x+1./2', 'x+3./4'],['x+3./4', '-x', 'x+3./4'],['x+3./4', '-x+1./2', 'x+1./4'],['-x', 'x+1./4', 'x+1./4'],['-x', 'x+3./4', 'x+3./4'],['-x+1./2', 'x+1./4', 'x+3./4'],['-x+1./2', 'x+3./4', 'x+1./4']],
                        '48f': [['x', 1/8, 1/8],['x', 5/8, 5/8],['x+1./2', 1/8, 5/8],['x+1./2', 5/8, 1/8],['-x+3./4', 5/8, 1/8],['-x+3./4', 1/8, 5/8],['-x+1./4', 5/8, 5/8],['-x+1./4', 1/8, 1/8],[1/8, 'x', 1/8],[1/8, 'x+1./2', 5/8],[5/8, 'x', 5/8],[5/8, 'x+1./2', 1/8],[1/8, '-x+3./4', 5/8],[1/8, '-x+1./4', 1/8],[5/8, '-x+3./4', 1/8],[5/8, '-x+1./4', 5/8],[1/8, 1/8, 'x'],[1/8, 5/8, 'x+1./2'],[5/8, 1/8, 'x+1./2'],[5/8, 5/8, 'x'],[5/8, 1/8, '-x+3./4'],[5/8, 5/8, '-x+1./4'],[1/8, 1/8, '-x+1./4'],[1/8, 5/8, '-x+3./4'],['-x', 7/8, 7/8],['-x', 3/8, 3/8],['-x+1./2', 7/8, 3/8],['-x+1./2', 3/8, 7/8],['x+1./4', 3/8, 7/8],['x+1./4', 7/8, 3/8],['x+3./4', 3/8, 3/8],['x+3./4', 7/8, 7/8],[7/8, '-x', 7/8],[7/8, '-x+1./2', 3/8],[3/8, '-x', 3/8],[3/8, '-x+1./2', 7/8],[7/8, 'x+1./4', 3/8],[7/8, 'x+3./4', 7/8],[3/8, 'x+1./4', 7/8],[3/8, 'x+3./4', 3/8],[7/8, 7/8, '-x'],[7/8, 3/8, '-x+1./2'],[3/8, 7/8, '-x+1./2'],[3/8, 3/8, '-x'],[3/8, 7/8, 'x+1./4'],[3/8, 3/8, 'x+3./4'],[7/8, 7/8, 'x+3./4'],[7/8, 3/8, 'x+1./4']],
                        '96g': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x+3./4', '-y+3./4', 'z'],['-x+3./4', '-y+1./4', 'z+1./2'],['-x+1./4', '-y+3./4', 'z+1./2'],['-x+1./4', '-y+1./4', 'z'],['-x+3./4', 'y', '-z+3./4'],['-x+3./4', 'y+1./2', '-z+1./4'],['-x+1./4', 'y', '-z+1./4'],['-x+1./4', 'y+1./2', '-z+3./4'],['x', '-y+3./4', '-z+3./4'],['x', '-y+1./4', '-z+1./4'],['x+1./2', '-y+3./4', '-z+1./4'],['x+1./2', '-y+1./4', '-z+3./4'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x+3./4', '-y+3./4'],['z', '-x+1./4', '-y+1./4'],['z+1./2', '-x+3./4', '-y+1./4'],['z+1./2', '-x+1./4', '-y+3./4'],['-z+3./4', '-x+3./4', 'y'],['-z+3./4', '-x+1./4', 'y+1./2'],['-z+1./4', '-x+3./4', 'y+1./2'],['-z+1./4', '-x+1./4', 'y'],['-z+3./4', 'x', '-y+3./4'],['-z+3./4', 'x+1./2', '-y+1./4'],['-z+1./4', 'x', '-y+1./4'],['-z+1./4', 'x+1./2', '-y+3./4'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y+3./4', 'z', '-x+3./4'],['-y+3./4', 'z+1./2', '-x+1./4'],['-y+1./4', 'z', '-x+1./4'],['-y+1./4', 'z+1./2', '-x+3./4'],['y', '-z+3./4', '-x+3./4'],['y', '-z+1./4', '-x+1./4'],['y+1./2', '-z+3./4', '-x+1./4'],['y+1./2', '-z+1./4', '-x+3./4'],['-y+3./4', '-z+3./4', 'x'],['-y+3./4', '-z+1./4', 'x+1./2'],['-y+1./4', '-z+3./4', 'x+1./2'],['-y+1./4', '-z+1./4', 'x'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x+1./4', 'y+1./4', '-z'],['x+1./4', 'y+3./4', '-z+1./2'],['x+3./4', 'y+1./4', '-z+1./2'],['x+3./4', 'y+3./4', '-z'],['x+1./4', '-y', 'z+1./4'],['x+1./4', '-y+1./2', 'z+3./4'],['x+3./4', '-y', 'z+3./4'],['x+3./4', '-y+1./2', 'z+1./4'],['-x', 'y+1./4', 'z+1./4'],['-x', 'y+3./4', 'z+3./4'],['-x+1./2', 'y+1./4', 'z+3./4'],['-x+1./2', 'y+3./4', 'z+1./4'],['-z', '-x', '-y'],['-z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x', '-y+1./2'],['-z+1./2', '-x+1./2', '-y'],['-z', 'x+1./4', 'y+1./4'],['-z', 'x+3./4', 'y+3./4'],['-z+1./2', 'x+1./4', 'y+3./4'],['-z+1./2', 'x+3./4', 'y+1./4'],['z+1./4', 'x+1./4', '-y'],['z+1./4', 'x+3./4', '-y+1./2'],['z+3./4', 'x+1./4', '-y+1./2'],['z+3./4', 'x+3./4', '-y'],['z+1./4', '-x', 'y+1./4'],['z+1./4', '-x+1./2', 'y+3./4'],['z+3./4', '-x', 'y+3./4'],['z+3./4', '-x+1./2', 'y+1./4'],['-y', '-z', '-x'],['-y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z', '-x+1./2'],['-y+1./2', '-z+1./2', '-x'],['y+1./4', '-z', 'x+1./4'],['y+1./4', '-z+1./2', 'x+3./4'],['y+3./4', '-z', 'x+3./4'],['y+3./4', '-z+1./2', 'x+1./4'],['-y', 'z+1./4', 'x+1./4'],['-y', 'z+3./4', 'x+3./4'],['-y+1./2', 'z+1./4', 'x+3./4'],['-y+1./2', 'z+3./4', 'x+1./4'],['y+1./4', 'z+1./4', '-x'],['y+1./4', 'z+3./4', '-x+1./2'],['y+3./4', 'z+1./4', '-x+1./2'],['y+3./4', 'z+3./4', '-x']]},
              '204': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 0., 1./2],[0., 1./2, 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                      '8c': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                      '12d': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2],[0., 0., 'x'],[1./2, 1./2, 'x+1./2'],[0., 0., '-x'],[1./2, 1./2, '-x+1./2']],
                      '12e': [['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0],[1./2, 'x', 0],[0., 'x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 1./2],[0., 1./2, 'x'],[1./2, 0., 'x+1./2'],[0., 1./2, '-x'],[1./2, 0., '-x+1./2']],
                      '16f': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', '-x'],['-x+1./2', 'x+1./2', '-x+1./2'],['x', '-x', '-x'],['x+1./2', '-x+1./2', '-x+1./2'],['-x', '-x', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x', 'x', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['x', '-x', 'x'],['x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', 'x'],['-x+1./2', 'x+1./2', 'x+1./2']],
                      '24g': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z+1./2'],[0., 'y', '-z'],[1./2, 'y+1./2', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z+1./2'],['z', 0., 'y'],['z+1./2', 1./2, 'y+1./2'],['z', 0., '-y'],['z+1./2', 1./2, '-y+1./2'],['-z', 0., 'y'],['-z+1./2', 1./2, 'y+1./2'],['-z', 0., '-y'],['-z+1./2', 1./2, '-y+1./2'],['y', 'z', 0],['y+1./2', 'z+1./2', 1./2],['-y', 'z', 0],['-y+1./2', 'z+1./2', 1./2],['y', '-z', 0],['y+1./2', '-z+1./2', 1./2],['-y', '-z', 0],['-y+1./2', '-z+1./2', 1./2]],
                      '48h': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z', '-x', '-y'],['z+1./2', '-x+1./2', '-y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y+1./2'],['-z', 'x', '-y'],['-z+1./2', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z', '-x'],['-y+1./2', 'z+1./2', '-x+1./2'],['y', '-z', '-x'],['y+1./2', '-z+1./2', '-x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['-z', '-x', '-y'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z', 'x', 'y'],['-z+1./2', 'x+1./2', 'y+1./2'],['z', 'x', '-y'],['z+1./2', 'x+1./2', '-y+1./2'],['z', '-x', 'y'],['z+1./2', '-x+1./2', 'y+1./2'],['-y', '-z', '-x'],['-y+1./2', '-z+1./2', '-x+1./2'],['y', '-z', 'x'],['y+1./2', '-z+1./2', 'x+1./2'],['-y', 'z', 'x'],['-y+1./2', 'z+1./2', 'x+1./2'],['y', 'z', '-x'],['y+1./2', 'z+1./2', '-x+1./2']]},
              '205': {'4a': [[0., 0., 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[1./2, 1./2, 0]],
                      '4b': [[1./2, 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 0],[0., 0., 1./2]],
                      '8c': [['x', 'x', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['-x', '-x', '-x'],['x+1./2', 'x', '-x+1./2'],['x', '-x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', 'x']],
                      '24d': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z+1./2', '-x+1./2', '-y'],['-z+1./2', '-x', 'y+1./2'],['-z', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['-y', 'z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y+1./2', '-z', 'x+1./2'],['-x', '-y', '-z'],['x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-z', '-x', '-y'],['-z+1./2', 'x+1./2', 'y'],['z+1./2', 'x', '-y+1./2'],['z', '-x+1./2', 'y+1./2'],['-y', '-z', '-x'],['y', '-z+1./2', 'x+1./2'],['-y+1./2', 'z+1./2', 'x'],['y+1./2', 'z', '-x+1./2']]},
              '206': {'8a': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                      '8b': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4]],
                      '16c': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x', '-x+1./2', 'x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x+1./2'],['-x', '-x', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', 'x', '-x+1./2'],['x', 'x+1./2', '-x'],['x', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x'],['-x+1./2', 'x+1./2', 'x'],['-x', 'x', 'x+1./2']],
                      '24d': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],[1./4, 'x', 0],[3./4, 'x+1./2', 1./2],[3./4, '-x+1./2', 0],[1./4, '-x', 1./2],[0., 1./4, 'x'],[1./2, 3./4, 'x+1./2'],[0., 3./4, '-x+1./2'],[1./2, 1./4, '-x'],['-x', 0., 3./4],['-x+1./2', 1./2, 1./4],['x+1./2', 0., 1./4],['x', 1./2, 3./4],[3./4, '-x', 0],[1./4, '-x+1./2', 1./2],[1./4, 'x+1./2', 0],[3./4, 'x', 1./2],[0., 3./4, '-x'],[1./2, 1./4, '-x+1./2'],[0., 1./4, 'x+1./2'],[1./2, 3./4, 'x']],
                      '48e': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z', '-x+1./2', 'y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y', '-z+1./2', 'x'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z'],['-x+1./2', 'y+1./2', 'z'],['-x', 'y', 'z+1./2'],['-z', '-x', '-y'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z+1./2', 'x+1./2', 'y'],['-z', 'x', 'y+1./2'],['z+1./2', 'x', '-y+1./2'],['z', 'x+1./2', '-y'],['z', '-x+1./2', 'y+1./2'],['z+1./2', '-x', 'y'],['-y', '-z', '-x'],['-y+1./2', '-z+1./2', '-x+1./2'],['y', '-z+1./2', 'x+1./2'],['y+1./2', '-z', 'x'],['-y+1./2', 'z+1./2', 'x'],['-y', 'z', 'x+1./2'],['y+1./2', 'z', '-x+1./2'],['y', 'z+1./2', '-x']]},
              '207': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 1./2]],
                      '3c': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '6e': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x']],
                      '6f': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2],[1./2, 1./2, 'x'],[1./2, 1./2, '-x']],
                      '8g': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x', 'x', '-x'],['-x', '-x', '-x'],['x', '-x', 'x'],['-x', 'x', 'x']],
                      '12h': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],[1./2, 'x', 0],[1./2, '-x', 0],['x', 0., 1./2],['-x', 0., 1./2],[0., 1./2, '-x'],[0., 1./2, 'x']],
                      '12i': [[0., 'y', 'y'],[0., '-y', 'y'],[0., 'y', '-y'],[0., '-y', '-y'],['y', 0., 'y'],['y', 0., '-y'],['-y', 0., 'y'],['-y', 0., '-y'],['y', 'y', 0],['-y', 'y', 0],['y', '-y', 0],['-y', '-y', 0]],
                      '12j': [[1./2, 'y', 'y'],[1./2, '-y', 'y'],[1./2, 'y', '-y'],[1./2, '-y', '-y'],['y', 1./2, 'y'],['y', 1./2, '-y'],['-y', 1./2, 'y'],['-y', 1./2, '-y'],['y', 'y', 1./2],['-y', 'y', 1./2],['y', '-y', 1./2],['-y', '-y', 1./2]],
                      '24k': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y', 'x', '-z'],['-y', '-x', '-z'],['y', '-x', 'z'],['-y', 'x', 'z'],['x', 'z', '-y'],['-x', 'z', 'y'],['-x', '-z', '-y'],['x', '-z', 'y'],['z', 'y', '-x'],['z', '-y', 'x'],['-z', 'y', 'x'],['-z', '-y', '-x']]},
              '208': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '4b': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4]],
                      '4c': [[3./4, 3./4, 3./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                      '6d': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 0],[1./2, 0., 0],[0., 0., 1./2]],
                      '6e': [[1./4, 0., 1./2],[3./4, 0., 1./2],[1./2, 1./4, 0],[1./2, 3./4, 0],[0., 1./2, 1./4],[0., 1./2, 3./4]],
                      '6f': [[1./4, 1./2, 0],[3./4, 1./2, 0],[0., 1./4, 1./2],[0., 3./4, 1./2],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                      '8g': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', 'x+1./2']],
                      '12h': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x'],[1./2, 'x+1./2', 1./2],[1./2, '-x+1./2', 1./2],['x+1./2', 1./2, 1./2],['-x+1./2', 1./2, 1./2],[1./2, 1./2, '-x+1./2'],[1./2, 1./2, 'x+1./2']],
                      '12i': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x'],[1./2, 'x+1./2', 0],[1./2, '-x+1./2', 0],['x+1./2', 0., 1./2],['-x+1./2', 0., 1./2],[0., 1./2, '-x+1./2'],[0., 1./2, 'x+1./2']],
                      '12j': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],[0., 'x+1./2', 1./2],[0., '-x+1./2', 1./2],['x+1./2', 1./2, 0],['-x+1./2', 1./2, 0],[1./2, 0., '-x+1./2'],[1./2, 0., 'x+1./2']],
                      '12k': [[1./4, 'y', '-y+1./2'],[3./4, '-y', '-y+1./2'],[3./4, 'y', 'y+1./2'],[1./4, '-y', 'y+1./2'],['-y+1./2', 1./4, 'y'],['-y+1./2', 3./4, '-y'],['y+1./2', 3./4, 'y'],['y+1./2', 1./4, '-y'],['y', '-y+1./2', 1./4],['-y', '-y+1./2', 3./4],['y', 'y+1./2', 3./4],['-y', 'y+1./2', 1./4]],
                      '12l': [[1./4, 'y', 'y+1./2'],[3./4, '-y', 'y+1./2'],[3./4, 'y', '-y+1./2'],[1./4, '-y', '-y+1./2'],['y+1./2', 1./4, 'y'],['y+1./2', 3./4, '-y'],['-y+1./2', 3./4, 'y'],['-y+1./2', 1./4, '-y'],['y', 'y+1./2', 1./4],['-y', 'y+1./2', 3./4],['y', '-y+1./2', 3./4],['-y', '-y+1./2', 1./4]],
                      '24m': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-y+1./2', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', '-y+1./2'],['-x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', '-z+1./2', 'y+1./2'],['z+1./2', 'y+1./2', '-x+1./2'],['z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', 'y+1./2', 'x+1./2'],['-z+1./2', '-y+1./2', '-x+1./2']]},
              '209': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '4b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '8c': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                      '24d': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0]],
                      '24e': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x']],
                      '32f': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x', 'x', '-x'],['x', 'x+1./2', '-x+1./2'],['x+1./2', 'x', '-x+1./2'],['x+1./2', 'x+1./2', '-x'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x', '-x', 'x'],['x', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x+1./2'],['x+1./2', '-x+1./2', 'x'],['-x', 'x', 'x'],['-x', 'x+1./2', 'x+1./2'],['-x+1./2', 'x', 'x+1./2'],['-x+1./2', 'x+1./2', 'x']],
                      '48g': [[0., 'y', 'y'],[0., 'y+1./2', 'y+1./2'],[1./2, 'y', 'y+1./2'],[1./2, 'y+1./2', 'y'],[0., '-y', 'y'],[0., '-y+1./2', 'y+1./2'],[1./2, '-y', 'y+1./2'],[1./2, '-y+1./2', 'y'],[0., 'y', '-y'],[0., 'y+1./2', '-y+1./2'],[1./2, 'y', '-y+1./2'],[1./2, 'y+1./2', '-y'],[0., '-y', '-y'],[0., '-y+1./2', '-y+1./2'],[1./2, '-y', '-y+1./2'],[1./2, '-y+1./2', '-y'],['y', 0., 'y'],['y', 1./2, 'y+1./2'],['y+1./2', 0., 'y+1./2'],['y+1./2', 1./2, 'y'],['y', 0., '-y'],['y', 1./2, '-y+1./2'],['y+1./2', 0., '-y+1./2'],['y+1./2', 1./2, '-y'],['-y', 0., 'y'],['-y', 1./2, 'y+1./2'],['-y+1./2', 0., 'y+1./2'],['-y+1./2', 1./2, 'y'],['-y', 0., '-y'],['-y', 1./2, '-y+1./2'],['-y+1./2', 0., '-y+1./2'],['-y+1./2', 1./2, '-y'],['y', 'y', 0],['y', 'y+1./2', 1./2],['y+1./2', 'y', 1./2],['y+1./2', 'y+1./2', 0],['-y', 'y', 0],['-y', 'y+1./2', 1./2],['-y+1./2', 'y', 1./2],['-y+1./2', 'y+1./2', 0],['y', '-y', 0],['y', '-y+1./2', 1./2],['y+1./2', '-y', 1./2],['y+1./2', '-y+1./2', 0],['-y', '-y', 0],['-y', '-y+1./2', 1./2],['-y+1./2', '-y', 1./2],['-y+1./2', '-y+1./2', 0]],
                      '48h': [[1./2, 'y', 'y'],[1./2, 'y+1./2', 'y+1./2'],[0., 'y', 'y+1./2'],[0., 'y+1./2', 'y'],[1./2, '-y', 'y'],[1./2, '-y+1./2', 'y+1./2'],[0., '-y', 'y+1./2'],[0., '-y+1./2', 'y'],[1./2, 'y', '-y'],[1./2, 'y+1./2', '-y+1./2'],[0., 'y', '-y+1./2'],[0., 'y+1./2', '-y'],[1./2, '-y', '-y'],[1./2, '-y+1./2', '-y+1./2'],[0., '-y', '-y+1./2'],[0., '-y+1./2', '-y'],['y', 1./2, 'y'],['y', 0., 'y+1./2'],['y+1./2', 1./2, 'y+1./2'],['y+1./2', 0., 'y'],['y', 1./2, '-y'],['y', 0., '-y+1./2'],['y+1./2', 1./2, '-y+1./2'],['y+1./2', 0., '-y'],['-y', 1./2, 'y'],['-y', 0., 'y+1./2'],['-y+1./2', 1./2, 'y+1./2'],['-y+1./2', 0., 'y'],['-y', 1./2, '-y'],['-y', 0., '-y+1./2'],['-y+1./2', 1./2, '-y+1./2'],['-y+1./2', 0., '-y'],['y', 'y', 1./2],['y', 'y+1./2', 0],['y+1./2', 'y', 0],['y+1./2', 'y+1./2', 1./2],['-y', 'y', 1./2],['-y', 'y+1./2', 0],['-y+1./2', 'y', 0],['-y+1./2', 'y+1./2', 1./2],['y', '-y', 1./2],['y', '-y+1./2', 0],['y+1./2', '-y', 0],['y+1./2', '-y+1./2', 1./2],['-y', '-y', 1./2],['-y', '-y+1./2', 0],['-y+1./2', '-y', 0],['-y+1./2', '-y+1./2', 1./2]],
                      '48i': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x'],[1./4, 'x', 3./4],[1./4, 'x+1./2', 1./4],[3./4, 'x', 1./4],[3./4, 'x+1./2', 3./4],[3./4, '-x', 3./4],[3./4, '-x+1./2', 1./4],[1./4, '-x', 1./4],[1./4, '-x+1./2', 3./4],['x', 1./4, 3./4],['x', 3./4, 1./4],['x+1./2', 1./4, 1./4],['x+1./2', 3./4, 3./4],['-x', 1./4, 1./4],['-x', 3./4, 3./4],['-x+1./2', 1./4, 3./4],['-x+1./2', 3./4, 1./4],[1./4, 1./4, '-x'],[1./4, 3./4, '-x+1./2'],[3./4, 1./4, '-x+1./2'],[3./4, 3./4, '-x'],[1./4, 3./4, 'x'],[1./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x']],
                      '96j': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y', 'x', '-z'],['y', 'x+1./2', '-z+1./2'],['y+1./2', 'x', '-z+1./2'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z'],['-y', '-x+1./2', '-z+1./2'],['-y+1./2', '-x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z'],['y', '-x', 'z'],['y', '-x+1./2', 'z+1./2'],['y+1./2', '-x', 'z+1./2'],['y+1./2', '-x+1./2', 'z'],['-y', 'x', 'z'],['-y', 'x+1./2', 'z+1./2'],['-y+1./2', 'x', 'z+1./2'],['-y+1./2', 'x+1./2', 'z'],['x', 'z', '-y'],['x', 'z+1./2', '-y+1./2'],['x+1./2', 'z', '-y+1./2'],['x+1./2', 'z+1./2', '-y'],['-x', 'z', 'y'],['-x', 'z+1./2', 'y+1./2'],['-x+1./2', 'z', 'y+1./2'],['-x+1./2', 'z+1./2', 'y'],['-x', '-z', '-y'],['-x', '-z+1./2', '-y+1./2'],['-x+1./2', '-z', '-y+1./2'],['-x+1./2', '-z+1./2', '-y'],['x', '-z', 'y'],['x', '-z+1./2', 'y+1./2'],['x+1./2', '-z', 'y+1./2'],['x+1./2', '-z+1./2', 'y'],['z', 'y', '-x'],['z', 'y+1./2', '-x+1./2'],['z+1./2', 'y', '-x+1./2'],['z+1./2', 'y+1./2', '-x'],['z', '-y', 'x'],['z', '-y+1./2', 'x+1./2'],['z+1./2', '-y', 'x+1./2'],['z+1./2', '-y+1./2', 'x'],['-z', 'y', 'x'],['-z', 'y+1./2', 'x+1./2'],['-z+1./2', 'y', 'x+1./2'],['-z+1./2', 'y+1./2', 'x'],['-z', '-y', '-x'],['-z', '-y+1./2', '-x+1./2'],['-z+1./2', '-y', '-x+1./2'],['-z+1./2', '-y+1./2', '-x']]},
              '210': {'8a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                      '8b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                      '16c': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 3/8, 5/8],[7/8, 7/8, 1/8],[3/8, 3/8, 1/8],[3/8, 7/8, 5/8],[3/8, 5/8, 7/8],[3/8, 1/8, 3/8],[7/8, 5/8, 3/8],[7/8, 1/8, 7/8],[5/8, 7/8, 3/8],[5/8, 3/8, 7/8],[1/8, 7/8, 7/8],[1/8, 3/8, 3/8]],
                      '16d': [[5/8, 5/8, 5/8],[5/8, 1/8, 1/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[3/8, 7/8, 1/8],[3/8, 3/8, 5/8],[7/8, 7/8, 5/8],[7/8, 3/8, 1/8],[7/8, 1/8, 3/8],[7/8, 5/8, 7/8],[3/8, 1/8, 7/8],[3/8, 5/8, 3/8],[1/8, 3/8, 7/8],[1/8, 7/8, 3/8],[5/8, 3/8, 3/8],[5/8, 7/8, 7/8]],
                      '32e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', 'x+1./2', '-x'],['-x+1./2', 'x', '-x+1./2'],['-x', 'x+1./2', '-x+1./2'],['-x', 'x', '-x'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+3./4', 'x+1./4', '-x+3./4'],['x+3./4', 'x+3./4', '-x+1./4'],['x+1./4', 'x+1./4', '-x+1./4'],['x+1./4', 'x+3./4', '-x+3./4'],['-x+1./4', '-x+1./4', '-x+1./4'],['-x+1./4', '-x+3./4', '-x+3./4'],['-x+3./4', '-x+1./4', '-x+3./4'],['-x+3./4', '-x+3./4', '-x+1./4'],['x+1./4', '-x+3./4', 'x+3./4'],['x+1./4', '-x+1./4', 'x+1./4'],['x+3./4', '-x+3./4', 'x+1./4'],['x+3./4', '-x+1./4', 'x+3./4'],['-x+3./4', 'x+3./4', 'x+1./4'],['-x+3./4', 'x+1./4', 'x+3./4'],['-x+1./4', 'x+3./4', 'x+3./4'],['-x+1./4', 'x+1./4', 'x+1./4']],
                      '48f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 0],['-x+1./2', 0., 1./2],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[1./2, 1./2, '-x'],[1./2, 0., '-x+1./2'],[0., 1./2, '-x+1./2'],[0., 0., '-x'],[3./4, 'x+1./4', 3./4],[3./4, 'x+3./4', 1./4],[1./4, 'x+1./4', 1./4],[1./4, 'x+3./4', 3./4],[1./4, '-x+1./4', 1./4],[1./4, '-x+3./4', 3./4],[3./4, '-x+1./4', 3./4],[3./4, '-x+3./4', 1./4],['x+3./4', 1./4, 3./4],['x+3./4', 3./4, 1./4],['x+1./4', 1./4, 1./4],['x+1./4', 3./4, 3./4],['-x+3./4', 3./4, 1./4],['-x+3./4', 1./4, 3./4],['-x+1./4', 3./4, 3./4],['-x+1./4', 1./4, 1./4],[3./4, 1./4, '-x+3./4'],[3./4, 3./4, '-x+1./4'],[1./4, 1./4, '-x+1./4'],[1./4, 3./4, '-x+3./4'],[1./4, 3./4, 'x+3./4'],[1./4, 1./4, 'x+1./4'],[3./4, 3./4, 'x+1./4'],[3./4, 1./4, 'x+3./4']],
                      '48g': [[1/8, 'y', '-y+1./4'],[1/8, 'y+1./2', '-y+3./4'],[5/8, 'y', '-y+3./4'],[5/8, 'y+1./2', '-y+1./4'],[7/8, '-y+1./2', '-y+3./4'],[7/8, '-y', '-y+1./4'],[3/8, '-y+1./2', '-y+1./4'],[3/8, '-y', '-y+3./4'],[3/8, 'y+1./2', 'y+3./4'],[3/8, 'y', 'y+1./4'],[7/8, 'y+1./2', 'y+1./4'],[7/8, 'y', 'y+3./4'],[5/8, '-y', 'y+1./4'],[5/8, '-y+1./2', 'y+3./4'],[1/8, '-y', 'y+3./4'],[1/8, '-y+1./2', 'y+1./4'],['-y+1./4', 1/8, 'y'],['-y+1./4', 5/8, 'y+1./2'],['-y+3./4', 1/8, 'y+1./2'],['-y+3./4', 5/8, 'y'],['-y+3./4', 7/8, '-y+1./2'],['-y+3./4', 3/8, '-y'],['-y+1./4', 7/8, '-y'],['-y+1./4', 3/8, '-y+1./2'],['y+3./4', 3/8, 'y+1./2'],['y+3./4', 7/8, 'y'],['y+1./4', 3/8, 'y'],['y+1./4', 7/8, 'y+1./2'],['y+1./4', 5/8, '-y'],['y+1./4', 1/8, '-y+1./2'],['y+3./4', 5/8, '-y+1./2'],['y+3./4', 1/8, '-y'],['y', '-y+1./4', 1/8],['y', '-y+3./4', 5/8],['y+1./2', '-y+1./4', 5/8],['y+1./2', '-y+3./4', 1/8],['-y+1./2', '-y+3./4', 7/8],['-y+1./2', '-y+1./4', 3/8],['-y', '-y+3./4', 3/8],['-y', '-y+1./4', 7/8],['y+1./2', 'y+3./4', 3/8],['y+1./2', 'y+1./4', 7/8],['y', 'y+3./4', 7/8],['y', 'y+1./4', 3/8],['-y', 'y+1./4', 5/8],['-y', 'y+3./4', 1/8],['-y+1./2', 'y+1./4', 1/8],['-y+1./2', 'y+3./4', 5/8]],
                      '96h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', '-z'],['-x+1./2', 'y', '-z+1./2'],['-x', 'y+1./2', '-z+1./2'],['-x', 'y', '-z'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['-z', '-x+1./2', 'y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', 'x+1./2', '-y'],['-z+1./2', 'x', '-y+1./2'],['-z', 'x+1./2', '-y+1./2'],['-z', 'x', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y+1./2', 'z+1./2', '-x'],['-y+1./2', 'z', '-x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y', 'z', '-x'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['-y', '-z+1./2', 'x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x'],['-y+1./2', '-z', 'x+1./2'],['y+3./4', 'x+1./4', '-z+3./4'],['y+3./4', 'x+3./4', '-z+1./4'],['y+1./4', 'x+1./4', '-z+1./4'],['y+1./4', 'x+3./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['-y+1./4', '-x+3./4', '-z+3./4'],['-y+3./4', '-x+1./4', '-z+3./4'],['-y+3./4', '-x+3./4', '-z+1./4'],['y+1./4', '-x+3./4', 'z+3./4'],['y+1./4', '-x+1./4', 'z+1./4'],['y+3./4', '-x+3./4', 'z+1./4'],['y+3./4', '-x+1./4', 'z+3./4'],['-y+3./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['-y+1./4', 'x+3./4', 'z+3./4'],['-y+1./4', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-y+3./4'],['x+3./4', 'z+3./4', '-y+1./4'],['x+1./4', 'z+1./4', '-y+1./4'],['x+1./4', 'z+3./4', '-y+3./4'],['-x+3./4', 'z+3./4', 'y+1./4'],['-x+3./4', 'z+1./4', 'y+3./4'],['-x+1./4', 'z+3./4', 'y+3./4'],['-x+1./4', 'z+1./4', 'y+1./4'],['-x+1./4', '-z+1./4', '-y+1./4'],['-x+1./4', '-z+3./4', '-y+3./4'],['-x+3./4', '-z+1./4', '-y+3./4'],['-x+3./4', '-z+3./4', '-y+1./4'],['x+1./4', '-z+3./4', 'y+3./4'],['x+1./4', '-z+1./4', 'y+1./4'],['x+3./4', '-z+3./4', 'y+1./4'],['x+3./4', '-z+1./4', 'y+3./4'],['z+3./4', 'y+1./4', '-x+3./4'],['z+3./4', 'y+3./4', '-x+1./4'],['z+1./4', 'y+1./4', '-x+1./4'],['z+1./4', 'y+3./4', '-x+3./4'],['z+1./4', '-y+3./4', 'x+3./4'],['z+1./4', '-y+1./4', 'x+1./4'],['z+3./4', '-y+3./4', 'x+1./4'],['z+3./4', '-y+1./4', 'x+3./4'],['-z+3./4', 'y+3./4', 'x+1./4'],['-z+3./4', 'y+1./4', 'x+3./4'],['-z+1./4', 'y+3./4', 'x+3./4'],['-z+1./4', 'y+1./4', 'x+1./4'],['-z+1./4', '-y+1./4', '-x+1./4'],['-z+1./4', '-y+3./4', '-x+3./4'],['-z+3./4', '-y+1./4', '-x+3./4'],['-z+3./4', '-y+3./4', '-x+1./4']]},
              '211': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 0., 1./2],[0., 1./2, 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                      '8c': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                      '12d': [[1./4, 1./2, 0],[3./4, 0., 1./2],[3./4, 1./2, 0],[1./4, 0., 1./2],[0., 1./4, 1./2],[1./2, 3./4, 0],[0., 3./4, 1./2],[1./2, 1./4, 0],[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4],[0., 1./2, 1./4]],
                      '12e': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2],[0., 0., 'x'],[1./2, 1./2, 'x+1./2'],[0., 0., '-x'],[1./2, 1./2, '-x+1./2']],
                      '16f': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', '-x'],['-x+1./2', 'x+1./2', '-x+1./2'],['x', '-x', '-x'],['x+1./2', '-x+1./2', '-x+1./2'],['x', 'x', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['-x', '-x', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x', '-x', 'x'],['x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', 'x'],['-x+1./2', 'x+1./2', 'x+1./2']],
                      '24g': [['x', 1./2, 0],['x+1./2', 0., 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2],[0., 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 1./2],[1./2, '-x+1./2', 0],[1./2, 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 'x', 0],[0., 'x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 1./2],['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0],[0., 1./2, '-x'],[1./2, 0., '-x+1./2'],[0., 1./2, 'x'],[1./2, 0., 'x+1./2']],
                      '24h': [[0., 'y', 'y'],[1./2, 'y+1./2', 'y+1./2'],[0., '-y', 'y'],[1./2, '-y+1./2', 'y+1./2'],[0., 'y', '-y'],[1./2, 'y+1./2', '-y+1./2'],[0., '-y', '-y'],[1./2, '-y+1./2', '-y+1./2'],['y', 0., 'y'],['y+1./2', 1./2, 'y+1./2'],['y', 0., '-y'],['y+1./2', 1./2, '-y+1./2'],['-y', 0., 'y'],['-y+1./2', 1./2, 'y+1./2'],['-y', 0., '-y'],['-y+1./2', 1./2, '-y+1./2'],['y', 'y', 0],['y+1./2', 'y+1./2', 1./2],['-y', 'y', 0],['-y+1./2', 'y+1./2', 1./2],['y', '-y', 0],['y+1./2', '-y+1./2', 1./2],['-y', '-y', 0],['-y+1./2', '-y+1./2', 1./2]],
                      '24i': [[1./4, 'y', '-y+1./2'],[3./4, 'y+1./2', '-y'],[3./4, '-y', '-y+1./2'],[1./4, '-y+1./2', '-y'],[3./4, 'y', 'y+1./2'],[1./4, 'y+1./2', 'y'],[1./4, '-y', 'y+1./2'],[3./4, '-y+1./2', 'y'],['-y+1./2', 1./4, 'y'],['-y', 3./4, 'y+1./2'],['-y+1./2', 3./4, '-y'],['-y', 1./4, '-y+1./2'],['y+1./2', 3./4, 'y'],['y', 1./4, 'y+1./2'],['y+1./2', 1./4, '-y'],['y', 3./4, '-y+1./2'],['y', '-y+1./2', 1./4],['y+1./2', '-y', 3./4],['-y', '-y+1./2', 3./4],['-y+1./2', '-y', 1./4],['y', 'y+1./2', 3./4],['y+1./2', 'y', 1./4],['-y', 'y+1./2', 1./4],['-y+1./2', 'y', 3./4]],
                      '48j': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z', '-x', '-y'],['z+1./2', '-x+1./2', '-y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y+1./2'],['-z', 'x', '-y'],['-z+1./2', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z', '-x'],['-y+1./2', 'z+1./2', '-x+1./2'],['y', '-z', '-x'],['y+1./2', '-z+1./2', '-x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x+1./2'],['y', 'x', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['x', 'z', '-y'],['x+1./2', 'z+1./2', '-y+1./2'],['-x', 'z', 'y'],['-x+1./2', 'z+1./2', 'y+1./2'],['-x', '-z', '-y'],['-x+1./2', '-z+1./2', '-y+1./2'],['x', '-z', 'y'],['x+1./2', '-z+1./2', 'y+1./2'],['z', 'y', '-x'],['z+1./2', 'y+1./2', '-x+1./2'],['z', '-y', 'x'],['z+1./2', '-y+1./2', 'x+1./2'],['-z', 'y', 'x'],['-z+1./2', 'y+1./2', 'x+1./2'],['-z', '-y', '-x'],['-z+1./2', '-y+1./2', '-x+1./2']]},
              '212': {'4a': [[1/8, 1/8, 1/8],[3/8, 7/8, 5/8],[7/8, 5/8, 3/8],[5/8, 3/8, 7/8]],
                      '4b': [[5/8, 5/8, 5/8],[7/8, 3/8, 1/8],[3/8, 1/8, 7/8],[1/8, 7/8, 3/8]],
                      '8c': [['x', 'x', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x+1./4', 'x+3./4', '-x+3./4'],['-x+1./4', '-x+1./4', '-x+1./4'],['x+3./4', '-x+3./4', 'x+1./4'],['-x+3./4', 'x+1./4', 'x+3./4']],
                      '12d': [[1/8, 'y', '-y+1./4'],[3/8, '-y', '-y+3./4'],[7/8, 'y+1./2', 'y+1./4'],[5/8, '-y+1./2', 'y+3./4'],['-y+1./4', 1/8, 'y'],['-y+3./4', 3/8, '-y'],['y+1./4', 7/8, 'y+1./2'],['y+3./4', 5/8, '-y+1./2'],['y', '-y+1./4', 1/8],['-y', '-y+3./4', 3/8],['y+1./2', 'y+1./4', 7/8],['-y+1./2', 'y+3./4', 5/8]],
                      '24e': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z+1./2', '-x+1./2', '-y'],['-z+1./2', '-x', 'y+1./2'],['-z', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['-y', 'z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y+1./2', '-z', 'x+1./2'],['y+1./4', 'x+3./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['y+3./4', '-x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['x+1./4', 'z+3./4', '-y+3./4'],['-x+3./4', 'z+1./4', 'y+3./4'],['-x+1./4', '-z+1./4', '-y+1./4'],['x+3./4', '-z+3./4', 'y+1./4'],['z+1./4', 'y+3./4', '-x+3./4'],['z+3./4', '-y+3./4', 'x+1./4'],['-z+3./4', 'y+1./4', 'x+3./4'],['-z+1./4', '-y+1./4', '-x+1./4']]},
              '213': {'4a': [[3/8, 3/8, 3/8],[1/8, 5/8, 7/8],[5/8, 7/8, 1/8],[7/8, 1/8, 5/8]],
                      '4b': [[7/8, 7/8, 7/8],[5/8, 1/8, 3/8],[1/8, 3/8, 5/8],[3/8, 5/8, 1/8]],
                      '8c': [['x', 'x', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x+3./4', 'x+1./4', '-x+1./4'],['-x+3./4', '-x+3./4', '-x+3./4'],['x+1./4', '-x+1./4', 'x+3./4'],['-x+1./4', 'x+3./4', 'x+1./4']],
                      '12d': [[1/8, 'y', 'y+1./4'],[3/8, '-y', 'y+3./4'],[7/8, 'y+1./2', '-y+1./4'],[5/8, '-y+1./2', '-y+3./4'],['y+1./4', 1/8, 'y'],['y+3./4', 3/8, '-y'],['-y+1./4', 7/8, 'y+1./2'],['-y+3./4', 5/8, '-y+1./2'],['y', 'y+1./4', 1/8],['-y', 'y+3./4', 3/8],['y+1./2', '-y+1./4', 7/8],['-y+1./2', '-y+3./4', 5/8]],
                      '24e': [['x', 'y', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z+1./2', '-x+1./2', '-y'],['-z+1./2', '-x', 'y+1./2'],['-z', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['-y', 'z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y+1./2', '-z', 'x+1./2'],['y+3./4', 'x+1./4', '-z+1./4'],['-y+3./4', '-x+3./4', '-z+3./4'],['y+1./4', '-x+1./4', 'z+3./4'],['-y+1./4', 'x+3./4', 'z+1./4'],['x+3./4', 'z+1./4', '-y+1./4'],['-x+1./4', 'z+3./4', 'y+1./4'],['-x+3./4', '-z+3./4', '-y+3./4'],['x+1./4', '-z+1./4', 'y+3./4'],['z+3./4', 'y+1./4', '-x+1./4'],['z+1./4', '-y+1./4', 'x+3./4'],['-z+1./4', 'y+3./4', 'x+1./4'],['-z+3./4', '-y+3./4', '-x+3./4']]},
              '214': {'8a': [[1/8, 1/8, 1/8],[5/8, 5/8, 5/8],[3/8, 7/8, 5/8],[7/8, 3/8, 1/8],[7/8, 5/8, 3/8],[3/8, 1/8, 7/8],[5/8, 3/8, 7/8],[1/8, 7/8, 3/8]],
                      '8b': [[7/8, 7/8, 7/8],[3/8, 3/8, 3/8],[5/8, 1/8, 3/8],[1/8, 5/8, 7/8],[1/8, 3/8, 5/8],[5/8, 7/8, 1/8],[3/8, 5/8, 1/8],[7/8, 1/8, 5/8]],
                      '12c': [[1/8, 0., 1./4],[5/8, 1./2, 3./4],[3/8, 0., 3./4],[7/8, 1./2, 1./4],[1./4, 1/8, 0],[3./4, 5/8, 1./2],[3./4, 3/8, 0],[1./4, 7/8, 1./2],[0., 1./4, 1/8],[1./2, 3./4, 5/8],[0., 3./4, 3/8],[1./2, 1./4, 7/8]],
                      '12d': [[5/8, 0., 1./4],[1/8, 1./2, 3./4],[7/8, 0., 3./4],[3/8, 1./2, 1./4],[1./4, 5/8, 0],[3./4, 1/8, 1./2],[3./4, 7/8, 0],[1./4, 3/8, 1./2],[0., 1./4, 5/8],[1./2, 3./4, 1/8],[0., 3./4, 7/8],[1./2, 1./4, 3/8]],
                      '16e': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x', '-x+1./2', 'x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x+1./2'],['x+3./4', 'x+1./4', '-x+1./4'],['x+1./4', 'x+3./4', '-x+3./4'],['-x+3./4', '-x+3./4', '-x+3./4'],['-x+1./4', '-x+1./4', '-x+1./4'],['x+1./4', '-x+1./4', 'x+3./4'],['x+3./4', '-x+3./4', 'x+1./4'],['-x+1./4', 'x+3./4', 'x+1./4'],['-x+3./4', 'x+1./4', 'x+3./4']],
                      '24f': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],[1./4, 'x', 0],[3./4, 'x+1./2', 1./2],[3./4, '-x+1./2', 0],[1./4, '-x', 1./2],[0., 1./4, 'x'],[1./2, 3./4, 'x+1./2'],[0., 3./4, '-x+1./2'],[1./2, 1./4, '-x'],[3./4, 'x+1./4', 0],[1./4, 'x+3./4', 1./2],[3./4, '-x+3./4', 1./2],[1./4, '-x+1./4', 0],['x+3./4', 1./2, 1./4],['x+1./4', 0., 3./4],['-x+1./4', 0., 1./4],['-x+3./4', 1./2, 3./4],[0., 1./4, '-x+1./4'],[1./2, 3./4, '-x+3./4'],[1./2, 1./4, 'x+3./4'],[0., 3./4, 'x+1./4']],
                      '24g': [[1/8, 'y', 'y+1./4'],[5/8, 'y+1./2', 'y+3./4'],[3/8, '-y', 'y+3./4'],[7/8, '-y+1./2', 'y+1./4'],[7/8, 'y+1./2', '-y+1./4'],[3/8, 'y', '-y+3./4'],[5/8, '-y+1./2', '-y+3./4'],[1/8, '-y', '-y+1./4'],['y+1./4', 1/8, 'y'],['y+3./4', 5/8, 'y+1./2'],['y+3./4', 3/8, '-y'],['y+1./4', 7/8, '-y+1./2'],['-y+1./4', 7/8, 'y+1./2'],['-y+3./4', 3/8, 'y'],['-y+3./4', 5/8, '-y+1./2'],['-y+1./4', 1/8, '-y'],['y', 'y+1./4', 1/8],['y+1./2', 'y+3./4', 5/8],['-y', 'y+3./4', 3/8],['-y+1./2', 'y+1./4', 7/8],['y+1./2', '-y+1./4', 7/8],['y', '-y+3./4', 3/8],['-y+1./2', '-y+3./4', 5/8],['-y', '-y+1./4', 1/8]],
                      '24h': [[1/8, 'y', '-y+1./4'],[5/8, 'y+1./2', '-y+3./4'],[3/8, '-y', '-y+3./4'],[7/8, '-y+1./2', '-y+1./4'],[7/8, 'y+1./2', 'y+1./4'],[3/8, 'y', 'y+3./4'],[5/8, '-y+1./2', 'y+3./4'],[1/8, '-y', 'y+1./4'],['-y+1./4', 1/8, 'y'],['-y+3./4', 5/8, 'y+1./2'],['-y+3./4', 3/8, '-y'],['-y+1./4', 7/8, '-y+1./2'],['y+1./4', 7/8, 'y+1./2'],['y+3./4', 3/8, 'y'],['y+3./4', 5/8, '-y+1./2'],['y+1./4', 1/8, '-y'],['y', '-y+1./4', 1/8],['y+1./2', '-y+3./4', 5/8],['-y', '-y+3./4', 3/8],['-y+1./2', '-y+1./4', 7/8],['y+1./2', 'y+1./4', 7/8],['y', 'y+3./4', 3/8],['-y+1./2', 'y+3./4', 5/8],['-y', 'y+1./4', 1/8]],
                      '48i': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z', '-x+1./2', 'y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y', '-z+1./2', 'x'],['y+3./4', 'x+1./4', '-z+1./4'],['y+1./4', 'x+3./4', '-z+3./4'],['-y+3./4', '-x+3./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['y+1./4', '-x+1./4', 'z+3./4'],['y+3./4', '-x+3./4', 'z+1./4'],['-y+1./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['x+3./4', 'z+1./4', '-y+1./4'],['x+1./4', 'z+3./4', '-y+3./4'],['-x+1./4', 'z+3./4', 'y+1./4'],['-x+3./4', 'z+1./4', 'y+3./4'],['-x+3./4', '-z+3./4', '-y+3./4'],['-x+1./4', '-z+1./4', '-y+1./4'],['x+1./4', '-z+1./4', 'y+3./4'],['x+3./4', '-z+3./4', 'y+1./4'],['z+3./4', 'y+1./4', '-x+1./4'],['z+1./4', 'y+3./4', '-x+3./4'],['z+1./4', '-y+1./4', 'x+3./4'],['z+3./4', '-y+3./4', 'x+1./4'],['-z+1./4', 'y+3./4', 'x+1./4'],['-z+3./4', 'y+1./4', 'x+3./4'],['-z+3./4', '-y+3./4', '-x+3./4'],['-z+1./4', '-y+1./4', '-x+1./4']]},
              '215': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 1./2]],
                      '3c': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '4e': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x']],
                      '6f': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x']],
                      '6g': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2],[1./2, 1./2, 'x'],[1./2, 1./2, '-x']],
                      '12h': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],[1./2, 'x', 0],[1./2, '-x', 0],['x', 0., 1./2],['-x', 0., 1./2],[0., 1./2, 'x'],[0., 1./2, '-x']],
                      '12i': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', '-z'],['x', '-x', '-z'],['z', 'x', 'x'],['z', '-x', '-x'],['-z', '-x', 'x'],['-z', 'x', '-x'],['x', 'z', 'x'],['-x', 'z', '-x'],['x', '-z', '-x'],['-x', '-z', 'x']],
                      '24j': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y', 'x', 'z'],['-y', '-x', 'z'],['y', '-x', '-z'],['-y', 'x', '-z'],['x', 'z', 'y'],['-x', 'z', '-y'],['-x', '-z', 'y'],['x', '-z', '-y'],['z', 'y', 'x'],['z', '-y', '-x'],['-z', 'y', '-x'],['-z', '-y', 'x']]},
              '216': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '4b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '4c': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                      '4d': [[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                      '16e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x']],
                      '24f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x']],
                      '24g': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x']],
                      '48h': [['x', 'x', 'z'],['x', 'x+1./2', 'z+1./2'],['x+1./2', 'x', 'z+1./2'],['x+1./2', 'x+1./2', 'z'],['-x', '-x', 'z'],['-x', '-x+1./2', 'z+1./2'],['-x+1./2', '-x', 'z+1./2'],['-x+1./2', '-x+1./2', 'z'],['-x', 'x', '-z'],['-x', 'x+1./2', '-z+1./2'],['-x+1./2', 'x', '-z+1./2'],['-x+1./2', 'x+1./2', '-z'],['x', '-x', '-z'],['x', '-x+1./2', '-z+1./2'],['x+1./2', '-x', '-z+1./2'],['x+1./2', '-x+1./2', '-z'],['z', 'x', 'x'],['z', 'x+1./2', 'x+1./2'],['z+1./2', 'x', 'x+1./2'],['z+1./2', 'x+1./2', 'x'],['z', '-x', '-x'],['z', '-x+1./2', '-x+1./2'],['z+1./2', '-x', '-x+1./2'],['z+1./2', '-x+1./2', '-x'],['-z', '-x', 'x'],['-z', '-x+1./2', 'x+1./2'],['-z+1./2', '-x', 'x+1./2'],['-z+1./2', '-x+1./2', 'x'],['-z', 'x', '-x'],['-z', 'x+1./2', '-x+1./2'],['-z+1./2', 'x', '-x+1./2'],['-z+1./2', 'x+1./2', '-x'],['x', 'z', 'x'],['x', 'z+1./2', 'x+1./2'],['x+1./2', 'z', 'x+1./2'],['x+1./2', 'z+1./2', 'x'],['-x', 'z', '-x'],['-x', 'z+1./2', '-x+1./2'],['-x+1./2', 'z', '-x+1./2'],['-x+1./2', 'z+1./2', '-x'],['x', '-z', '-x'],['x', '-z+1./2', '-x+1./2'],['x+1./2', '-z', '-x+1./2'],['x+1./2', '-z+1./2', '-x'],['-x', '-z', 'x'],['-x', '-z+1./2', 'x+1./2'],['-x+1./2', '-z', 'x+1./2'],['-x+1./2', '-z+1./2', 'x']],
                      '96i': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y', 'x', 'z'],['y', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z+1./2'],['y+1./2', 'x+1./2', 'z'],['-y', '-x', 'z'],['-y', '-x+1./2', 'z+1./2'],['-y+1./2', '-x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['y', '-x', '-z'],['y', '-x+1./2', '-z+1./2'],['y+1./2', '-x', '-z+1./2'],['y+1./2', '-x+1./2', '-z'],['-y', 'x', '-z'],['-y', 'x+1./2', '-z+1./2'],['-y+1./2', 'x', '-z+1./2'],['-y+1./2', 'x+1./2', '-z'],['x', 'z', 'y'],['x', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y+1./2'],['x+1./2', 'z+1./2', 'y'],['-x', 'z', '-y'],['-x', 'z+1./2', '-y+1./2'],['-x+1./2', 'z', '-y+1./2'],['-x+1./2', 'z+1./2', '-y'],['-x', '-z', 'y'],['-x', '-z+1./2', 'y+1./2'],['-x+1./2', '-z', 'y+1./2'],['-x+1./2', '-z+1./2', 'y'],['x', '-z', '-y'],['x', '-z+1./2', '-y+1./2'],['x+1./2', '-z', '-y+1./2'],['x+1./2', '-z+1./2', '-y'],['z', 'y', 'x'],['z', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x+1./2'],['z+1./2', 'y+1./2', 'x'],['z', '-y', '-x'],['z', '-y+1./2', '-x+1./2'],['z+1./2', '-y', '-x+1./2'],['z+1./2', '-y+1./2', '-x'],['-z', 'y', '-x'],['-z', 'y+1./2', '-x+1./2'],['-z+1./2', 'y', '-x+1./2'],['-z+1./2', 'y+1./2', '-x'],['-z', '-y', 'x'],['-z', '-y+1./2', 'x+1./2'],['-z+1./2', '-y', 'x+1./2'],['-z+1./2', '-y+1./2', 'x']]},
              '217': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 0., 1./2],[0., 1./2, 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                      '8c': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', '-x'],['-x+1./2', 'x+1./2', '-x+1./2'],['x', '-x', '-x'],['x+1./2', '-x+1./2', '-x+1./2']],
                      '12d': [[1./4, 1./2, 0],[3./4, 0., 1./2],[3./4, 1./2, 0],[1./4, 0., 1./2],[0., 1./4, 1./2],[1./2, 3./4, 0],[0., 3./4, 1./2],[1./2, 1./4, 0],[1./2, 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4],[0., 1./2, 1./4]],
                      '12e': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2],[0., 0., 'x'],[1./2, 1./2, 'x+1./2'],[0., 0., '-x'],[1./2, 1./2, '-x+1./2']],
                      '24f': [['x', 1./2, 0],['x+1./2', 0., 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2],[0., 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 1./2],[1./2, '-x+1./2', 0],[1./2, 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 'x', 0],[0., 'x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 1./2],['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0],[0., 1./2, 'x'],[1./2, 0., 'x+1./2'],[0., 1./2, '-x'],[1./2, 0., '-x+1./2']],
                      '24g': [['x', 'x', 'z'],['x+1./2', 'x+1./2', 'z+1./2'],['-x', '-x', 'z'],['-x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', '-z'],['-x+1./2', 'x+1./2', '-z+1./2'],['x', '-x', '-z'],['x+1./2', '-x+1./2', '-z+1./2'],['z', 'x', 'x'],['z+1./2', 'x+1./2', 'x+1./2'],['z', '-x', '-x'],['z+1./2', '-x+1./2', '-x+1./2'],['-z', '-x', 'x'],['-z+1./2', '-x+1./2', 'x+1./2'],['-z', 'x', '-x'],['-z+1./2', 'x+1./2', '-x+1./2'],['x', 'z', 'x'],['x+1./2', 'z+1./2', 'x+1./2'],['-x', 'z', '-x'],['-x+1./2', 'z+1./2', '-x+1./2'],['x', '-z', '-x'],['x+1./2', '-z+1./2', '-x+1./2'],['-x', '-z', 'x'],['-x+1./2', '-z+1./2', 'x+1./2']],
                      '48h': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z', '-x', '-y'],['z+1./2', '-x+1./2', '-y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y+1./2'],['-z', 'x', '-y'],['-z+1./2', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z', '-x'],['-y+1./2', 'z+1./2', '-x+1./2'],['y', '-z', '-x'],['y+1./2', '-z+1./2', '-x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x+1./2'],['y', 'x', 'z'],['y+1./2', 'x+1./2', 'z+1./2'],['-y', '-x', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['x', 'z', 'y'],['x+1./2', 'z+1./2', 'y+1./2'],['-x', 'z', '-y'],['-x+1./2', 'z+1./2', '-y+1./2'],['-x', '-z', 'y'],['-x+1./2', '-z+1./2', 'y+1./2'],['x', '-z', '-y'],['x+1./2', '-z+1./2', '-y+1./2'],['z', 'y', 'x'],['z+1./2', 'y+1./2', 'x+1./2'],['z', '-y', '-x'],['z+1./2', '-y+1./2', '-x+1./2'],['-z', 'y', '-x'],['-z+1./2', 'y+1./2', '-x+1./2'],['-z', '-y', 'x'],['-z+1./2', '-y+1./2', 'x+1./2']]},
              '218': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 0],[1./2, 0., 0],[0., 0., 1./2]],
                      '6c': [[1./4, 1./2, 0],[3./4, 1./2, 0],[0., 1./4, 1./2],[0., 3./4, 1./2],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                      '6d': [[1./4, 0., 1./2],[3./4, 0., 1./2],[1./2, 1./4, 0],[1./2, 3./4, 0],[0., 1./2, 1./4],[0., 1./2, 3./4]],
                      '8e': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x+1./2', 'x+1./2'],['x+1./2', '-x+1./2', '-x+1./2'],['-x+1./2', 'x+1./2', '-x+1./2']],
                      '12f': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x'],[1./2, 'x+1./2', 1./2],[1./2, '-x+1./2', 1./2],['x+1./2', 1./2, 1./2],['-x+1./2', 1./2, 1./2],[1./2, 1./2, 'x+1./2'],[1./2, 1./2, '-x+1./2']],
                      '12g': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],[0., 'x+1./2', 1./2],[0., '-x+1./2', 1./2],['x+1./2', 1./2, 0],['-x+1./2', 1./2, 0],[1./2, 0., 'x+1./2'],[1./2, 0., '-x+1./2']],
                      '12h': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x'],[1./2, 'x+1./2', 0],[1./2, '-x+1./2', 0],['x+1./2', 0., 1./2],['-x+1./2', 0., 1./2],[0., 1./2, 'x+1./2'],[0., 1./2, '-x+1./2']],
                      '24i': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y+1./2', 'x+1./2', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', 'z+1./2', '-y+1./2'],['-x+1./2', '-z+1./2', 'y+1./2'],['x+1./2', '-z+1./2', '-y+1./2'],['z+1./2', 'y+1./2', 'x+1./2'],['z+1./2', '-y+1./2', '-x+1./2'],['-z+1./2', 'y+1./2', '-x+1./2'],['-z+1./2', '-y+1./2', 'x+1./2']]},
              '219': {'8a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '8b': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                      '24c': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0]],
                      '24d': [[1./4, 0., 0],[1./4, 1./2, 1./2],[3./4, 0., 1./2],[3./4, 1./2, 0],[3./4, 0., 0],[3./4, 1./2, 1./2],[1./4, 0., 1./2],[1./4, 1./2, 0],[0., 1./4, 0],[0., 3./4, 1./2],[1./2, 1./4, 1./2],[1./2, 3./4, 0],[0., 3./4, 0],[0., 1./4, 1./2],[1./2, 3./4, 1./2],[1./2, 1./4, 0],[0., 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4],[1./2, 1./2, 1./4],[0., 0., 3./4],[0., 1./2, 1./4],[1./2, 0., 1./4],[1./2, 1./2, 3./4]],
                      '32e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x+1./2', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x'],['x', 'x+1./2', 'x'],['x', 'x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x'],['-x', '-x+1./2', 'x'],['-x', '-x', 'x+1./2'],['x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x'],['x', '-x+1./2', '-x'],['x', '-x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['-x', 'x+1./2', '-x'],['-x', 'x', '-x+1./2']],
                      '48f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x'],[1./2, 'x+1./2', 1./2],[1./2, 'x', 0],[0., 'x+1./2', 0],[0., 'x', 1./2],[1./2, '-x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 0],[0., '-x', 1./2],['x+1./2', 1./2, 1./2],['x+1./2', 0., 0],['x', 1./2, 0],['x', 0., 1./2],['-x+1./2', 1./2, 1./2],['-x+1./2', 0., 0],['-x', 1./2, 0],['-x', 0., 1./2],[1./2, 1./2, 'x+1./2'],[1./2, 0., 'x'],[0., 1./2, 'x'],[0., 0., 'x+1./2'],[1./2, 1./2, '-x+1./2'],[1./2, 0., '-x'],[0., 1./2, '-x'],[0., 0., '-x+1./2']],
                      '48g': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x'],[3./4, 'x+1./2', 3./4],[3./4, 'x', 1./4],[1./4, 'x+1./2', 1./4],[1./4, 'x', 3./4],[1./4, '-x+1./2', 3./4],[1./4, '-x', 1./4],[3./4, '-x+1./2', 1./4],[3./4, '-x', 3./4],['x+1./2', 3./4, 3./4],['x+1./2', 1./4, 1./4],['x', 3./4, 1./4],['x', 1./4, 3./4],['-x+1./2', 3./4, 1./4],['-x+1./2', 1./4, 3./4],['-x', 3./4, 3./4],['-x', 1./4, 1./4],[3./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x'],[1./4, 3./4, 'x'],[1./4, 1./4, 'x+1./2'],[3./4, 1./4, '-x+1./2'],[3./4, 3./4, '-x'],[1./4, 1./4, '-x'],[1./4, 3./4, '-x+1./2']],
                      '96h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z'],['y', 'x+1./2', 'z'],['y', 'x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['-y+1./2', '-x', 'z'],['-y', '-x+1./2', 'z'],['-y', '-x', 'z+1./2'],['y+1./2', '-x+1./2', '-z+1./2'],['y+1./2', '-x', '-z'],['y', '-x+1./2', '-z'],['y', '-x', '-z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', 'x', '-z'],['-y', 'x+1./2', '-z'],['-y', 'x', '-z+1./2'],['x+1./2', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y'],['x', 'z+1./2', 'y'],['x', 'z', 'y+1./2'],['-x+1./2', 'z+1./2', '-y+1./2'],['-x+1./2', 'z', '-y'],['-x', 'z+1./2', '-y'],['-x', 'z', '-y+1./2'],['-x+1./2', '-z+1./2', 'y+1./2'],['-x+1./2', '-z', 'y'],['-x', '-z+1./2', 'y'],['-x', '-z', 'y+1./2'],['x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', '-z', '-y'],['x', '-z+1./2', '-y'],['x', '-z', '-y+1./2'],['z+1./2', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x'],['z', 'y+1./2', 'x'],['z', 'y', 'x+1./2'],['z+1./2', '-y+1./2', '-x+1./2'],['z+1./2', '-y', '-x'],['z', '-y+1./2', '-x'],['z', '-y', '-x+1./2'],['-z+1./2', 'y+1./2', '-x+1./2'],['-z+1./2', 'y', '-x'],['-z', 'y+1./2', '-x'],['-z', 'y', '-x+1./2'],['-z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', '-y', 'x'],['-z', '-y+1./2', 'x'],['-z', '-y', 'x+1./2']]},
              '220': {'12a': [[3/8, 0., 1./4],[7/8, 1./2, 3./4],[1/8, 0., 3./4],[5/8, 1./2, 1./4],[1./4, 3/8, 0],[3./4, 7/8, 1./2],[3./4, 1/8, 0],[1./4, 5/8, 1./2],[0., 1./4, 3/8],[1./2, 3./4, 7/8],[0., 3./4, 1/8],[1./2, 1./4, 5/8]],
                      '12b': [[7/8, 0., 1./4],[3/8, 1./2, 3./4],[5/8, 0., 3./4],[1/8, 1./2, 1./4],[1./4, 7/8, 0],[3./4, 3/8, 1./2],[3./4, 5/8, 0],[1./4, 1/8, 1./2],[0., 1./4, 7/8],[1./2, 3./4, 3/8],[0., 3./4, 5/8],[1./2, 1./4, 1/8]],
                      '16c': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x', '-x+1./2', 'x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x+1./2'],['x+1./4', 'x+1./4', 'x+1./4'],['x+3./4', 'x+3./4', 'x+3./4'],['-x+1./4', '-x+3./4', 'x+3./4'],['-x+3./4', '-x+1./4', 'x+1./4'],['x+3./4', '-x+1./4', '-x+3./4'],['x+1./4', '-x+3./4', '-x+1./4'],['-x+3./4', 'x+3./4', '-x+1./4'],['-x+1./4', 'x+1./4', '-x+3./4']],
                      '24d': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],[1./4, 'x', 0],[3./4, 'x+1./2', 1./2],[3./4, '-x+1./2', 0],[1./4, '-x', 1./2],[0., 1./4, 'x'],[1./2, 3./4, 'x+1./2'],[0., 3./4, '-x+1./2'],[1./2, 1./4, '-x'],[1./4, 'x+1./4', 1./2],[3./4, 'x+3./4', 0],[1./4, '-x+3./4', 0],[3./4, '-x+1./4', 1./2],['x+1./4', 1./2, 1./4],['x+3./4', 0., 3./4],['-x+3./4', 0., 1./4],['-x+1./4', 1./2, 3./4],[1./2, 1./4, 'x+1./4'],[0., 3./4, 'x+3./4'],[0., 1./4, '-x+3./4'],[1./2, 3./4, '-x+1./4']],
                      '48e': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z', '-x+1./2', 'y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y', '-z+1./2', 'x'],['y+1./4', 'x+1./4', 'z+1./4'],['y+3./4', 'x+3./4', 'z+3./4'],['-y+1./4', '-x+3./4', 'z+3./4'],['-y+3./4', '-x+1./4', 'z+1./4'],['y+3./4', '-x+1./4', '-z+3./4'],['y+1./4', '-x+3./4', '-z+1./4'],['-y+3./4', 'x+3./4', '-z+1./4'],['-y+1./4', 'x+1./4', '-z+3./4'],['x+1./4', 'z+1./4', 'y+1./4'],['x+3./4', 'z+3./4', 'y+3./4'],['-x+3./4', 'z+3./4', '-y+1./4'],['-x+1./4', 'z+1./4', '-y+3./4'],['-x+1./4', '-z+3./4', 'y+3./4'],['-x+3./4', '-z+1./4', 'y+1./4'],['x+3./4', '-z+1./4', '-y+3./4'],['x+1./4', '-z+3./4', '-y+1./4'],['z+1./4', 'y+1./4', 'x+1./4'],['z+3./4', 'y+3./4', 'x+3./4'],['z+3./4', '-y+1./4', '-x+3./4'],['z+1./4', '-y+3./4', '-x+1./4'],['-z+3./4', 'y+3./4', '-x+1./4'],['-z+1./4', 'y+1./4', '-x+3./4'],['-z+1./4', '-y+3./4', 'x+3./4'],['-z+3./4', '-y+1./4', 'x+1./4']]},
              '221': {'1a': [[0., 0., 0]],
                      '1b': [[1./2, 1./2, 1./2]],
                      '3c': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '3d': [[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '6e': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x']],
                      '6f': [['x', 1./2, 1./2],['-x', 1./2, 1./2],[1./2, 'x', 1./2],[1./2, '-x', 1./2],[1./2, 1./2, 'x'],[1./2, 1./2, '-x']],
                      '8g': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x', 'x', '-x'],['-x', '-x', '-x'],['x', '-x', 'x'],['-x', 'x', 'x']],
                      '12h': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],[1./2, 'x', 0],[1./2, '-x', 0],['x', 0., 1./2],['-x', 0., 1./2],[0., 1./2, '-x'],[0., 1./2, 'x']],
                      '12i': [[0., 'y', 'y'],[0., '-y', 'y'],[0., 'y', '-y'],[0., '-y', '-y'],['y', 0., 'y'],['y', 0., '-y'],['-y', 0., 'y'],['-y', 0., '-y'],['y', 'y', 0],['-y', 'y', 0],['y', '-y', 0],['-y', '-y', 0]],
                      '12j': [[1./2, 'y', 'y'],[1./2, '-y', 'y'],[1./2, 'y', '-y'],[1./2, '-y', '-y'],['y', 1./2, 'y'],['y', 1./2, '-y'],['-y', 1./2, 'y'],['-y', 1./2, '-y'],['y', 'y', 1./2],['-y', 'y', 1./2],['y', '-y', 1./2],['-y', '-y', 1./2]],
                      '24k': [[0., 'y', 'z'],[0., '-y', 'z'],[0., 'y', '-z'],[0., '-y', '-z'],['z', 0., 'y'],['z', 0., '-y'],['-z', 0., 'y'],['-z', 0., '-y'],['y', 'z', 0],['-y', 'z', 0],['y', '-z', 0],['-y', '-z', 0],['y', 0., '-z'],['-y', 0., '-z'],['y', 0., 'z'],['-y', 0., 'z'],[0., 'z', '-y'],[0., 'z', 'y'],[0., '-z', '-y'],[0., '-z', 'y'],['z', 'y', 0],['z', '-y', 0],['-z', 'y', 0],['-z', '-y', 0]],
                      '24l': [[1./2, 'y', 'z'],[1./2, '-y', 'z'],[1./2, 'y', '-z'],[1./2, '-y', '-z'],['z', 1./2, 'y'],['z', 1./2, '-y'],['-z', 1./2, 'y'],['-z', 1./2, '-y'],['y', 'z', 1./2],['-y', 'z', 1./2],['y', '-z', 1./2],['-y', '-z', 1./2],['y', 1./2, '-z'],['-y', 1./2, '-z'],['y', 1./2, 'z'],['-y', 1./2, 'z'],[1./2, 'z', '-y'],[1./2, 'z', 'y'],[1./2, '-z', '-y'],[1./2, '-z', 'y'],['z', 'y', 1./2],['z', '-y', 1./2],['-z', 'y', 1./2],['-z', '-y', 1./2]],
                      '24m': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', '-z'],['x', '-x', '-z'],['z', 'x', 'x'],['z', '-x', '-x'],['-z', '-x', 'x'],['-z', 'x', '-x'],['x', 'z', 'x'],['-x', 'z', '-x'],['x', '-z', '-x'],['-x', '-z', 'x'],['x', 'x', '-z'],['-x', '-x', '-z'],['x', '-x', 'z'],['-x', 'x', 'z'],['x', 'z', '-x'],['-x', 'z', 'x'],['-x', '-z', '-x'],['x', '-z', 'x'],['z', 'x', '-x'],['z', '-x', 'x'],['-z', 'x', 'x'],['-z', '-x', '-x']],
                      '48n': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y', 'x', '-z'],['-y', '-x', '-z'],['y', '-x', 'z'],['-y', 'x', 'z'],['x', 'z', '-y'],['-x', 'z', 'y'],['-x', '-z', '-y'],['x', '-z', 'y'],['z', 'y', '-x'],['z', '-y', 'x'],['-z', 'y', 'x'],['-z', '-y', '-x'],['-x', '-y', '-z'],['x', 'y', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-z', '-x', '-y'],['-z', 'x', 'y'],['z', 'x', '-y'],['z', '-x', 'y'],['-y', '-z', '-x'],['y', '-z', 'x'],['-y', 'z', 'x'],['y', 'z', '-x'],['-y', '-x', 'z'],['y', 'x', 'z'],['-y', 'x', '-z'],['y', '-x', '-z'],['-x', '-z', 'y'],['x', '-z', '-y'],['x', 'z', 'y'],['-x', 'z', '-y'],['-z', '-y', 'x'],['-z', 'y', '-x'],['z', '-y', '-x'],['z', 'y', 'x']]},
              '222:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '6b': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                        '8c': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                        '12d': [[1./4, 0., 1./2],[3./4, 0., 1./2],[1./2, 1./4, 0],[1./2, 3./4, 0],[0., 1./2, 1./4],[0., 1./2, 3./4],[0., 1./4, 1./2],[0., 3./4, 1./2],[1./4, 1./2, 0],[3./4, 1./2, 0],[1./2, 0., 3./4],[1./2, 0., 1./4]],
                        '12e': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x'],['-x+1./2', 1./2, 1./2],['x+1./2', 1./2, 1./2],[1./2, '-x+1./2', 1./2],[1./2, 'x+1./2', 1./2],[1./2, 1./2, '-x+1./2'],[1./2, 1./2, 'x+1./2']],
                        '16f': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x', 'x', '-x'],['-x', '-x', '-x'],['x', '-x', 'x'],['-x', 'x', 'x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x+1./2', 'x+1./2'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', '-x+1./2']],
                        '24g': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x'],[0., 'x', 1./2],[0., '-x', 1./2],['x', 1./2, 0],['-x', 1./2, 0],[1./2, 0., '-x'],[1./2, 0., 'x'],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0],[0., '-x+1./2', 1./2],[0., 'x+1./2', 1./2],[1./2, 0., '-x+1./2'],[1./2, 0., 'x+1./2'],[1./2, '-x+1./2', 0],[1./2, 'x+1./2', 0],['-x+1./2', 0., 1./2],['x+1./2', 0., 1./2],[0., 1./2, 'x+1./2'],[0., 1./2, '-x+1./2']],
                        '24h': [[0., 'y', 'y'],[0., '-y', 'y'],[0., 'y', '-y'],[0., '-y', '-y'],['y', 0., 'y'],['y', 0., '-y'],['-y', 0., 'y'],['-y', 0., '-y'],['y', 'y', 0],['-y', 'y', 0],['y', '-y', 0],['-y', '-y', 0],[1./2, '-y+1./2', '-y+1./2'],[1./2, 'y+1./2', '-y+1./2'],[1./2, '-y+1./2', 'y+1./2'],[1./2, 'y+1./2', 'y+1./2'],['-y+1./2', 1./2, '-y+1./2'],['-y+1./2', 1./2, 'y+1./2'],['y+1./2', 1./2, '-y+1./2'],['y+1./2', 1./2, 'y+1./2'],['-y+1./2', '-y+1./2', 1./2],['y+1./2', '-y+1./2', 1./2],['-y+1./2', 'y+1./2', 1./2],['y+1./2', 'y+1./2', 1./2]],
                        '48i': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y', 'x', '-z'],['-y', '-x', '-z'],['y', '-x', 'z'],['-y', 'x', 'z'],['x', 'z', '-y'],['-x', 'z', 'y'],['-x', '-z', '-y'],['x', '-z', 'y'],['z', 'y', '-x'],['z', '-y', 'x'],['-z', 'y', 'x'],['-z', '-y', '-x'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', 'x+1./2', '-y+1./2'],['z+1./2', '-x+1./2', 'y+1./2'],['-y+1./2', '-z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', 'x+1./2'],['-y+1./2', 'z+1./2', 'x+1./2'],['y+1./2', 'z+1./2', '-x+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['y+1./2', '-x+1./2', '-z+1./2'],['-x+1./2', '-z+1./2', 'y+1./2'],['x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', 'z+1./2', '-y+1./2'],['-z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', 'y+1./2', '-x+1./2'],['z+1./2', '-y+1./2', '-x+1./2'],['z+1./2', 'y+1./2', 'x+1./2']]},
              '222:2': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '6b': [[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4]],
                        '8c': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2],[0., 0., 1./2],[1./2, 1./2, 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                        '12d': [[0., 3./4, 1./4],[1./2, 3./4, 1./4],[1./4, 0., 3./4],[1./4, 1./2, 3./4],[3./4, 1./4, 0],[3./4, 1./4, 1./2],[3./4, 0., 1./4],[3./4, 1./2, 1./4],[0., 1./4, 3./4],[1./2, 1./4, 3./4],[1./4, 3./4, 1./2],[1./4, 3./4, 0]],
                        '12e': [['x', 1./4, 1./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, '-x+1./2', 1./4],[1./4, 1./4, 'x'],[1./4, 1./4, '-x+1./2'],['-x', 3./4, 3./4],['x+1./2', 3./4, 3./4],[3./4, '-x', 3./4],[3./4, 'x+1./2', 3./4],[3./4, 3./4, '-x'],[3./4, 3./4, 'x+1./2']],
                        '16f': [['x', 'x', 'x'],['-x+1./2', '-x+1./2', 'x'],['-x+1./2', 'x', '-x+1./2'],['x', '-x+1./2', '-x+1./2'],['x', 'x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x+1./2'],['x', '-x+1./2', 'x'],['-x+1./2', 'x', 'x'],['-x', '-x', '-x'],['x+1./2', 'x+1./2', '-x'],['x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', 'x+1./2'],['-x', '-x', 'x+1./2'],['x+1./2', 'x+1./2', 'x+1./2'],['-x', 'x+1./2', '-x'],['x+1./2', '-x', '-x']],
                        '24g': [['x', 3./4, 1./4],['-x+1./2', 3./4, 1./4],[1./4, 'x', 3./4],[1./4, '-x+1./2', 3./4],[3./4, 1./4, 'x'],[3./4, 1./4, '-x+1./2'],[3./4, 'x', 1./4],[3./4, '-x+1./2', 1./4],['x', 1./4, 3./4],['-x+1./2', 1./4, 3./4],[1./4, 3./4, '-x+1./2'],[1./4, 3./4, 'x'],['-x', 1./4, 3./4],['x+1./2', 1./4, 3./4],[3./4, '-x', 1./4],[3./4, 'x+1./2', 1./4],[1./4, 3./4, '-x'],[1./4, 3./4, 'x+1./2'],[1./4, '-x', 3./4],[1./4, 'x+1./2', 3./4],['-x', 3./4, 1./4],['x+1./2', 3./4, 1./4],[3./4, 1./4, 'x+1./2'],[3./4, 1./4, '-x']],
                        '24h': [[1./4, 'y', 'y'],[1./4, '-y+1./2', 'y'],[1./4, 'y', '-y+1./2'],[1./4, '-y+1./2', '-y+1./2'],['y', 1./4, 'y'],['y', 1./4, '-y+1./2'],['-y+1./2', 1./4, 'y'],['-y+1./2', 1./4, '-y+1./2'],['y', 'y', 1./4],['-y+1./2', 'y', 1./4],['y', '-y+1./2', 1./4],['-y+1./2', '-y+1./2', 1./4],[3./4, '-y', '-y'],[3./4, 'y+1./2', '-y'],[3./4, '-y', 'y+1./2'],[3./4, 'y+1./2', 'y+1./2'],['-y', 3./4, '-y'],['-y', 3./4, 'y+1./2'],['y+1./2', 3./4, '-y'],['y+1./2', 3./4, 'y+1./2'],['-y', '-y', 3./4],['y+1./2', '-y', 3./4],['-y', 'y+1./2', 3./4],['y+1./2', 'y+1./2', 3./4]],
                        '48i': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z+1./2', 'x', '-y+1./2'],['y', 'z', 'x'],['-y+1./2', 'z', '-x+1./2'],['y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y', 'x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['y', '-x+1./2', 'z'],['-y+1./2', 'x', 'z'],['x', 'z', '-y+1./2'],['-x+1./2', 'z', 'y'],['-x+1./2', '-z+1./2', '-y+1./2'],['x', '-z+1./2', 'y'],['z', 'y', '-x+1./2'],['z', '-y+1./2', 'x'],['-z+1./2', 'y', 'x'],['-z+1./2', '-y+1./2', '-x+1./2'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2'],['-z', '-x', '-y'],['-z', 'x+1./2', 'y+1./2'],['z+1./2', 'x+1./2', '-y'],['z+1./2', '-x', 'y+1./2'],['-y', '-z', '-x'],['y+1./2', '-z', 'x+1./2'],['-y', 'z+1./2', 'x+1./2'],['y+1./2', 'z+1./2', '-x'],['-y', '-x', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2'],['-y', 'x+1./2', '-z'],['y+1./2', '-x', '-z'],['-x', '-z', 'y+1./2'],['x+1./2', '-z', '-y'],['x+1./2', 'z+1./2', 'y+1./2'],['-x', 'z+1./2', '-y'],['-z', '-y', 'x+1./2'],['-z', 'y+1./2', '-x'],['z+1./2', '-y', '-x'],['z+1./2', 'y+1./2', 'x+1./2']]},
              '223': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 0],[1./2, 0., 0],[0., 0., 1./2]],
                      '6c': [[1./4, 0., 1./2],[3./4, 0., 1./2],[1./2, 1./4, 0],[1./2, 3./4, 0],[0., 1./2, 1./4],[0., 1./2, 3./4]],
                      '6d': [[1./4, 1./2, 0],[3./4, 1./2, 0],[0., 1./4, 1./2],[0., 3./4, 1./2],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                      '8e': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4],[3./4, 3./4, 3./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                      '12f': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x'],[1./2, 'x+1./2', 1./2],[1./2, '-x+1./2', 1./2],['x+1./2', 1./2, 1./2],['-x+1./2', 1./2, 1./2],[1./2, 1./2, '-x+1./2'],[1./2, 1./2, 'x+1./2']],
                      '12g': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x'],[1./2, 'x+1./2', 0],[1./2, '-x+1./2', 0],['x+1./2', 0., 1./2],['-x+1./2', 0., 1./2],[0., 1./2, '-x+1./2'],[0., 1./2, 'x+1./2']],
                      '12h': [['x', 1./2, 0],['-x', 1./2, 0],[0., 'x', 1./2],[0., '-x', 1./2],[1./2, 0., 'x'],[1./2, 0., '-x'],[0., 'x+1./2', 1./2],[0., '-x+1./2', 1./2],['x+1./2', 1./2, 0],['-x+1./2', 1./2, 0],[1./2, 0., '-x+1./2'],[1./2, 0., 'x+1./2']],
                      '16i': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', 'x+1./2'],['-x', '-x', '-x'],['x', 'x', '-x'],['x', '-x', 'x'],['-x', 'x', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', '-x+1./2']],
                      '24j': [[1./4, 'y', 'y+1./2'],[3./4, '-y', 'y+1./2'],[3./4, 'y', '-y+1./2'],[1./4, '-y', '-y+1./2'],['y+1./2', 1./4, 'y'],['y+1./2', 3./4, '-y'],['-y+1./2', 3./4, 'y'],['-y+1./2', 1./4, '-y'],['y', 'y+1./2', 1./4],['-y', 'y+1./2', 3./4],['y', '-y+1./2', 3./4],['-y', '-y+1./2', 1./4],[3./4, '-y', '-y+1./2'],[1./4, 'y', '-y+1./2'],[1./4, '-y', 'y+1./2'],[3./4, 'y', 'y+1./2'],['-y+1./2', 3./4, '-y'],['-y+1./2', 1./4, 'y'],['y+1./2', 1./4, '-y'],['y+1./2', 3./4, 'y'],['-y', '-y+1./2', 3./4],['y', '-y+1./2', 1./4],['-y', 'y+1./2', 1./4],['y', 'y+1./2', 3./4]],
                      '24k': [[0., 'y', 'z'],[0., '-y', 'z'],[0., 'y', '-z'],[0., '-y', '-z'],['z', 0., 'y'],['z', 0., '-y'],['-z', 0., 'y'],['-z', 0., '-y'],['y', 'z', 0],['-y', 'z', 0],['y', '-z', 0],['-y', '-z', 0],['y+1./2', 1./2, '-z+1./2'],['-y+1./2', 1./2, '-z+1./2'],['y+1./2', 1./2, 'z+1./2'],['-y+1./2', 1./2, 'z+1./2'],[1./2, 'z+1./2', '-y+1./2'],[1./2, 'z+1./2', 'y+1./2'],[1./2, '-z+1./2', '-y+1./2'],[1./2, '-z+1./2', 'y+1./2'],['z+1./2', 'y+1./2', 1./2],['z+1./2', '-y+1./2', 1./2],['-z+1./2', 'y+1./2', 1./2],['-z+1./2', '-y+1./2', 1./2]],
                      '48l': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-y+1./2', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', '-y+1./2'],['-x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', '-z+1./2', 'y+1./2'],['z+1./2', 'y+1./2', '-x+1./2'],['z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', 'y+1./2', 'x+1./2'],['-z+1./2', '-y+1./2', '-x+1./2'],['-x', '-y', '-z'],['x', 'y', '-z'],['x', '-y', 'z'],['-x', 'y', 'z'],['-z', '-x', '-y'],['-z', 'x', 'y'],['z', 'x', '-y'],['z', '-x', 'y'],['-y', '-z', '-x'],['y', '-z', 'x'],['-y', 'z', 'x'],['y', 'z', '-x'],['-y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['y+1./2', '-x+1./2', '-z+1./2'],['-x+1./2', '-z+1./2', 'y+1./2'],['x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', 'z+1./2', '-y+1./2'],['-z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', 'y+1./2', '-x+1./2'],['z+1./2', '-y+1./2', '-x+1./2'],['z+1./2', 'y+1./2', 'x+1./2']]},
              '224:1': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                        '4b': [[1./4, 1./4, 1./4],[3./4, 3./4, 1./4],[3./4, 1./4, 3./4],[1./4, 3./4, 3./4]],
                        '4c': [[3./4, 3./4, 3./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4]],
                        '6d': [[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[0., 1./2, 0],[1./2, 0., 0],[0., 0., 1./2]],
                        '8e': [['x', 'x', 'x'],['-x', '-x', 'x'],['-x', 'x', '-x'],['x', '-x', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', 'x+1./2', 'x+1./2']],
                        '12f': [[1./4, 0., 1./2],[3./4, 0., 1./2],[1./2, 1./4, 0],[1./2, 3./4, 0],[0., 1./2, 1./4],[0., 1./2, 3./4],[1./4, 1./2, 0],[3./4, 1./2, 0],[0., 1./4, 1./2],[0., 3./4, 1./2],[1./2, 0., 1./4],[1./2, 0., 3./4]],
                        '12g': [['x', 0., 0],['-x', 0., 0],[0., 'x', 0],[0., '-x', 0],[0., 0., 'x'],[0., 0., '-x'],[1./2, 'x+1./2', 1./2],[1./2, '-x+1./2', 1./2],['x+1./2', 1./2, 1./2],['-x+1./2', 1./2, 1./2],[1./2, 1./2, '-x+1./2'],[1./2, 1./2, 'x+1./2']],
                        '24h': [['x', 0., 1./2],['-x', 0., 1./2],[1./2, 'x', 0],[1./2, '-x', 0],[0., 1./2, 'x'],[0., 1./2, '-x'],[1./2, 'x+1./2', 0],[1./2, '-x+1./2', 0],['x+1./2', 0., 1./2],['-x+1./2', 0., 1./2],[0., 1./2, '-x+1./2'],[0., 1./2, 'x+1./2'],['-x+1./2', 1./2, 0],['x+1./2', 1./2, 0],[0., '-x+1./2', 1./2],[0., 'x+1./2', 1./2],[1./2, 0., '-x+1./2'],[1./2, 0., 'x+1./2'],[0., '-x', 1./2],[0., 'x', 1./2],['-x', 1./2, 0],['x', 1./2, 0],[1./2, 0., 'x'],[1./2, 0., '-x']],
                        '24i': [[1./4, 'y', '-y+1./2'],[3./4, '-y', '-y+1./2'],[3./4, 'y', 'y+1./2'],[1./4, '-y', 'y+1./2'],['-y+1./2', 1./4, 'y'],['-y+1./2', 3./4, '-y'],['y+1./2', 3./4, 'y'],['y+1./2', 1./4, '-y'],['y', '-y+1./2', 1./4],['-y', '-y+1./2', 3./4],['y', 'y+1./2', 3./4],['-y', 'y+1./2', 1./4],[1./4, '-y+1./2', 'y'],[3./4, 'y+1./2', 'y'],[3./4, '-y+1./2', '-y'],[1./4, 'y+1./2', '-y'],['y', 1./4, '-y+1./2'],['y', 3./4, 'y+1./2'],['-y', 3./4, '-y+1./2'],['-y', 1./4, 'y+1./2'],['-y+1./2', 'y', 1./4],['y+1./2', 'y', 3./4],['-y+1./2', '-y', 3./4],['y+1./2', '-y', 1./4]],
                        '24j': [[1./4, 'y', 'y+1./2'],[3./4, '-y', 'y+1./2'],[3./4, 'y', '-y+1./2'],[1./4, '-y', '-y+1./2'],['y+1./2', 1./4, 'y'],['y+1./2', 3./4, '-y'],['-y+1./2', 3./4, 'y'],['-y+1./2', 1./4, '-y'],['y', 'y+1./2', 1./4],['-y', 'y+1./2', 3./4],['y', '-y+1./2', 3./4],['-y', '-y+1./2', 1./4],[1./4, '-y+1./2', '-y'],[3./4, 'y+1./2', '-y'],[3./4, '-y+1./2', 'y'],[1./4, 'y+1./2', 'y'],['-y', 1./4, '-y+1./2'],['-y', 3./4, 'y+1./2'],['y', 3./4, '-y+1./2'],['y', 1./4, 'y+1./2'],['-y+1./2', '-y', 1./4],['y+1./2', '-y', 3./4],['-y+1./2', 'y', 3./4],['y+1./2', 'y', 1./4]],
                        '24k': [['x', 'x', 'z'],['-x', '-x', 'z'],['-x', 'x', '-z'],['x', '-x', '-z'],['z', 'x', 'x'],['z', '-x', '-x'],['-z', '-x', 'x'],['-z', 'x', '-x'],['x', 'z', 'x'],['-x', 'z', '-x'],['x', '-z', '-x'],['-x', '-z', 'x'],['x+1./2', 'x+1./2', '-z+1./2'],['-x+1./2', '-x+1./2', '-z+1./2'],['x+1./2', '-x+1./2', 'z+1./2'],['-x+1./2', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', '-x+1./2'],['-x+1./2', 'z+1./2', 'x+1./2'],['-x+1./2', '-z+1./2', '-x+1./2'],['x+1./2', '-z+1./2', 'x+1./2'],['z+1./2', 'x+1./2', '-x+1./2'],['z+1./2', '-x+1./2', 'x+1./2'],['-z+1./2', 'x+1./2', 'x+1./2'],['-z+1./2', '-x+1./2', '-x+1./2']],
                        '48l': [['x', 'y', 'z'],['-x', '-y', 'z'],['-x', 'y', '-z'],['x', '-y', '-z'],['z', 'x', 'y'],['z', '-x', '-y'],['-z', '-x', 'y'],['-z', 'x', '-y'],['y', 'z', 'x'],['-y', 'z', '-x'],['y', '-z', '-x'],['-y', '-z', 'x'],['y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['-y+1./2', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', '-y+1./2'],['-x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', '-z+1./2', 'y+1./2'],['z+1./2', 'y+1./2', '-x+1./2'],['z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', 'y+1./2', 'x+1./2'],['-z+1./2', '-y+1./2', '-x+1./2'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y+1./2', '-z+1./2'],['x+1./2', '-y+1./2', 'z+1./2'],['-x+1./2', 'y+1./2', 'z+1./2'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', 'x+1./2', '-y+1./2'],['z+1./2', '-x+1./2', 'y+1./2'],['-y+1./2', '-z+1./2', '-x+1./2'],['y+1./2', '-z+1./2', 'x+1./2'],['-y+1./2', 'z+1./2', 'x+1./2'],['y+1./2', 'z+1./2', '-x+1./2'],['-y', '-x', 'z'],['y', 'x', 'z'],['-y', 'x', '-z'],['y', '-x', '-z'],['-x', '-z', 'y'],['x', '-z', '-y'],['x', 'z', 'y'],['-x', 'z', '-y'],['-z', '-y', 'x'],['-z', 'y', '-x'],['z', '-y', '-x'],['z', 'y', 'x']]},
              '224:2': {'2a': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                        '4b': [[0., 0., 0],[1./2, 1./2, 0],[1./2, 0., 1./2],[0., 1./2, 1./2]],
                        '4c': [[1./2, 1./2, 1./2],[0., 0., 1./2],[0., 1./2, 0],[1./2, 0., 0]],
                        '6d': [[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[1./4, 1./4, 3./4]],
                        '8e': [['x', 'x', 'x'],['-x+1./2', '-x+1./2', 'x'],['-x+1./2', 'x', '-x+1./2'],['x', '-x+1./2', '-x+1./2'],['x+1./2', 'x+1./2', '-x'],['-x', '-x', '-x'],['x+1./2', '-x', 'x+1./2'],['-x', 'x+1./2', 'x+1./2']],
                        '12f': [[1./2, 1./4, 3./4],[0., 1./4, 3./4],[3./4, 1./2, 1./4],[3./4, 0., 1./4],[1./4, 3./4, 1./2],[1./4, 3./4, 0],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[1./4, 1./2, 3./4],[1./4, 0., 3./4],[3./4, 1./4, 1./2],[3./4, 1./4, 0]],
                        '12g': [['x', 1./4, 1./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, '-x+1./2', 1./4],[1./4, 1./4, 'x'],[1./4, 1./4, '-x+1./2'],[3./4, 'x+1./2', 3./4],[3./4, '-x', 3./4],['x+1./2', 3./4, 3./4],['-x', 3./4, 3./4],[3./4, 3./4, '-x'],[3./4, 3./4, 'x+1./2']],
                        '24h': [['x', 1./4, 3./4],['-x+1./2', 1./4, 3./4],[3./4, 'x', 1./4],[3./4, '-x+1./2', 1./4],[1./4, 3./4, 'x'],[1./4, 3./4, '-x+1./2'],[3./4, 'x+1./2', 1./4],[3./4, '-x', 1./4],['x+1./2', 1./4, 3./4],['-x', 1./4, 3./4],[1./4, 3./4, '-x'],[1./4, 3./4, 'x+1./2'],['-x', 3./4, 1./4],['x+1./2', 3./4, 1./4],[1./4, '-x', 3./4],[1./4, 'x+1./2', 3./4],[3./4, 1./4, '-x'],[3./4, 1./4, 'x+1./2'],[1./4, '-x+1./2', 3./4],[1./4, 'x', 3./4],['-x+1./2', 3./4, 1./4],['x', 3./4, 1./4],[3./4, 1./4, 'x'],[3./4, 1./4, '-x+1./2']],
                        '24i': [[1./2, 'y', 'y+1./2'],[0., '-y+1./2', 'y+1./2'],[0., 'y', '-y'],[1./2, '-y+1./2', '-y'],['y+1./2', 1./2, 'y'],['y+1./2', 0., '-y+1./2'],['-y', 0., 'y'],['-y', 1./2, '-y+1./2'],['y', 'y+1./2', 1./2],['-y+1./2', 'y+1./2', 0],['y', '-y', 0],['-y+1./2', '-y', 1./2],[1./2, '-y', '-y+1./2'],[0., 'y+1./2', '-y+1./2'],[0., '-y', 'y'],[1./2, 'y+1./2', 'y'],['-y+1./2', 1./2, '-y'],['-y+1./2', 0., 'y+1./2'],['y', 0., '-y'],['y', 1./2, 'y+1./2'],['-y', '-y+1./2', 1./2],['y+1./2', '-y+1./2', 0],['-y', 'y', 0],['y+1./2', 'y', 1./2]],
                        '24j': [[1./2, 'y', '-y'],[0., '-y+1./2', '-y'],[0., 'y', 'y+1./2'],[1./2, '-y+1./2', 'y+1./2'],['-y', 1./2, 'y'],['-y', 0., '-y+1./2'],['y+1./2', 0., 'y'],['y+1./2', 1./2, '-y+1./2'],['y', '-y', 1./2],['-y+1./2', '-y', 0],['y', 'y+1./2', 0],['-y+1./2', 'y+1./2', 1./2],[1./2, '-y', 'y'],[0., 'y+1./2', 'y'],[0., '-y', '-y+1./2'],[1./2, 'y+1./2', '-y+1./2'],['y', 1./2, '-y'],['y', 0., 'y+1./2'],['-y+1./2', 0., '-y'],['-y+1./2', 1./2, 'y+1./2'],['-y', 'y', 1./2],['y+1./2', 'y', 0],['-y', '-y+1./2', 0],['y+1./2', '-y+1./2', 1./2]],
                        '24k': [['x', 'x', 'z'],['-x+1./2', '-x+1./2', 'z'],['-x+1./2', 'x', '-z+1./2'],['x', '-x+1./2', '-z+1./2'],['z', 'x', 'x'],['z', '-x+1./2', '-x+1./2'],['-z+1./2', '-x+1./2', 'x'],['-z+1./2', 'x', '-x+1./2'],['x', 'z', 'x'],['-x+1./2', 'z', '-x+1./2'],['x', '-z+1./2', '-x+1./2'],['-x+1./2', '-z+1./2', 'x'],['x+1./2', 'x+1./2', '-z'],['-x', '-x', '-z'],['x+1./2', '-x', 'z+1./2'],['-x', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', '-x'],['-x', 'z+1./2', 'x+1./2'],['-x', '-z', '-x'],['x+1./2', '-z', 'x+1./2'],['z+1./2', 'x+1./2', '-x'],['z+1./2', '-x', 'x+1./2'],['-z', 'x+1./2', 'x+1./2'],['-z', '-x', '-x']],
                        '48l': [['x', 'y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', 'y', '-z+1./2'],['x', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z+1./2', 'x', '-y+1./2'],['y', 'z', 'x'],['-y+1./2', 'z', '-x+1./2'],['y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z'],['y+1./2', '-x', 'z+1./2'],['-y', 'x+1./2', 'z+1./2'],['x+1./2', 'z+1./2', '-y'],['-x', 'z+1./2', 'y+1./2'],['-x', '-z', '-y'],['x+1./2', '-z', 'y+1./2'],['z+1./2', 'y+1./2', '-x'],['z+1./2', '-y', 'x+1./2'],['-z', 'y+1./2', 'x+1./2'],['-z', '-y', '-x'],['-x', '-y', '-z'],['x+1./2', 'y+1./2', '-z'],['x+1./2', '-y', 'z+1./2'],['-x', 'y+1./2', 'z+1./2'],['-z', '-x', '-y'],['-z', 'x+1./2', 'y+1./2'],['z+1./2', 'x+1./2', '-y'],['z+1./2', '-x', 'y+1./2'],['-y', '-z', '-x'],['y+1./2', '-z', 'x+1./2'],['-y', 'z+1./2', 'x+1./2'],['y+1./2', 'z+1./2', '-x'],['-y+1./2', '-x+1./2', 'z'],['y', 'x', 'z'],['-y+1./2', 'x', '-z+1./2'],['y', '-x+1./2', '-z+1./2'],['-x+1./2', '-z+1./2', 'y'],['x', '-z+1./2', '-y+1./2'],['x', 'z', 'y'],['-x+1./2', 'z', '-y+1./2'],['-z+1./2', '-y+1./2', 'x'],['-z+1./2', 'y', '-x+1./2'],['z', '-y+1./2', '-x+1./2'],['z', 'y', 'x']]},
              '225': {'4a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0]],
                      '4b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '8c': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[1./4, 3./4, 1./4],[3./4, 1./4, 1./4],[3./4, 3./4, 3./4]],
                      '24d': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0]],
                      '24e': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x']],
                      '32f': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x', 'x', '-x'],['x', 'x+1./2', '-x+1./2'],['x+1./2', 'x', '-x+1./2'],['x+1./2', 'x+1./2', '-x'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x', '-x', 'x'],['x', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x+1./2'],['x+1./2', '-x+1./2', 'x'],['-x', 'x', 'x'],['-x', 'x+1./2', 'x+1./2'],['-x+1./2', 'x', 'x+1./2'],['-x+1./2', 'x+1./2', 'x']],
                      '48g': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x'],[1./4, 'x', 3./4],[1./4, 'x+1./2', 1./4],[3./4, 'x', 1./4],[3./4, 'x+1./2', 3./4],[3./4, '-x', 3./4],[3./4, '-x+1./2', 1./4],[1./4, '-x', 1./4],[1./4, '-x+1./2', 3./4],['x', 1./4, 3./4],['x', 3./4, 1./4],['x+1./2', 1./4, 1./4],['x+1./2', 3./4, 3./4],['-x', 1./4, 1./4],['-x', 3./4, 3./4],['-x+1./2', 1./4, 3./4],['-x+1./2', 3./4, 1./4],[1./4, 1./4, '-x'],[1./4, 3./4, '-x+1./2'],[3./4, 1./4, '-x+1./2'],[3./4, 3./4, '-x'],[1./4, 3./4, 'x'],[1./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x']],
                      '48h': [[0., 'y', 'y'],[0., 'y+1./2', 'y+1./2'],[1./2, 'y', 'y+1./2'],[1./2, 'y+1./2', 'y'],[0., '-y', 'y'],[0., '-y+1./2', 'y+1./2'],[1./2, '-y', 'y+1./2'],[1./2, '-y+1./2', 'y'],[0., 'y', '-y'],[0., 'y+1./2', '-y+1./2'],[1./2, 'y', '-y+1./2'],[1./2, 'y+1./2', '-y'],[0., '-y', '-y'],[0., '-y+1./2', '-y+1./2'],[1./2, '-y', '-y+1./2'],[1./2, '-y+1./2', '-y'],['y', 0., 'y'],['y', 1./2, 'y+1./2'],['y+1./2', 0., 'y+1./2'],['y+1./2', 1./2, 'y'],['y', 0., '-y'],['y', 1./2, '-y+1./2'],['y+1./2', 0., '-y+1./2'],['y+1./2', 1./2, '-y'],['-y', 0., 'y'],['-y', 1./2, 'y+1./2'],['-y+1./2', 0., 'y+1./2'],['-y+1./2', 1./2, 'y'],['-y', 0., '-y'],['-y', 1./2, '-y+1./2'],['-y+1./2', 0., '-y+1./2'],['-y+1./2', 1./2, '-y'],['y', 'y', 0],['y', 'y+1./2', 1./2],['y+1./2', 'y', 1./2],['y+1./2', 'y+1./2', 0],['-y', 'y', 0],['-y', 'y+1./2', 1./2],['-y+1./2', 'y', 1./2],['-y+1./2', 'y+1./2', 0],['y', '-y', 0],['y', '-y+1./2', 1./2],['y+1./2', '-y', 1./2],['y+1./2', '-y+1./2', 0],['-y', '-y', 0],['-y', '-y+1./2', 1./2],['-y+1./2', '-y', 1./2],['-y+1./2', '-y+1./2', 0]],
                      '48i': [[1./2, 'y', 'y'],[1./2, 'y+1./2', 'y+1./2'],[0., 'y', 'y+1./2'],[0., 'y+1./2', 'y'],[1./2, '-y', 'y'],[1./2, '-y+1./2', 'y+1./2'],[0., '-y', 'y+1./2'],[0., '-y+1./2', 'y'],[1./2, 'y', '-y'],[1./2, 'y+1./2', '-y+1./2'],[0., 'y', '-y+1./2'],[0., 'y+1./2', '-y'],[1./2, '-y', '-y'],[1./2, '-y+1./2', '-y+1./2'],[0., '-y', '-y+1./2'],[0., '-y+1./2', '-y'],['y', 1./2, 'y'],['y', 0., 'y+1./2'],['y+1./2', 1./2, 'y+1./2'],['y+1./2', 0., 'y'],['y', 1./2, '-y'],['y', 0., '-y+1./2'],['y+1./2', 1./2, '-y+1./2'],['y+1./2', 0., '-y'],['-y', 1./2, 'y'],['-y', 0., 'y+1./2'],['-y+1./2', 1./2, 'y+1./2'],['-y+1./2', 0., 'y'],['-y', 1./2, '-y'],['-y', 0., '-y+1./2'],['-y+1./2', 1./2, '-y+1./2'],['-y+1./2', 0., '-y'],['y', 'y', 1./2],['y', 'y+1./2', 0],['y+1./2', 'y', 0],['y+1./2', 'y+1./2', 1./2],['-y', 'y', 1./2],['-y', 'y+1./2', 0],['-y+1./2', 'y', 0],['-y+1./2', 'y+1./2', 1./2],['y', '-y', 1./2],['y', '-y+1./2', 0],['y+1./2', '-y', 0],['y+1./2', '-y+1./2', 1./2],['-y', '-y', 1./2],['-y', '-y+1./2', 0],['-y+1./2', '-y', 0],['-y+1./2', '-y+1./2', 1./2]],
                      '96j': [[0., 'y', 'z'],[0., 'y+1./2', 'z+1./2'],[1./2, 'y', 'z+1./2'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[0., '-y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[1./2, '-y+1./2', 'z'],[0., 'y', '-z'],[0., 'y+1./2', '-z+1./2'],[1./2, 'y', '-z+1./2'],[1./2, 'y+1./2', '-z'],[0., '-y', '-z'],[0., '-y+1./2', '-z+1./2'],[1./2, '-y', '-z+1./2'],[1./2, '-y+1./2', '-z'],['z', 0., 'y'],['z', 1./2, 'y+1./2'],['z+1./2', 0., 'y+1./2'],['z+1./2', 1./2, 'y'],['z', 0., '-y'],['z', 1./2, '-y+1./2'],['z+1./2', 0., '-y+1./2'],['z+1./2', 1./2, '-y'],['-z', 0., 'y'],['-z', 1./2, 'y+1./2'],['-z+1./2', 0., 'y+1./2'],['-z+1./2', 1./2, 'y'],['-z', 0., '-y'],['-z', 1./2, '-y+1./2'],['-z+1./2', 0., '-y+1./2'],['-z+1./2', 1./2, '-y'],['y', 'z', 0],['y', 'z+1./2', 1./2],['y+1./2', 'z', 1./2],['y+1./2', 'z+1./2', 0],['-y', 'z', 0],['-y', 'z+1./2', 1./2],['-y+1./2', 'z', 1./2],['-y+1./2', 'z+1./2', 0],['y', '-z', 0],['y', '-z+1./2', 1./2],['y+1./2', '-z', 1./2],['y+1./2', '-z+1./2', 0],['-y', '-z', 0],['-y', '-z+1./2', 1./2],['-y+1./2', '-z', 1./2],['-y+1./2', '-z+1./2', 0],['y', 0., '-z'],['y', 1./2, '-z+1./2'],['y+1./2', 0., '-z+1./2'],['y+1./2', 1./2, '-z'],['-y', 0., '-z'],['-y', 1./2, '-z+1./2'],['-y+1./2', 0., '-z+1./2'],['-y+1./2', 1./2, '-z'],['y', 0., 'z'],['y', 1./2, 'z+1./2'],['y+1./2', 0., 'z+1./2'],['y+1./2', 1./2, 'z'],['-y', 0., 'z'],['-y', 1./2, 'z+1./2'],['-y+1./2', 0., 'z+1./2'],['-y+1./2', 1./2, 'z'],[0., 'z', '-y'],[0., 'z+1./2', '-y+1./2'],[1./2, 'z', '-y+1./2'],[1./2, 'z+1./2', '-y'],[0., 'z', 'y'],[0., 'z+1./2', 'y+1./2'],[1./2, 'z', 'y+1./2'],[1./2, 'z+1./2', 'y'],[0., '-z', '-y'],[0., '-z+1./2', '-y+1./2'],[1./2, '-z', '-y+1./2'],[1./2, '-z+1./2', '-y'],[0., '-z', 'y'],[0., '-z+1./2', 'y+1./2'],[1./2, '-z', 'y+1./2'],[1./2, '-z+1./2', 'y'],['z', 'y', 0],['z', 'y+1./2', 1./2],['z+1./2', 'y', 1./2],['z+1./2', 'y+1./2', 0],['z', '-y', 0],['z', '-y+1./2', 1./2],['z+1./2', '-y', 1./2],['z+1./2', '-y+1./2', 0],['-z', 'y', 0],['-z', 'y+1./2', 1./2],['-z+1./2', 'y', 1./2],['-z+1./2', 'y+1./2', 0],['-z', '-y', 0],['-z', '-y+1./2', 1./2],['-z+1./2', '-y', 1./2],['-z+1./2', '-y+1./2', 0]],
                      '96k': [['x', 'x', 'z'],['x', 'x+1./2', 'z+1./2'],['x+1./2', 'x', 'z+1./2'],['x+1./2', 'x+1./2', 'z'],['-x', '-x', 'z'],['-x', '-x+1./2', 'z+1./2'],['-x+1./2', '-x', 'z+1./2'],['-x+1./2', '-x+1./2', 'z'],['-x', 'x', '-z'],['-x', 'x+1./2', '-z+1./2'],['-x+1./2', 'x', '-z+1./2'],['-x+1./2', 'x+1./2', '-z'],['x', '-x', '-z'],['x', '-x+1./2', '-z+1./2'],['x+1./2', '-x', '-z+1./2'],['x+1./2', '-x+1./2', '-z'],['z', 'x', 'x'],['z', 'x+1./2', 'x+1./2'],['z+1./2', 'x', 'x+1./2'],['z+1./2', 'x+1./2', 'x'],['z', '-x', '-x'],['z', '-x+1./2', '-x+1./2'],['z+1./2', '-x', '-x+1./2'],['z+1./2', '-x+1./2', '-x'],['-z', '-x', 'x'],['-z', '-x+1./2', 'x+1./2'],['-z+1./2', '-x', 'x+1./2'],['-z+1./2', '-x+1./2', 'x'],['-z', 'x', '-x'],['-z', 'x+1./2', '-x+1./2'],['-z+1./2', 'x', '-x+1./2'],['-z+1./2', 'x+1./2', '-x'],['x', 'z', 'x'],['x', 'z+1./2', 'x+1./2'],['x+1./2', 'z', 'x+1./2'],['x+1./2', 'z+1./2', 'x'],['-x', 'z', '-x'],['-x', 'z+1./2', '-x+1./2'],['-x+1./2', 'z', '-x+1./2'],['-x+1./2', 'z+1./2', '-x'],['x', '-z', '-x'],['x', '-z+1./2', '-x+1./2'],['x+1./2', '-z', '-x+1./2'],['x+1./2', '-z+1./2', '-x'],['-x', '-z', 'x'],['-x', '-z+1./2', 'x+1./2'],['-x+1./2', '-z', 'x+1./2'],['-x+1./2', '-z+1./2', 'x'],['x', 'x', '-z'],['x', 'x+1./2', '-z+1./2'],['x+1./2', 'x', '-z+1./2'],['x+1./2', 'x+1./2', '-z'],['-x', '-x', '-z'],['-x', '-x+1./2', '-z+1./2'],['-x+1./2', '-x', '-z+1./2'],['-x+1./2', '-x+1./2', '-z'],['x', '-x', 'z'],['x', '-x+1./2', 'z+1./2'],['x+1./2', '-x', 'z+1./2'],['x+1./2', '-x+1./2', 'z'],['-x', 'x', 'z'],['-x', 'x+1./2', 'z+1./2'],['-x+1./2', 'x', 'z+1./2'],['-x+1./2', 'x+1./2', 'z'],['x', 'z', '-x'],['x', 'z+1./2', '-x+1./2'],['x+1./2', 'z', '-x+1./2'],['x+1./2', 'z+1./2', '-x'],['-x', 'z', 'x'],['-x', 'z+1./2', 'x+1./2'],['-x+1./2', 'z', 'x+1./2'],['-x+1./2', 'z+1./2', 'x'],['-x', '-z', '-x'],['-x', '-z+1./2', '-x+1./2'],['-x+1./2', '-z', '-x+1./2'],['-x+1./2', '-z+1./2', '-x'],['x', '-z', 'x'],['x', '-z+1./2', 'x+1./2'],['x+1./2', '-z', 'x+1./2'],['x+1./2', '-z+1./2', 'x'],['z', 'x', '-x'],['z', 'x+1./2', '-x+1./2'],['z+1./2', 'x', '-x+1./2'],['z+1./2', 'x+1./2', '-x'],['z', '-x', 'x'],['z', '-x+1./2', 'x+1./2'],['z+1./2', '-x', 'x+1./2'],['z+1./2', '-x+1./2', 'x'],['-z', 'x', 'x'],['-z', 'x+1./2', 'x+1./2'],['-z+1./2', 'x', 'x+1./2'],['-z+1./2', 'x+1./2', 'x'],['-z', '-x', '-x'],['-z', '-x+1./2', '-x+1./2'],['-z+1./2', '-x', '-x+1./2'],['-z+1./2', '-x+1./2', '-x']],
                      '192l': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y', 'x', '-z'],['y', 'x+1./2', '-z+1./2'],['y+1./2', 'x', '-z+1./2'],['y+1./2', 'x+1./2', '-z'],['-y', '-x', '-z'],['-y', '-x+1./2', '-z+1./2'],['-y+1./2', '-x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z'],['y', '-x', 'z'],['y', '-x+1./2', 'z+1./2'],['y+1./2', '-x', 'z+1./2'],['y+1./2', '-x+1./2', 'z'],['-y', 'x', 'z'],['-y', 'x+1./2', 'z+1./2'],['-y+1./2', 'x', 'z+1./2'],['-y+1./2', 'x+1./2', 'z'],['x', 'z', '-y'],['x', 'z+1./2', '-y+1./2'],['x+1./2', 'z', '-y+1./2'],['x+1./2', 'z+1./2', '-y'],['-x', 'z', 'y'],['-x', 'z+1./2', 'y+1./2'],['-x+1./2', 'z', 'y+1./2'],['-x+1./2', 'z+1./2', 'y'],['-x', '-z', '-y'],['-x', '-z+1./2', '-y+1./2'],['-x+1./2', '-z', '-y+1./2'],['-x+1./2', '-z+1./2', '-y'],['x', '-z', 'y'],['x', '-z+1./2', 'y+1./2'],['x+1./2', '-z', 'y+1./2'],['x+1./2', '-z+1./2', 'y'],['z', 'y', '-x'],['z', 'y+1./2', '-x+1./2'],['z+1./2', 'y', '-x+1./2'],['z+1./2', 'y+1./2', '-x'],['z', '-y', 'x'],['z', '-y+1./2', 'x+1./2'],['z+1./2', '-y', 'x+1./2'],['z+1./2', '-y+1./2', 'x'],['-z', 'y', 'x'],['-z', 'y+1./2', 'x+1./2'],['-z+1./2', 'y', 'x+1./2'],['-z+1./2', 'y+1./2', 'x'],['-z', '-y', '-x'],['-z', '-y+1./2', '-x+1./2'],['-z+1./2', '-y', '-x+1./2'],['-z+1./2', '-y+1./2', '-x'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z'],['x', 'y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x', 'y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-z', '-x', '-y'],['-z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x', '-y+1./2'],['-z+1./2', '-x+1./2', '-y'],['-z', 'x', 'y'],['-z', 'x+1./2', 'y+1./2'],['-z+1./2', 'x', 'y+1./2'],['-z+1./2', 'x+1./2', 'y'],['z', 'x', '-y'],['z', 'x+1./2', '-y+1./2'],['z+1./2', 'x', '-y+1./2'],['z+1./2', 'x+1./2', '-y'],['z', '-x', 'y'],['z', '-x+1./2', 'y+1./2'],['z+1./2', '-x', 'y+1./2'],['z+1./2', '-x+1./2', 'y'],['-y', '-z', '-x'],['-y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z', '-x+1./2'],['-y+1./2', '-z+1./2', '-x'],['y', '-z', 'x'],['y', '-z+1./2', 'x+1./2'],['y+1./2', '-z', 'x+1./2'],['y+1./2', '-z+1./2', 'x'],['-y', 'z', 'x'],['-y', 'z+1./2', 'x+1./2'],['-y+1./2', 'z', 'x+1./2'],['-y+1./2', 'z+1./2', 'x'],['y', 'z', '-x'],['y', 'z+1./2', '-x+1./2'],['y+1./2', 'z', '-x+1./2'],['y+1./2', 'z+1./2', '-x'],['-y', '-x', 'z'],['-y', '-x+1./2', 'z+1./2'],['-y+1./2', '-x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['y', 'x', 'z'],['y', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z+1./2'],['y+1./2', 'x+1./2', 'z'],['-y', 'x', '-z'],['-y', 'x+1./2', '-z+1./2'],['-y+1./2', 'x', '-z+1./2'],['-y+1./2', 'x+1./2', '-z'],['y', '-x', '-z'],['y', '-x+1./2', '-z+1./2'],['y+1./2', '-x', '-z+1./2'],['y+1./2', '-x+1./2', '-z'],['-x', '-z', 'y'],['-x', '-z+1./2', 'y+1./2'],['-x+1./2', '-z', 'y+1./2'],['-x+1./2', '-z+1./2', 'y'],['x', '-z', '-y'],['x', '-z+1./2', '-y+1./2'],['x+1./2', '-z', '-y+1./2'],['x+1./2', '-z+1./2', '-y'],['x', 'z', 'y'],['x', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y+1./2'],['x+1./2', 'z+1./2', 'y'],['-x', 'z', '-y'],['-x', 'z+1./2', '-y+1./2'],['-x+1./2', 'z', '-y+1./2'],['-x+1./2', 'z+1./2', '-y'],['-z', '-y', 'x'],['-z', '-y+1./2', 'x+1./2'],['-z+1./2', '-y', 'x+1./2'],['-z+1./2', '-y+1./2', 'x'],['-z', 'y', '-x'],['-z', 'y+1./2', '-x+1./2'],['-z+1./2', 'y', '-x+1./2'],['-z+1./2', 'y+1./2', '-x'],['z', '-y', '-x'],['z', '-y+1./2', '-x+1./2'],['z+1./2', '-y', '-x+1./2'],['z+1./2', '-y+1./2', '-x'],['z', 'y', 'x'],['z', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x+1./2'],['z+1./2', 'y+1./2', 'x']]},
              '226': {'8a': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4]],
                      '8b': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2]],
                      '24c': [[1./4, 0., 0],[1./4, 1./2, 1./2],[3./4, 0., 1./2],[3./4, 1./2, 0],[3./4, 0., 0],[3./4, 1./2, 1./2],[1./4, 0., 1./2],[1./4, 1./2, 0],[0., 1./4, 0],[0., 3./4, 1./2],[1./2, 1./4, 1./2],[1./2, 3./4, 0],[0., 3./4, 0],[0., 1./4, 1./2],[1./2, 3./4, 1./2],[1./2, 1./4, 0],[0., 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4],[1./2, 1./2, 1./4],[0., 0., 3./4],[0., 1./2, 1./4],[1./2, 0., 1./4],[1./2, 1./2, 3./4]],
                      '24d': [[0., 1./4, 1./4],[0., 3./4, 3./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0]],
                      '48e': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 0],['-x', 1./2, 1./2],['-x+1./2', 0., 1./2],['-x+1./2', 1./2, 0],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[0., 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., '-x+1./2'],[1./2, 1./2, '-x'],[1./2, 'x+1./2', 1./2],[1./2, 'x', 0],[0., 'x+1./2', 0],[0., 'x', 1./2],[1./2, '-x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 0],[0., '-x', 1./2],['x+1./2', 1./2, 1./2],['x+1./2', 0., 0],['x', 1./2, 0],['x', 0., 1./2],['-x+1./2', 1./2, 1./2],['-x+1./2', 0., 0],['-x', 1./2, 0],['-x', 0., 1./2],[1./2, 1./2, '-x+1./2'],[1./2, 0., '-x'],[0., 1./2, '-x'],[0., 0., '-x+1./2'],[1./2, 1./2, 'x+1./2'],[1./2, 0., 'x'],[0., 1./2, 'x'],[0., 0., 'x+1./2']],
                      '48f': [['x', 1./4, 1./4],['x', 3./4, 3./4],['x+1./2', 1./4, 3./4],['x+1./2', 3./4, 1./4],['-x', 3./4, 1./4],['-x', 1./4, 3./4],['-x+1./2', 3./4, 3./4],['-x+1./2', 1./4, 1./4],[1./4, 'x', 1./4],[1./4, 'x+1./2', 3./4],[3./4, 'x', 3./4],[3./4, 'x+1./2', 1./4],[1./4, '-x', 3./4],[1./4, '-x+1./2', 1./4],[3./4, '-x', 1./4],[3./4, '-x+1./2', 3./4],[1./4, 1./4, 'x'],[1./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x'],[3./4, 1./4, '-x'],[3./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x'],['-x', 3./4, 3./4],['-x', 1./4, 1./4],['-x+1./2', 3./4, 1./4],['-x+1./2', 1./4, 3./4],['x', 1./4, 3./4],['x', 3./4, 1./4],['x+1./2', 1./4, 1./4],['x+1./2', 3./4, 3./4],[3./4, '-x', 3./4],[3./4, '-x+1./2', 1./4],[1./4, '-x', 1./4],[1./4, '-x+1./2', 3./4],[3./4, 'x', 1./4],[3./4, 'x+1./2', 3./4],[1./4, 'x', 3./4],[1./4, 'x+1./2', 1./4],[3./4, 3./4, '-x'],[3./4, 1./4, '-x+1./2'],[1./4, 3./4, '-x+1./2'],[1./4, 1./4, '-x'],[1./4, 3./4, 'x'],[1./4, 1./4, 'x+1./2'],[3./4, 3./4, 'x+1./2'],[3./4, 1./4, 'x']],
                      '64g': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x'],['-x', 'x', '-x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x+1./2'],['-x+1./2', 'x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['x+1./2', 'x', '-x'],['x', 'x+1./2', '-x'],['x', 'x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x'],['-x', '-x+1./2', '-x'],['-x', '-x', '-x+1./2'],['x+1./2', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x'],['x', '-x+1./2', 'x'],['x', '-x', 'x+1./2'],['-x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', 'x', 'x'],['-x', 'x+1./2', 'x'],['-x', 'x', 'x+1./2'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x', 'x', '-x'],['x', 'x+1./2', '-x+1./2'],['x+1./2', 'x', '-x+1./2'],['x+1./2', 'x+1./2', '-x'],['x', '-x', 'x'],['x', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x+1./2'],['x+1./2', '-x+1./2', 'x'],['-x', 'x', 'x'],['-x', 'x+1./2', 'x+1./2'],['-x+1./2', 'x', 'x+1./2'],['-x+1./2', 'x+1./2', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x'],['-x', '-x+1./2', 'x'],['-x', '-x', 'x+1./2'],['x+1./2', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x'],['x', 'x+1./2', 'x'],['x', 'x', 'x+1./2'],['-x+1./2', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['-x', 'x+1./2', '-x'],['-x', 'x', '-x+1./2'],['x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', '-x', '-x'],['x', '-x+1./2', '-x'],['x', '-x', '-x+1./2']],
                      '96h': [[1./4, 'y', 'y'],[1./4, 'y+1./2', 'y+1./2'],[3./4, 'y', 'y+1./2'],[3./4, 'y+1./2', 'y'],[3./4, '-y', 'y'],[3./4, '-y+1./2', 'y+1./2'],[1./4, '-y', 'y+1./2'],[1./4, '-y+1./2', 'y'],[3./4, 'y', '-y'],[3./4, 'y+1./2', '-y+1./2'],[1./4, 'y', '-y+1./2'],[1./4, 'y+1./2', '-y'],[1./4, '-y', '-y'],[1./4, '-y+1./2', '-y+1./2'],[3./4, '-y', '-y+1./2'],[3./4, '-y+1./2', '-y'],['y', 1./4, 'y'],['y', 3./4, 'y+1./2'],['y+1./2', 1./4, 'y+1./2'],['y+1./2', 3./4, 'y'],['y', 3./4, '-y'],['y', 1./4, '-y+1./2'],['y+1./2', 3./4, '-y+1./2'],['y+1./2', 1./4, '-y'],['-y', 3./4, 'y'],['-y', 1./4, 'y+1./2'],['-y+1./2', 3./4, 'y+1./2'],['-y+1./2', 1./4, 'y'],['-y', 1./4, '-y'],['-y', 3./4, '-y+1./2'],['-y+1./2', 1./4, '-y+1./2'],['-y+1./2', 3./4, '-y'],['y', 'y', 1./4],['y', 'y+1./2', 3./4],['y+1./2', 'y', 3./4],['y+1./2', 'y+1./2', 1./4],['-y', 'y', 3./4],['-y', 'y+1./2', 1./4],['-y+1./2', 'y', 1./4],['-y+1./2', 'y+1./2', 3./4],['y', '-y', 3./4],['y', '-y+1./2', 1./4],['y+1./2', '-y', 1./4],['y+1./2', '-y+1./2', 3./4],['-y', '-y', 1./4],['-y', '-y+1./2', 3./4],['-y+1./2', '-y', 3./4],['-y+1./2', '-y+1./2', 1./4],[3./4, '-y', '-y'],[3./4, '-y+1./2', '-y+1./2'],[1./4, '-y', '-y+1./2'],[1./4, '-y+1./2', '-y'],[1./4, 'y', '-y'],[1./4, 'y+1./2', '-y+1./2'],[3./4, 'y', '-y+1./2'],[3./4, 'y+1./2', '-y'],[1./4, '-y', 'y'],[1./4, '-y+1./2', 'y+1./2'],[3./4, '-y', 'y+1./2'],[3./4, '-y+1./2', 'y'],[3./4, 'y', 'y'],[3./4, 'y+1./2', 'y+1./2'],[1./4, 'y', 'y+1./2'],[1./4, 'y+1./2', 'y'],['-y', 3./4, '-y'],['-y', 1./4, '-y+1./2'],['-y+1./2', 3./4, '-y+1./2'],['-y+1./2', 1./4, '-y'],['-y', 1./4, 'y'],['-y', 3./4, 'y+1./2'],['-y+1./2', 1./4, 'y+1./2'],['-y+1./2', 3./4, 'y'],['y', 1./4, '-y'],['y', 3./4, '-y+1./2'],['y+1./2', 1./4, '-y+1./2'],['y+1./2', 3./4, '-y'],['y', 3./4, 'y'],['y', 1./4, 'y+1./2'],['y+1./2', 3./4, 'y+1./2'],['y+1./2', 1./4, 'y'],['-y', '-y', 3./4],['-y', '-y+1./2', 1./4],['-y+1./2', '-y', 1./4],['-y+1./2', '-y+1./2', 3./4],['y', '-y', 1./4],['y', '-y+1./2', 3./4],['y+1./2', '-y', 3./4],['y+1./2', '-y+1./2', 1./4],['-y', 'y', 1./4],['-y', 'y+1./2', 3./4],['-y+1./2', 'y', 3./4],['-y+1./2', 'y+1./2', 1./4],['y', 'y', 3./4],['y', 'y+1./2', 1./4],['y+1./2', 'y', 1./4],['y+1./2', 'y+1./2', 3./4]],
                      '96i': [[0., 'y', 'z'],[0., 'y+1./2', 'z+1./2'],[1./2, 'y', 'z+1./2'],[1./2, 'y+1./2', 'z'],[0., '-y', 'z'],[0., '-y+1./2', 'z+1./2'],[1./2, '-y', 'z+1./2'],[1./2, '-y+1./2', 'z'],[0., 'y', '-z'],[0., 'y+1./2', '-z+1./2'],[1./2, 'y', '-z+1./2'],[1./2, 'y+1./2', '-z'],[0., '-y', '-z'],[0., '-y+1./2', '-z+1./2'],[1./2, '-y', '-z+1./2'],[1./2, '-y+1./2', '-z'],['z', 0., 'y'],['z', 1./2, 'y+1./2'],['z+1./2', 0., 'y+1./2'],['z+1./2', 1./2, 'y'],['z', 0., '-y'],['z', 1./2, '-y+1./2'],['z+1./2', 0., '-y+1./2'],['z+1./2', 1./2, '-y'],['-z', 0., 'y'],['-z', 1./2, 'y+1./2'],['-z+1./2', 0., 'y+1./2'],['-z+1./2', 1./2, 'y'],['-z', 0., '-y'],['-z', 1./2, '-y+1./2'],['-z+1./2', 0., '-y+1./2'],['-z+1./2', 1./2, '-y'],['y', 'z', 0],['y', 'z+1./2', 1./2],['y+1./2', 'z', 1./2],['y+1./2', 'z+1./2', 0],['-y', 'z', 0],['-y', 'z+1./2', 1./2],['-y+1./2', 'z', 1./2],['-y+1./2', 'z+1./2', 0],['y', '-z', 0],['y', '-z+1./2', 1./2],['y+1./2', '-z', 1./2],['y+1./2', '-z+1./2', 0],['-y', '-z', 0],['-y', '-z+1./2', 1./2],['-y+1./2', '-z', 1./2],['-y+1./2', '-z+1./2', 0],['y+1./2', 1./2, '-z+1./2'],['y+1./2', 0., '-z'],['y', 1./2, '-z'],['y', 0., '-z+1./2'],['-y+1./2', 1./2, '-z+1./2'],['-y+1./2', 0., '-z'],['-y', 1./2, '-z'],['-y', 0., '-z+1./2'],['y+1./2', 1./2, 'z+1./2'],['y+1./2', 0., 'z'],['y', 1./2, 'z'],['y', 0., 'z+1./2'],['-y+1./2', 1./2, 'z+1./2'],['-y+1./2', 0., 'z'],['-y', 1./2, 'z'],['-y', 0., 'z+1./2'],[1./2, 'z+1./2', '-y+1./2'],[1./2, 'z', '-y'],[0., 'z+1./2', '-y'],[0., 'z', '-y+1./2'],[1./2, 'z+1./2', 'y+1./2'],[1./2, 'z', 'y'],[0., 'z+1./2', 'y'],[0., 'z', 'y+1./2'],[1./2, '-z+1./2', '-y+1./2'],[1./2, '-z', '-y'],[0., '-z+1./2', '-y'],[0., '-z', '-y+1./2'],[1./2, '-z+1./2', 'y+1./2'],[1./2, '-z', 'y'],[0., '-z+1./2', 'y'],[0., '-z', 'y+1./2'],['z+1./2', 'y+1./2', 1./2],['z+1./2', 'y', 0],['z', 'y+1./2', 0],['z', 'y', 1./2],['z+1./2', '-y+1./2', 1./2],['z+1./2', '-y', 0],['z', '-y+1./2', 0],['z', '-y', 1./2],['-z+1./2', 'y+1./2', 1./2],['-z+1./2', 'y', 0],['-z', 'y+1./2', 0],['-z', 'y', 1./2],['-z+1./2', '-y+1./2', 1./2],['-z+1./2', '-y', 0],['-z', '-y+1./2', 0],['-z', '-y', 1./2]],
                      '192j': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', '-y+1./2', 'z'],['-x', 'y', '-z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z+1./2'],['-x+1./2', 'y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['-z', '-x', 'y'],['-z', '-x+1./2', 'y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', '-x+1./2', 'y'],['-z', 'x', '-y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y+1./2'],['-z+1./2', 'x+1./2', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y', 'z', '-x'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x+1./2'],['-y+1./2', 'z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['-y', '-z', 'x'],['-y', '-z+1./2', 'x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y+1./2', '-z+1./2', 'x'],['y+1./2', 'x+1./2', '-z+1./2'],['y+1./2', 'x', '-z'],['y', 'x+1./2', '-z'],['y', 'x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', '-x', '-z'],['-y', '-x+1./2', '-z'],['-y', '-x', '-z+1./2'],['y+1./2', '-x+1./2', 'z+1./2'],['y+1./2', '-x', 'z'],['y', '-x+1./2', 'z'],['y', '-x', 'z+1./2'],['-y+1./2', 'x+1./2', 'z+1./2'],['-y+1./2', 'x', 'z'],['-y', 'x+1./2', 'z'],['-y', 'x', 'z+1./2'],['x+1./2', 'z+1./2', '-y+1./2'],['x+1./2', 'z', '-y'],['x', 'z+1./2', '-y'],['x', 'z', '-y+1./2'],['-x+1./2', 'z+1./2', 'y+1./2'],['-x+1./2', 'z', 'y'],['-x', 'z+1./2', 'y'],['-x', 'z', 'y+1./2'],['-x+1./2', '-z+1./2', '-y+1./2'],['-x+1./2', '-z', '-y'],['-x', '-z+1./2', '-y'],['-x', '-z', '-y+1./2'],['x+1./2', '-z+1./2', 'y+1./2'],['x+1./2', '-z', 'y'],['x', '-z+1./2', 'y'],['x', '-z', 'y+1./2'],['z+1./2', 'y+1./2', '-x+1./2'],['z+1./2', 'y', '-x'],['z', 'y+1./2', '-x'],['z', 'y', '-x+1./2'],['z+1./2', '-y+1./2', 'x+1./2'],['z+1./2', '-y', 'x'],['z', '-y+1./2', 'x'],['z', '-y', 'x+1./2'],['-z+1./2', 'y+1./2', 'x+1./2'],['-z+1./2', 'y', 'x'],['-z', 'y+1./2', 'x'],['-z', 'y', 'x+1./2'],['-z+1./2', '-y+1./2', '-x+1./2'],['-z+1./2', '-y', '-x'],['-z', '-y+1./2', '-x'],['-z', '-y', '-x+1./2'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x', 'y', '-z'],['x', 'y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x+1./2', 'y+1./2', '-z'],['x', '-y', 'z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z+1./2'],['x+1./2', '-y+1./2', 'z'],['-x', 'y', 'z'],['-x', 'y+1./2', 'z+1./2'],['-x+1./2', 'y', 'z+1./2'],['-x+1./2', 'y+1./2', 'z'],['-z', '-x', '-y'],['-z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x', '-y+1./2'],['-z+1./2', '-x+1./2', '-y'],['-z', 'x', 'y'],['-z', 'x+1./2', 'y+1./2'],['-z+1./2', 'x', 'y+1./2'],['-z+1./2', 'x+1./2', 'y'],['z', 'x', '-y'],['z', 'x+1./2', '-y+1./2'],['z+1./2', 'x', '-y+1./2'],['z+1./2', 'x+1./2', '-y'],['z', '-x', 'y'],['z', '-x+1./2', 'y+1./2'],['z+1./2', '-x', 'y+1./2'],['z+1./2', '-x+1./2', 'y'],['-y', '-z', '-x'],['-y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z', '-x+1./2'],['-y+1./2', '-z+1./2', '-x'],['y', '-z', 'x'],['y', '-z+1./2', 'x+1./2'],['y+1./2', '-z', 'x+1./2'],['y+1./2', '-z+1./2', 'x'],['-y', 'z', 'x'],['-y', 'z+1./2', 'x+1./2'],['-y+1./2', 'z', 'x+1./2'],['-y+1./2', 'z+1./2', 'x'],['y', 'z', '-x'],['y', 'z+1./2', '-x+1./2'],['y+1./2', 'z', '-x+1./2'],['y+1./2', 'z+1./2', '-x'],['-y+1./2', '-x+1./2', 'z+1./2'],['-y+1./2', '-x', 'z'],['-y', '-x+1./2', 'z'],['-y', '-x', 'z+1./2'],['y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z'],['y', 'x+1./2', 'z'],['y', 'x', 'z+1./2'],['-y+1./2', 'x+1./2', '-z+1./2'],['-y+1./2', 'x', '-z'],['-y', 'x+1./2', '-z'],['-y', 'x', '-z+1./2'],['y+1./2', '-x+1./2', '-z+1./2'],['y+1./2', '-x', '-z'],['y', '-x+1./2', '-z'],['y', '-x', '-z+1./2'],['-x+1./2', '-z+1./2', 'y+1./2'],['-x+1./2', '-z', 'y'],['-x', '-z+1./2', 'y'],['-x', '-z', 'y+1./2'],['x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', '-z', '-y'],['x', '-z+1./2', '-y'],['x', '-z', '-y+1./2'],['x+1./2', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y'],['x', 'z+1./2', 'y'],['x', 'z', 'y+1./2'],['-x+1./2', 'z+1./2', '-y+1./2'],['-x+1./2', 'z', '-y'],['-x', 'z+1./2', '-y'],['-x', 'z', '-y+1./2'],['-z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', '-y', 'x'],['-z', '-y+1./2', 'x'],['-z', '-y', 'x+1./2'],['-z+1./2', 'y+1./2', '-x+1./2'],['-z+1./2', 'y', '-x'],['-z', 'y+1./2', '-x'],['-z', 'y', '-x+1./2'],['z+1./2', '-y+1./2', '-x+1./2'],['z+1./2', '-y', '-x'],['z', '-y+1./2', '-x'],['z', '-y', '-x+1./2'],['z+1./2', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x'],['z', 'y+1./2', 'x'],['z', 'y', 'x+1./2']]},
              '227:1': {'8a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 1./4],[1./4, 3./4, 3./4]],
                        '8b': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                        '16c': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 3/8, 5/8],[7/8, 7/8, 1/8],[3/8, 3/8, 1/8],[3/8, 7/8, 5/8],[3/8, 5/8, 7/8],[3/8, 1/8, 3/8],[7/8, 5/8, 3/8],[7/8, 1/8, 7/8],[5/8, 7/8, 3/8],[5/8, 3/8, 7/8],[1/8, 7/8, 7/8],[1/8, 3/8, 3/8]],
                        '16d': [[5/8, 5/8, 5/8],[5/8, 1/8, 1/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[3/8, 7/8, 1/8],[3/8, 3/8, 5/8],[7/8, 7/8, 5/8],[7/8, 3/8, 1/8],[7/8, 1/8, 3/8],[7/8, 5/8, 7/8],[3/8, 1/8, 7/8],[3/8, 5/8, 3/8],[1/8, 3/8, 7/8],[1/8, 7/8, 3/8],[5/8, 3/8, 3/8],[5/8, 7/8, 7/8]],
                        '32e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', 'x+1./2', '-x'],['-x+1./2', 'x', '-x+1./2'],['-x', 'x+1./2', '-x+1./2'],['-x', 'x', '-x'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+3./4', 'x+1./4', '-x+3./4'],['x+3./4', 'x+3./4', '-x+1./4'],['x+1./4', 'x+1./4', '-x+1./4'],['x+1./4', 'x+3./4', '-x+3./4'],['-x+1./4', '-x+1./4', '-x+1./4'],['-x+1./4', '-x+3./4', '-x+3./4'],['-x+3./4', '-x+1./4', '-x+3./4'],['-x+3./4', '-x+3./4', '-x+1./4'],['x+1./4', '-x+3./4', 'x+3./4'],['x+1./4', '-x+1./4', 'x+1./4'],['x+3./4', '-x+3./4', 'x+1./4'],['x+3./4', '-x+1./4', 'x+3./4'],['-x+3./4', 'x+3./4', 'x+1./4'],['-x+3./4', 'x+1./4', 'x+3./4'],['-x+1./4', 'x+3./4', 'x+3./4'],['-x+1./4', 'x+1./4', 'x+1./4']],
                        '48f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 0],['-x+1./2', 0., 1./2],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[1./2, 1./2, '-x'],[1./2, 0., '-x+1./2'],[0., 1./2, '-x+1./2'],[0., 0., '-x'],[3./4, 'x+1./4', 3./4],[3./4, 'x+3./4', 1./4],[1./4, 'x+1./4', 1./4],[1./4, 'x+3./4', 3./4],[1./4, '-x+1./4', 1./4],[1./4, '-x+3./4', 3./4],[3./4, '-x+1./4', 3./4],[3./4, '-x+3./4', 1./4],['x+3./4', 1./4, 3./4],['x+3./4', 3./4, 1./4],['x+1./4', 1./4, 1./4],['x+1./4', 3./4, 3./4],['-x+3./4', 3./4, 1./4],['-x+3./4', 1./4, 3./4],['-x+1./4', 3./4, 3./4],['-x+1./4', 1./4, 1./4],[3./4, 1./4, '-x+3./4'],[3./4, 3./4, '-x+1./4'],[1./4, 1./4, '-x+1./4'],[1./4, 3./4, '-x+3./4'],[1./4, 3./4, 'x+3./4'],[1./4, 1./4, 'x+1./4'],[3./4, 3./4, 'x+1./4'],[3./4, 1./4, 'x+3./4']],
                        '96g': [['x', 'x', 'z'],['x', 'x+1./2', 'z+1./2'],['x+1./2', 'x', 'z+1./2'],['x+1./2', 'x+1./2', 'z'],['-x', '-x+1./2', 'z+1./2'],['-x', '-x', 'z'],['-x+1./2', '-x+1./2', 'z'],['-x+1./2', '-x', 'z+1./2'],['-x+1./2', 'x+1./2', '-z'],['-x+1./2', 'x', '-z+1./2'],['-x', 'x+1./2', '-z+1./2'],['-x', 'x', '-z'],['x+1./2', '-x', '-z+1./2'],['x+1./2', '-x+1./2', '-z'],['x', '-x', '-z'],['x', '-x+1./2', '-z+1./2'],['z', 'x', 'x'],['z', 'x+1./2', 'x+1./2'],['z+1./2', 'x', 'x+1./2'],['z+1./2', 'x+1./2', 'x'],['z+1./2', '-x', '-x+1./2'],['z+1./2', '-x+1./2', '-x'],['z', '-x', '-x'],['z', '-x+1./2', '-x+1./2'],['-z', '-x+1./2', 'x+1./2'],['-z', '-x', 'x'],['-z+1./2', '-x+1./2', 'x'],['-z+1./2', '-x', 'x+1./2'],['-z+1./2', 'x+1./2', '-x'],['-z+1./2', 'x', '-x+1./2'],['-z', 'x+1./2', '-x+1./2'],['-z', 'x', '-x'],['x', 'z', 'x'],['x', 'z+1./2', 'x+1./2'],['x+1./2', 'z', 'x+1./2'],['x+1./2', 'z+1./2', 'x'],['-x+1./2', 'z+1./2', '-x'],['-x+1./2', 'z', '-x+1./2'],['-x', 'z+1./2', '-x+1./2'],['-x', 'z', '-x'],['x+1./2', '-z', '-x+1./2'],['x+1./2', '-z+1./2', '-x'],['x', '-z', '-x'],['x', '-z+1./2', '-x+1./2'],['-x', '-z+1./2', 'x+1./2'],['-x', '-z', 'x'],['-x+1./2', '-z+1./2', 'x'],['-x+1./2', '-z', 'x+1./2'],['x+3./4', 'x+1./4', '-z+3./4'],['x+3./4', 'x+3./4', '-z+1./4'],['x+1./4', 'x+1./4', '-z+1./4'],['x+1./4', 'x+3./4', '-z+3./4'],['-x+1./4', '-x+1./4', '-z+1./4'],['-x+1./4', '-x+3./4', '-z+3./4'],['-x+3./4', '-x+1./4', '-z+3./4'],['-x+3./4', '-x+3./4', '-z+1./4'],['x+1./4', '-x+3./4', 'z+3./4'],['x+1./4', '-x+1./4', 'z+1./4'],['x+3./4', '-x+3./4', 'z+1./4'],['x+3./4', '-x+1./4', 'z+3./4'],['-x+3./4', 'x+3./4', 'z+1./4'],['-x+3./4', 'x+1./4', 'z+3./4'],['-x+1./4', 'x+3./4', 'z+3./4'],['-x+1./4', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-x+3./4'],['x+3./4', 'z+3./4', '-x+1./4'],['x+1./4', 'z+1./4', '-x+1./4'],['x+1./4', 'z+3./4', '-x+3./4'],['-x+3./4', 'z+3./4', 'x+1./4'],['-x+3./4', 'z+1./4', 'x+3./4'],['-x+1./4', 'z+3./4', 'x+3./4'],['-x+1./4', 'z+1./4', 'x+1./4'],['-x+1./4', '-z+1./4', '-x+1./4'],['-x+1./4', '-z+3./4', '-x+3./4'],['-x+3./4', '-z+1./4', '-x+3./4'],['-x+3./4', '-z+3./4', '-x+1./4'],['x+1./4', '-z+3./4', 'x+3./4'],['x+1./4', '-z+1./4', 'x+1./4'],['x+3./4', '-z+3./4', 'x+1./4'],['x+3./4', '-z+1./4', 'x+3./4'],['z+3./4', 'x+1./4', '-x+3./4'],['z+3./4', 'x+3./4', '-x+1./4'],['z+1./4', 'x+1./4', '-x+1./4'],['z+1./4', 'x+3./4', '-x+3./4'],['z+1./4', '-x+3./4', 'x+3./4'],['z+1./4', '-x+1./4', 'x+1./4'],['z+3./4', '-x+3./4', 'x+1./4'],['z+3./4', '-x+1./4', 'x+3./4'],['-z+3./4', 'x+3./4', 'x+1./4'],['-z+3./4', 'x+1./4', 'x+3./4'],['-z+1./4', 'x+3./4', 'x+3./4'],['-z+1./4', 'x+1./4', 'x+1./4'],['-z+1./4', '-x+1./4', '-x+1./4'],['-z+1./4', '-x+3./4', '-x+3./4'],['-z+3./4', '-x+1./4', '-x+3./4'],['-z+3./4', '-x+3./4', '-x+1./4']],
                        '96h': [[1/8, 'y', '-y+1./4'],[1/8, 'y+1./2', '-y+3./4'],[5/8, 'y', '-y+3./4'],[5/8, 'y+1./2', '-y+1./4'],[7/8, '-y+1./2', '-y+3./4'],[7/8, '-y', '-y+1./4'],[3/8, '-y+1./2', '-y+1./4'],[3/8, '-y', '-y+3./4'],[3/8, 'y+1./2', 'y+3./4'],[3/8, 'y', 'y+1./4'],[7/8, 'y+1./2', 'y+1./4'],[7/8, 'y', 'y+3./4'],[5/8, '-y', 'y+1./4'],[5/8, '-y+1./2', 'y+3./4'],[1/8, '-y', 'y+3./4'],[1/8, '-y+1./2', 'y+1./4'],['-y+1./4', 1/8, 'y'],['-y+1./4', 5/8, 'y+1./2'],['-y+3./4', 1/8, 'y+1./2'],['-y+3./4', 5/8, 'y'],['-y+3./4', 7/8, '-y+1./2'],['-y+3./4', 3/8, '-y'],['-y+1./4', 7/8, '-y'],['-y+1./4', 3/8, '-y+1./2'],['y+3./4', 3/8, 'y+1./2'],['y+3./4', 7/8, 'y'],['y+1./4', 3/8, 'y'],['y+1./4', 7/8, 'y+1./2'],['y+1./4', 5/8, '-y'],['y+1./4', 1/8, '-y+1./2'],['y+3./4', 5/8, '-y+1./2'],['y+3./4', 1/8, '-y'],['y', '-y+1./4', 1/8],['y', '-y+3./4', 5/8],['y+1./2', '-y+1./4', 5/8],['y+1./2', '-y+3./4', 1/8],['-y+1./2', '-y+3./4', 7/8],['-y+1./2', '-y+1./4', 3/8],['-y', '-y+3./4', 3/8],['-y', '-y+1./4', 7/8],['y+1./2', 'y+3./4', 3/8],['y+1./2', 'y+1./4', 7/8],['y', 'y+3./4', 7/8],['y', 'y+1./4', 3/8],['-y', 'y+1./4', 5/8],['-y', 'y+3./4', 1/8],['-y+1./2', 'y+1./4', 1/8],['-y+1./2', 'y+3./4', 5/8],[1/8, '-y+1./4', 'y'],[1/8, '-y+3./4', 'y+1./2'],[5/8, '-y+1./4', 'y+1./2'],[5/8, '-y+3./4', 'y'],[3/8, 'y+3./4', 'y+1./2'],[3/8, 'y+1./4', 'y'],[7/8, 'y+3./4', 'y'],[7/8, 'y+1./4', 'y+1./2'],[7/8, '-y+3./4', '-y+1./2'],[7/8, '-y+1./4', '-y'],[3/8, '-y+3./4', '-y'],[3/8, '-y+1./4', '-y+1./2'],[5/8, 'y+1./4', '-y'],[5/8, 'y+3./4', '-y+1./2'],[1/8, 'y+1./4', '-y+1./2'],[1/8, 'y+3./4', '-y'],['y', 1/8, '-y+1./4'],['y', 5/8, '-y+3./4'],['y+1./2', 1/8, '-y+3./4'],['y+1./2', 5/8, '-y+1./4'],['y+1./2', 3/8, 'y+3./4'],['y+1./2', 7/8, 'y+1./4'],['y', 3/8, 'y+1./4'],['y', 7/8, 'y+3./4'],['-y+1./2', 7/8, '-y+3./4'],['-y+1./2', 3/8, '-y+1./4'],['-y', 7/8, '-y+1./4'],['-y', 3/8, '-y+3./4'],['-y', 5/8, 'y+1./4'],['-y', 1/8, 'y+3./4'],['-y+1./2', 5/8, 'y+3./4'],['-y+1./2', 1/8, 'y+1./4'],['-y+1./4', 'y', 1/8],['-y+1./4', 'y+1./2', 5/8],['-y+3./4', 'y', 5/8],['-y+3./4', 'y+1./2', 1/8],['y+3./4', 'y+1./2', 3/8],['y+3./4', 'y', 7/8],['y+1./4', 'y+1./2', 7/8],['y+1./4', 'y', 3/8],['-y+3./4', '-y+1./2', 7/8],['-y+3./4', '-y', 3/8],['-y+1./4', '-y+1./2', 3/8],['-y+1./4', '-y', 7/8],['y+1./4', '-y', 5/8],['y+1./4', '-y+1./2', 1/8],['y+3./4', '-y', 1/8],['y+3./4', '-y+1./2', 5/8]],
                        '192i': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', '-z'],['-x+1./2', 'y', '-z+1./2'],['-x', 'y+1./2', '-z+1./2'],['-x', 'y', '-z'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['-z', '-x+1./2', 'y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', 'x+1./2', '-y'],['-z+1./2', 'x', '-y+1./2'],['-z', 'x+1./2', '-y+1./2'],['-z', 'x', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y+1./2', 'z+1./2', '-x'],['-y+1./2', 'z', '-x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y', 'z', '-x'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['-y', '-z+1./2', 'x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x'],['-y+1./2', '-z', 'x+1./2'],['y+3./4', 'x+1./4', '-z+3./4'],['y+3./4', 'x+3./4', '-z+1./4'],['y+1./4', 'x+1./4', '-z+1./4'],['y+1./4', 'x+3./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['-y+1./4', '-x+3./4', '-z+3./4'],['-y+3./4', '-x+1./4', '-z+3./4'],['-y+3./4', '-x+3./4', '-z+1./4'],['y+1./4', '-x+3./4', 'z+3./4'],['y+1./4', '-x+1./4', 'z+1./4'],['y+3./4', '-x+3./4', 'z+1./4'],['y+3./4', '-x+1./4', 'z+3./4'],['-y+3./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['-y+1./4', 'x+3./4', 'z+3./4'],['-y+1./4', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-y+3./4'],['x+3./4', 'z+3./4', '-y+1./4'],['x+1./4', 'z+1./4', '-y+1./4'],['x+1./4', 'z+3./4', '-y+3./4'],['-x+3./4', 'z+3./4', 'y+1./4'],['-x+3./4', 'z+1./4', 'y+3./4'],['-x+1./4', 'z+3./4', 'y+3./4'],['-x+1./4', 'z+1./4', 'y+1./4'],['-x+1./4', '-z+1./4', '-y+1./4'],['-x+1./4', '-z+3./4', '-y+3./4'],['-x+3./4', '-z+1./4', '-y+3./4'],['-x+3./4', '-z+3./4', '-y+1./4'],['x+1./4', '-z+3./4', 'y+3./4'],['x+1./4', '-z+1./4', 'y+1./4'],['x+3./4', '-z+3./4', 'y+1./4'],['x+3./4', '-z+1./4', 'y+3./4'],['z+3./4', 'y+1./4', '-x+3./4'],['z+3./4', 'y+3./4', '-x+1./4'],['z+1./4', 'y+1./4', '-x+1./4'],['z+1./4', 'y+3./4', '-x+3./4'],['z+1./4', '-y+3./4', 'x+3./4'],['z+1./4', '-y+1./4', 'x+1./4'],['z+3./4', '-y+3./4', 'x+1./4'],['z+3./4', '-y+1./4', 'x+3./4'],['-z+3./4', 'y+3./4', 'x+1./4'],['-z+3./4', 'y+1./4', 'x+3./4'],['-z+1./4', 'y+3./4', 'x+3./4'],['-z+1./4', 'y+1./4', 'x+1./4'],['-z+1./4', '-y+1./4', '-x+1./4'],['-z+1./4', '-y+3./4', '-x+3./4'],['-z+3./4', '-y+1./4', '-x+3./4'],['-z+3./4', '-y+3./4', '-x+1./4'],['-x+1./4', '-y+1./4', '-z+1./4'],['-x+1./4', '-y+3./4', '-z+3./4'],['-x+3./4', '-y+1./4', '-z+3./4'],['-x+3./4', '-y+3./4', '-z+1./4'],['x+1./4', 'y+3./4', '-z+3./4'],['x+1./4', 'y+1./4', '-z+1./4'],['x+3./4', 'y+3./4', '-z+1./4'],['x+3./4', 'y+1./4', '-z+3./4'],['x+3./4', '-y+3./4', 'z+1./4'],['x+3./4', '-y+1./4', 'z+3./4'],['x+1./4', '-y+3./4', 'z+3./4'],['x+1./4', '-y+1./4', 'z+1./4'],['-x+3./4', 'y+1./4', 'z+3./4'],['-x+3./4', 'y+3./4', 'z+1./4'],['-x+1./4', 'y+1./4', 'z+1./4'],['-x+1./4', 'y+3./4', 'z+3./4'],['-z+1./4', '-x+1./4', '-y+1./4'],['-z+1./4', '-x+3./4', '-y+3./4'],['-z+3./4', '-x+1./4', '-y+3./4'],['-z+3./4', '-x+3./4', '-y+1./4'],['-z+3./4', 'x+1./4', 'y+3./4'],['-z+3./4', 'x+3./4', 'y+1./4'],['-z+1./4', 'x+1./4', 'y+1./4'],['-z+1./4', 'x+3./4', 'y+3./4'],['z+1./4', 'x+3./4', '-y+3./4'],['z+1./4', 'x+1./4', '-y+1./4'],['z+3./4', 'x+3./4', '-y+1./4'],['z+3./4', 'x+1./4', '-y+3./4'],['z+3./4', '-x+3./4', 'y+1./4'],['z+3./4', '-x+1./4', 'y+3./4'],['z+1./4', '-x+3./4', 'y+3./4'],['z+1./4', '-x+1./4', 'y+1./4'],['-y+1./4', '-z+1./4', '-x+1./4'],['-y+1./4', '-z+3./4', '-x+3./4'],['-y+3./4', '-z+1./4', '-x+3./4'],['-y+3./4', '-z+3./4', '-x+1./4'],['y+3./4', '-z+3./4', 'x+1./4'],['y+3./4', '-z+1./4', 'x+3./4'],['y+1./4', '-z+3./4', 'x+3./4'],['y+1./4', '-z+1./4', 'x+1./4'],['-y+3./4', 'z+1./4', 'x+3./4'],['-y+3./4', 'z+3./4', 'x+1./4'],['-y+1./4', 'z+1./4', 'x+1./4'],['-y+1./4', 'z+3./4', 'x+3./4'],['y+1./4', 'z+3./4', '-x+3./4'],['y+1./4', 'z+1./4', '-x+1./4'],['y+3./4', 'z+3./4', '-x+1./4'],['y+3./4', 'z+1./4', '-x+3./4'],['-y+1./2', '-x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z'],['-y', '-x', 'z'],['-y', '-x+1./2', 'z+1./2'],['y', 'x', 'z'],['y', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z+1./2'],['y+1./2', 'x+1./2', 'z'],['-y', 'x+1./2', '-z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z'],['-y+1./2', 'x', '-z+1./2'],['y+1./2', '-x+1./2', '-z'],['y+1./2', '-x', '-z+1./2'],['y', '-x+1./2', '-z+1./2'],['y', '-x', '-z'],['-x+1./2', '-z', 'y+1./2'],['-x+1./2', '-z+1./2', 'y'],['-x', '-z', 'y'],['-x', '-z+1./2', 'y+1./2'],['x+1./2', '-z+1./2', '-y'],['x+1./2', '-z', '-y+1./2'],['x', '-z+1./2', '-y+1./2'],['x', '-z', '-y'],['x', 'z', 'y'],['x', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y+1./2'],['x+1./2', 'z+1./2', 'y'],['-x', 'z+1./2', '-y+1./2'],['-x', 'z', '-y'],['-x+1./2', 'z+1./2', '-y'],['-x+1./2', 'z', '-y+1./2'],['-z+1./2', '-y', 'x+1./2'],['-z+1./2', '-y+1./2', 'x'],['-z', '-y', 'x'],['-z', '-y+1./2', 'x+1./2'],['-z', 'y+1./2', '-x+1./2'],['-z', 'y', '-x'],['-z+1./2', 'y+1./2', '-x'],['-z+1./2', 'y', '-x+1./2'],['z+1./2', '-y+1./2', '-x'],['z+1./2', '-y', '-x+1./2'],['z', '-y+1./2', '-x+1./2'],['z', '-y', '-x'],['z', 'y', 'x'],['z', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x+1./2'],['z+1./2', 'y+1./2', 'x']]},
              '227:2': {'8a': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 3/8, 3/8],[7/8, 7/8, 7/8],[3/8, 3/8, 7/8],[3/8, 7/8, 3/8]],
                        '8b': [[3/8, 3/8, 3/8],[3/8, 7/8, 7/8],[7/8, 3/8, 7/8],[7/8, 7/8, 3/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[5/8, 5/8, 5/8],[5/8, 1/8, 1/8]],
                        '16c': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[1./4, 1./2, 3./4],[1./4, 0., 1./4],[3./4, 1./2, 1./4],[3./4, 0., 3./4],[1./2, 3./4, 1./4],[1./2, 1./4, 3./4],[0., 3./4, 3./4],[0., 1./4, 1./4]],
                        '16d': [[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./4, 3./4, 0],[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 0],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[0., 1./4, 3./4],[0., 3./4, 1./4],[1./2, 1./4, 1./4],[1./2, 3./4, 3./4]],
                        '32e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x+3./4', '-x+1./4', 'x+1./2'],['-x+3./4', '-x+3./4', 'x'],['-x+1./4', '-x+1./4', 'x'],['-x+1./4', '-x+3./4', 'x+1./2'],['-x+1./4', 'x+1./2', '-x+3./4'],['-x+1./4', 'x', '-x+1./4'],['-x+3./4', 'x+1./2', '-x+1./4'],['-x+3./4', 'x', '-x+3./4'],['x+1./2', '-x+3./4', '-x+1./4'],['x+1./2', '-x+1./4', '-x+3./4'],['x', '-x+3./4', '-x+3./4'],['x', '-x+1./4', '-x+1./4'],['x+3./4', 'x+1./4', '-x+1./2'],['x+3./4', 'x+3./4', '-x'],['x+1./4', 'x+1./4', '-x'],['x+1./4', 'x+3./4', '-x+1./2'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x+1./4', '-x+1./2', 'x+3./4'],['x+1./4', '-x', 'x+1./4'],['x+3./4', '-x+1./2', 'x+1./4'],['x+3./4', '-x', 'x+3./4'],['-x+1./2', 'x+3./4', 'x+1./4'],['-x+1./2', 'x+1./4', 'x+3./4'],['-x', 'x+3./4', 'x+3./4'],['-x', 'x+1./4', 'x+1./4']],
                        '48f': [['x', 1/8, 1/8],['x', 5/8, 5/8],['x+1./2', 1/8, 5/8],['x+1./2', 5/8, 1/8],['-x+3./4', 1/8, 5/8],['-x+3./4', 5/8, 1/8],['-x+1./4', 1/8, 1/8],['-x+1./4', 5/8, 5/8],[1/8, 'x', 1/8],[1/8, 'x+1./2', 5/8],[5/8, 'x', 5/8],[5/8, 'x+1./2', 1/8],[5/8, '-x+3./4', 1/8],[5/8, '-x+1./4', 5/8],[1/8, '-x+3./4', 5/8],[1/8, '-x+1./4', 1/8],[1/8, 1/8, 'x'],[1/8, 5/8, 'x+1./2'],[5/8, 1/8, 'x+1./2'],[5/8, 5/8, 'x'],[1/8, 5/8, '-x+3./4'],[1/8, 1/8, '-x+1./4'],[5/8, 5/8, '-x+1./4'],[5/8, 1/8, '-x+3./4'],[7/8, 'x+1./4', 3/8],[7/8, 'x+3./4', 7/8],[3/8, 'x+1./4', 7/8],[3/8, 'x+3./4', 3/8],[7/8, '-x', 7/8],[7/8, '-x+1./2', 3/8],[3/8, '-x', 3/8],[3/8, '-x+1./2', 7/8],['x+3./4', 3/8, 3/8],['x+3./4', 7/8, 7/8],['x+1./4', 3/8, 7/8],['x+1./4', 7/8, 3/8],['-x+1./2', 7/8, 3/8],['-x+1./2', 3/8, 7/8],['-x', 7/8, 7/8],['-x', 3/8, 3/8],[7/8, 3/8, '-x+1./2'],[7/8, 7/8, '-x'],[3/8, 3/8, '-x'],[3/8, 7/8, '-x+1./2'],[3/8, 3/8, 'x+3./4'],[3/8, 7/8, 'x+1./4'],[7/8, 3/8, 'x+1./4'],[7/8, 7/8, 'x+3./4']],
                        '96g': [['x', 'x', 'z'],['x', 'x+1./2', 'z+1./2'],['x+1./2', 'x', 'z+1./2'],['x+1./2', 'x+1./2', 'z'],['-x+3./4', '-x+1./4', 'z+1./2'],['-x+3./4', '-x+3./4', 'z'],['-x+1./4', '-x+1./4', 'z'],['-x+1./4', '-x+3./4', 'z+1./2'],['-x+1./4', 'x+1./2', '-z+3./4'],['-x+1./4', 'x', '-z+1./4'],['-x+3./4', 'x+1./2', '-z+1./4'],['-x+3./4', 'x', '-z+3./4'],['x+1./2', '-x+3./4', '-z+1./4'],['x+1./2', '-x+1./4', '-z+3./4'],['x', '-x+3./4', '-z+3./4'],['x', '-x+1./4', '-z+1./4'],['z', 'x', 'x'],['z', 'x+1./2', 'x+1./2'],['z+1./2', 'x', 'x+1./2'],['z+1./2', 'x+1./2', 'x'],['z+1./2', '-x+3./4', '-x+1./4'],['z+1./2', '-x+1./4', '-x+3./4'],['z', '-x+3./4', '-x+3./4'],['z', '-x+1./4', '-x+1./4'],['-z+3./4', '-x+1./4', 'x+1./2'],['-z+3./4', '-x+3./4', 'x'],['-z+1./4', '-x+1./4', 'x'],['-z+1./4', '-x+3./4', 'x+1./2'],['-z+1./4', 'x+1./2', '-x+3./4'],['-z+1./4', 'x', '-x+1./4'],['-z+3./4', 'x+1./2', '-x+1./4'],['-z+3./4', 'x', '-x+3./4'],['x', 'z', 'x'],['x', 'z+1./2', 'x+1./2'],['x+1./2', 'z', 'x+1./2'],['x+1./2', 'z+1./2', 'x'],['-x+1./4', 'z+1./2', '-x+3./4'],['-x+1./4', 'z', '-x+1./4'],['-x+3./4', 'z+1./2', '-x+1./4'],['-x+3./4', 'z', '-x+3./4'],['x+1./2', '-z+3./4', '-x+1./4'],['x+1./2', '-z+1./4', '-x+3./4'],['x', '-z+3./4', '-x+3./4'],['x', '-z+1./4', '-x+1./4'],['-x+3./4', '-z+1./4', 'x+1./2'],['-x+3./4', '-z+3./4', 'x'],['-x+1./4', '-z+1./4', 'x'],['-x+1./4', '-z+3./4', 'x+1./2'],['x+3./4', 'x+1./4', '-z+1./2'],['x+3./4', 'x+3./4', '-z'],['x+1./4', 'x+1./4', '-z'],['x+1./4', 'x+3./4', '-z+1./2'],['-x', '-x', '-z'],['-x', '-x+1./2', '-z+1./2'],['-x+1./2', '-x', '-z+1./2'],['-x+1./2', '-x+1./2', '-z'],['x+1./4', '-x+1./2', 'z+3./4'],['x+1./4', '-x', 'z+1./4'],['x+3./4', '-x+1./2', 'z+1./4'],['x+3./4', '-x', 'z+3./4'],['-x+1./2', 'x+3./4', 'z+1./4'],['-x+1./2', 'x+1./4', 'z+3./4'],['-x', 'x+3./4', 'z+3./4'],['-x', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-x+1./2'],['x+3./4', 'z+3./4', '-x'],['x+1./4', 'z+1./4', '-x'],['x+1./4', 'z+3./4', '-x+1./2'],['-x+1./2', 'z+3./4', 'x+1./4'],['-x+1./2', 'z+1./4', 'x+3./4'],['-x', 'z+3./4', 'x+3./4'],['-x', 'z+1./4', 'x+1./4'],['-x', '-z', '-x'],['-x', '-z+1./2', '-x+1./2'],['-x+1./2', '-z', '-x+1./2'],['-x+1./2', '-z+1./2', '-x'],['x+1./4', '-z+1./2', 'x+3./4'],['x+1./4', '-z', 'x+1./4'],['x+3./4', '-z+1./2', 'x+1./4'],['x+3./4', '-z', 'x+3./4'],['z+3./4', 'x+1./4', '-x+1./2'],['z+3./4', 'x+3./4', '-x'],['z+1./4', 'x+1./4', '-x'],['z+1./4', 'x+3./4', '-x+1./2'],['z+1./4', '-x+1./2', 'x+3./4'],['z+1./4', '-x', 'x+1./4'],['z+3./4', '-x+1./2', 'x+1./4'],['z+3./4', '-x', 'x+3./4'],['-z+1./2', 'x+3./4', 'x+1./4'],['-z+1./2', 'x+1./4', 'x+3./4'],['-z', 'x+3./4', 'x+3./4'],['-z', 'x+1./4', 'x+1./4'],['-z', '-x', '-x'],['-z', '-x+1./2', '-x+1./2'],['-z+1./2', '-x', '-x+1./2'],['-z+1./2', '-x+1./2', '-x']],
                        '96h': [[0., 'y', '-y'],[0., 'y+1./2', '-y+1./2'],[1./2, 'y', '-y+1./2'],[1./2, 'y+1./2', '-y'],[3./4, '-y+1./4', '-y+1./2'],[3./4, '-y+3./4', '-y'],[1./4, '-y+1./4', '-y'],[1./4, '-y+3./4', '-y+1./2'],[1./4, 'y+1./2', 'y+3./4'],[1./4, 'y', 'y+1./4'],[3./4, 'y+1./2', 'y+1./4'],[3./4, 'y', 'y+3./4'],[1./2, '-y+3./4', 'y+1./4'],[1./2, '-y+1./4', 'y+3./4'],[0., '-y+3./4', 'y+3./4'],[0., '-y+1./4', 'y+1./4'],['-y', 0., 'y'],['-y', 1./2, 'y+1./2'],['-y+1./2', 0., 'y+1./2'],['-y+1./2', 1./2, 'y'],['-y+1./2', 3./4, '-y+1./4'],['-y+1./2', 1./4, '-y+3./4'],['-y', 3./4, '-y+3./4'],['-y', 1./4, '-y+1./4'],['y+3./4', 1./4, 'y+1./2'],['y+3./4', 3./4, 'y'],['y+1./4', 1./4, 'y'],['y+1./4', 3./4, 'y+1./2'],['y+1./4', 1./2, '-y+3./4'],['y+1./4', 0., '-y+1./4'],['y+3./4', 1./2, '-y+1./4'],['y+3./4', 0., '-y+3./4'],['y', '-y', 0],['y', '-y+1./2', 1./2],['y+1./2', '-y', 1./2],['y+1./2', '-y+1./2', 0],['-y+1./4', '-y+1./2', 3./4],['-y+1./4', '-y', 1./4],['-y+3./4', '-y+1./2', 1./4],['-y+3./4', '-y', 3./4],['y+1./2', 'y+3./4', 1./4],['y+1./2', 'y+1./4', 3./4],['y', 'y+3./4', 3./4],['y', 'y+1./4', 1./4],['-y+3./4', 'y+1./4', 1./2],['-y+3./4', 'y+3./4', 0],['-y+1./4', 'y+1./4', 0],['-y+1./4', 'y+3./4', 1./2],[0., '-y', 'y'],[0., '-y+1./2', 'y+1./2'],[1./2, '-y', 'y+1./2'],[1./2, '-y+1./2', 'y'],[1./4, 'y+3./4', 'y+1./2'],[1./4, 'y+1./4', 'y'],[3./4, 'y+3./4', 'y'],[3./4, 'y+1./4', 'y+1./2'],[3./4, '-y+1./2', '-y+1./4'],[3./4, '-y', '-y+3./4'],[1./4, '-y+1./2', '-y+3./4'],[1./4, '-y', '-y+1./4'],[1./2, 'y+1./4', '-y+3./4'],[1./2, 'y+3./4', '-y+1./4'],[0., 'y+1./4', '-y+1./4'],[0., 'y+3./4', '-y+3./4'],['y', 0., '-y'],['y', 1./2, '-y+1./2'],['y+1./2', 0., '-y+1./2'],['y+1./2', 1./2, '-y'],['y+1./2', 1./4, 'y+3./4'],['y+1./2', 3./4, 'y+1./4'],['y', 1./4, 'y+1./4'],['y', 3./4, 'y+3./4'],['-y+1./4', 3./4, '-y+1./2'],['-y+1./4', 1./4, '-y'],['-y+3./4', 3./4, '-y'],['-y+3./4', 1./4, '-y+1./2'],['-y+3./4', 1./2, 'y+1./4'],['-y+3./4', 0., 'y+3./4'],['-y+1./4', 1./2, 'y+3./4'],['-y+1./4', 0., 'y+1./4'],['-y', 'y', 0],['-y', 'y+1./2', 1./2],['-y+1./2', 'y', 1./2],['-y+1./2', 'y+1./2', 0],['y+3./4', 'y+1./2', 1./4],['y+3./4', 'y', 3./4],['y+1./4', 'y+1./2', 3./4],['y+1./4', 'y', 1./4],['-y+1./2', '-y+1./4', 3./4],['-y+1./2', '-y+3./4', 1./4],['-y', '-y+1./4', 1./4],['-y', '-y+3./4', 3./4],['y+1./4', '-y+3./4', 1./2],['y+1./4', '-y+1./4', 0],['y+3./4', '-y+3./4', 0],['y+3./4', '-y+1./4', 1./2]],
                        '192i': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x+3./4', '-y+1./4', 'z+1./2'],['-x+3./4', '-y+3./4', 'z'],['-x+1./4', '-y+1./4', 'z'],['-x+1./4', '-y+3./4', 'z+1./2'],['-x+1./4', 'y+1./2', '-z+3./4'],['-x+1./4', 'y', '-z+1./4'],['-x+3./4', 'y+1./2', '-z+1./4'],['-x+3./4', 'y', '-z+3./4'],['x+1./2', '-y+3./4', '-z+1./4'],['x+1./2', '-y+1./4', '-z+3./4'],['x', '-y+3./4', '-z+3./4'],['x', '-y+1./4', '-z+1./4'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z+1./2', '-x+3./4', '-y+1./4'],['z+1./2', '-x+1./4', '-y+3./4'],['z', '-x+3./4', '-y+3./4'],['z', '-x+1./4', '-y+1./4'],['-z+3./4', '-x+1./4', 'y+1./2'],['-z+3./4', '-x+3./4', 'y'],['-z+1./4', '-x+1./4', 'y'],['-z+1./4', '-x+3./4', 'y+1./2'],['-z+1./4', 'x+1./2', '-y+3./4'],['-z+1./4', 'x', '-y+1./4'],['-z+3./4', 'x+1./2', '-y+1./4'],['-z+3./4', 'x', '-y+3./4'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y+1./4', 'z+1./2', '-x+3./4'],['-y+1./4', 'z', '-x+1./4'],['-y+3./4', 'z+1./2', '-x+1./4'],['-y+3./4', 'z', '-x+3./4'],['y+1./2', '-z+3./4', '-x+1./4'],['y+1./2', '-z+1./4', '-x+3./4'],['y', '-z+3./4', '-x+3./4'],['y', '-z+1./4', '-x+1./4'],['-y+3./4', '-z+1./4', 'x+1./2'],['-y+3./4', '-z+3./4', 'x'],['-y+1./4', '-z+1./4', 'x'],['-y+1./4', '-z+3./4', 'x+1./2'],['y+3./4', 'x+1./4', '-z+1./2'],['y+3./4', 'x+3./4', '-z'],['y+1./4', 'x+1./4', '-z'],['y+1./4', 'x+3./4', '-z+1./2'],['-y', '-x', '-z'],['-y', '-x+1./2', '-z+1./2'],['-y+1./2', '-x', '-z+1./2'],['-y+1./2', '-x+1./2', '-z'],['y+1./4', '-x+1./2', 'z+3./4'],['y+1./4', '-x', 'z+1./4'],['y+3./4', '-x+1./2', 'z+1./4'],['y+3./4', '-x', 'z+3./4'],['-y+1./2', 'x+3./4', 'z+1./4'],['-y+1./2', 'x+1./4', 'z+3./4'],['-y', 'x+3./4', 'z+3./4'],['-y', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-y+1./2'],['x+3./4', 'z+3./4', '-y'],['x+1./4', 'z+1./4', '-y'],['x+1./4', 'z+3./4', '-y+1./2'],['-x+1./2', 'z+3./4', 'y+1./4'],['-x+1./2', 'z+1./4', 'y+3./4'],['-x', 'z+3./4', 'y+3./4'],['-x', 'z+1./4', 'y+1./4'],['-x', '-z', '-y'],['-x', '-z+1./2', '-y+1./2'],['-x+1./2', '-z', '-y+1./2'],['-x+1./2', '-z+1./2', '-y'],['x+1./4', '-z+1./2', 'y+3./4'],['x+1./4', '-z', 'y+1./4'],['x+3./4', '-z+1./2', 'y+1./4'],['x+3./4', '-z', 'y+3./4'],['z+3./4', 'y+1./4', '-x+1./2'],['z+3./4', 'y+3./4', '-x'],['z+1./4', 'y+1./4', '-x'],['z+1./4', 'y+3./4', '-x+1./2'],['z+1./4', '-y+1./2', 'x+3./4'],['z+1./4', '-y', 'x+1./4'],['z+3./4', '-y+1./2', 'x+1./4'],['z+3./4', '-y', 'x+3./4'],['-z+1./2', 'y+3./4', 'x+1./4'],['-z+1./2', 'y+1./4', 'x+3./4'],['-z', 'y+3./4', 'x+3./4'],['-z', 'y+1./4', 'x+1./4'],['-z', '-y', '-x'],['-z', '-y+1./2', '-x+1./2'],['-z+1./2', '-y', '-x+1./2'],['-z+1./2', '-y+1./2', '-x'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x+1./4', 'y+3./4', '-z+1./2'],['x+1./4', 'y+1./4', '-z'],['x+3./4', 'y+3./4', '-z'],['x+3./4', 'y+1./4', '-z+1./2'],['x+3./4', '-y+1./2', 'z+1./4'],['x+3./4', '-y', 'z+3./4'],['x+1./4', '-y+1./2', 'z+3./4'],['x+1./4', '-y', 'z+1./4'],['-x+1./2', 'y+1./4', 'z+3./4'],['-x+1./2', 'y+3./4', 'z+1./4'],['-x', 'y+1./4', 'z+1./4'],['-x', 'y+3./4', 'z+3./4'],['-z', '-x', '-y'],['-z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x', '-y+1./2'],['-z+1./2', '-x+1./2', '-y'],['-z+1./2', 'x+1./4', 'y+3./4'],['-z+1./2', 'x+3./4', 'y+1./4'],['-z', 'x+1./4', 'y+1./4'],['-z', 'x+3./4', 'y+3./4'],['z+1./4', 'x+3./4', '-y+1./2'],['z+1./4', 'x+1./4', '-y'],['z+3./4', 'x+3./4', '-y'],['z+3./4', 'x+1./4', '-y+1./2'],['z+3./4', '-x+1./2', 'y+1./4'],['z+3./4', '-x', 'y+3./4'],['z+1./4', '-x+1./2', 'y+3./4'],['z+1./4', '-x', 'y+1./4'],['-y', '-z', '-x'],['-y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z', '-x+1./2'],['-y+1./2', '-z+1./2', '-x'],['y+3./4', '-z+1./2', 'x+1./4'],['y+3./4', '-z', 'x+3./4'],['y+1./4', '-z+1./2', 'x+3./4'],['y+1./4', '-z', 'x+1./4'],['-y+1./2', 'z+1./4', 'x+3./4'],['-y+1./2', 'z+3./4', 'x+1./4'],['-y', 'z+1./4', 'x+1./4'],['-y', 'z+3./4', 'x+3./4'],['y+1./4', 'z+3./4', '-x+1./2'],['y+1./4', 'z+1./4', '-x'],['y+3./4', 'z+3./4', '-x'],['y+3./4', 'z+1./4', '-x+1./2'],['-y+1./4', '-x+3./4', 'z+1./2'],['-y+1./4', '-x+1./4', 'z'],['-y+3./4', '-x+3./4', 'z'],['-y+3./4', '-x+1./4', 'z+1./2'],['y', 'x', 'z'],['y', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z+1./2'],['y+1./2', 'x+1./2', 'z'],['-y+3./4', 'x+1./2', '-z+1./4'],['-y+3./4', 'x', '-z+3./4'],['-y+1./4', 'x+1./2', '-z+3./4'],['-y+1./4', 'x', '-z+1./4'],['y+1./2', '-x+1./4', '-z+3./4'],['y+1./2', '-x+3./4', '-z+1./4'],['y', '-x+1./4', '-z+1./4'],['y', '-x+3./4', '-z+3./4'],['-x+1./4', '-z+3./4', 'y+1./2'],['-x+1./4', '-z+1./4', 'y'],['-x+3./4', '-z+3./4', 'y'],['-x+3./4', '-z+1./4', 'y+1./2'],['x+1./2', '-z+1./4', '-y+3./4'],['x+1./2', '-z+3./4', '-y+1./4'],['x', '-z+1./4', '-y+1./4'],['x', '-z+3./4', '-y+3./4'],['x', 'z', 'y'],['x', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y+1./2'],['x+1./2', 'z+1./2', 'y'],['-x+3./4', 'z+1./2', '-y+1./4'],['-x+3./4', 'z', '-y+3./4'],['-x+1./4', 'z+1./2', '-y+3./4'],['-x+1./4', 'z', '-y+1./4'],['-z+1./4', '-y+3./4', 'x+1./2'],['-z+1./4', '-y+1./4', 'x'],['-z+3./4', '-y+3./4', 'x'],['-z+3./4', '-y+1./4', 'x+1./2'],['-z+3./4', 'y+1./2', '-x+1./4'],['-z+3./4', 'y', '-x+3./4'],['-z+1./4', 'y+1./2', '-x+3./4'],['-z+1./4', 'y', '-x+1./4'],['z+1./2', '-y+1./4', '-x+3./4'],['z+1./2', '-y+3./4', '-x+1./4'],['z', '-y+1./4', '-x+1./4'],['z', '-y+3./4', '-x+3./4'],['z', 'y', 'x'],['z', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x+1./2'],['z+1./2', 'y+1./2', 'x']]},
              '228:1': {'16a': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4],[0., 1./2, 0],[0., 0., 1./2],[1./2, 1./2, 1./2],[1./2, 0., 0]],
                        '32b': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 3/8, 5/8],[7/8, 7/8, 1/8],[3/8, 3/8, 1/8],[3/8, 7/8, 5/8],[3/8, 5/8, 7/8],[3/8, 1/8, 3/8],[7/8, 5/8, 3/8],[7/8, 1/8, 7/8],[5/8, 7/8, 3/8],[5/8, 3/8, 7/8],[1/8, 7/8, 7/8],[1/8, 3/8, 3/8],[5/8, 5/8, 5/8],[5/8, 1/8, 1/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[7/8, 3/8, 1/8],[7/8, 7/8, 5/8],[3/8, 3/8, 5/8],[3/8, 7/8, 1/8],[3/8, 1/8, 7/8],[3/8, 5/8, 3/8],[7/8, 1/8, 3/8],[7/8, 5/8, 7/8],[1/8, 7/8, 3/8],[1/8, 3/8, 7/8],[5/8, 7/8, 7/8],[5/8, 3/8, 3/8]],
                        '32c': [[3/8, 3/8, 3/8],[3/8, 7/8, 7/8],[7/8, 3/8, 7/8],[7/8, 7/8, 3/8],[5/8, 1/8, 7/8],[5/8, 5/8, 3/8],[1/8, 1/8, 3/8],[1/8, 5/8, 7/8],[1/8, 7/8, 5/8],[1/8, 3/8, 1/8],[5/8, 7/8, 1/8],[5/8, 3/8, 5/8],[7/8, 5/8, 1/8],[7/8, 1/8, 5/8],[3/8, 5/8, 5/8],[3/8, 1/8, 1/8],[1/8, 5/8, 3/8],[1/8, 1/8, 7/8],[5/8, 5/8, 7/8],[5/8, 1/8, 3/8],[7/8, 7/8, 7/8],[7/8, 3/8, 3/8],[3/8, 7/8, 3/8],[3/8, 3/8, 7/8],[5/8, 3/8, 1/8],[5/8, 7/8, 5/8],[1/8, 3/8, 5/8],[1/8, 7/8, 1/8],[3/8, 1/8, 5/8],[3/8, 5/8, 1/8],[7/8, 1/8, 1/8],[7/8, 5/8, 5/8]],
                        '48d': [[1./4, 0., 0],[1./4, 1./2, 1./2],[3./4, 0., 1./2],[3./4, 1./2, 0],[3./4, 1./2, 1./2],[3./4, 0., 0],[1./4, 1./2, 0],[1./4, 0., 1./2],[0., 1./4, 0],[0., 3./4, 1./2],[1./2, 1./4, 1./2],[1./2, 3./4, 0],[1./2, 3./4, 1./2],[1./2, 1./4, 0],[0., 3./4, 0],[0., 1./4, 1./2],[0., 0., 1./4],[0., 1./2, 3./4],[1./2, 0., 3./4],[1./2, 1./2, 1./4],[1./2, 1./2, 3./4],[1./2, 0., 1./4],[0., 1./2, 1./4],[0., 0., 3./4],[3./4, 1./2, 3./4],[3./4, 0., 1./4],[1./4, 1./2, 1./4],[1./4, 0., 3./4],[1./4, 0., 1./4],[1./4, 1./2, 3./4],[3./4, 0., 3./4],[3./4, 1./2, 1./4],[0., 1./4, 3./4],[0., 3./4, 1./4],[1./2, 1./4, 1./4],[1./2, 3./4, 3./4],[1./2, 3./4, 1./4],[1./2, 1./4, 3./4],[0., 3./4, 3./4],[0., 1./4, 1./4],[3./4, 1./4, 1./2],[3./4, 3./4, 0],[1./4, 1./4, 0],[1./4, 3./4, 1./2],[1./4, 3./4, 0],[1./4, 1./4, 1./2],[3./4, 3./4, 1./2],[3./4, 1./4, 0]],
                        '64e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x', '-x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x'],['-x+1./2', '-x', 'x+1./2'],['-x+1./2', 'x+1./2', '-x'],['-x+1./2', 'x', '-x+1./2'],['-x', 'x+1./2', '-x+1./2'],['-x', 'x', '-x'],['x+1./2', '-x', '-x+1./2'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x'],['x', '-x+1./2', '-x+1./2'],['x+3./4', 'x+1./4', '-x+3./4'],['x+3./4', 'x+3./4', '-x+1./4'],['x+1./4', 'x+1./4', '-x+1./4'],['x+1./4', 'x+3./4', '-x+3./4'],['-x+1./4', '-x+1./4', '-x+1./4'],['-x+1./4', '-x+3./4', '-x+3./4'],['-x+3./4', '-x+1./4', '-x+3./4'],['-x+3./4', '-x+3./4', '-x+1./4'],['x+1./4', '-x+3./4', 'x+3./4'],['x+1./4', '-x+1./4', 'x+1./4'],['x+3./4', '-x+3./4', 'x+1./4'],['x+3./4', '-x+1./4', 'x+3./4'],['-x+3./4', 'x+3./4', 'x+1./4'],['-x+3./4', 'x+1./4', 'x+3./4'],['-x+1./4', 'x+3./4', 'x+3./4'],['-x+1./4', 'x+1./4', 'x+1./4'],['-x+3./4', '-x+3./4', '-x+3./4'],['-x+3./4', '-x+1./4', '-x+1./4'],['-x+1./4', '-x+3./4', '-x+1./4'],['-x+1./4', '-x+1./4', '-x+3./4'],['x+3./4', 'x+1./4', '-x+1./4'],['x+3./4', 'x+3./4', '-x+3./4'],['x+1./4', 'x+1./4', '-x+3./4'],['x+1./4', 'x+3./4', '-x+1./4'],['x+1./4', '-x+1./4', 'x+3./4'],['x+1./4', '-x+3./4', 'x+1./4'],['x+3./4', '-x+1./4', 'x+1./4'],['x+3./4', '-x+3./4', 'x+3./4'],['-x+1./4', 'x+3./4', 'x+1./4'],['-x+1./4', 'x+1./4', 'x+3./4'],['-x+3./4', 'x+3./4', 'x+3./4'],['-x+3./4', 'x+1./4', 'x+1./4'],['-x', '-x+1./2', 'x'],['-x', '-x', 'x+1./2'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x'],['x', 'x+1./2', 'x'],['x', 'x', 'x+1./2'],['-x+1./2', 'x', '-x'],['-x+1./2', 'x+1./2', '-x+1./2'],['-x', 'x', '-x+1./2'],['-x', 'x+1./2', '-x'],['x', '-x', '-x+1./2'],['x', '-x+1./2', '-x'],['x+1./2', '-x', '-x'],['x+1./2', '-x+1./2', '-x+1./2']],
                        '96f': [['x', 0., 0],['x', 1./2, 1./2],['x+1./2', 0., 1./2],['x+1./2', 1./2, 0],['-x', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 0],['-x+1./2', 0., 1./2],[0., 'x', 0],[0., 'x+1./2', 1./2],[1./2, 'x', 1./2],[1./2, 'x+1./2', 0],[1./2, '-x', 1./2],[1./2, '-x+1./2', 0],[0., '-x', 0],[0., '-x+1./2', 1./2],[0., 0., 'x'],[0., 1./2, 'x+1./2'],[1./2, 0., 'x+1./2'],[1./2, 1./2, 'x'],[1./2, 1./2, '-x'],[1./2, 0., '-x+1./2'],[0., 1./2, '-x+1./2'],[0., 0., '-x'],[3./4, 'x+1./4', 3./4],[3./4, 'x+3./4', 1./4],[1./4, 'x+1./4', 1./4],[1./4, 'x+3./4', 3./4],[1./4, '-x+1./4', 1./4],[1./4, '-x+3./4', 3./4],[3./4, '-x+1./4', 3./4],[3./4, '-x+3./4', 1./4],['x+3./4', 1./4, 3./4],['x+3./4', 3./4, 1./4],['x+1./4', 1./4, 1./4],['x+1./4', 3./4, 3./4],['-x+3./4', 3./4, 1./4],['-x+3./4', 1./4, 3./4],['-x+1./4', 3./4, 3./4],['-x+1./4', 1./4, 1./4],[3./4, 1./4, '-x+3./4'],[3./4, 3./4, '-x+1./4'],[1./4, 1./4, '-x+1./4'],[1./4, 3./4, '-x+3./4'],[1./4, 3./4, 'x+3./4'],[1./4, 1./4, 'x+1./4'],[3./4, 3./4, 'x+1./4'],[3./4, 1./4, 'x+3./4'],['-x+3./4', 3./4, 3./4],['-x+3./4', 1./4, 1./4],['-x+1./4', 3./4, 1./4],['-x+1./4', 1./4, 3./4],['x+3./4', 1./4, 1./4],['x+3./4', 3./4, 3./4],['x+1./4', 1./4, 3./4],['x+1./4', 3./4, 1./4],[3./4, '-x+3./4', 3./4],[3./4, '-x+1./4', 1./4],[1./4, '-x+3./4', 1./4],[1./4, '-x+1./4', 3./4],[1./4, 'x+3./4', 1./4],[1./4, 'x+1./4', 3./4],[3./4, 'x+3./4', 3./4],[3./4, 'x+1./4', 1./4],[3./4, 3./4, '-x+3./4'],[3./4, 1./4, '-x+1./4'],[1./4, 3./4, '-x+1./4'],[1./4, 1./4, '-x+3./4'],[1./4, 1./4, 'x+3./4'],[1./4, 3./4, 'x+1./4'],[3./4, 1./4, 'x+1./4'],[3./4, 3./4, 'x+3./4'],[0., '-x+1./2', 0],[0., '-x', 1./2],[1./2, '-x+1./2', 1./2],[1./2, '-x', 0],[1./2, 'x+1./2', 1./2],[1./2, 'x', 0],[0., 'x+1./2', 0],[0., 'x', 1./2],['-x', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 1./2],['-x+1./2', 0., 0],['x', 0., 1./2],['x', 1./2, 0],['x+1./2', 0., 0],['x+1./2', 1./2, 1./2],[0., 1./2, 'x'],[0., 0., 'x+1./2'],[1./2, 1./2, 'x+1./2'],[1./2, 0., 'x'],[1./2, 0., '-x'],[1./2, 1./2, '-x+1./2'],[0., 0., '-x+1./2'],[0., 1./2, '-x']],
                        '96g': [[1/8, 'y', '-y+1./4'],[1/8, 'y+1./2', '-y+3./4'],[5/8, 'y', '-y+3./4'],[5/8, 'y+1./2', '-y+1./4'],[7/8, '-y+1./2', '-y+3./4'],[7/8, '-y', '-y+1./4'],[3/8, '-y+1./2', '-y+1./4'],[3/8, '-y', '-y+3./4'],[3/8, 'y+1./2', 'y+3./4'],[3/8, 'y', 'y+1./4'],[7/8, 'y+1./2', 'y+1./4'],[7/8, 'y', 'y+3./4'],[5/8, '-y', 'y+1./4'],[5/8, '-y+1./2', 'y+3./4'],[1/8, '-y', 'y+3./4'],[1/8, '-y+1./2', 'y+1./4'],['-y+1./4', 1/8, 'y'],['-y+1./4', 5/8, 'y+1./2'],['-y+3./4', 1/8, 'y+1./2'],['-y+3./4', 5/8, 'y'],['-y+3./4', 7/8, '-y+1./2'],['-y+3./4', 3/8, '-y'],['-y+1./4', 7/8, '-y'],['-y+1./4', 3/8, '-y+1./2'],['y+3./4', 3/8, 'y+1./2'],['y+3./4', 7/8, 'y'],['y+1./4', 3/8, 'y'],['y+1./4', 7/8, 'y+1./2'],['y+1./4', 5/8, '-y'],['y+1./4', 1/8, '-y+1./2'],['y+3./4', 5/8, '-y+1./2'],['y+3./4', 1/8, '-y'],['y', '-y+1./4', 1/8],['y', '-y+3./4', 5/8],['y+1./2', '-y+1./4', 5/8],['y+1./2', '-y+3./4', 1/8],['-y+1./2', '-y+3./4', 7/8],['-y+1./2', '-y+1./4', 3/8],['-y', '-y+3./4', 3/8],['-y', '-y+1./4', 7/8],['y+1./2', 'y+3./4', 3/8],['y+1./2', 'y+1./4', 7/8],['y', 'y+3./4', 7/8],['y', 'y+1./4', 3/8],['-y', 'y+1./4', 5/8],['-y', 'y+3./4', 1/8],['-y+1./2', 'y+1./4', 1/8],['-y+1./2', 'y+3./4', 5/8],[5/8, '-y+3./4', 'y+1./2'],[5/8, '-y+1./4', 'y'],[1/8, '-y+3./4', 'y'],[1/8, '-y+1./4', 'y+1./2'],[7/8, 'y+1./4', 'y'],[7/8, 'y+3./4', 'y+1./2'],[3/8, 'y+1./4', 'y+1./2'],[3/8, 'y+3./4', 'y'],[3/8, '-y+1./4', '-y'],[3/8, '-y+3./4', '-y+1./2'],[7/8, '-y+1./4', '-y+1./2'],[7/8, '-y+3./4', '-y'],[1/8, 'y+3./4', '-y+1./2'],[1/8, 'y+1./4', '-y'],[5/8, 'y+3./4', '-y'],[5/8, 'y+1./4', '-y+1./2'],['y+1./2', 5/8, '-y+3./4'],['y+1./2', 1/8, '-y+1./4'],['y', 5/8, '-y+1./4'],['y', 1/8, '-y+3./4'],['y', 7/8, 'y+1./4'],['y', 3/8, 'y+3./4'],['y+1./2', 7/8, 'y+3./4'],['y+1./2', 3/8, 'y+1./4'],['-y', 3/8, '-y+1./4'],['-y', 7/8, '-y+3./4'],['-y+1./2', 3/8, '-y+3./4'],['-y+1./2', 7/8, '-y+1./4'],['-y+1./2', 1/8, 'y+3./4'],['-y+1./2', 5/8, 'y+1./4'],['-y', 1/8, 'y+1./4'],['-y', 5/8, 'y+3./4'],['-y+3./4', 'y+1./2', 5/8],['-y+3./4', 'y', 1/8],['-y+1./4', 'y+1./2', 1/8],['-y+1./4', 'y', 5/8],['y+1./4', 'y', 7/8],['y+1./4', 'y+1./2', 3/8],['y+3./4', 'y', 3/8],['y+3./4', 'y+1./2', 7/8],['-y+1./4', '-y', 3/8],['-y+1./4', '-y+1./2', 7/8],['-y+3./4', '-y', 7/8],['-y+3./4', '-y+1./2', 3/8],['y+3./4', '-y+1./2', 1/8],['y+3./4', '-y', 5/8],['y+1./4', '-y+1./2', 5/8],['y+1./4', '-y', 1/8]],
                        '192h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x', '-y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z'],['-x+1./2', '-y', 'z+1./2'],['-x+1./2', 'y+1./2', '-z'],['-x+1./2', 'y', '-z+1./2'],['-x', 'y+1./2', '-z+1./2'],['-x', 'y', '-z'],['x+1./2', '-y', '-z+1./2'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z'],['x', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z+1./2', '-x', '-y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y'],['z', '-x+1./2', '-y+1./2'],['-z', '-x+1./2', 'y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y'],['-z+1./2', '-x', 'y+1./2'],['-z+1./2', 'x+1./2', '-y'],['-z+1./2', 'x', '-y+1./2'],['-z', 'x+1./2', '-y+1./2'],['-z', 'x', '-y'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y+1./2', 'z+1./2', '-x'],['-y+1./2', 'z', '-x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y', 'z', '-x'],['y+1./2', '-z', '-x+1./2'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x'],['y', '-z+1./2', '-x+1./2'],['-y', '-z+1./2', 'x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x'],['-y+1./2', '-z', 'x+1./2'],['y+3./4', 'x+1./4', '-z+3./4'],['y+3./4', 'x+3./4', '-z+1./4'],['y+1./4', 'x+1./4', '-z+1./4'],['y+1./4', 'x+3./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['-y+1./4', '-x+3./4', '-z+3./4'],['-y+3./4', '-x+1./4', '-z+3./4'],['-y+3./4', '-x+3./4', '-z+1./4'],['y+1./4', '-x+3./4', 'z+3./4'],['y+1./4', '-x+1./4', 'z+1./4'],['y+3./4', '-x+3./4', 'z+1./4'],['y+3./4', '-x+1./4', 'z+3./4'],['-y+3./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['-y+1./4', 'x+3./4', 'z+3./4'],['-y+1./4', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-y+3./4'],['x+3./4', 'z+3./4', '-y+1./4'],['x+1./4', 'z+1./4', '-y+1./4'],['x+1./4', 'z+3./4', '-y+3./4'],['-x+3./4', 'z+3./4', 'y+1./4'],['-x+3./4', 'z+1./4', 'y+3./4'],['-x+1./4', 'z+3./4', 'y+3./4'],['-x+1./4', 'z+1./4', 'y+1./4'],['-x+1./4', '-z+1./4', '-y+1./4'],['-x+1./4', '-z+3./4', '-y+3./4'],['-x+3./4', '-z+1./4', '-y+3./4'],['-x+3./4', '-z+3./4', '-y+1./4'],['x+1./4', '-z+3./4', 'y+3./4'],['x+1./4', '-z+1./4', 'y+1./4'],['x+3./4', '-z+3./4', 'y+1./4'],['x+3./4', '-z+1./4', 'y+3./4'],['z+3./4', 'y+1./4', '-x+3./4'],['z+3./4', 'y+3./4', '-x+1./4'],['z+1./4', 'y+1./4', '-x+1./4'],['z+1./4', 'y+3./4', '-x+3./4'],['z+1./4', '-y+3./4', 'x+3./4'],['z+1./4', '-y+1./4', 'x+1./4'],['z+3./4', '-y+3./4', 'x+1./4'],['z+3./4', '-y+1./4', 'x+3./4'],['-z+3./4', 'y+3./4', 'x+1./4'],['-z+3./4', 'y+1./4', 'x+3./4'],['-z+1./4', 'y+3./4', 'x+3./4'],['-z+1./4', 'y+1./4', 'x+1./4'],['-z+1./4', '-y+1./4', '-x+1./4'],['-z+1./4', '-y+3./4', '-x+3./4'],['-z+3./4', '-y+1./4', '-x+3./4'],['-z+3./4', '-y+3./4', '-x+1./4'],['-x+3./4', '-y+3./4', '-z+3./4'],['-x+3./4', '-y+1./4', '-z+1./4'],['-x+1./4', '-y+3./4', '-z+1./4'],['-x+1./4', '-y+1./4', '-z+3./4'],['x+3./4', 'y+1./4', '-z+1./4'],['x+3./4', 'y+3./4', '-z+3./4'],['x+1./4', 'y+1./4', '-z+3./4'],['x+1./4', 'y+3./4', '-z+1./4'],['x+1./4', '-y+1./4', 'z+3./4'],['x+1./4', '-y+3./4', 'z+1./4'],['x+3./4', '-y+1./4', 'z+1./4'],['x+3./4', '-y+3./4', 'z+3./4'],['-x+1./4', 'y+3./4', 'z+1./4'],['-x+1./4', 'y+1./4', 'z+3./4'],['-x+3./4', 'y+3./4', 'z+3./4'],['-x+3./4', 'y+1./4', 'z+1./4'],['-z+3./4', '-x+3./4', '-y+3./4'],['-z+3./4', '-x+1./4', '-y+1./4'],['-z+1./4', '-x+3./4', '-y+1./4'],['-z+1./4', '-x+1./4', '-y+3./4'],['-z+1./4', 'x+3./4', 'y+1./4'],['-z+1./4', 'x+1./4', 'y+3./4'],['-z+3./4', 'x+3./4', 'y+3./4'],['-z+3./4', 'x+1./4', 'y+1./4'],['z+3./4', 'x+1./4', '-y+1./4'],['z+3./4', 'x+3./4', '-y+3./4'],['z+1./4', 'x+1./4', '-y+3./4'],['z+1./4', 'x+3./4', '-y+1./4'],['z+1./4', '-x+1./4', 'y+3./4'],['z+1./4', '-x+3./4', 'y+1./4'],['z+3./4', '-x+1./4', 'y+1./4'],['z+3./4', '-x+3./4', 'y+3./4'],['-y+3./4', '-z+3./4', '-x+3./4'],['-y+3./4', '-z+1./4', '-x+1./4'],['-y+1./4', '-z+3./4', '-x+1./4'],['-y+1./4', '-z+1./4', '-x+3./4'],['y+1./4', '-z+1./4', 'x+3./4'],['y+1./4', '-z+3./4', 'x+1./4'],['y+3./4', '-z+1./4', 'x+1./4'],['y+3./4', '-z+3./4', 'x+3./4'],['-y+1./4', 'z+3./4', 'x+1./4'],['-y+1./4', 'z+1./4', 'x+3./4'],['-y+3./4', 'z+3./4', 'x+3./4'],['-y+3./4', 'z+1./4', 'x+1./4'],['y+3./4', 'z+1./4', '-x+1./4'],['y+3./4', 'z+3./4', '-x+3./4'],['y+1./4', 'z+1./4', '-x+3./4'],['y+1./4', 'z+3./4', '-x+1./4'],['-y', '-x+1./2', 'z'],['-y', '-x', 'z+1./2'],['-y+1./2', '-x+1./2', 'z+1./2'],['-y+1./2', '-x', 'z'],['y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z'],['y', 'x+1./2', 'z'],['y', 'x', 'z+1./2'],['-y+1./2', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['-y', 'x', '-z+1./2'],['-y', 'x+1./2', '-z'],['y', '-x', '-z+1./2'],['y', '-x+1./2', '-z'],['y+1./2', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-x', '-z+1./2', 'y'],['-x', '-z', 'y+1./2'],['-x+1./2', '-z+1./2', 'y+1./2'],['-x+1./2', '-z', 'y'],['x', '-z', '-y+1./2'],['x', '-z+1./2', '-y'],['x+1./2', '-z', '-y'],['x+1./2', '-z+1./2', '-y+1./2'],['x+1./2', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y'],['x', 'z+1./2', 'y'],['x', 'z', 'y+1./2'],['-x+1./2', 'z', '-y'],['-x+1./2', 'z+1./2', '-y+1./2'],['-x', 'z', '-y+1./2'],['-x', 'z+1./2', '-y'],['-z', '-y+1./2', 'x'],['-z', '-y', 'x+1./2'],['-z+1./2', '-y+1./2', 'x+1./2'],['-z+1./2', '-y', 'x'],['-z+1./2', 'y', '-x'],['-z+1./2', 'y+1./2', '-x+1./2'],['-z', 'y', '-x+1./2'],['-z', 'y+1./2', '-x'],['z', '-y', '-x+1./2'],['z', '-y+1./2', '-x'],['z+1./2', '-y', '-x'],['z+1./2', '-y+1./2', '-x+1./2'],['z+1./2', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x'],['z', 'y+1./2', 'x'],['z', 'y', 'x+1./2']]},
              '228:2': {'16a': [[1/8, 1/8, 1/8],[1/8, 5/8, 5/8],[5/8, 1/8, 5/8],[5/8, 5/8, 1/8],[7/8, 3/8, 7/8],[7/8, 7/8, 3/8],[3/8, 3/8, 3/8],[3/8, 7/8, 7/8],[7/8, 7/8, 7/8],[7/8, 3/8, 3/8],[3/8, 7/8, 3/8],[3/8, 3/8, 7/8],[1/8, 5/8, 1/8],[1/8, 1/8, 5/8],[5/8, 5/8, 5/8],[5/8, 1/8, 1/8]],
                        '32b': [[1./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 3./4],[3./4, 3./4, 1./4],[0., 1./2, 3./4],[0., 0., 1./4],[1./2, 1./2, 1./4],[1./2, 0., 3./4],[1./2, 3./4, 0],[1./2, 1./4, 1./2],[0., 3./4, 1./2],[0., 1./4, 0],[3./4, 0., 1./2],[3./4, 1./2, 0],[1./4, 0., 0],[1./4, 1./2, 1./2],[3./4, 3./4, 3./4],[3./4, 1./4, 1./4],[1./4, 3./4, 1./4],[1./4, 1./4, 3./4],[0., 1./2, 1./4],[0., 0., 3./4],[1./2, 1./2, 3./4],[1./2, 0., 1./4],[1./2, 1./4, 0],[1./2, 3./4, 1./2],[0., 1./4, 1./2],[0., 3./4, 0],[1./4, 0., 1./2],[1./4, 1./2, 0],[3./4, 0., 0],[3./4, 1./2, 1./2]],
                        '32c': [[0., 0., 0],[0., 1./2, 1./2],[1./2, 0., 1./2],[1./2, 1./2, 0],[1./4, 3./4, 1./2],[1./4, 1./4, 0],[3./4, 3./4, 0],[3./4, 1./4, 1./2],[3./4, 1./2, 1./4],[3./4, 0., 3./4],[1./4, 1./2, 3./4],[1./4, 0., 1./4],[1./2, 1./4, 3./4],[1./2, 3./4, 1./4],[0., 1./4, 1./4],[0., 3./4, 3./4],[3./4, 1./4, 0],[3./4, 3./4, 1./2],[1./4, 1./4, 1./2],[1./4, 3./4, 0],[1./2, 1./2, 1./2],[1./2, 0., 0],[0., 1./2, 0],[0., 0., 1./2],[1./4, 0., 3./4],[1./4, 1./2, 1./4],[3./4, 0., 1./4],[3./4, 1./2, 3./4],[0., 3./4, 1./4],[0., 1./4, 3./4],[1./2, 3./4, 3./4],[1./2, 1./4, 1./4]],
                        '48d': [[7/8, 1/8, 1/8],[7/8, 5/8, 5/8],[3/8, 1/8, 5/8],[3/8, 5/8, 1/8],[3/8, 5/8, 5/8],[3/8, 1/8, 1/8],[7/8, 5/8, 1/8],[7/8, 1/8, 5/8],[1/8, 7/8, 1/8],[1/8, 3/8, 5/8],[5/8, 7/8, 5/8],[5/8, 3/8, 1/8],[5/8, 3/8, 5/8],[5/8, 7/8, 1/8],[1/8, 3/8, 1/8],[1/8, 7/8, 5/8],[1/8, 1/8, 7/8],[1/8, 5/8, 3/8],[5/8, 1/8, 3/8],[5/8, 5/8, 7/8],[5/8, 5/8, 3/8],[5/8, 1/8, 7/8],[1/8, 5/8, 7/8],[1/8, 1/8, 3/8],[7/8, 1/8, 7/8],[7/8, 5/8, 3/8],[3/8, 1/8, 3/8],[3/8, 5/8, 7/8],[3/8, 5/8, 3/8],[3/8, 1/8, 7/8],[7/8, 5/8, 7/8],[7/8, 1/8, 3/8],[5/8, 3/8, 7/8],[5/8, 7/8, 3/8],[1/8, 3/8, 3/8],[1/8, 7/8, 7/8],[1/8, 7/8, 3/8],[1/8, 3/8, 7/8],[5/8, 7/8, 7/8],[5/8, 3/8, 3/8],[7/8, 3/8, 1/8],[7/8, 7/8, 5/8],[3/8, 3/8, 5/8],[3/8, 7/8, 1/8],[3/8, 7/8, 5/8],[3/8, 3/8, 1/8],[7/8, 7/8, 1/8],[7/8, 3/8, 5/8]],
                        '64e': [['x', 'x', 'x'],['x', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x+1./2'],['x+1./2', 'x+1./2', 'x'],['-x+1./4', '-x+3./4', 'x+1./2'],['-x+1./4', '-x+1./4', 'x'],['-x+3./4', '-x+3./4', 'x'],['-x+3./4', '-x+1./4', 'x+1./2'],['-x+3./4', 'x+1./2', '-x+1./4'],['-x+3./4', 'x', '-x+3./4'],['-x+1./4', 'x+1./2', '-x+3./4'],['-x+1./4', 'x', '-x+1./4'],['x+1./2', '-x+1./4', '-x+3./4'],['x+1./2', '-x+3./4', '-x+1./4'],['x', '-x+1./4', '-x+1./4'],['x', '-x+3./4', '-x+3./4'],['x+3./4', 'x+1./4', '-x'],['x+3./4', 'x+3./4', '-x+1./2'],['x+1./4', 'x+1./4', '-x+1./2'],['x+1./4', 'x+3./4', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x'],['-x', '-x+1./2', '-x'],['-x', '-x', '-x+1./2'],['x+1./4', '-x', 'x+3./4'],['x+1./4', '-x+1./2', 'x+1./4'],['x+3./4', '-x', 'x+1./4'],['x+3./4', '-x+1./2', 'x+3./4'],['-x', 'x+3./4', 'x+1./4'],['-x', 'x+1./4', 'x+3./4'],['-x+1./2', 'x+3./4', 'x+3./4'],['-x+1./2', 'x+1./4', 'x+1./4'],['-x', '-x', '-x'],['-x', '-x+1./2', '-x+1./2'],['-x+1./2', '-x', '-x+1./2'],['-x+1./2', '-x+1./2', '-x'],['x+3./4', 'x+1./4', '-x+1./2'],['x+3./4', 'x+3./4', '-x'],['x+1./4', 'x+1./4', '-x'],['x+1./4', 'x+3./4', '-x+1./2'],['x+1./4', '-x+1./2', 'x+3./4'],['x+1./4', '-x', 'x+1./4'],['x+3./4', '-x+1./2', 'x+1./4'],['x+3./4', '-x', 'x+3./4'],['-x+1./2', 'x+3./4', 'x+1./4'],['-x+1./2', 'x+1./4', 'x+3./4'],['-x', 'x+3./4', 'x+3./4'],['-x', 'x+1./4', 'x+1./4'],['-x+1./4', '-x+3./4', 'x'],['-x+1./4', '-x+1./4', 'x+1./2'],['-x+3./4', '-x+3./4', 'x+1./2'],['-x+3./4', '-x+1./4', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['x+1./2', 'x', 'x'],['x', 'x+1./2', 'x'],['x', 'x', 'x+1./2'],['-x+3./4', 'x', '-x+1./4'],['-x+3./4', 'x+1./2', '-x+3./4'],['-x+1./4', 'x', '-x+3./4'],['-x+1./4', 'x+1./2', '-x+1./4'],['x', '-x+1./4', '-x+3./4'],['x', '-x+3./4', '-x+1./4'],['x+1./2', '-x+1./4', '-x+1./4'],['x+1./2', '-x+3./4', '-x+3./4']],
                        '96f': [['x', 1/8, 1/8],['x', 5/8, 5/8],['x+1./2', 1/8, 5/8],['x+1./2', 5/8, 1/8],['-x+1./4', 5/8, 5/8],['-x+1./4', 1/8, 1/8],['-x+3./4', 5/8, 1/8],['-x+3./4', 1/8, 5/8],[1/8, 'x', 1/8],[1/8, 'x+1./2', 5/8],[5/8, 'x', 5/8],[5/8, 'x+1./2', 1/8],[5/8, '-x+1./4', 5/8],[5/8, '-x+3./4', 1/8],[1/8, '-x+1./4', 1/8],[1/8, '-x+3./4', 5/8],[1/8, 1/8, 'x'],[1/8, 5/8, 'x+1./2'],[5/8, 1/8, 'x+1./2'],[5/8, 5/8, 'x'],[5/8, 5/8, '-x+1./4'],[5/8, 1/8, '-x+3./4'],[1/8, 5/8, '-x+3./4'],[1/8, 1/8, '-x+1./4'],[7/8, 'x+1./4', 7/8],[7/8, 'x+3./4', 3/8],[3/8, 'x+1./4', 3/8],[3/8, 'x+3./4', 7/8],[3/8, '-x+1./2', 3/8],[3/8, '-x', 7/8],[7/8, '-x+1./2', 7/8],[7/8, '-x', 3/8],['x+3./4', 3/8, 7/8],['x+3./4', 7/8, 3/8],['x+1./4', 3/8, 3/8],['x+1./4', 7/8, 7/8],['-x', 7/8, 3/8],['-x', 3/8, 7/8],['-x+1./2', 7/8, 7/8],['-x+1./2', 3/8, 3/8],[7/8, 3/8, '-x'],[7/8, 7/8, '-x+1./2'],[3/8, 3/8, '-x+1./2'],[3/8, 7/8, '-x'],[3/8, 7/8, 'x+3./4'],[3/8, 3/8, 'x+1./4'],[7/8, 7/8, 'x+1./4'],[7/8, 3/8, 'x+3./4'],['-x', 7/8, 7/8],['-x', 3/8, 3/8],['-x+1./2', 7/8, 3/8],['-x+1./2', 3/8, 7/8],['x+3./4', 3/8, 3/8],['x+3./4', 7/8, 7/8],['x+1./4', 3/8, 7/8],['x+1./4', 7/8, 3/8],[7/8, '-x', 7/8],[7/8, '-x+1./2', 3/8],[3/8, '-x', 3/8],[3/8, '-x+1./2', 7/8],[3/8, 'x+3./4', 3/8],[3/8, 'x+1./4', 7/8],[7/8, 'x+3./4', 7/8],[7/8, 'x+1./4', 3/8],[7/8, 7/8, '-x'],[7/8, 3/8, '-x+1./2'],[3/8, 7/8, '-x+1./2'],[3/8, 3/8, '-x'],[3/8, 3/8, 'x+3./4'],[3/8, 7/8, 'x+1./4'],[7/8, 3/8, 'x+1./4'],[7/8, 7/8, 'x+3./4'],[1/8, '-x+3./4', 1/8],[1/8, '-x+1./4', 5/8],[5/8, '-x+3./4', 5/8],[5/8, '-x+1./4', 1/8],[5/8, 'x+1./2', 5/8],[5/8, 'x', 1/8],[1/8, 'x+1./2', 1/8],[1/8, 'x', 5/8],['-x+1./4', 5/8, 1/8],['-x+1./4', 1/8, 5/8],['-x+3./4', 5/8, 5/8],['-x+3./4', 1/8, 1/8],['x', 1/8, 5/8],['x', 5/8, 1/8],['x+1./2', 1/8, 1/8],['x+1./2', 5/8, 5/8],[1/8, 5/8, 'x'],[1/8, 1/8, 'x+1./2'],[5/8, 5/8, 'x+1./2'],[5/8, 1/8, 'x'],[5/8, 1/8, '-x+1./4'],[5/8, 5/8, '-x+3./4'],[1/8, 1/8, '-x+3./4'],[1/8, 5/8, '-x+1./4']],
                        '96g': [[1./4, 'y', '-y'],[1./4, 'y+1./2', '-y+1./2'],[3./4, 'y', '-y+1./2'],[3./4, 'y+1./2', '-y'],[0., '-y+3./4', '-y+1./2'],[0., '-y+1./4', '-y'],[1./2, '-y+3./4', '-y'],[1./2, '-y+1./4', '-y+1./2'],[1./2, 'y+1./2', 'y+1./4'],[1./2, 'y', 'y+3./4'],[0., 'y+1./2', 'y+3./4'],[0., 'y', 'y+1./4'],[3./4, '-y+1./4', 'y+3./4'],[3./4, '-y+3./4', 'y+1./4'],[1./4, '-y+1./4', 'y+1./4'],[1./4, '-y+3./4', 'y+3./4'],['-y', 1./4, 'y'],['-y', 3./4, 'y+1./2'],['-y+1./2', 1./4, 'y+1./2'],['-y+1./2', 3./4, 'y'],['-y+1./2', 0., '-y+3./4'],['-y+1./2', 1./2, '-y+1./4'],['-y', 0., '-y+1./4'],['-y', 1./2, '-y+3./4'],['y+1./4', 1./2, 'y+1./2'],['y+1./4', 0., 'y'],['y+3./4', 1./2, 'y'],['y+3./4', 0., 'y+1./2'],['y+3./4', 3./4, '-y+1./4'],['y+3./4', 1./4, '-y+3./4'],['y+1./4', 3./4, '-y+3./4'],['y+1./4', 1./4, '-y+1./4'],['y', '-y', 1./4],['y', '-y+1./2', 3./4],['y+1./2', '-y', 3./4],['y+1./2', '-y+1./2', 1./4],['-y+3./4', '-y+1./2', 0],['-y+3./4', '-y', 1./2],['-y+1./4', '-y+1./2', 1./2],['-y+1./4', '-y', 0],['y+1./2', 'y+1./4', 1./2],['y+1./2', 'y+3./4', 0],['y', 'y+1./4', 0],['y', 'y+3./4', 1./2],['-y+1./4', 'y+3./4', 3./4],['-y+1./4', 'y+1./4', 1./4],['-y+3./4', 'y+3./4', 1./4],['-y+3./4', 'y+1./4', 3./4],[3./4, '-y', 'y'],[3./4, '-y+1./2', 'y+1./2'],[1./4, '-y', 'y+1./2'],[1./4, '-y+1./2', 'y'],[0., 'y+1./4', 'y+1./2'],[0., 'y+3./4', 'y'],[1./2, 'y+1./4', 'y'],[1./2, 'y+3./4', 'y+1./2'],[1./2, '-y+1./2', '-y+3./4'],[1./2, '-y', '-y+1./4'],[0., '-y+1./2', '-y+1./4'],[0., '-y', '-y+3./4'],[1./4, 'y+3./4', '-y+1./4'],[1./4, 'y+1./4', '-y+3./4'],[3./4, 'y+3./4', '-y+3./4'],[3./4, 'y+1./4', '-y+1./4'],['y', 3./4, '-y'],['y', 1./4, '-y+1./2'],['y+1./2', 3./4, '-y+1./2'],['y+1./2', 1./4, '-y'],['y+1./2', 0., 'y+1./4'],['y+1./2', 1./2, 'y+3./4'],['y', 0., 'y+3./4'],['y', 1./2, 'y+1./4'],['-y+3./4', 1./2, '-y+1./2'],['-y+3./4', 0., '-y'],['-y+1./4', 1./2, '-y'],['-y+1./4', 0., '-y+1./2'],['-y+1./4', 1./4, 'y+3./4'],['-y+1./4', 3./4, 'y+1./4'],['-y+3./4', 1./4, 'y+1./4'],['-y+3./4', 3./4, 'y+3./4'],['-y', 'y', 3./4],['-y', 'y+1./2', 1./4],['-y+1./2', 'y', 1./4],['-y+1./2', 'y+1./2', 3./4],['y+1./4', 'y+1./2', 0],['y+1./4', 'y', 1./2],['y+3./4', 'y+1./2', 1./2],['y+3./4', 'y', 0],['-y+1./2', '-y+3./4', 1./2],['-y+1./2', '-y+1./4', 0],['-y', '-y+3./4', 0],['-y', '-y+1./4', 1./2],['y+3./4', '-y+1./4', 1./4],['y+3./4', '-y+3./4', 3./4],['y+1./4', '-y+1./4', 3./4],['y+1./4', '-y+3./4', 1./4]],
                        '192h': [['x', 'y', 'z'],['x', 'y+1./2', 'z+1./2'],['x+1./2', 'y', 'z+1./2'],['x+1./2', 'y+1./2', 'z'],['-x+1./4', '-y+3./4', 'z+1./2'],['-x+1./4', '-y+1./4', 'z'],['-x+3./4', '-y+3./4', 'z'],['-x+3./4', '-y+1./4', 'z+1./2'],['-x+3./4', 'y+1./2', '-z+1./4'],['-x+3./4', 'y', '-z+3./4'],['-x+1./4', 'y+1./2', '-z+3./4'],['-x+1./4', 'y', '-z+1./4'],['x+1./2', '-y+1./4', '-z+3./4'],['x+1./2', '-y+3./4', '-z+1./4'],['x', '-y+1./4', '-z+1./4'],['x', '-y+3./4', '-z+3./4'],['z', 'x', 'y'],['z', 'x+1./2', 'y+1./2'],['z+1./2', 'x', 'y+1./2'],['z+1./2', 'x+1./2', 'y'],['z+1./2', '-x+1./4', '-y+3./4'],['z+1./2', '-x+3./4', '-y+1./4'],['z', '-x+1./4', '-y+1./4'],['z', '-x+3./4', '-y+3./4'],['-z+1./4', '-x+3./4', 'y+1./2'],['-z+1./4', '-x+1./4', 'y'],['-z+3./4', '-x+3./4', 'y'],['-z+3./4', '-x+1./4', 'y+1./2'],['-z+3./4', 'x+1./2', '-y+1./4'],['-z+3./4', 'x', '-y+3./4'],['-z+1./4', 'x+1./2', '-y+3./4'],['-z+1./4', 'x', '-y+1./4'],['y', 'z', 'x'],['y', 'z+1./2', 'x+1./2'],['y+1./2', 'z', 'x+1./2'],['y+1./2', 'z+1./2', 'x'],['-y+3./4', 'z+1./2', '-x+1./4'],['-y+3./4', 'z', '-x+3./4'],['-y+1./4', 'z+1./2', '-x+3./4'],['-y+1./4', 'z', '-x+1./4'],['y+1./2', '-z+1./4', '-x+3./4'],['y+1./2', '-z+3./4', '-x+1./4'],['y', '-z+1./4', '-x+1./4'],['y', '-z+3./4', '-x+3./4'],['-y+1./4', '-z+3./4', 'x+1./2'],['-y+1./4', '-z+1./4', 'x'],['-y+3./4', '-z+3./4', 'x'],['-y+3./4', '-z+1./4', 'x+1./2'],['y+3./4', 'x+1./4', '-z'],['y+3./4', 'x+3./4', '-z+1./2'],['y+1./4', 'x+1./4', '-z+1./2'],['y+1./4', 'x+3./4', '-z'],['-y+1./2', '-x+1./2', '-z+1./2'],['-y+1./2', '-x', '-z'],['-y', '-x+1./2', '-z'],['-y', '-x', '-z+1./2'],['y+1./4', '-x', 'z+3./4'],['y+1./4', '-x+1./2', 'z+1./4'],['y+3./4', '-x', 'z+1./4'],['y+3./4', '-x+1./2', 'z+3./4'],['-y', 'x+3./4', 'z+1./4'],['-y', 'x+1./4', 'z+3./4'],['-y+1./2', 'x+3./4', 'z+3./4'],['-y+1./2', 'x+1./4', 'z+1./4'],['x+3./4', 'z+1./4', '-y'],['x+3./4', 'z+3./4', '-y+1./2'],['x+1./4', 'z+1./4', '-y+1./2'],['x+1./4', 'z+3./4', '-y'],['-x', 'z+3./4', 'y+1./4'],['-x', 'z+1./4', 'y+3./4'],['-x+1./2', 'z+3./4', 'y+3./4'],['-x+1./2', 'z+1./4', 'y+1./4'],['-x+1./2', '-z+1./2', '-y+1./2'],['-x+1./2', '-z', '-y'],['-x', '-z+1./2', '-y'],['-x', '-z', '-y+1./2'],['x+1./4', '-z', 'y+3./4'],['x+1./4', '-z+1./2', 'y+1./4'],['x+3./4', '-z', 'y+1./4'],['x+3./4', '-z+1./2', 'y+3./4'],['z+3./4', 'y+1./4', '-x'],['z+3./4', 'y+3./4', '-x+1./2'],['z+1./4', 'y+1./4', '-x+1./2'],['z+1./4', 'y+3./4', '-x'],['z+1./4', '-y', 'x+3./4'],['z+1./4', '-y+1./2', 'x+1./4'],['z+3./4', '-y', 'x+1./4'],['z+3./4', '-y+1./2', 'x+3./4'],['-z', 'y+3./4', 'x+1./4'],['-z', 'y+1./4', 'x+3./4'],['-z+1./2', 'y+3./4', 'x+3./4'],['-z+1./2', 'y+1./4', 'x+1./4'],['-z+1./2', '-y+1./2', '-x+1./2'],['-z+1./2', '-y', '-x'],['-z', '-y+1./2', '-x'],['-z', '-y', '-x+1./2'],['-x', '-y', '-z'],['-x', '-y+1./2', '-z+1./2'],['-x+1./2', '-y', '-z+1./2'],['-x+1./2', '-y+1./2', '-z'],['x+3./4', 'y+1./4', '-z+1./2'],['x+3./4', 'y+3./4', '-z'],['x+1./4', 'y+1./4', '-z'],['x+1./4', 'y+3./4', '-z+1./2'],['x+1./4', '-y+1./2', 'z+3./4'],['x+1./4', '-y', 'z+1./4'],['x+3./4', '-y+1./2', 'z+1./4'],['x+3./4', '-y', 'z+3./4'],['-x+1./2', 'y+3./4', 'z+1./4'],['-x+1./2', 'y+1./4', 'z+3./4'],['-x', 'y+3./4', 'z+3./4'],['-x', 'y+1./4', 'z+1./4'],['-z', '-x', '-y'],['-z', '-x+1./2', '-y+1./2'],['-z+1./2', '-x', '-y+1./2'],['-z+1./2', '-x+1./2', '-y'],['-z+1./2', 'x+3./4', 'y+1./4'],['-z+1./2', 'x+1./4', 'y+3./4'],['-z', 'x+3./4', 'y+3./4'],['-z', 'x+1./4', 'y+1./4'],['z+3./4', 'x+1./4', '-y+1./2'],['z+3./4', 'x+3./4', '-y'],['z+1./4', 'x+1./4', '-y'],['z+1./4', 'x+3./4', '-y+1./2'],['z+1./4', '-x+1./2', 'y+3./4'],['z+1./4', '-x', 'y+1./4'],['z+3./4', '-x+1./2', 'y+1./4'],['z+3./4', '-x', 'y+3./4'],['-y', '-z', '-x'],['-y', '-z+1./2', '-x+1./2'],['-y+1./2', '-z', '-x+1./2'],['-y+1./2', '-z+1./2', '-x'],['y+1./4', '-z+1./2', 'x+3./4'],['y+1./4', '-z', 'x+1./4'],['y+3./4', '-z+1./2', 'x+1./4'],['y+3./4', '-z', 'x+3./4'],['-y+1./2', 'z+3./4', 'x+1./4'],['-y+1./2', 'z+1./4', 'x+3./4'],['-y', 'z+3./4', 'x+3./4'],['-y', 'z+1./4', 'x+1./4'],['y+3./4', 'z+1./4', '-x+1./2'],['y+3./4', 'z+3./4', '-x'],['y+1./4', 'z+1./4', '-x'],['y+1./4', 'z+3./4', '-x+1./2'],['-y+1./4', '-x+3./4', 'z'],['-y+1./4', '-x+1./4', 'z+1./2'],['-y+3./4', '-x+3./4', 'z+1./2'],['-y+3./4', '-x+1./4', 'z'],['y+1./2', 'x+1./2', 'z+1./2'],['y+1./2', 'x', 'z'],['y', 'x+1./2', 'z'],['y', 'x', 'z+1./2'],['-y+3./4', 'x', '-z+1./4'],['-y+3./4', 'x+1./2', '-z+3./4'],['-y+1./4', 'x', '-z+3./4'],['-y+1./4', 'x+1./2', '-z+1./4'],['y', '-x+1./4', '-z+3./4'],['y', '-x+3./4', '-z+1./4'],['y+1./2', '-x+1./4', '-z+1./4'],['y+1./2', '-x+3./4', '-z+3./4'],['-x+1./4', '-z+3./4', 'y'],['-x+1./4', '-z+1./4', 'y+1./2'],['-x+3./4', '-z+3./4', 'y+1./2'],['-x+3./4', '-z+1./4', 'y'],['x', '-z+1./4', '-y+3./4'],['x', '-z+3./4', '-y+1./4'],['x+1./2', '-z+1./4', '-y+1./4'],['x+1./2', '-z+3./4', '-y+3./4'],['x+1./2', 'z+1./2', 'y+1./2'],['x+1./2', 'z', 'y'],['x', 'z+1./2', 'y'],['x', 'z', 'y+1./2'],['-x+3./4', 'z', '-y+1./4'],['-x+3./4', 'z+1./2', '-y+3./4'],['-x+1./4', 'z', '-y+3./4'],['-x+1./4', 'z+1./2', '-y+1./4'],['-z+1./4', '-y+3./4', 'x'],['-z+1./4', '-y+1./4', 'x+1./2'],['-z+3./4', '-y+3./4', 'x+1./2'],['-z+3./4', '-y+1./4', 'x'],['-z+3./4', 'y', '-x+1./4'],['-z+3./4', 'y+1./2', '-x+3./4'],['-z+1./4', 'y', '-x+3./4'],['-z+1./4', 'y+1./2', '-x+1./4'],['z', '-y+1./4', '-x+3./4'],['z', '-y+3./4', '-x+1./4'],['z+1./2', '-y+1./4', '-x+1./4'],['z+1./2', '-y+3./4', '-x+3./4'],['z+1./2', 'y+1./2', 'x+1./2'],['z+1./2', 'y', 'x'],['z', 'y+1./2', 'x'],['z', 'y', 'x+1./2']]},
              '229': {'2a': [[0., 0., 0],[1./2, 1./2, 1./2]],
                      '6b': [[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 0., 1./2],[0., 1./2, 0],[1./2, 1./2, 0],[0., 0., 1./2]],
                      '8c': [[1./4, 1./4, 1./4],[3./4, 3./4, 3./4],[3./4, 3./4, 1./4],[1./4, 1./4, 3./4],[3./4, 1./4, 3./4],[1./4, 3./4, 1./4],[1./4, 3./4, 3./4],[3./4, 1./4, 1./4]],
                      '12d': [[1./4, 0., 1./2],[3./4, 1./2, 0],[3./4, 0., 1./2],[1./4, 1./2, 0],[1./2, 1./4, 0],[0., 3./4, 1./2],[1./2, 3./4, 0],[0., 1./4, 1./2],[0., 1./2, 1./4],[1./2, 0., 3./4],[0., 1./2, 3./4],[1./2, 0., 1./4]],
                      '12e': [['x', 0., 0],['x+1./2', 1./2, 1./2],['-x', 0., 0],['-x+1./2', 1./2, 1./2],[0., 'x', 0],[1./2, 'x+1./2', 1./2],[0., '-x', 0],[1./2, '-x+1./2', 1./2],[0., 0., 'x'],[1./2, 1./2, 'x+1./2'],[0., 0., '-x'],[1./2, 1./2, '-x+1./2']],
                      '16f': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x', '-x', 'x'],['-x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', '-x'],['-x+1./2', 'x+1./2', '-x+1./2'],['x', '-x', '-x'],['x+1./2', '-x+1./2', '-x+1./2'],['x', 'x', '-x'],['x+1./2', 'x+1./2', '-x+1./2'],['-x', '-x', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x', '-x', 'x'],['x+1./2', '-x+1./2', 'x+1./2'],['-x', 'x', 'x'],['-x+1./2', 'x+1./2', 'x+1./2']],
                      '24g': [['x', 0., 1./2],['x+1./2', 1./2, 0],['-x', 0., 1./2],['-x+1./2', 1./2, 0],[1./2, 'x', 0],[0., 'x+1./2', 1./2],[1./2, '-x', 0],[0., '-x+1./2', 1./2],[0., 1./2, 'x'],[1./2, 0., 'x+1./2'],[0., 1./2, '-x'],[1./2, 0., '-x+1./2'],[0., 'x', 1./2],[1./2, 'x+1./2', 0],[0., '-x', 1./2],[1./2, '-x+1./2', 0],['x', 1./2, 0],['x+1./2', 0., 1./2],['-x', 1./2, 0],['-x+1./2', 0., 1./2],[1./2, 0., '-x'],[0., 1./2, '-x+1./2'],[1./2, 0., 'x'],[0., 1./2, 'x+1./2']],
                      '24h': [[0., 'y', 'y'],[1./2, 'y+1./2', 'y+1./2'],[0., '-y', 'y'],[1./2, '-y+1./2', 'y+1./2'],[0., 'y', '-y'],[1./2, 'y+1./2', '-y+1./2'],[0., '-y', '-y'],[1./2, '-y+1./2', '-y+1./2'],['y', 0., 'y'],['y+1./2', 1./2, 'y+1./2'],['y', 0., '-y'],['y+1./2', 1./2, '-y+1./2'],['-y', 0., 'y'],['-y+1./2', 1./2, 'y+1./2'],['-y', 0., '-y'],['-y+1./2', 1./2, '-y+1./2'],['y', 'y', 0],['y+1./2', 'y+1./2', 1./2],['-y', 'y', 0],['-y+1./2', 'y+1./2', 1./2],['y', '-y', 0],['y+1./2', '-y+1./2', 1./2],['-y', '-y', 0],['-y+1./2', '-y+1./2', 1./2]],
                      '48i': [[1./4, 'y', '-y+1./2'],[3./4, 'y+1./2', '-y'],[3./4, '-y', '-y+1./2'],[1./4, '-y+1./2', '-y'],[3./4, 'y', 'y+1./2'],[1./4, 'y+1./2', 'y'],[1./4, '-y', 'y+1./2'],[3./4, '-y+1./2', 'y'],['-y+1./2', 1./4, 'y'],['-y', 3./4, 'y+1./2'],['-y+1./2', 3./4, '-y'],['-y', 1./4, '-y+1./2'],['y+1./2', 3./4, 'y'],['y', 1./4, 'y+1./2'],['y+1./2', 1./4, '-y'],['y', 3./4, '-y+1./2'],['y', '-y+1./2', 1./4],['y+1./2', '-y', 3./4],['-y', '-y+1./2', 3./4],['-y+1./2', '-y', 1./4],['y', 'y+1./2', 3./4],['y+1./2', 'y', 1./4],['-y', 'y+1./2', 1./4],['-y+1./2', 'y', 3./4],[3./4, '-y', 'y+1./2'],[1./4, '-y+1./2', 'y'],[1./4, 'y', 'y+1./2'],[3./4, 'y+1./2', 'y'],[1./4, '-y', '-y+1./2'],[3./4, '-y+1./2', '-y'],[3./4, 'y', '-y+1./2'],[1./4, 'y+1./2', '-y'],['y+1./2', 3./4, '-y'],['y', 1./4, '-y+1./2'],['y+1./2', 1./4, 'y'],['y', 3./4, 'y+1./2'],['-y+1./2', 1./4, '-y'],['-y', 3./4, '-y+1./2'],['-y+1./2', 3./4, 'y'],['-y', 1./4, 'y+1./2'],['-y', 'y+1./2', 3./4],['-y+1./2', 'y', 1./4],['y', 'y+1./2', 1./4],['y+1./2', 'y', 3./4],['-y', '-y+1./2', 1./4],['-y+1./2', '-y', 3./4],['y', '-y+1./2', 3./4],['y+1./2', '-y', 1./4]],
                      '48j': [[0., 'y', 'z'],[1./2, 'y+1./2', 'z+1./2'],[0., '-y', 'z'],[1./2, '-y+1./2', 'z+1./2'],[0., 'y', '-z'],[1./2, 'y+1./2', '-z+1./2'],[0., '-y', '-z'],[1./2, '-y+1./2', '-z+1./2'],['z', 0., 'y'],['z+1./2', 1./2, 'y+1./2'],['z', 0., '-y'],['z+1./2', 1./2, '-y+1./2'],['-z', 0., 'y'],['-z+1./2', 1./2, 'y+1./2'],['-z', 0., '-y'],['-z+1./2', 1./2, '-y+1./2'],['y', 'z', 0],['y+1./2', 'z+1./2', 1./2],['-y', 'z', 0],['-y+1./2', 'z+1./2', 1./2],['y', '-z', 0],['y+1./2', '-z+1./2', 1./2],['-y', '-z', 0],['-y+1./2', '-z+1./2', 1./2],['y', 0., '-z'],['y+1./2', 1./2, '-z+1./2'],['-y', 0., '-z'],['-y+1./2', 1./2, '-z+1./2'],['y', 0., 'z'],['y+1./2', 1./2, 'z+1./2'],['-y', 0., 'z'],['-y+1./2', 1./2, 'z+1./2'],[0., 'z', '-y'],[1./2, 'z+1./2', '-y+1./2'],[0., 'z', 'y'],[1./2, 'z+1./2', 'y+1./2'],[0., '-z', '-y'],[1./2, '-z+1./2', '-y+1./2'],[0., '-z', 'y'],[1./2, '-z+1./2', 'y+1./2'],['z', 'y', 0],['z+1./2', 'y+1./2', 1./2],['z', '-y', 0],['z+1./2', '-y+1./2', 1./2],['-z', 'y', 0],['-z+1./2', 'y+1./2', 1./2],['-z', '-y', 0],['-z+1./2', '-y+1./2', 1./2]],
                      '48k': [['x', 'x', 'z'],['x+1./2', 'x+1./2', 'z+1./2'],['-x', '-x', 'z'],['-x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', '-z'],['-x+1./2', 'x+1./2', '-z+1./2'],['x', '-x', '-z'],['x+1./2', '-x+1./2', '-z+1./2'],['z', 'x', 'x'],['z+1./2', 'x+1./2', 'x+1./2'],['z', '-x', '-x'],['z+1./2', '-x+1./2', '-x+1./2'],['-z', '-x', 'x'],['-z+1./2', '-x+1./2', 'x+1./2'],['-z', 'x', '-x'],['-z+1./2', 'x+1./2', '-x+1./2'],['x', 'z', 'x'],['x+1./2', 'z+1./2', 'x+1./2'],['-x', 'z', '-x'],['-x+1./2', 'z+1./2', '-x+1./2'],['x', '-z', '-x'],['x+1./2', '-z+1./2', '-x+1./2'],['-x', '-z', 'x'],['-x+1./2', '-z+1./2', 'x+1./2'],['x', 'x', '-z'],['x+1./2', 'x+1./2', '-z+1./2'],['-x', '-x', '-z'],['-x+1./2', '-x+1./2', '-z+1./2'],['x', '-x', 'z'],['x+1./2', '-x+1./2', 'z+1./2'],['-x', 'x', 'z'],['-x+1./2', 'x+1./2', 'z+1./2'],['x', 'z', '-x'],['x+1./2', 'z+1./2', '-x+1./2'],['-x', 'z', 'x'],['-x+1./2', 'z+1./2', 'x+1./2'],['-x', '-z', '-x'],['-x+1./2', '-z+1./2', '-x+1./2'],['x', '-z', 'x'],['x+1./2', '-z+1./2', 'x+1./2'],['z', 'x', '-x'],['z+1./2', 'x+1./2', '-x+1./2'],['z', '-x', 'x'],['z+1./2', '-x+1./2', 'x+1./2'],['-z', 'x', 'x'],['-z+1./2', 'x+1./2', 'x+1./2'],['-z', '-x', '-x'],['-z+1./2', '-x+1./2', '-x+1./2']],
                      '96l': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x', '-y', 'z'],['-x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', '-z'],['-x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', '-z'],['x+1./2', '-y+1./2', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z', '-x', '-y'],['z+1./2', '-x+1./2', '-y+1./2'],['-z', '-x', 'y'],['-z+1./2', '-x+1./2', 'y+1./2'],['-z', 'x', '-y'],['-z+1./2', 'x+1./2', '-y+1./2'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z', '-x'],['-y+1./2', 'z+1./2', '-x+1./2'],['y', '-z', '-x'],['y+1./2', '-z+1./2', '-x+1./2'],['-y', '-z', 'x'],['-y+1./2', '-z+1./2', 'x+1./2'],['y', 'x', '-z'],['y+1./2', 'x+1./2', '-z+1./2'],['-y', '-x', '-z'],['-y+1./2', '-x+1./2', '-z+1./2'],['y', '-x', 'z'],['y+1./2', '-x+1./2', 'z+1./2'],['-y', 'x', 'z'],['-y+1./2', 'x+1./2', 'z+1./2'],['x', 'z', '-y'],['x+1./2', 'z+1./2', '-y+1./2'],['-x', 'z', 'y'],['-x+1./2', 'z+1./2', 'y+1./2'],['-x', '-z', '-y'],['-x+1./2', '-z+1./2', '-y+1./2'],['x', '-z', 'y'],['x+1./2', '-z+1./2', 'y+1./2'],['z', 'y', '-x'],['z+1./2', 'y+1./2', '-x+1./2'],['z', '-y', 'x'],['z+1./2', '-y+1./2', 'x+1./2'],['-z', 'y', 'x'],['-z+1./2', 'y+1./2', 'x+1./2'],['-z', '-y', '-x'],['-z+1./2', '-y+1./2', '-x+1./2'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x', 'y', '-z'],['x+1./2', 'y+1./2', '-z+1./2'],['x', '-y', 'z'],['x+1./2', '-y+1./2', 'z+1./2'],['-x', 'y', 'z'],['-x+1./2', 'y+1./2', 'z+1./2'],['-z', '-x', '-y'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z', 'x', 'y'],['-z+1./2', 'x+1./2', 'y+1./2'],['z', 'x', '-y'],['z+1./2', 'x+1./2', '-y+1./2'],['z', '-x', 'y'],['z+1./2', '-x+1./2', 'y+1./2'],['-y', '-z', '-x'],['-y+1./2', '-z+1./2', '-x+1./2'],['y', '-z', 'x'],['y+1./2', '-z+1./2', 'x+1./2'],['-y', 'z', 'x'],['-y+1./2', 'z+1./2', 'x+1./2'],['y', 'z', '-x'],['y+1./2', 'z+1./2', '-x+1./2'],['-y', '-x', 'z'],['-y+1./2', '-x+1./2', 'z+1./2'],['y', 'x', 'z'],['y+1./2', 'x+1./2', 'z+1./2'],['-y', 'x', '-z'],['-y+1./2', 'x+1./2', '-z+1./2'],['y', '-x', '-z'],['y+1./2', '-x+1./2', '-z+1./2'],['-x', '-z', 'y'],['-x+1./2', '-z+1./2', 'y+1./2'],['x', '-z', '-y'],['x+1./2', '-z+1./2', '-y+1./2'],['x', 'z', 'y'],['x+1./2', 'z+1./2', 'y+1./2'],['-x', 'z', '-y'],['-x+1./2', 'z+1./2', '-y+1./2'],['-z', '-y', 'x'],['-z+1./2', '-y+1./2', 'x+1./2'],['-z', 'y', '-x'],['-z+1./2', 'y+1./2', '-x+1./2'],['z', '-y', '-x'],['z+1./2', '-y+1./2', '-x+1./2'],['z', 'y', 'x'],['z+1./2', 'y+1./2', 'x+1./2']]},
              '230': {'16a': [[0., 0., 0],[1./2, 1./2, 1./2],[1./2, 0., 1./2],[0., 1./2, 0],[0., 1./2, 1./2],[1./2, 0., 0],[1./2, 1./2, 0],[0., 0., 1./2],[3./4, 1./4, 1./4],[1./4, 3./4, 3./4],[3./4, 3./4, 3./4],[1./4, 1./4, 1./4],[1./4, 1./4, 3./4],[3./4, 3./4, 1./4],[1./4, 3./4, 1./4],[3./4, 1./4, 3./4]],
                      '16b': [[1/8, 1/8, 1/8],[5/8, 5/8, 5/8],[3/8, 7/8, 5/8],[7/8, 3/8, 1/8],[7/8, 5/8, 3/8],[3/8, 1/8, 7/8],[5/8, 3/8, 7/8],[1/8, 7/8, 3/8],[7/8, 7/8, 7/8],[3/8, 3/8, 3/8],[5/8, 1/8, 3/8],[1/8, 5/8, 7/8],[1/8, 3/8, 5/8],[5/8, 7/8, 1/8],[3/8, 5/8, 1/8],[7/8, 1/8, 5/8]],
                      '24c': [[1/8, 0., 1./4],[5/8, 1./2, 3./4],[3/8, 0., 3./4],[7/8, 1./2, 1./4],[1./4, 1/8, 0],[3./4, 5/8, 1./2],[3./4, 3/8, 0],[1./4, 7/8, 1./2],[0., 1./4, 1/8],[1./2, 3./4, 5/8],[0., 3./4, 3/8],[1./2, 1./4, 7/8],[7/8, 0., 3./4],[3/8, 1./2, 1./4],[5/8, 0., 1./4],[1/8, 1./2, 3./4],[3./4, 7/8, 0],[1./4, 3/8, 1./2],[1./4, 5/8, 0],[3./4, 1/8, 1./2],[0., 3./4, 7/8],[1./2, 1./4, 3/8],[0., 1./4, 5/8],[1./2, 3./4, 1/8]],
                      '24d': [[3/8, 0., 1./4],[7/8, 1./2, 3./4],[1/8, 0., 3./4],[5/8, 1./2, 1./4],[1./4, 3/8, 0],[3./4, 7/8, 1./2],[3./4, 1/8, 0],[1./4, 5/8, 1./2],[0., 1./4, 3/8],[1./2, 3./4, 7/8],[0., 3./4, 1/8],[1./2, 1./4, 5/8],[3./4, 5/8, 0],[1./4, 1/8, 1./2],[3./4, 3/8, 1./2],[1./4, 7/8, 0],[1/8, 1./2, 1./4],[5/8, 0., 3./4],[7/8, 0., 1./4],[3/8, 1./2, 3./4],[0., 1./4, 7/8],[1./2, 3./4, 3/8],[1./2, 1./4, 1/8],[0., 3./4, 5/8]],
                      '32e': [['x', 'x', 'x'],['x+1./2', 'x+1./2', 'x+1./2'],['-x+1./2', '-x', 'x+1./2'],['-x', '-x+1./2', 'x'],['-x', 'x+1./2', '-x+1./2'],['-x+1./2', 'x', '-x'],['x+1./2', '-x+1./2', '-x'],['x', '-x', '-x+1./2'],['x+3./4', 'x+1./4', '-x+1./4'],['x+1./4', 'x+3./4', '-x+3./4'],['-x+3./4', '-x+3./4', '-x+3./4'],['-x+1./4', '-x+1./4', '-x+1./4'],['x+1./4', '-x+1./4', 'x+3./4'],['x+3./4', '-x+3./4', 'x+1./4'],['-x+1./4', 'x+3./4', 'x+1./4'],['-x+3./4', 'x+1./4', 'x+3./4'],['-x', '-x', '-x'],['-x+1./2', '-x+1./2', '-x+1./2'],['x+1./2', 'x', '-x+1./2'],['x', 'x+1./2', '-x'],['x', '-x+1./2', 'x+1./2'],['x+1./2', '-x', 'x'],['-x+1./2', 'x+1./2', 'x'],['-x', 'x', 'x+1./2'],['-x+1./4', '-x+3./4', 'x+3./4'],['-x+3./4', '-x+1./4', 'x+1./4'],['x+1./4', 'x+1./4', 'x+1./4'],['x+3./4', 'x+3./4', 'x+3./4'],['-x+3./4', 'x+3./4', '-x+1./4'],['-x+1./4', 'x+1./4', '-x+3./4'],['x+3./4', '-x+1./4', '-x+3./4'],['x+1./4', '-x+3./4', '-x+1./4']],
                      '48f': [['x', 0., 1./4],['x+1./2', 1./2, 3./4],['-x+1./2', 0., 3./4],['-x', 1./2, 1./4],[1./4, 'x', 0],[3./4, 'x+1./2', 1./2],[3./4, '-x+1./2', 0],[1./4, '-x', 1./2],[0., 1./4, 'x'],[1./2, 3./4, 'x+1./2'],[0., 3./4, '-x+1./2'],[1./2, 1./4, '-x'],[3./4, 'x+1./4', 0],[1./4, 'x+3./4', 1./2],[3./4, '-x+3./4', 1./2],[1./4, '-x+1./4', 0],['x+3./4', 1./2, 1./4],['x+1./4', 0., 3./4],['-x+1./4', 0., 1./4],['-x+3./4', 1./2, 3./4],[0., 1./4, '-x+1./4'],[1./2, 3./4, '-x+3./4'],[1./2, 1./4, 'x+3./4'],[0., 3./4, 'x+1./4'],['-x', 0., 3./4],['-x+1./2', 1./2, 1./4],['x+1./2', 0., 1./4],['x', 1./2, 3./4],[3./4, '-x', 0],[1./4, '-x+1./2', 1./2],[1./4, 'x+1./2', 0],[3./4, 'x', 1./2],[0., 3./4, '-x'],[1./2, 1./4, '-x+1./2'],[0., 1./4, 'x+1./2'],[1./2, 3./4, 'x'],[1./4, '-x+3./4', 0],[3./4, '-x+1./4', 1./2],[1./4, 'x+1./4', 1./2],[3./4, 'x+3./4', 0],['-x+1./4', 1./2, 3./4],['-x+3./4', 0., 1./4],['x+3./4', 0., 3./4],['x+1./4', 1./2, 1./4],[0., 3./4, 'x+3./4'],[1./2, 1./4, 'x+1./4'],[1./2, 3./4, '-x+1./4'],[0., 1./4, '-x+3./4']],
                      '48g': [[1/8, 'y', '-y+1./4'],[5/8, 'y+1./2', '-y+3./4'],[3/8, '-y', '-y+3./4'],[7/8, '-y+1./2', '-y+1./4'],[7/8, 'y+1./2', 'y+1./4'],[3/8, 'y', 'y+3./4'],[5/8, '-y+1./2', 'y+3./4'],[1/8, '-y', 'y+1./4'],['-y+1./4', 1/8, 'y'],['-y+3./4', 5/8, 'y+1./2'],['-y+3./4', 3/8, '-y'],['-y+1./4', 7/8, '-y+1./2'],['y+1./4', 7/8, 'y+1./2'],['y+3./4', 3/8, 'y'],['y+3./4', 5/8, '-y+1./2'],['y+1./4', 1/8, '-y'],['y', '-y+1./4', 1/8],['y+1./2', '-y+3./4', 5/8],['-y', '-y+3./4', 3/8],['-y+1./2', '-y+1./4', 7/8],['y+1./2', 'y+1./4', 7/8],['y', 'y+3./4', 3/8],['-y+1./2', 'y+3./4', 5/8],['-y', 'y+1./4', 1/8],[7/8, '-y', 'y+3./4'],[3/8, '-y+1./2', 'y+1./4'],[5/8, 'y', 'y+1./4'],[1/8, 'y+1./2', 'y+3./4'],[1/8, '-y+1./2', '-y+3./4'],[5/8, '-y', '-y+1./4'],[3/8, 'y+1./2', '-y+1./4'],[7/8, 'y', '-y+3./4'],['y+3./4', 7/8, '-y'],['y+1./4', 3/8, '-y+1./2'],['y+1./4', 5/8, 'y'],['y+3./4', 1/8, 'y+1./2'],['-y+3./4', 1/8, '-y+1./2'],['-y+1./4', 5/8, '-y'],['-y+1./4', 3/8, 'y+1./2'],['-y+3./4', 7/8, 'y'],['-y', 'y+3./4', 7/8],['-y+1./2', 'y+1./4', 3/8],['y', 'y+1./4', 5/8],['y+1./2', 'y+3./4', 1/8],['-y+1./2', '-y+3./4', 1/8],['-y', '-y+1./4', 5/8],['y+1./2', '-y+1./4', 3/8],['y', '-y+3./4', 7/8]],
                      '96h': [['x', 'y', 'z'],['x+1./2', 'y+1./2', 'z+1./2'],['-x+1./2', '-y', 'z+1./2'],['-x', '-y+1./2', 'z'],['-x', 'y+1./2', '-z+1./2'],['-x+1./2', 'y', '-z'],['x+1./2', '-y+1./2', '-z'],['x', '-y', '-z+1./2'],['z', 'x', 'y'],['z+1./2', 'x+1./2', 'y+1./2'],['z+1./2', '-x+1./2', '-y'],['z', '-x', '-y+1./2'],['-z+1./2', '-x', 'y+1./2'],['-z', '-x+1./2', 'y'],['-z', 'x+1./2', '-y+1./2'],['-z+1./2', 'x', '-y'],['y', 'z', 'x'],['y+1./2', 'z+1./2', 'x+1./2'],['-y', 'z+1./2', '-x+1./2'],['-y+1./2', 'z', '-x'],['y+1./2', '-z+1./2', '-x'],['y', '-z', '-x+1./2'],['-y+1./2', '-z', 'x+1./2'],['-y', '-z+1./2', 'x'],['y+3./4', 'x+1./4', '-z+1./4'],['y+1./4', 'x+3./4', '-z+3./4'],['-y+3./4', '-x+3./4', '-z+3./4'],['-y+1./4', '-x+1./4', '-z+1./4'],['y+1./4', '-x+1./4', 'z+3./4'],['y+3./4', '-x+3./4', 'z+1./4'],['-y+1./4', 'x+3./4', 'z+1./4'],['-y+3./4', 'x+1./4', 'z+3./4'],['x+3./4', 'z+1./4', '-y+1./4'],['x+1./4', 'z+3./4', '-y+3./4'],['-x+1./4', 'z+3./4', 'y+1./4'],['-x+3./4', 'z+1./4', 'y+3./4'],['-x+3./4', '-z+3./4', '-y+3./4'],['-x+1./4', '-z+1./4', '-y+1./4'],['x+1./4', '-z+1./4', 'y+3./4'],['x+3./4', '-z+3./4', 'y+1./4'],['z+3./4', 'y+1./4', '-x+1./4'],['z+1./4', 'y+3./4', '-x+3./4'],['z+1./4', '-y+1./4', 'x+3./4'],['z+3./4', '-y+3./4', 'x+1./4'],['-z+1./4', 'y+3./4', 'x+1./4'],['-z+3./4', 'y+1./4', 'x+3./4'],['-z+3./4', '-y+3./4', '-x+3./4'],['-z+1./4', '-y+1./4', '-x+1./4'],['-x', '-y', '-z'],['-x+1./2', '-y+1./2', '-z+1./2'],['x+1./2', 'y', '-z+1./2'],['x', 'y+1./2', '-z'],['x', '-y+1./2', 'z+1./2'],['x+1./2', '-y', 'z'],['-x+1./2', 'y+1./2', 'z'],['-x', 'y', 'z+1./2'],['-z', '-x', '-y'],['-z+1./2', '-x+1./2', '-y+1./2'],['-z+1./2', 'x+1./2', 'y'],['-z', 'x', 'y+1./2'],['z+1./2', 'x', '-y+1./2'],['z', 'x+1./2', '-y'],['z', '-x+1./2', 'y+1./2'],['z+1./2', '-x', 'y'],['-y', '-z', '-x'],['-y+1./2', '-z+1./2', '-x+1./2'],['y', '-z+1./2', 'x+1./2'],['y+1./2', '-z', 'x'],['-y+1./2', 'z+1./2', 'x'],['-y', 'z', 'x+1./2'],['y+1./2', 'z', '-x+1./2'],['y', 'z+1./2', '-x'],['-y+1./4', '-x+3./4', 'z+3./4'],['-y+3./4', '-x+1./4', 'z+1./4'],['y+1./4', 'x+1./4', 'z+1./4'],['y+3./4', 'x+3./4', 'z+3./4'],['-y+3./4', 'x+3./4', '-z+1./4'],['-y+1./4', 'x+1./4', '-z+3./4'],['y+3./4', '-x+1./4', '-z+3./4'],['y+1./4', '-x+3./4', '-z+1./4'],['-x+1./4', '-z+3./4', 'y+3./4'],['-x+3./4', '-z+1./4', 'y+1./4'],['x+3./4', '-z+1./4', '-y+3./4'],['x+1./4', '-z+3./4', '-y+1./4'],['x+1./4', 'z+1./4', 'y+1./4'],['x+3./4', 'z+3./4', 'y+3./4'],['-x+3./4', 'z+3./4', '-y+1./4'],['-x+1./4', 'z+1./4', '-y+3./4'],['-z+1./4', '-y+3./4', 'x+3./4'],['-z+3./4', '-y+1./4', 'x+1./4'],['-z+3./4', 'y+3./4', '-x+1./4'],['-z+1./4', 'y+1./4', '-x+3./4'],['z+3./4', '-y+1./4', '-x+3./4'],['z+1./4', '-y+3./4', '-x+1./4'],['z+1./4', 'y+1./4', 'x+1./4'],['z+3./4', 'y+3./4', 'x+3./4']]}}

## P primitive; I body centered; F face centered; A centered on A faces only;
## B centered on B faces only; C centered on C faces only; R rhombohedral
SPACEGROUPS = {'1':['B1','C1','A1','F1','I1'],
               '2':['B-1','C-1','P-1','A-1','F-1','I-1','P1','P111','F1-11'],
               '3':['B121','C112','P2','A211','P112','P121','P211'],
               '4':['B1211','C1121','P21','A2111','P1121','P1211','P2111','P12121211'],
               '5':['A121','B112','C2','A112','B211','C121','C211','F112','F121','F211','I112','I121','I211','B2','I2'],
               '6':['B1m1','C11m','Pm','Am11','P11m','P1m1','Pm11'],
               '7':['Ad11','B1a1','C11d','Ab11','B1d1','C11a','P11a','P11b','P11n','P1a1','P1c1','P1n1','Pb11','Pc','Pc11','Pn11'],
               '8':['A1m1','B11m','Cm','A11m','Bm11','C1m1','Cm11','F11m','F1m1','Fm11','I11m','I1m1','Im11'],
               '9':['A1a1','B11b','Cc','A11a','Bb11','C1c1','Cc11','F11d','F1d1','Fd11','I11a','I11b','I1a1','Ib11'],
               '10':['B12/m1','C112/m','P2/m','A2/m11','P112/m','P12/m1','P2/m11'],
               '11':['B121/m1','C1121/m','P21/m','A21/m11','P1121/m','P121/m1','P21/m11','P1-11','P112/c'],
               '12':['A12/m1','B112/m','C2/m','A112/m','B2/m11','C12/m1','C2/m11','F112/m','F12/m1','F2/m11','I112/m','I12/m1','I2/m11','C1-11','C111'],
               '13':['A2/d11','B12/a1','C112/d','A2/b11','B12/d1','C112/a','P112/a','P112/b','P112/n','P12/a1','P12/c1','P12/n1','P2/b11','P2/c','P2/c11','P2/n11'],
               '14':['A21/d11','B121/a1','C1121/d','A21/b11','B121/d1','C1121/a','P1121/a','P1121/b','P1121/n','P1121/b','P121/a1','P121/c1','P121/n1','P121/c1','P21/b11','P21/c11','P21/n11','P21/c','P121/a1','P121/n1','P21/b11','P1121/n','B121/d1','B121/m1','P21/n11','P1121/a','P21/c','P21/n','B1121/d'],
               '15':['A12/a1','A12/n1','B2/b11','A112/a','B112/b','B112/n','C112/a','C12/c1','C2/c','C2/c11','F112/d','F12/d1','F2/d11','I112/a','I112/b','I12/a1','I12/c1','I2/b11','P1b1'],
               '16':['P222'],
               '17':['P2212','P222_1','P2122','P2221'],
               '18':['P2_122_1','P21212','P22121','P2_12_12','P21221','P22_12_1'],
               '19':['P212121','P2_12_12_1'],
               '20':['A2122','B2212','C2221','A2_122','C222_1','C2221','C1211'],
               '21':['B222','C222','A222'],
               '22':['F222'],
               '23':['I222'],
               '24':['I212121','I2_12_12_1'],
               '25':['Pm2m','Pmm2','P2mm'],
               '26':['P2_1ma','P21am','Pb21m','P2_1am','P21ma','Pb2_1m','Pcm2_1','Pcm21','Pm21b','Pmc2_1','Pmc21'],
               '27':['Pb2b','Pcc2','P2aa'],
               '28':['P2cm','P2mb','Pc2m','P2cm','Pbm2','Pbm2','Pm2a','Pma2'],
               '29':['P2_1ca','P21ab','Pbc2_1','P2_1ab','P21ca','Pb21a','Pbc21','Pc2_1b','Pc21b','Pca2_1','Pca21'],
               '30':['P2an','P2na','Pn2b','P2an','Pb2n','Pcn2','Pnc2'],
               '31':['P2_1nm','P21mn','Pmn2_1','P2_1mn','P21nm','Pm21n','Pmn21','Pn2_1m','Pn21m','Pnm2_1','Pnm21'],
               '32':['Pba2','Pc2a','P2cb'],
               '33':['P2_1nb','P21cn','Pbn21','P2_1cn','P21nb','Pbn2_1','Pc2_1n','Pc21n','Pn2_1a','Pn21a','Pna21'],
               '34':['P2nn','Pn2n','P2nn','Pnn2'],
               '35':['A2mm','Bm2m','A2mm','Cmm2'],
               '36':['A2_1ma','A21am','Bb21m','A2_1am','A21ma','Bb2_1m','Bm21b','Ccm2_1','Ccm21','Cmc2_1','Cmc21'],
               '37':['Bb2b','Ccc2','A2aa'],
               '38':['Amm2','B2mm','Cm2m','Am2m','Bmm2','C2mm'],
               '39':['Abm2','Aem2','C2ma','Ab2m','B2am','Bma2','Cm2a'],
               '40':['Ama2','B2mb','Cc2m','Am2a','Bbm2','C2cm'],
               '41':['Aba2','Aea2','C2ca','Ab2a','B2ab','Bba2','C2cb','Cc2a'],
               '42':['Fmm2','Fm2m','Fmm2','F2mm'],
               '43':['Fd2d','Fdd2','F2dd','Fdd211'],
               '44':['Im2m','Imm2','I2mm'],
               '45':['Ib2a','Iba2','I2aa'],
               '46':['I2cm','I2ma','Ibm2','I2am','I2mb','Ib2m','Im2a','Ima2'],
               '47':['Pmmm','P2/m2/m2/m'],
               '48':['Pnnn','P2/n2/n2/n'],
               '49':['Pbmb','Pccm','P2/c2/c2/m','Pmaa','Pccm'],
               '50':['Pban','Pcna','P2/b2/a2/n','Pncb'],
               '51':['Pbmm','Pcmm','Pmma','P21/m2/m2/a','Pmam','Pmcm','Pmmb'],
               '52':['Pbnn','Pcnn','Pnna','P2/n21/n2/a','Pnan','Pncn','Pnnb'],
               '53':['P21/c2/c2/a','Pbmn','Pmna','P2/m2/n21/a','Pcnm','Pman','Pncm','Pnmb'],
               '54':['Pbab','Pbcb','Pbaa','Pcca','Pccb'],
               '55':['Pbam','Pcma','P21/b21/a2/m','Pmcb'],
               '56':['Pbnb','Pccn','P21/c21/c2/n','Pnaa'],
               '57':['Pbcm','Pbma','Pmab','P2/b21/c21/m','Pcam','Pcmb','Pmca'],
               '58':['Pmnn','Pnmn','P21/n21/n2/m','Pnnm'],
               '59':['Pmmn','Pmnm','P21/m21/m2/n','Pnmm'],
               '60':['Pbcn','Pbna','Pnab','P21/b2/c21/n','Pcan','Pcnb','Pnca'],
               '61':['P21/b21/c21/a','Pbca','Pcab'],
               '62':['P1nam1','P1nma1','P21/n21/m21/a','Pbnm','Pcmn','Pmcn','Pmnb','Pnam','Pnma'],
               '63':['Amam','Amma','Bbmm','Bmmb','C2/m2/c21/m','Ccmm','Cmcm'],
               '64':['Abam','Abma','Acam','Bbam','Bbcm','Bmab','C2/m2/c21/a','Ccma','Ccmb','Cmca','Cmce'],
               '65':['Ammm','C2/m2/m2/m','Cmmm'],
               '66':['Amaa','Bbmb','C2/c2/c2/m','Cccm'],
               '67':['Abmm','Acmm','Bmam','C2/m2/m2/e','Cmma','Cmme','Cmmb'],
               '68':['Abaa','Bbab','C2/c2/c2/e','Ccca','Ccce'],
               '69':['F2/m2/m2/m','Fmmm'],
               '70':['F2/d2/d2/d','Fddd'],
               '71':['I2/m2/m2/m','Immm'],
               '72':['I2/b2/a2/m','Ibam','Ibma','Icma','Imaa','Imab','Imcb'],
               '73':['I2/b2/c2/a','Ibca'],
               '74':['I1mma1','I2/m2/m2/a','Ibmm','Imam','Imcm','Imma'],
               '75':['C4','P4'],
               '76':['C41','P4_1','P41'],
               '77':['C42','P4_2','P42'],
               '78':['C43','P4_3','P43'],
               '79':['F4','I4'],
               '80':['F41','I41'],
               '81':['C-4','P-4'],
               '82':['F-4','I-4'],
               '83':['C4/m','P4/m'],
               '84':['C42/m','P4_2/m','P42/m'],
               '85':['C4/a','P4/n'],
               '86':['C42/a','P4_2/n','P42/n'],
               '87':['F4/m','I4/m'],
               '88':['F41/d','I4_1/a','I41/a'],
               '89':['C422','P422'],
               '90':['C4221','P4212'],
               '91':['C4122','P4_122','P4122'],
               '92':['C41221','P4_12_12','P41212'],
               '93':['C4222','P4222'],
               '94':['C42221','P42212'],
               '95':['C4322','P4_322','P4322'],
               '96':['C43221','P4_32_12','P43212'],
               '97':['F422','I422'],
               '98':['F4122','I4_122','I4122'],
               '99':['C4mm','P4mm'],
               '100':['C4mb','P4bm'],
               '101':['C42mc','P42cm'],
               '102':['C42mn','P4_2nm','P42nm'],
               '103':['C4cc','P4cc'],
               '104':['C4cn','P4nc'],
               '105':['C42cm','P4_2mc','P42mc'],
               '106':['C42cb','P42bc'],
               '107':['F4mm','I4mm'],
               '108':['F4mc','I4cm'],
               '109':['F41dm','I41md'],
               '110':['F41dc','I4_1cd','I41cd'],
               '111':['C-4m2','P-42m','P42m'],
               '112':['C-4c2','P-42c','P42c'],
               '113':['C-4m21','P-421m','P421m'],
               '114':['C-4c21','P-421c','P421c'],
               '115':['C-42m','P-4m2','P4m2'],
               '116':['C-42c','P-4c2','P4c2'],
               '117':['C-42b','P-4b2','P4b2'],
               '118':['C-42n','P-4n2','P4n2'],
               '119':['F-42m','I-4m2','I4m2'],
               '120':['F-42c','I-4c2','I4c2'],
               '121':['F-4m2','I-42m','I42m'],
               '122':['F-4d2','I-42d','I42d'],
               '123':['C4/mmm','P4/m2/m2/m','P4/mmm'],
               '124':['C4/mcc','P4/m2/c2/c','P4/mcc'],
               '125':['C4/amb','P4/n2/b2/m','P4/nbm'],
               '126':['C4/acn','P4/n2/n2/c','P4/nnc'],
               '127':['C4/mmb','P4/m21/b2/m','P4/mbm'],
               '128':['C4/mcn','P4/m21/n2/c','P4/mnc'],
               '129':['C4/amm','P4/n21/m2/m','P4/nmm'],
               '130':['C4/acc','P4/n21/c2/c','P4/ncc'],
               '131':['C42/mcm','P4_2/mmc','P42/m2/m2/c','P42/mmc'],
               '132':['C42/mmc','P4_2/mcm','P42/m2/c2/m','P42/mcm'],
               '133':['C42/acb','P4_2/nbc','P42/n2/b2/c','P42/nbc'],
               '134':['C42/anm','P4_2/nnm','P42/n2/n2/m','P42/nnm'],
               '135':['C42/mcb','P42/m21/b2/c','P42/mbc'],
               '136':['C42/mmn','P4_2/mnm','P42/m21/n2/m','P42/mnm'],
               '137':['C42/acm','P4_2/nmc','P42/n21/m2/c','P42/nmc'],
               '138':['C42/amc','P4_2/ncm','P42/n21/c2/m','P42/ncm'],
               '139':['F4/mmm','I4/m2/m2/m','I4/mmm'],
               '140':['F4/mmc','I4/m2/c2/m','I4/mcm'],
               '141':['141','F41/ddm','I4_1/amd','I41/a2/m2/d','I41/amd'],
               '142':['F41/ddc','I4_1/acd','I41/a2/c2/d','I41/acd','I2/a'],
               '143':['H3','P3'],
               '144':['H31','P3_1','P31'],
               '145':['H32','P3_2','P32'],
               '146':['R3'],
               '147':['H-3','P-3'],
               '148':['R-3'],
               '149':['H321','P312'],
               '150':['H312','P321'],
               '151':['H3121','P3_112','P3112'],
               '152':['H3112','P3_121','P3121'],
               '153':['H3221','P3_212','P3212'],
               '154':['H3212','P3_221','P3221'],
               '155':['R32'],
               '156':['H31m','P3m1'],
               '157':['H3m1','P31m'],
               '158':['H31c','P3c1'],
               '159':['H3c1','P31c'],
               '160':['R3m'],
               '161':['R3c'],
               '162':['H-3m1','P-31m'],
               '163':['H-3c1','P-31c'],
               '164':['H-31m','P-3m1'],
               '165':['H-31c','P-3c1'],
               '166':['R-3m'],
               '167:H':['R-32/c','R-3c','R-3cH','R-3H'],
               '167:R':['R-3cR'],
               '168':['P6'],
               '169':['P6_1','P61'],
               '170':['P6_5','P65'],
               '171':['P62'],
               '172':['P64'],
               '173':['P6_3','P63'],
               '174':['P-6'],
               '175':['P6/m'],
               '176':['P63/m','P6_3/m'],
               '177':['P622'],
               '178':['P6122'],
               '179':['P6_522','P6522'],
               '180':['P6_222','P6222'],
               '181':['P6_422','P6422'],
               '182':['P6_322','P6322'],
               '183':['P6mm'],
               '184':['P6cc'],
               '185':['P6_3cm','P63cm'],
               '186':['P6_3mc','P63mc'],
               '187':['P-6m2','P6m2'],
               '188':['P-6c2','P6c2'],
               '189':['P-62m','P62m'],
               '190':['P-62c','P62c'],
               '191':['P6/m2/m2/m','P6/mmm'],
               '192':['P6/mcc','P6/m2/c2/c'],
               '193':['P6_3/mcm','P63/m2/c2/m','P63/mcm'],
               '194':['P6_3/mmc','P63/m2/m2/c','P63/mmc'],
               '195':['P23'],
               '196':['F23'],
               '197':['I23'],
               '198':['P2_13','P213'],
               '199':['I2_13','I213'],
               '200':['Pm-3','Pm3'],
               '201':['Pn-3','Pn3'],
               '202':['Fm-3','Fm3'],
               '203':['Fd-3','Fd3'],
               '204':['Im-3','Im3'],
               '205':['Pa-3','Pa3'],
               '206':['Ia-3','Ia3','I21/a-3'],
               '207':['P432'],
               '208':['P4_232','P4232'],
               '209':['F432'],
               '210':['F4132'],
               '211':['I432','Pm-3m'],
               '212':['P4_332','P4332'],
               '213':['P4_132','P4132'],
               '214':['I4_132','I4132'],
               '215':['P-43m','P43m'],
               '216':['F-43m','F43m'],
               '217':['I-43m','I43m'],
               '218':['P-43n','P43n'],
               '219':['F-43c','F43c'],
               '220':['I-43d','I43d'],
               '221':['P4/m-32/m','Pm-3m','Pm3m'],
               '222':['Pn-3n','Pn3n'],
               '223':['Pm-3n','Pm3n'],
               '224':['Pn-3m','Pn3m'],
               '225':['Fm3m','Fm-3m'],
               '226':['Fm-3c','Fm3c'],
               '227':['Fd-3m','Fd3m'],
               '228':['Fd-3c','Fd3c'],
               '229':['Im-3m','Im3m'],
               '230':['Ia-3d','Ia3d']}

##########################################################################
# FUNCTIONS


class Symmetry(object):
    def __init__(self):
        self.no = None
        self.name = None
        self.type = None
        self.xyz = None
        self.xyz_id = None

class Atom(object):
    def __init__(self):
        self.site_H = None
        self.B_iso = None
        self.label = None
        self.label2 = None
        self.occupancy = None
        self.fract_x = None
        self.fract_y = None
        self.fract_z = None
        self.oxid_no = None
        #### gives 4a or 8c, etc.
        self.symm_multi = None
        self.symm_wyckoff = None

class Citation(object):
    def __init__(self):
        self.year = None
        self.author = None
        self.journal = None
        self.journal_id = None
        self.title = None
        self.id = None
        self.volume = None
        self.first_page = None
        self.last_page = None

class XRDCIF(object):
    '''
    CIF data class for XRD

    Attributes:
    ------------
    * unitcell       = [5.4, 5.4, 5.4, 90, 90, 90] # [a,b,c,alpha,beta,gamma]
    * symmetry.no    = 225                         # space group number
    * symmetry.name  = 'F m 3 m'                   # space group name

    Methods
    -------
    read_ciffile(ciffile)
    read_ciftext(ciftext)
    structure_factors(wavelength=1.54056, q_min=0.2, q_max=10.0)

    mkak 2017.04.13
    '''

    def __init__(self, text=None):

        self.label   = None
        self.formula = None
        self.ciffile = None
        self.id_no   = None

        self.unitcell = np.array([0,0,0,0,0,0], dtype=np.float32)
        self.density  = None
        self.volume   = None

        self.symmetry    = Symmetry()
        self.atom        = Atom()
        self.publication = Citation()
        self.formula = None
        self.elem_uvw = {}
        if text is not None:
            self.read_ciftext(text)

    def read_ciffile(self, ciffile, verbose=False):
        if not os.path.exists(ciffile):
            print("cif file '%s' not found" % repr(ciffile))
            return

        if verbose:
            print( '\n=== Reading file: %s\n' % ciffile)

        self.ciffile = ciffile
        textlines = []
        # hack? to ignore multiple 'chemical_formula_sum',
        # has can happen with AMCSD
        has_formula_sum = False
        with open(ciffile, 'r') as fh:
            for line in fh.readlines():
                if line.startswith('_chemical_formula_sum'):
                    if not has_formula_sum:
                        textlines.append(line)
                    has_formula_sum = True
                else:
                    textlines.append(line)
        self.read_ciftext(''.join(textlines))

    def read_ciftext(self, ciftext, verbose=False):
        if not HAS_CifFile:
            print('must install pycifrw to extract structure factors')
            return

        cf = CifFile.ReadCif(StringIO(ciftext))
        key = cf.keys()[0]
        for k0 in cf[key].keys():
            k = k0.lower()
            ## name/label
            if k in ('_chemical_name_mineral', '_chemical_name_systematic',
                     '_amcsd_formula_title'):
                self.label = cf[key][k]
            elif k in ('_chemical_formula_structural','_chemical_formula_sum'):
                self.formula = cf[key][k]
            elif k in ('_database_code_amcsd','_database_code_icsd'):
                self.id_no = int(cf[key][k])

            ## unit cell information
            elif k == '_cell_length_a':
                if cf[key][k] == '3,2518':
                    self.unitcell[0] = '3.2518' ## typo in cif 14130
                elif cf[key][k][-1] == ',':
                    self.unitcell[0] = re.sub(r"\(\w*\)", r"", cf[key][k][0:-1])
                else:
                    self.unitcell[0] = re.sub(r"\(\w*\)", r"", cf[key][k])
            elif k == '_cell_length_b':
                if cf[key][k][-1] == ',':
                    self.unitcell[1] = re.sub(r"\(\w*\)", r"", cf[key][k][0:-1])
                else:
                    self.unitcell[1] =re.sub(r"\(\w*\)", r"", cf[key][k])
            elif k == '_cell_length_c':
                self.unitcell[2] = re.sub(r"\(\w*\)", r"", cf[key][k])
            elif k == '_cell_angle_alpha':
                self.unitcell[3] = re.sub(r"\(\w*\)", r"", cf[key][k])
            elif k == '_cell_angle_beta':
                self.unitcell[4] = re.sub(r"\(\w*\)", r"", cf[key][k])
            elif k == '_cell_angle_gamma':
                self.unitcell[5] = re.sub(r"\(\w*\)", r"", cf[key][k])
            elif k == '_exptl_crystal_density_diffrn':
                self.density = cf[key][k]
            elif k == '_cell_volume':
                self.volume = cf[key][k]

            ## per atom in unit cell information
            elif k == '_atom_site_label':
                atoms = cf[key][k]
                for a,atmlbl in enumerate(atoms):
                    if atmlbl not in elem_symbol:
                        if '-' in atmlbl:
                            for e,echatm in enumerate(atmlbl.split('-')):
                                try:
                                    junk = int(echatm) ## skip this - not an element
                                except:
                                    addatm = check_elemsym(echatm)
                                    if e == 0:
                                        newlbl = addatm
                                    else:
                                        newlbl = '%s-%s' % (newlbl,addatm)
                                    atmlbl = newlbl
                        else:
                            atmlbl = check_elemsym(atmlbl)
                        atoms[a] = atmlbl
                self.atom.label = atoms
            elif k == '_atom_site_type_symbol' or  k == '_atom_type_symbol':
                self.atom.label2 = cf[key][k]
            elif k == '_atom_site_fract_x':
                self.atom.fract_x = [float(re.sub(r"\(\w*\)", r"", x)) for x in cf[key][k]]
                #self.atom.fract_x = [float(x) for x in cf[key][k]]
            elif k == '_atom_site_fract_y':
                self.atom.fract_y = [float(re.sub(r"\(\w*\)", r"", y)) for y in cf[key][k]]
                #self.atom.fract_y = [float(y) for y in cf[key][k]]
            elif k == '_atom_site_fract_z':
                self.atom.fract_z = [float(re.sub(r"\(\w*\)", r"", z)) for z in cf[key][k]]
                #self.atom.fract_z = [float(z) for z in cf[key][k]]
            elif k == '_atom_site_wyckoff_symbol':
                self.atom.symm_wyckoff = cf[key][k]
            elif k == '_atom_type_oxidation_number':
                self.atom.oxid_no = cf[key][k]
            elif k == '_atom_site_attached_hydrogens':
                self.atom.site_H = cf[key][k]
            elif k == '_atom_site_b_iso_or_equiv':
                self.atom.B_iso = cf[key][k]
            elif k == '_atom_site_occupancy':
                self.atom.occupancy = cf[key][k]
            elif k == '_atom_site_symmetry_multiplicity':
                self.atom.symm_multi = cf[key][k]

            ## crystal symmetry information
            elif k == '_symmetry_int_tables_number':
                self.symmetry.no = cf[key][k]
            elif k == '_symmetry_space_group_name_h-m':
                self.symmetry.name = cf[key][k].replace(' ', '')
                #self.symmetry.name = cf[key][k]
            elif k == '_chemical_name_structure_type':
                self.symmetry.type = cf[key][k]
            elif k == '_space_group_symop_operation_xyz' or k == '_symmetry_equiv_pos_as_xyz':
                self.symmetry.xyz = cf[key][k]
            elif k == '_symmetry_equiv_pos_site_id':
                self.symmetry.xyz_id = cf[key][k]

            ## publication information for citation
            elif k == '_citation_id':
                self.publication.id = cf[key][k]
            elif k == '_citation_journal_full' or k == '_journal_name_full':
                self.publication.journal = cf[key][k]
            elif k == '_citation_journal_id_astm':
                self.publication.journal_id = cf[key][k]
            elif k == '_citation_journal_volume' or k == '_journal_volume':
                self.publication.volume = cf[key][k]
            elif k == '_citation_page_first' or k == '_journal_page_first':
                self.publication.first_page = cf[key][k]
            elif k == '_citation_page_last' or k == '_journal_page_last':
                self.publication.last_page = cf[key][k]
            elif k == '_citation_year' or k == '_journal_year':
                self.publication.year = cf[key][k]
            elif k == '_publ_author_name':
                self.publication.author = cf[key][k]
            elif k == '_publ_section_title':
                self.publication.title = cf[key][k]

            ## '_audit_creation_date','_audit_update_record' : source specific (kit)
            ## '_cell_formula_units_z'   : not sure what this is (kit)
            ## '_refine_ls_r_factor_all' : not sure what this is (kit)

        # set spacegroup number
        if self.symmetry.no is None and self.symmetry.name is not None:
            name = self.symmetry.name.replace(' ', '')
            for no, sgroup in SPACEGROUPS.items():
                if name in sgroup:
                    self.symmetry.no = no
                    break
        self.set_wyckoff()
        self.check_atoms()

    def check_atoms(self):
        for i,atom in enumerate(zip(self.atom.label, self.atom.symm_wyckoff)):
            el,sym = atom
            if len(el.split('-')) > 1:
                for j,e in enumerate(el.split('-')):
                    if j == 0:
                        self.atom.label[i] = e
                    else:
                        self.atom.label += [e]
                        self.atom.symm_wyckoff += [self.atom.symm_wyckoff[i]]
                        self.atom.fract_x += [self.atom.fract_x[i]]
                        self.atom.fract_y += [self.atom.fract_y[i]]
                        self.atom.fract_z += [self.atom.fract_z[i]]

        for i,atom in enumerate(zip(self.atom.label,self.atom.symm_wyckoff)):
            el,sym = atom
            coor_list = []
            x,y,z = self.atom.fract_x[i],self.atom.fract_y[i],self.atom.fract_z[i]

            if sym not in SPGRP_SYMM[self.symm_key].keys():
                sym = self.atom.symm_multi[i]+sym
                self.atom.symm_wyckoff[i]  = sym

            for coord in SPGRP_SYMM[self.symm_key][sym]:
                a = list(coord)
                for i,ai in enumerate(a):
                    if type(ai) == str:
                        a[i] = eval(ai)
                coor_list += [a]
            self.elem_uvw.update({el:coor_list})

    def set_wyckoff(self):
        self.symm_key = []
        if str(self.symmetry.no) not in SPGRP_SYMM.keys():
             for key in SPGRP_SYMM.keys():
                 if key.startswith(str(self.symmetry.no)+':'):
                     self.symm_key += [key]
        else:
            self.symm_key = [str(self.symmetry.no)]

        ## cheating for now - just picks first in list
        self.symm_key = sorted(np.array(self.symm_key))[0]

        if self.atom.symm_wyckoff is None and self.atom.label is not None:
            self.atom.symm_wyckoff = []

            for i,atom in enumerate(self.atom.label):
                x,y,z = self.atom.fract_x[i],self.atom.fract_y[i],self.atom.fract_z[i]
                min_match = 4
                match = None
                for key in SPGRP_SYMM[self.symm_key].keys():
                    a = list(SPGRP_SYMM[self.symm_key][key][0])
                    str_ct = 0
                    for i,ai in enumerate(a):
                        if type(ai) == str:
                            a[i] = eval(ai)
                            str_ct += 1
                    diff = np.linalg.norm(np.array(a)-np.array([x,y,z]))
                    if diff < 0.0001 and str_ct < min_match:
                        min_match,match = str_ct,key
                if match is not None:
                    self.atom.symm_wyckoff += [match]
                    self.symmetry.no = self.symm_key
                else:
                    self.atom.symm_wyckoff += ['error']


    def calc_q(self, q_min=0.2, q_max=10.2):
        """
        """
        hkl_list = generate_hkl(positive_only=True)

        dhkl = d_from_hkl(hkl_list, *self.unitcell)
        qhkl = q_from_d(dhkl)

        F2hkl = np.zeros(len(hkl_list))

        ## removes q values outside of range
        ii, jj = qhkl < q_max, qhkl > q_min
        ii = jj*ii

        for i, hkl in enumerate(hkl_list):
            Fhkl = 0
            if ii[i]:
                for el in self.atom.label:  ## loops through each element
                    for uvw in self.elem_uvw[el]: ## loops through each position in unit cell
                        hukvlw = hkl[0]*uvw[0]+hkl[1]*uvw[1]+hkl[2]*uvw[2]## (hu+kv+lw)
                        Fhkl = Fhkl + (np.exp(2*1j*PI*hukvlw)).real
            if abs(Fhkl) > 1e-5:
                F2hkl[i] = np.float(Fhkl**2)

        ## removes zero value structure factors
        ii = ii*(F2hkl > 0.001)

        qarr = np.array(qhkl[ii], dtype=np.float64)

        return sorted(np.array(list(set(qarr))))


    def structure_factors(self, wavelength=None, energy=None, qmin=0.2, qmax=10.2):
        if not HAS_CifFile:
            z = np.zeros(2, dtype=np.float64)
            o = np.ones(2, dtype=np.float64)
            return StructureFactor(q=z, intensity=z, hkl=z, d=z, f2hkl=z,
                                   twotheta=z, degen=z, lorentz=o,
                                   wavelength=o, energy=o)



        if energy is not None:
            wavelength = lambda_from_E(energy, E_units='eV')
        if wavelength is None:
            wavelength = 1.0
        if energy is None:
            energy = E_from_lambda(wavelength, E_units='eV')

        hkls = generate_hkl(hmax=12, kmax=12, lmax=12, positive_only=False)
        dhkl = d_from_hkl(hkls, *self.unitcell)
        qhkl = q_from_d(dhkl)
        f2hkl = np.zeros(len(hkls))

        ## removes q values outside of range
        ii, jj = qhkl < qmax, qhkl > qmin
        ii = jj*ii
        # pre-calculate form factors
        f0vals = {}
        f1vals, f2vals = {}, {}
        for el in self.atom.label:
            f0vals[el] = f0(el, np.array(qhkl/(4*PI)))
            f1vals[el] = f1_chantler(el, energy)
            f2vals[el] = f2_chantler(el, energy)

        for i, hkl in enumerate(hkls):
            if not ii[i]:
                continue
            fhkl = 0.0
            for el in self.atom.label:
                fval = f0vals[el][i] + f1vals[el] - 1j*f2vals[el]
                for uvw in self.elem_uvw[el]:
                    # (hu+kv+lw)
                    hukvlw = hkl[0]*uvw[0] + hkl[1]*uvw[1] + hkl[2]*uvw[2]
                    fhkl += fval*np.exp(2*1j*PI*hukvlw)
            f2hkl[i] = (fhkl*fhkl.conjugate()).real

        ## removes zero value structure factors
        ii = ii*(f2hkl > 1.e-4)

        # push q values to large ints to better find duplicates
        qhkl   = [int(round(_q*1.e7)) for _q in qhkl[ii]]
        hkl    = hkls[ii]
        f2hkl  = f2hkl[ii]

        # find duplicates, set degen
        qwork, degen, f2work, hklwork = [], [], [], []
        for i, _q in enumerate(qhkl):
            if _q in qwork:
                degen[qwork.index(_q)] += 1
            else:
                qwork.append(_q)
                degen.append(1)
                f2work.append(f2hkl[i])
                hklwork.append(hkl[i])

        qorder = np.argsort(qwork)
        qhkl  = np.array(qwork, dtype=np.float64)[qorder]/1.e7
        degen = np.array(degen)[qorder]
        f2hkl = np.array(f2work)[qorder]
        hkl   = abs(np.array(hklwork)[qorder])

        twotheta = twth_from_q(qhkl, wavelength)
        if np.any(np.isnan(twotheta)):
            nan_mask = np.where(np.isfinite(twotheta))
            twotheta = twotheta[nan_mask]
            hkl  = hkl[nan_mask]
            qhkl = qhkl[nan_mask]
            f2hkl  = f2hkl[nan_mask]
            degen = degen[nan_mask]

        dhkl  = d_from_q(qhkl)
        lap_corr = self.correction_factor(twotheta)

        ihkl = degen * f2hkl * lap_corr
        if self.volume is not None:
            ihkl *= wavelength**3 / float(self.volume)

        return StructureFactor(q=qhkl, intensity=ihkl, hkl=hkl, d=dhkl,
                               f2hkl=f2hkl, twotheta=twotheta, degen=degen,
                               lorentz=lap_corr, wavelength=wavelength,
                               energy=energy)

    def correction_factor(self, twth):
        ## calculates Lorentz and Polarization corrections
        twth = PI*twth/180
        return (1+np.cos(twth)**2)/(np.sin(twth/2)**2*np.cos(twth/2))

def check_elemsym(atom):

    match_list = []
    loops = 0
    while len(match_list) == 0:
        loops += 1
        if atom.startswith('Wat'):
            match_list += ['H-H-O']
        elif atom.startswith('OH'):
            match_list += ['H-O']
        elif atom.startswith('D'): ## this is a type in cif 01158,02658
            match_list += ['H']
        elif atom.startswith('CH4'): ## this is a type in cif 04526
            match_list += ['C-H-H-H-H']
        else:
            for e in elem_symbol:
                if atom.startswith(e):
                    match_list += [e]
        alen,ai = 0,0
        for i,match in enumerate(match_list):
            if len(match) > alen: alen,ai = len(match),i
        if len(match_list) == 0:
            atom = re.sub(r'([0-9])', r'', atom).title()
        if loops > 2:
            return atom
    return match_list[ai]

def removeNonAscii(s):
    return "".join(i for i in s if ord(i)<128)

def create_xrdcif(filename=None, text=None):
    """
    create CIF representation from CIF filename, text of CIF file,

    Arguments
    ---------
    ciffile
    """
    if text is None and filename is not None and os.path.exists(filename):
        with open(filename, 'r') as fh:
            text = fh.read()

    return XRDCIF(text=text)
