/* https://github.com/tylabs/quicksand Exploit detection signatures 
 * v    */

rule warning_exec_ocx_object {
	meta:
		is_exploit = true
		is_warning = false
		is_feature = true
		rank = 5
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "OLE application command"
		mitre = "T1204.002"
	strings:
		$s1 = "w:ocx w:data=\"DATA:application/x-oleobject"
	condition:
            1 of them
}




rule warning_scriptbridge {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "ScriptBridge may load remote exploit"
		mitre = "T1204.002"

	strings:
		$s1 = "ScriptBridge.ScriptBridge.1"

	condition:
            1 of them
}



rule exploit_cve_2006_2492 {
	meta:
		is_exploit = true
		is_warning = false
		is_feature = false
		rank = 10
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "smarttag overflow CVE-2006-2492"
		mitre = "T1203 T1204.002"
	strings:
		$s1 = {0600DDC6040011000100D65A12000000000001000000060000000300}
		$s2 = {0600C8BE1B0008000200685B1200}
		$s3 = "0600DDC6040011000100D65A12000000000001000000060000000300"
		$s4 = "0600C8BE1B0008000200685B1200"
	condition:
            1 of them
}

rule exploit_cve_2009_3129 {
	meta:
		is_exploit = true
		is_warning = false
		is_feature = false
		rank = 10
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "excel buffer overflow CVE-2009-3129"
		mitre = "T1203 T1204.002"
	strings:
		$s1 = {4F7269656E746174696F6E??504F33}
		$s2 = "4F7269656E746174696F6E??504F33"
	condition:
            1 of them
}


rule warning_embedded_flash {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Embedded Flash data"
		mitre = "T1204.002"
	strings:
		$cws = {66556655??????00435753}
		$fws = {66556655??????00465753}
		$zws = {66556655??????005a5753}
		$control = "CONTROL ShockwaveFlash.ShockwaveFlash"
		$jit = {076A69745F656767}
		$generic = "ShockwaveFlash.ShockwaveFlash."
		$genericw = "ShockwaveFlash" wide
		$generich = "53686F636B77617665466C6173682E53686F636B77617665466C6173682E"

	condition:
            1 of them
}



rule exploit_cve_2011_0609 {
	meta:
		is_exploit = true
		is_warning = false
		is_feature = false
		rank = 10
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Flash exploit CVE-2011-0609"
		mitre = "T1203 T1204.002"

	strings:
		$s1 = {4657530947CB0000480140005A0000190100441108000000BF141CCB0000000000000010002E00060080804094A8D0A001808004100002000000121212E24130F00931343134313431343134313431343134313431343134313431343134313431343134313431343134313431343134313431343134}
		$s2 = {34363537353330394541433730303030373830303036343030303030304338303030303032443031303034343131313830303030303034333032463446344634383630363036303230303031303030304646303931303030303030303033303030313030383630363036303130303032303030303430303030303030424631313235}
		$s3 = {3941303139413031394130313941303139064C6F61646572}

	condition:
            1 of them
}

rule exploit_cve_2011_0611 {
	meta:
		is_exploit = true
		is_warning = false
		is_feature = false
		rank = 10
		revision = "1"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Flash exploit CVE-2011-061"
		mitre = "T1203 T1204.002"

	strings:
		$s1 = {7772697465427974650541727261799817343635373533304143433035303030303738}
		$s2 = {5131645443737746414142346E453155625778545A52512B743733742B3362744B4E30596E617767552F414452654D5848334777597276757737597A643743674A734A6C76643174374E716D393959576D4B676B5A7674686C68446942556E344D694645453030514659306D456F664A2B4F45504D55594E6F69614C526D4E696A4D45494444665065652B3139663534652B35356E764F63383578376532766732514551504148514C6B45384248683175303937414B7741654943394F6A336579756277574E52793141564A475939326D4777444832794278794147636569424250524348}
		$s3 = {343635373533304143433035303030303738303030353546303030303046413030303030313830313030343431313030303030303030334630334137303530303030393630433030303530303037393543333743313330374642433337433133304531323944303230303443303439443032303031383030383831353030303930303431}
		$s4 = {3063306330633063306330633063306306537472696E6706}
		$s5 = {410042004300440045004600470048004900A18E110064656661756C74}
		$s6 = {00414243444500566B6475686752656D686677317375727772777C73680064656661756C740067657453697A650047647768317375727772777C73680077777273757277}
		$s7 = "AAB4AAVfAAAPoAAAGAEARBEAAAAAPwOnBQAAlgwABQAHlcN8Ewf7w3wTDhKdAgBMBJ0CABgAiBUACQBBAEIAQwBEAEUARgBHAEgASQChjhEAZGVmYXVsdAABAAQqAAIAmAGWCgAHWMBJSAenP7a3YJ0CAAAAmQIASQBAlgUABxZ0cAtMYp0CAAwAhwEAAxeHAQABlgoAB"
    

	condition:
            1 of them
}
    
    
    
rule exploit_cve_2012_0754 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Flash exploit malformed mp4 CVE-2012-0754"
	mitre = "T1203 T1204.002"

    strings:
        $s1 = {537472696E6706586D6C537766094D6F766965436C6970076A69745F656767086368696C645265660D446973706C61794F626A656374074D79566964656F05566964656F044D794E430D4E6574436F6E6E656374696F6E}
    condition:
        1 of them
}


rule exploit_cve_2010_3333 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
	release = "lite"
        author = "@tylabs"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "RTF stack overflow pFragments CVE-2010-3333"
	mitre = "T1203 T1204.002"

    strings:
        $s1 = /sn .{1,300}?pFragments.{1,700}?sv .{1,200}?[a-zA-Z0-9\*\+]{50}?/
        $s2 = "\\sn\\*\\sn-pFragments"
        $s3 = /pFragments.{1,200}?\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x61\x0D\x0A/
        $s4 = /sn pfragments.{1,30}?11111111/
        $s5 = /sn[\W]{1,20}?pFragments/
        $s6 = "\\sn9pFRagMEnTS"
        $s7 = {5C736E34096D656E7473}
    condition:
        1 of them
}
    
    


    
    
rule warning_rtf_embedded_file {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 2
        revision = "2"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_warning"
        desc = "RTF embedded file package"
	mitre = "T1027"
    strings:
        $s1 = /objdata.{1,300}\w*5\w*0\w*6\w*1\w*6\w*3\w*6\w*b\w*6\w*1\w*6\w*7\w*6\w*5\w*0\w*0/
        $s2 = "\\objclass Word.Document"
    condition:
        1 of them
}

    
rule exploit_MS12_060_tomato_garden {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Office exploit MSCOMCTL.OCX Toolbar MS12-060 Tomato Garden campaign"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "CONTROL MSComctlLib.Toolbar.2"
        $s2 = "Toolbar1, 0, 0, MSComctlLib, Toolbar"
        $s3 = "MSComctlLib.Toolbar.2"
        $s4 = {4D53436F6D63746C4C69622E546F6F6C6261722E32}
	$s5 = "4D53436F6D63746C4C69622E546F6F6C6261722E32"
    condition:
        1 of them
}

    


rule warning_office_encrypted_doc {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 1
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "office encrypted document"
	mitre = "T1027"
    strings:
        $s1 = {4D006900630072006F0073006F0066007400200042006100730065002000430072007900700074006F0067007200610070006800690063002000500072006F0076006900640065007200200076}
        $s2 = {45006E006300720079007000740065006400530075006D006D006100720079}
    condition:
        1 of them
}

   rule warning_office_encrypted_unsupported_xor {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 1
        revision = "1"
        date = "Oct 10 2020"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "office obfuscated document 2003 XOR with VelvetSweatshop"
	mitre = "T1027"
	ref = "e595a9c76519a346225c6c155dd0b942dbf24cb935c71bb32a376e651427e6ee"
    strings:
        $s1 = { 2F 00 06 00 00 00 59 B3 0A 9A }
    condition:
        1 of them
}
 
    
    
rule exploit_cve_2012_1535 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Flash exploit CVE-2012-1535"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = {4578616D706C650B6372656174654C696E65730968656170537072617908686578546F42696E076D782E636F72650A49466C6578417373657409466F6E7441737365740A666C6173682E74657874}
        $s2 = {454D4245444445445F4346460A666F6E744C6F6F6B75700D456C656D656E74466F726D617408666F6E7453697A650B54657874456C656D656E7407636F6E74656E740E637265617465546578744C696E6508546578744C696E650178017906686569676874086164644368696C6406456E6469616E0D4C4954544C455F454E4449414E06656E6469616E223063306330633063}
    condition:
        1 of them
}


    
    
rule exploit_cve_2013_0634 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Flash exploit CVE-2013-0634 memory corruption"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = {8A23ABA78A01908B23EED461D8872396A39A02F48523A1F94AB48323FBE0E303}
    condition:
        1 of them
}
   
rule exploit_cve_2012_5054 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Flash exploit CVE-2012-5054 Matrix3D"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = {7772697465446F75626C65084D61747269783344064F626A6563740B666C6173682E6D6564696105536F756E640C666C6173682E73797374656D0C4361706162696C69746965730776657273696F6E0B746F4C6F776572436173651077696E}
    condition:
        1 of them
}
    
    
    

    
    
rule exploit_cve_2012_1856 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Office exploit MSCOMCTL.OCX TabStrip CVE-2012-1856"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "MSComctlLib.TabStrip"
        $s2 = "4d53436f6d63746c4c69622e546162537472697" nocase
        $s3 = "9665fb1e7c85d111b16a00c0f0283628" nocase
        $s4 = "1EFB6596-857C-11D1-B16A-00C0F0283628" nocase

    condition:
        1 of them
    }
    
    
rule warning_mime_mso_embedded_flash {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 1
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
	release = "lite"
        sigtype = "cryptam_exploit"
        desc = "office embedded Flash in MSO file"
	mitre = "T1204.002"
    strings:
        $s1 = "D27CDB6E-AE6D-11CF-96B8-444553540000" nocase
    condition:
        1 of them
}


    
rule exploit_cve_2012_0158 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "2"
        date = "Sep 01 2020"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Office exploit MSCOMCTL.OCX RCE CVE-2012-0158"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = /objdata.{1,100}?53436F6D63746C4C69622E4C/
        $s2 = "ListView2, 1, 1, MSComctlLib, ListView"
        $s3 = "ListView1, 1, 0, MSComctlLib, ListView"
        $s4 = /0000000000000000000000000000000000000000000000.{1,300}?49746D736400000002000000010000000C000000436F626A/
        $s5 = /MSComctlLib.ListViewCtrl.{1,25}?objdata/
        $s6 = "MSComctlLib.ListViewCtrl.2"
        $s7 = {4C00690073007400560069006500770041}
        $s8 = {ECBD010005009017190000000800000049746D736400000002000000010000000C000000436F626A??0000008282000082820000000000000000000000000000????????90}
        //$s9 = {3131313131313131310D0D0D1320434F4E54524F4C204D53436F6D63746C4C69622E4C697374566965774374726C2E32}
        $s10 = "978C9E23-D4B0-11CE-BF2D-00AA003F40D0" nocase
        $s11 = "BDD1F04B-858B-11D1-B16A-00C0F0283628" nocase
        $s12 = "C74190B6-8589-11D1-B16A-00C0F0283628" nocase
        $s13 = "996BF5E0-8044-4650-ADEB-0B013914E99C" nocase
        $s14 = "9181DC5F-E07D-418A-ACA6-8EEA1ECB8E9E" nocase
        $s15 = "\\7300740056006\\"
        $s16 = "4C69{\\*}7374566"
        $s17 = "4C0069007300740056006900650077004" nocase
        $s18 = "4BF0D1BD8B85D111B16A00C0F0283628" nocase
        $s19 = {4BF0D1BD8B85D111B16A00C0F0283628}
        $s20 = "COMCTL.TreeCtrl.1"
        $s21 = {434F4D43544C2E547265654374726C2E31}
	$s22 = "4D53436F6D63746C4C69622E4C697374566965774374726C2E" nocase
	$s23 = "MSComctlLib.ListViewCtrl.0"
	$s24 = {4D 53 43 6F 6D 63 74 6C 4C 69 62 2E 4C 69 73 74 56 69 65 77 43 74 72 6C 2E 30}
	$s25 = "4D53436F6D63746C4C69622E4C697374566965774374726C2E30" nocase
condition:
        1 of them
}
    
    
    
rule warning_activex_exec {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 8
        revision = "3"
        date = "Oct 11 2017"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015, 2017. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "ActiveX content"
	mitre = "T1059"
    strings:
        $s1 = "MSComctlLib.TreeCtrl.2"
        $s2 = "4D53436F6D63746C4C69622E547265654374726C2E32" nocase
        $s3 = "B69041C78985D111B16A00AA003F40D0" nocase
    $s4 = {B69041C78985D111B16A00AA003F40D0}
    $s5 = "C74190B6-8589-11D1-B16A-00AA003F40D0" nocase
    $s6 = "C74190B6-8589-11D1-B16A-00C0F0283628" nocase
    $s7 = {B69041C78985D111B16A00C0F0283628}
    $s8 = "B69041C78985D111B16A00C0F0283628" nocase
    $s9 = "objclass MSComctlLib.ImageComboCtl.2"
    $s10 = "MSComctlLib.ImageComboCtl.2"
    $s11 = {00 4D 53 43 6F 6D 63 74 6C 4C 69 62 2E 49 6D 61 67
        65 43 6F 6D 62 6F 43 74 6C}
    $s12 = {49006D0061006700650043006F006D0062006F00430074006C002000}
    $s13 = "TreeView1, 0, 0, MSComctlLib, TreeView"
    $s14 = "new ActiveXObject"
    $s15 = "<ax:ocx ax:classid=" ascii nocase

    
    condition:
        1 of them
    }



    rule warning_vb_potential_heapspray {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 2
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
	release = "lite"
        sigtype = "cryptam_exploit"
        desc = "office heap spray"
	mitre = "T1059.005"
    strings:
        $s1 = "90909090EB7F414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141414141412490909090"
    condition:
        1 of them
    }
    
    
    rule exploit_cve_2013_3906 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Office exploit TIFF CVE-2013-3906"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = {49492A00C8490000803FE0503824160D0784426150B864361D0F8844625138A4562D178C466351B8E4763D1F90486452392418012794496552B964B65D2F984C665339A4D66D379C4E6753B9E4F67D3FA05068543A25168D47A4526954BA65361D2894D3AA553AA556AD57AC566B55BAE576BD5FB0586C563B2596CD67B25424F68B65B6DD6FB85C6E573BA5D6ED77BC5E6F57BBE5F64751BF6070583C26170D87C4627158BC66371D8FA5DA80190CA6572D97CC667359BC5404803FE0503824160D0784426150B864361D0F88446251}
        $s2 = {49492a000800000002000e010200fc3a0000260000006987040001000000223b00007c5a00000a0a0a0a0a}
        $s3 = /jpegblip.{1,20}?49492a00cf660000ffff/
        
    condition:
        1 of them
}
    
    
    
    rule warning_package_manager_embedded {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 1
        revision = "2"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "Office package manager may load unsafe content such as shell scripts"
	mitre = "T1027"
    strings:
        $s1 = "0003000C-0000-0000-c000-000000000046" nocase
        $s2 = "0c00030000000000c000000000000046"
        $s3 = {0c00030000000000c000000000000046}
	$s4 = "20a70df22fc0ce11927b0800095ae340" nocase
	$s5 = {20a70df22fc0ce11927b0800095ae340}
        $s7 = "Packager Shell Object" ascii wide
        
    condition:
        1 of them
    }
    

rule warning_ole2link_embedded {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 1
        revision = "3"
        date = "September 12 2017"
        author = "David Cannings"
        copyright = "source https://github.com/nccgroup/Cyber-Defence/blob/master/Technical%20Notes/Office%20zero-day%20(April%202017)/2017-04%20Office%20OLE2Link%20zero-day%20v0.4.md"
        tlp = "white"
        sigtype = "cryptam_warning"
        desc = "Office OLE2Link unsafe content such as remote risky content"
	mitre = "T1027 T1204.002"
    strings:
        // Parsers will open files without the full 'rtf'
        $header_rtf = "{\\rt" nocase
        $header_office = { D0 CF 11 E0 }
        $header_xml = "<?xml version=" nocase wide ascii

        // Marks of embedded data (reduce FPs)
        // RTF format
        $embedded_object   = "\\object" nocase
        $embedded_objdata  = "\\objdata" nocase
        $embedded_ocx      = "\\objocx" nocase
        $embedded_objclass = "\\objclass" nocase
        $embedded_oleclass = "\\oleclsid" nocase
    
        // XML Office documents
        $embedded_axocx      = "<ax:ocx"  nocase wide ascii
        $embedded_axclassid  = "ax:classid"  nocase wide ascii

        // OLE format
        $embedded_root_entry = "Root Entry" wide
        $embedded_comp_obj   = "Comp Obj" wide
        $embedded_obj_info   = "Obj Info" wide
        $embedded_ole10      = "Ole10Native" wide

        $data0 = "00000300-0000-0000-C000-000000000046" nocase wide ascii
        $data1 = { 0003000000000000C000000000000046 }
        $data2 = "OLE2Link" nocase wide ascii
        $data3 = "4f4c45324c696e6b" nocase wide ascii
        $data4 = "StdOleLink" nocase wide ascii
        $data5 = "5374644f6c654c696e6b" nocase wide ascii

      condition:
        // Mandatory header plus sign of embedding, then any of the others
        1 of ($header*) and 1 of ($embedded*) 
            and (1 of ($data*))
    }


    rule exploit_eicar_test_file {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "eicar test signature"
	mitre = ""
    strings:
        $s1 = "EICAR-STANDARD-ANTIVIRUS-TEST-FILE"
        
    condition:
        $s1
    }




rule warning_vb_macro {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 3
		revision = "2"
		date = "Oct 5 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Visual Basic macro"
		mitre = "T1059.005"
	strings:
		$s1 = "Name=\"VBAProject\"" nocase
		$s2 = "OLE Automation" wide nocase
		$s3 = "Visual Basic For Applications" wide nocase
		$s5 = "VBA6\\VBE6.DLL" wide nocase
		$s6 = "000204EF-0000-0000-C000-000000000046" ascii wide
		$s7 = "00020430-0000-0000-C000-000000000046" ascii wide
		$s8 = {000204EF00000000C000000000000046}
		$s9 = {0002043000000000C000000000000046}
		$s10 = "000204EF00000000C000000000000046"
		$s11 = "0002043000000000C000000000000046"
		$s12 = "wne:vbaSuppData" nocase
		$s13 = "wne:macroName" nocase
		$s14 = "e VB_Nam" ascii wide

	condition:
            1 of them
}


rule warning_js_embed {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 3
		revision = "1"
		date = "Apr 12 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Embedded js"
		mitre = "T1059.007"
	strings:
		$s1 = {6a 73 00}
		$s2 = "Package"
		$s3 = {2e 00 6a 00 73}
		$s4 = "Ole10Native" wide
	condition:
            3 of them
}

rule warning_vbs_embed {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 3
		revision = "1"
		date = "May 18 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Embedded vbs"
		mitre = "T1059.005"
	strings:
		$s1 = {2e 76 62 73 00}
		$s2 = "Package"
		$s3 = {2e 00 76 00 62 00 73}
		$s4 = "Ole10Native" wide
	condition:
            3 of them
}



rule exploit_activex_execute_shell {
	meta:
		is_exploit = true
		is_warning = true
		is_feature = true
		rank = 3
		revision = "2"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Visual Basic execute shell"
		mitre = "T1059.005 T1059.003"
	strings:
		$s1 = "Wscript.Shell" nocase
		$s2 = "netsh firewall set opmode mode=disable" nocase
		$s3 = "Shell" nocase
		$s4 = "CreateObject" nocase
		$s5 = "GetObject" nocase
		$s6 = "SendKeys" nocase
		$s7 = "MacScript" nocase
		$s8 = "FollowHyperlink" nocase
		$s9 = "CreateThread" nocase
		$s10 = "ShellExecute" nocase
		$s11 = "shell.application" nocase
	condition:
            (warning_vb_macro or warning_js_embed) and 1 of them
}



rule warning_vb_autoopen {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 3
		revision = "3"
		date = "Oct 9 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Visual Basic macro"
		mitre = "T1059.005"
	strings:
		$s1 = "Document_Open" ascii wide nocase
		$s2 = "AutoOpen" ascii wide nocase
		$s3 = "Document_Close" ascii wide nocase
		$s4 = "AutoExec" ascii wide nocase
		$s5 = "Auto_Open" ascii wide nocase
		$s6 = "AutoClose" ascii wide nocase
		$s7 = "Auto_Close" ascii wide nocase
		$s8 = "DocumentBeforeClose" ascii wide nocase
		$s9 = "DocumentChange" ascii wide nocase
		$s10 = "Document_New" ascii wide nocase
		$s11 = "NewDocument" ascii wide nocase
		$s12 = "Workbook_Open" ascii wide nocase
		$s13 = "Workbook_Close" ascii wide nocase
		$s14 = "Img_Painted" ascii wide nocase

	condition:
            warning_vb_macro and 1 of them
}


rule warning_vb_fileio {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 4
		revision = "2"
		date = "July 29 2015"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Visual Basic file io"
		mitre = "T1059.005"

	strings:
		$s1 = "Scripting.FileSystemObject" nocase
		$s2 = "OpenTextFile"
		$s3 = "FileCopy"
		$s4 = "CopyFile"
		$s5 = "Kill"
		$s6 = "CreateTextFile"
		$s7 = "VirtualAlloc"
		$s8 = "RtlMoveMemory"
		$s9 = "URLDownloadToFileA"
		$s10 = "AltStartupPath"
		$s11 = "URLDownloadToFileA"
		$s12 = "ADODB.Stream"
		$s13 = "WriteText"
		$s14 = "SaveToFile"
		$s15 = "SaveAs"
		$s16 = "SaveAsRTF"
		$s17 = "FileSaveAs"
		$s18 = "MkDir"
		$s19 = "RmDir"
		$s20 = "SaveSetting"
		$s21 = "SetAttr"
	condition:
            warning_vb_macro and 1 of them
}





rule exploit_cve_2014_1761 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "2"
        date = "September 23 2016"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "RTF memory corruption listoverridecount CVE-2012-2539 CVE-2014-1761"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = /listoverridecount([1-9]{2}|0[0-9]{2})/
        $s2 = "\\listoverridecount25"
    
    condition:
        1 of them
}

    
rule exploit_cve_2014_4114 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 29 2015"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Office package manager executes .inf files CVE-2014-4114, CVE-2014-6352"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = /\.inf\x00.{1,64}?\[Version/
        $s2 = /\x00\\\\.{1,64}?\.inf\x00/
        $s3 = "7EBEFBC0-3200-11d2-B4c2-00A0C9697D17" nocase
        $s4 = /\.lnk\x00/
        
    condition:
        1 of them
}



rule exploit_cve_2015_0097 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "July 31 2015"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
	ref = "https://www.fireeye.com/blog/threat-research/2015/07/cve-2015-0097_exploi.html"
	hashes = "8a1fd4182bea2cc5a33319769dbf501c"
        desc = "Microsoft Word Local Zone Remote Code Execution Vulnerability CVE-2015-0097"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "MSScriptControl.ScriptControl"
        $s2 = "\"ADODB.Recordset\""
        
    condition:
        all of them
}


rule exploit_cve_2015_1770 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Aug 26 2015"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Office exploit MSCOMCTL.OCX TabStrip CVE-2012-1856"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "CDDBCC7C-BE18-4A58-9CBF-D62A012272CE" nocase
        $s2 = "osf.Sandbox.1" ascii wide
        $s3 = "CDDBCC7CBE184A589CBFD62A012272CE" nocase
        $s4 = {CDDBCC7CBE184A589CBFD62A012272CE}
	//$s5 = "9368265E-85FE-11d1-8BE3-0000F8754DA1" nocase
	$s6 = "CDDBCC7C-BE18-4A58-9CBF-D62A012272CE" nocase



    condition:
        1 of them
    }


rule exploit_cve_2015_1641 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Aug 30 2015"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "SmartTag element attribute parsing vulnerability CVE-2015-1641"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "&#xBD50;&#x7C38;" nocase
        $s2 = "&#xBD68;&#x7C38;" nocase
        $s3 = "&#xBD60;&#x7C38;" nocase
        $s4 = "&#xBD80;&#x7C38;" nocase

    condition:
        1 of them
    }


rule exploit_cve_2015_2424 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Sep 30 2015"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Task Symbol buffer overflow CVE-2015-2424 CVE-2015-1642"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "Control.TaskSymbol.1" ascii wide
	$s2 = "44f9a03b-a3ec-4f3b-9364-08e0007f21df" nocase
	$s3 = "44f9a03ba3ec4f3b936408e0007f21df" nocase
	$s4 = {44f9a03ba3ec4f3b936408e0007f21df}
	$s5 = "MMC.IconControl.1" ascii wide
	$s6 = "b0395da5-6a15-4e44-9f36-9a9dc7a2f341" nocase
	$s7 = "b0395da56a154e449f369a9dc7a2f341" nocase
	$s8 = {b0395da56a154e449f369a9dc7a2f341}


    condition:
        1 of them
    }




rule exploit_cve_2015_2545 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Dec 19 2015"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "SmartTag element attribute parsing vulnerability CVE-2015-1641"
	mitre = "T1203 T1204.002"
    strings:
        $r1 = /forall.{1,30}1145324612/
        $r2 = /forall.{1,30}integertype/
        $s1 = "ffffff7fbebafeca" nocase

    condition:
        uint16(0) == 0x2125 and 1 of them
    }


rule warning_cve_2016_0021 {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = false
        rank = 10
        revision = "1"
        date = "Apr 7 2016"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2016. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Memory corruption vuln access freed heap memory CVE-2016_0021"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "InfoPath.DesignerExcelImport"
        $s2 = "InfoPath.DesignerWordImport"
        $h1 = "496e666f506174682e44657369676e6572457863656c496d706f7274" nocase
        $h2 = "496e666f506174682e44657369676e6572576f7264496d706f7274" nocase

    condition:
        1 of them

    }


rule exploit_cve_2016_4117 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Oct 27 2016"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Flash type confusion vulnerability CVE-2016-4117"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "FFFFFFFF0F29010105060000001014140A140E40012940A0042958FFFFFFFF"
        $h1 = {FFFFFFFF0F29010105060000001014140A140E40012940A0042958FFFFFFFF}

    condition:
	1 of them
    }

rule exploit_lnk_execute_shell {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = true
        rank = 10
        revision = "1"
        date = "Dec 09 2016"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2016. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = ".lnk files execute code"
	mitre = "T1203 T1204.002"
    strings:
	$s1 = /\.lnk\x00/
        //$s2 = {4C 00 00 00}
	$s3 = {00 0? 14 02 00 00 00 00 00 C0 00 00 00 00 00 00 46}
	$s4 = "cmd.exe" nocase ascii wide

    condition:
	all of them
    }



rule exploit_cve_2017_0199 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Apr 11 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Packager remote htmlfile inclusion"
	mitre = "T1203 T1204.002"
    strings:
	$s1 = "3050F5C8-98B5-11CF-BB82-00AA00BDCE0B" nocase
	$s2 = "3050F5C898B511CFBB8200AA00BDCE0B" nocase
	$s3 = {3050F5C898B511CFBB8200AA00BDCE0B}
	$s4 = "C8F55030-B598-CF11-BB82-00AA00BDCE0B" nocase
	$s5 = "C8F55030B598CF11BB8200AA00BDCE0B" nocase
	$s6 = {C8F55030B598CF11BB8200AA00BDCE0B}
	$s7 = "25336920-03F9-11CF-8FD0-00AA00686F13" nocase
	$s8 = "2533692003F911CF8FD000AA00686F13" nocase
	$s9 = {2533692003F911CF8FD000AA00686F13}
	$s10 = "20693325-F903-CF11-8FD0-00AA00686F13" nocase
	$s11 = "20693325F903CF118FD000AA00686F13" nocase
	$s12 = {20693325F903CF118FD000AA00686F13}
	$s13 = "25336921-03F9-11CF-8FD0-00AA00686F13" nocase
	$s14 = "2533692103F911CF8FD000AA00686F13" nocase
	$s15 = {2533692103F911CF8FD000AA00686F13}
	$s16 = "21693325-F903-CF11-8FD0-00AA00686F13" nocase
	$s17 = "21693325F903CF118FD000AA00686F13" nocase
	$s18 = {21693325F903CF118FD000AA00686F13}
	$s19 = "3050F3D9-98B5-11CF-BB82-00AA00BDCE0B" nocase
	$s20 = "3050F3D998B511CFBB8200AA00BDCE0B" nocase
	$s21 = {3050F3D998B511CFBB8200AA00BDCE0B}
	$s22 = "D9F35030-B598-CF11-BB82-00AA00BDCE0B" nocase
	$s23 = "D9F35030B598CF11BB8200AA00BDCE0B" nocase
	$s24 = {D9F35030B598CF11BB8200AA00BDCE0B}
	$s25 = "8856F961-340A-11D0-A96B-00C04FD705A2" nocase
	$s26 = "8856F961340A11D0A96B00C04FD705A2" nocase
	$s27 = {8856F961340A11D0A96B00C04FD705A2}
	$s28 = "61F95688-0a34-D011-A96B-00C04FD705A2" nocase
	$s29 = "61F956880a34D011A96B00C04FD705A2" nocase
	$s30 = {61F956880a34D011A96B00C04FD705A2}
	$s31= "2D360200-FFF5-11d1-8d03-00a0c959bc0a" nocase
	$s32= "2D360200FFF511d18d0300a0c959bc0a" nocase
	$s33= {2D360200FFF511d18d0300a0c959bc0a}
	$s34 = "0002362D-F5FF-D111-8d03-00a0c959bc0a" nocase
	$s35 = "0002362DF5FFD1118d0300a0c959bc0a" nocase
	$s36 = {0002362DF5FFD1118d0300a0c959bc0a}
	$s37 = "ECABAFC6-7F19-11D2-978E-0000F8757E2A" nocase
	$s38 = "ECABAFC67F1911D2978E0000F8757E2A" nocase
	$s39 = {ECABAFC67F1911D2978E0000F8757E2A}
	$s40 = "06290BD3-48AA-11D2-8432-006008C3FBFC" nocase
	$s41 = "06290BD348AA11D28432006008C3FBFC" nocase
	$s42 = {06290BD348AA11D28432006008C3FBFC}
	$r1 = "http" nocase wide
	$o1 = "OLE2Link"
	$r2 = "68007400740070"
	$r3 = "ftp" nocase wide
	$o2 = "4f4c45324c696e6b" nocase
        $o3 = "00000300-0000-0000-c000-000000000046" nocase
        $o4 = "0000030000000000c000000000000046" nocase
        $o5 = "objupdate" nocase
        $o6 = "00030000-0000-0000-c000-000000000046" nocase
        $o7 = "0003000000000000c000000000000046" nocase
        //$o8 = {0003000000000000c000000000000046}
	$m1 = "e0c9ea79-f9ba-ce11-8c82-00aa004ba90b" nocase
	$m2 = "e0c9ea79f9bace118c8200aa004ba90b" nocase
	$m3 = {e0c9ea79f9bace118c8200aa004ba90b}
	$m4 = "79eac9e0-baf9-11ce-8c82-00aa004ba90b" nocase


    condition:
        (1 of ($s*) or 1 of ($m*)) and 1 of ($r*) and 1 of ($o*) 
    }



rule exploit_cve_2017_8570 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "2"
        date = "Jan 30 2019"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Packager remote htmlfile inclusion"
	link = "http://justhaifei1.blogspot.ca/2017/07/bypassing-microsofts-cve-2017-0199-patch.html"
	mitre = "T1203 T1204.002"
    strings:
	$s1 = "06290BD2-48AA-11D2-8432-006008C3FBFC" nocase
	$s2 = "06290BD248AA11D28432006008C3FBFC" nocase
	$s3 = {06290BD248AA11D28432006008C3FBFC}

        $s4 = "09030000-0000-0000-c000-000000000046" nocase
        $s5 = "0903000000000000c000000000000046" nocase
        $s6 = {0903000000000000c000000000000046}
        $s7 = "03030000-0000-0000-c000-000000000046" nocase
        $s8 = "0303000000000000c000000000000046" nocase
        $s9 = {0303000000000000c000000000000046}
	$s10 = "C6AFABEC197FD211978E0000F8757E2A" nocase
	$s11 = "C6AFABE-C197-FD21-1978-E0000F8757E2A"
	$s12 = {C6AFABEC197FD211978E0000F8757E2A}

	$u1 = "Target" nocase
	$u2 = "script:" nocase
	$u3 = "External" nocase
	$u4 = "xml" nocase


    condition:
        any of ($s*) or all of ($u*)
    }


rule exploit_cve_2017_0261 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "May 11 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "EPS reuse after free"
	mitre = "T1203 T1204.002"
    strings:
	$h1 = "%!PS-Adobe-" nocase
        $s1 = "closefile"
	$s2 = "restore"
	$s3 = "2147483647"

    condition:
	$h1 at 0 and all of ($s*)
    }

rule exploit_cve_2017_0262 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "May 11 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "EPS exploit type confusion"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "repeat 1 array 226545696 forall"
	$s2 = "00d0800d30d0800d000000000200000010d0800d020000003cd0800d0005000000000000000000005cd0800d00000300000000000000000020d0800d3cd0800d6cd0800d00000000f0ffff7f50d0800d00000000f1ffff7f"
	$s3 = "bytesavailable"
	$s4 = "globaldict"

    condition:
	3 of ($s*)
    }

rule warning_EPS_xor_exec {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 5
        revision = "2"
        date = "Oct 13 2017"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "EPS obfuscation using xor and exec"
	mitre = "T1027"
    strings:
	$h1 = "%!PS-Adobe-" nocase
        $s1 = "mod get xor"
	$s2 = "exec"
	$s3 = "quit"

    condition:
	$h1 at 0 and all of ($s*)
    }


rule exploit_cve_2016_7193 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "Jun 9 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "Track Changes ListNum length overflow"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "\\dfrxst192\\dfrxst12\\dfrxst12\\dfrxst12"
	$s2 = "\\dfrxst100}}"

    condition:
	1 of them
    }


rule warning_office_encrypted_dataspace {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = true
        rank = 1
        revision = "1"
        date = "June 9 2017"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2015. All rights reserved."
        tlp = "white"
        sigtype = "cryptam_exploit"
        desc = "office encrypted dataspace"
	mitre = "T1027"
    strings:
        $s1 = "EncryptedPackage" wide
        $s2 = "DataSpaces" wide
        $s3 = "StrongEncryptionDataSpace" wide
	$u1 = "FF9A3F03-56EF-4613-BDD5-5A41C1D07246" wide ascii
	$u2 = {FF9A3F0356EF4613BDD55A41C1D07246}
	$u3 = {033f9affef561346bdd55A41C1D07246}

    condition:
        2 of ($s*) or 1 of ($u*)
}



rule warning_flash_obfuscation {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "1"
		date = "June 9 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Flash exec from string"
		mitre = "T1027"
	strings:
		$h1 = {435753}
		$h2 = {465753}
		$h3 = {5a5753}
		$s1 = "charCodeAt"
		$s2 = "currentDomain"
		$s3 = "loadBytes"
		$s4 = "addChild"
		$s5 = "ApplicationDomain"
		$s6 = "LoaderContext"


	condition:
            1 of ($h*) and 3 of ($s*)
}



rule warning_openxml_remote_smb
{
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "2"
		date = "June 18 2017"
		author = "@tylabs"
		release = "lite"
		hash = "34a11f3d68fd6cdef04b6df17bbe8f4d"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "External file inclusion may leak smb credentials"
		mitre = "T1187"
	strings: 
		$a = "schemas.openxmlformats.org" ascii nocase
		$b = "TargetMode=\"External\"" ascii nocase
		$c = /Target="file:[\\\/]+[0-9hf\.]+/

	condition:
		all of them
}

rule warning_SAXXMLReader
{
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 0
		revision = "1"
		date = "July 26 2020"
		author = "@tylabs"
		release = "lite"
		hash = "574c0c60df82b3d79937eaacddf83e3d"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "amber"
		sigtype = "cryptam_exploit"
		desc = "Msxml2.SAXXMLReader class used in exploit obfuscation"
		mitre = "T1027"
	strings: 
		$a = "Msxml2.SAXXMLReader.6.0" ascii nocase
		$b = "0C6AD98892F1D411A65F0040963251E5" ascii nocase
		$c = {0C6AD98892F1D411A65F0040963251E5}

	condition:
		any of them
}




rule exploit_cve_2017_8759 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "September 12 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "green"
        sigtype = "cryptam_exploit"
        desc = "OLE WSDL Parser Code Injection in PrintClientProxy CVE-2017-8759"
	mitre = "T1203 T1204.002"
    strings:
        $c5 = "wsdl=" ascii wide nocase
        $c7 = "wsdl=http" ascii wide nocase
	$c1 = "ECABB0C7-7F19-11D2-978E-0000F8757E2A"
	$c2 = "SoapMoniker"
	$c3 = "c7b0abec-197f-d211-978e-0000f8757e2a"
	$c4 = "c7b0abec197fd211978e0000f8757e2a"
	$c6 = {c7b0abec197fd211978e0000f8757e2a}
        
    condition:
        warning_ole2link_embedded and 1 of ($c*)
}


rule warning_js_inzip {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 3
		revision = "1"
		date = "Oct 9 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Embedded js"
		mitre = "T1059.007"
	strings:
		$h1 = "PK"
		$s1 = {2e6a730a0020}
		$s2 = {2e6a73ad}
	condition:
            $h1 at 0 and all of ($s*)
}


rule warning_excel_dde_exec {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "4"
		date = "Nov 13 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "Embedded shell command"
		mitre = "T1059.003"
	strings:
	 $header_rtf = "{\\rt" nocase
        $header_office = { D0 CF 11 E0 }
        	$header_xml = "<?xml version=" nocase wide ascii
		$dde_xml = "instrText>DDE" nocase
		$dde_ole = "DDEAUTO" nocase
		$dde_alt = "DDE " nocase
		$dde_alt2 = "\"DDE\" " nocase
		$dde_alt3 = "DDE(" nocase
		$dde_alt4 = "msexcel|" nocase
		$dde_alt5 = "cmd|" nocase
		$dde_cmd1 = "system32\\cmd.exe" nocase
		$dde_cmd2 = "system32\\\\cmd.exe" nocase
		$dde_cmd3 = "system32\\cmd.exe" nocase
		//$dde_reg = /[dD](<[^>]*>){0,5}[dD](<[^>]*>){0,5}[Ee]/
		//$dde_obj = "<w:instrText>\"Dd</w:instrText>" nocase
	condition:
            1 of ($header*) and 1 of ($dde*)
}


rule warning_excel_dde_auto {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Oct 17 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "if present overrides warning message"
		mitre = "T1059.003"
	strings:
    	$header_xml = "<?xml version=" nocase wide ascii
		$ddelink_xml = "<w:updateFields w:val=\"true\"/>" nocase wide ascii
		
	condition:
		all of them
}


rule exploit_cve_2017_11826 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "2"
        date = "Oct 12 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "red"
        sigtype = "cryptam_exploit"
        desc = "xml nested font tag confusion"
	mitre = "T1203 T1204.002"
    strings:
        $header_xml = "<?xml version=" nocase wide ascii
        $s1 = /<w:font [^\/>]+>/ //font tags ending in /> are safe
	$s2 = "</w:font>" ascii nocase

    condition:
	$header_xml and $s1 and (#s1 > #s2)
    }


rule warning_rtf_objupdate {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Nov 20 2017"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "update RTF object may load malicious content"
		mitre = "T1027"
	strings:
    	$header_xml = "{\\rt" nocase
		$upd = "\\objupdate" nocase
		
	condition:
		all of them
}

rule exploit_CVE_2018_0798 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "2"
        date = "Nov 20 2017"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "red"
        sigtype = "cryptam_exploit"
        desc = "Equation editor buffer overflow"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "Equation.3" ascii wide
        $s2 = {02 ce 02 00 00 00 00 00 c0 00 00 00 00 00 00 46}
	$s3 = "Equation Native" ascii wide
        $s4 = "Equation.2" ascii wide
 

    condition:
	1 of them
    }

rule exploit_cve_2017_11882 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "3"
        date = "Oct 16 2020"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "red"
        sigtype = "cryptam_exploit"
        desc = "Equation editor buffer overflow"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = "Equation.3" ascii wide
	$s2 = {41 41 12 0c 43 00} //WinExec
        $s3 = "Equation.2"  ascii wide
        $s4 = {02 ce 02 00 00 00 00 00 c0 00 00 00 00 00 00 46}
        $s5 = "02ce020000000000c000000000000046" nocase
	$s6 = "7b00021700-0000-0000-C000-000000000046" nocase
	$s7 = "7b0002170000000000C000000000000046" nocase
	$s8 = {7b0002170000000000c000000000000046}



    condition:
	1 of them
    }


rule warning_powershell_strings {
    meta:
        is_exploit = false
        is_warning = true
        is_feature = false
        rank = 5
        revision = "1"
        date = "Feb 15 2018"
        author = "@tylabs"
	release = "lite"
        copyright = "QuickSand.io (c) Copyright 2017. All rights reserved."
        tlp = "red"
        sigtype = "cryptam_exploit"
        desc = "Powershell"
	mitre = "T1059.001"
    strings:
        $s1 = "powershell.exe"
	$s2 = "-nop -w hidden -encodedcommand"
	$s3 = "Package"
	$s4 = "Ole10Native" wide


    condition:
	3 of them
    }


rule exploit_cve_2018_8174 {
    meta:
        is_exploit = true
        is_warning = false
        is_feature = false
        rank = 10
        revision = "1"
        date = "May 17 2018"
        author = "@tylabs"
	release = "full"
        copyright = "QuickSand.io (c) Copyright 2018. All rights reserved."
        tlp = "red"
        sigtype = "cryptam_exploit"
        desc = "Moniker exploit MSHTML COM CVE-2018-8174"
	mitre = "T1203 T1204.002"
    strings:
        $s1 = {E0 C9 EA 79 F9 BA CE 11 8C 82 00 AA 00 4B A9 0B}
	$s2 = "E0 C9 EA 79 F9 BA CE 11 8C 82 00 AA 00 4B A9 0B"

    condition:
	any of them
    }

rule warning_xml_script {
	meta:
		is_exe = true
		revision = "100"
		rank = 10
		type = "vb"
		date = "Jan 30 2019"
		author = "@tylabs"
		copyright = "QuickSand.io 2019"
		tlp = "green"
		mitre = "T1059.007"
	strings:
		$s1 = "<vt:lpstr>script"
		$s2 = "\"%73%63%72%69%70%74:%68%74%74%70"
	condition:
            1 of them
}

rule warning_openxml_remote_script
{
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "1"
		date = "Feb 6 2019"
		author = "@tylabs"
		release = "lite"
		hash = "0f74d8a880c638a05ba959c0d5b2fec6"
		copyright = "QuickSand.io (c) Copyright 2019. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "External script inclusion"
		mitre = "T1059.007"
	strings: 
		$a = "schemas.openxmlformats.org" ascii nocase
		$b = "TargetMode=\"External\"" ascii nocase
		$c = /Target\s*=\s*\"(s|%73)(c|%63)(r|%72)(i|%69)(p|%70)(t|%74):[0-9hf%]/

	condition:
		all of them
}

rule warning_rtf_remote_template
{
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "2"
		date = "Sep 9 2020"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1221"
		desc = "External template inclusion"
	strings: 
		$s1 = /\\\*\\template\s*(h|H|%68|%48)/
		$s2 = /\{\\\*\\template\s*(h|H|%68|%48)[^\}]{8,255}/


	condition:
		any of them
}



rule warning_openxml_remote_template
{
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "2"
		date = "Sep 7 2020"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1221"
		desc = "External template inclusion"
	strings: 
		$s1 = "schemas.openxmlformats.org" ascii nocase
		$s2 = "TargetMode=\"External\"" ascii nocase
		$s3 = /Target\s*=\s*[\'\"](h|H|%68|%48)/
		$s4 = "relationships/attachedTemplate" nocase
		$o = /Target\s*=\s*[\'\"](h|H|%68|%48)[^\'\"]{8,255}[\'\"]/



	condition:
		all of ($s*) or 4 of them
}

rule warning_hwp_active_content
{
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Jul 28 2020"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059.007"
		desc = "JavaScript in HWP Document"
	strings: 
		$a = "DefaultJScript" wide nocase
		$b = "JScriptVersion" wide nocase

	condition:
		any of them
}


rule warning_postscript_exec {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Jul 28 2020"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059"
		desc = "Postscript exec"
	strings: 
		$a = "pop exec" ascii nocase

	condition:
		any of them
}

rule warning_postscript_xor {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Jul 28 2020"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1027"
		desc = "Postscript xor could be used for obfuscation"
	strings: 
		$a = "and get xor" ascii nocase

	condition:
		any of them
}


rule warning_postscript_nopsled {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Jul 28 2020"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2020. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1027"
		desc = "Postscript buffer overflow"
	strings: 
		$a = "4747474747474747474747474747474747474747474747474747474747474747474747474747474747479090909090" ascii nocase

	condition:
		any of them
}

rule exploit_cve_2017_11774 {
	meta:
		is_exploit = true
		is_warning = false
		is_feature = false
		rank = 10
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "CVE-2017-11774 Outlook attack"
		mitre = "T1137"
		ref = "38bd9e647609d121621fc817ab2fdb5b58e9a2ac6c2f6640c36bc2164e7d54f1"
	strings:
		$s1 = "\\Outlook\\WebView\\Inbox"
		$s2 = "\\Outlook\\Security"
		$s3 = "\\Outlook\\WebView\\Calendar"
	condition:
            1 of them
}


rule warning_possible_guardrail {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = false
		rank = 1
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "accessing userdnsdomain"
		mitre = "T1480"
		ref = "7c37d91b143c91adc8226e332d8085ea"
	strings:
		$s1 = "USERDNSDOMAIN"
		$s2 = "ExpandEnvironmentStrings"
	condition:
            all of them
}

rule warning_possible_vbapurging {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = false
		rank = 5
		revision = "1"
		date = "Jul 4 2021"
		author = "https://www.fireeye.com/blog/threat-research/2020/11/purgalicious-vba-macro-obfuscation-with-vba-purging.html"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "macro source may have been removed"
		mitre = "T1564.007"
	strings:
		$vba_proj = { 5F 00 56 00 42 00 41 00 5F 00 50 00 52 00 4F 00 4A 00 45 00 43 00 54 00 00 00 00 00 00 00 00 00 }
	condition:
		uint32(0) == 0xe011cfd0 and ( uint32(@vba_proj[1] + 0x78) == 0x07 )
}


rule warning_possible_vbapurged {

	meta:
		is_exploit = false
		is_warning = true
		is_feature = false
		rank = 5
		revision = "1"
		date = "Jul 4 2021"
		author = "https://www.fireeye.com/blog/threat-research/2020/11/purgalicious-vba-macro-obfuscation-with-vba-purging.html"
		release = "lite"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "white"
		sigtype = "cryptam_exploit"
		desc = "macro source removed with tool"
		mitre = "T1564.007"

	strings:
		$vba_proj = { 5F 00 56 00 42 00 41 00 5F 00 50 00 52 00 4F 00 4A 00 45 00 43 00 54 00 00 00 00 00 00 00 00 00 }
		$cc61 = {CC 61 FF FF 00 00 00}
	condition:
		uint32(0) == 0xe011cfd0 and ( uint32(@vba_proj[1] + 0x78) >= 0x07 ) and ( uint32(@vba_proj[1] + 0x78) < 0xff ) and $cc61
}



rule warning_excel_macro {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 1
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059.005"
		desc = "Excel 4.0 macro"
	strings: 
		$a = "Excel 4.0 macro" ascii nocase

	condition:
		any of them
}

rule warning_excel_exec {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059.005"
		desc = "Excel 4.0 macro execute"
	strings: 
		$a = "Excel 4.0 macro" ascii nocase
		$b = "exec(" ascii nocase

	condition:
		all of them
}

rule warning_excel_register {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059.005"
		desc = "Excel 4.0 macro register dll"
	strings: 
		$a = "Excel 4.0 macro" ascii nocase
		$b = "register(" ascii nocase

	condition:
		all of them
}

rule warning_excel_formula_mod {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 2
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059.005"
		desc = "Excel 4.0 macro register dll"
	strings: 
		$a = "Excel 4.0 macro" ascii nocase
		$b = "FORMULA.FILL(" ascii nocase

	condition:
		all of them
}

rule warning_excel_auto_open {
	meta:
		is_exploit = false
		is_warning = true
		is_feature = true
		rank = 5
		revision = "1"
		date = "Jul 4 2021"
		author = "@tylabs"
		release = "full"
		copyright = "QuickSand.io (c) Copyright 2021. All rights reserved."
		tlp = "green"
		sigtype = "cryptam_exploit"
		mitre = "T1059.005"
		desc = "Excel 4.0 macro Auto_Open"
	strings: 
		$a = "Excel 4.0 macro" ascii nocase
		$b = "Auto_Open" ascii nocase

	condition:
		all of them
}
