"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessPoint = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const core = require("aws-cdk-lib");
const aws_s3objectlambda_1 = require("aws-cdk-lib/aws-s3objectlambda");
class AccessPointBase extends core.Resource {
    /** Implement the {@link IAccessPoint.domainName} field. */
    get domainName() {
        const urlSuffix = this.stack.urlSuffix;
        return `${this.accessPointName}-${this.stack.account}.s3-object-lambda.${urlSuffix}`;
    }
    /** Implement the {@link IAccessPoint.regionalDomainName} field. */
    get regionalDomainName() {
        const urlSuffix = this.stack.urlSuffix;
        const region = this.stack.region;
        return `${this.accessPointName}-${this.stack.account}.s3-object-lambda.${region}.${urlSuffix}`;
    }
    /** Implement the {@link IAccessPoint.virtualHostedUrlForObject} method. */
    virtualHostedUrlForObject(key, options) {
        var _b;
        const domainName = ((_b = options === null || options === void 0 ? void 0 : options.regional) !== null && _b !== void 0 ? _b : true) ? this.regionalDomainName : this.domainName;
        const prefix = `https://${domainName}`;
        if (!key) {
            return prefix;
        }
        if (key.startsWith('/')) {
            key = key.slice(1);
        }
        if (key.endsWith('/')) {
            key = key.slice(0, -1);
        }
        return `${prefix}/${key}`;
    }
}
/**
 * Checks the access point name against the rules in https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-names
 * @param name The name of the access point
 */
function validateAccessPointName(name) {
    if (name.length < 3 || name.length > 50) {
        throw new Error('Access point name must be between 3 and 50 characters long');
    }
    if (name.endsWith('-s3alias')) {
        throw new Error('Access point name cannot end with the suffix -s3alias');
    }
    if (name[0] === '-' || name[name.length - 1] === '-') {
        throw new Error('Access point name cannot begin or end with a dash');
    }
    if (!/^[0-9a-z](.(?![\.A-Z_]))+[0-9a-z]$/.test(name)) {
        throw new Error('Access point name must begin with a number or lowercase letter and not contain underscores, uppercase letters, or periods');
    }
}
/**
  * An S3 object lambda access point for intercepting and
  * transforming `GetObject` requests.
  */
class AccessPoint extends AccessPointBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.accessPointName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointProps(props);
        if (props.accessPointName) {
            validateAccessPointName(props.accessPointName);
        }
        const supporting = new s3.CfnAccessPoint(this, 'SupportingAccessPoint', {
            bucket: props.bucket.bucketName,
        });
        const allowedFeatures = [];
        if (props.supportsGetObjectPartNumber) {
            allowedFeatures.push('GetObject-PartNumber');
        }
        if (props.supportsGetObjectRange) {
            allowedFeatures.push('GetObject-Range');
        }
        const accessPoint = new aws_s3objectlambda_1.CfnAccessPoint(this, id, {
            name: this.physicalName,
            objectLambdaConfiguration: {
                allowedFeatures,
                cloudWatchMetricsEnabled: props.cloudWatchMetricsEnabled,
                supportingAccessPoint: supporting.attrArn,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: {
                            AwsLambda: {
                                FunctionArn: props.handler.functionArn,
                                FunctionPayload: props.payload ? JSON.stringify(props.payload) : undefined,
                            },
                        },
                    },
                ],
            },
        });
        this.accessPointName = accessPoint.ref;
        this.accessPointArn = accessPoint.attrArn;
        this.accessPointCreationDate = accessPoint.attrCreationDate;
        props.handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3-object-lambda:WriteGetObjectResponse'],
            resources: ['*'],
        }));
    }
    /**
     * Reference an existing AccessPoint defined outside of the CDK code.
     */
    static fromAccessPointAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointAttributes(attrs);
        const arn = core.Arn.split(attrs.accessPointArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arn.resourceName) {
            throw new Error('Unable to parse acess point name');
        }
        const name = arn.resourceName;
        class Import extends AccessPointBase {
            constructor() {
                super(...arguments);
                this.accessPointArn = attrs.accessPointArn;
                this.accessPointCreationDate = attrs.accessPointCreationDate;
                this.accessPointName = name;
            }
        }
        return new Import(scope, id);
    }
}
exports.AccessPoint = AccessPoint;
_a = JSII_RTTI_SYMBOL_1;
AccessPoint[_a] = { fqn: "@aws-cdk/aws-s3objectlambda-alpha.AccessPoint", version: "2.19.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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