"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenizedStringFragments = void 0;
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === 'token') {
            return first.token;
        }
        return undefined;
    }
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    get length() {
        return this.fragments.length;
    }
    addLiteral(lit) {
        this.fragments.push({ type: 'literal', lit });
    }
    addToken(token) {
        this.fragments.push({ type: 'token', token });
    }
    addIntrinsic(value) {
        this.fragments.push({ type: 'intrinsic', value });
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === 'token') {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case 'literal':
                    ret.addLiteral(f.lit);
                    break;
                case 'token':
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case 'intrinsic':
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case 'literal': return fragment.lit;
        case 'token': return fragment.token.toString();
        case 'intrinsic': return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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