"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unionValidator = exports.requireProperty = exports.requiredValidator = exports.propertyValidator = exports.hashValidator = exports.listValidator = exports.validateCfnTag = exports.validateObject = exports.validateDate = exports.validateBoolean = exports.validateNumber = exports.validateString = exports.canInspect = exports.VALIDATION_SUCCESS = exports.ValidationResults = exports.ValidationResult = exports.unionMapper = exports.hashMapper = exports.listMapper = exports.cfnTagToCloudFormation = exports.dateToCloudFormation = exports.numberToCloudFormation = exports.objectToCloudFormation = exports.booleanToCloudFormation = exports.stringToCloudFormation = void 0;
function identity(x) {
    return x;
}
exports.stringToCloudFormation = identity;
exports.booleanToCloudFormation = identity;
exports.objectToCloudFormation = identity;
exports.numberToCloudFormation = identity;
/**
 * The date needs to be formatted as an ISO date in UTC
 *
 * Some usage sites require a date, some require a timestamp. We'll
 * always output a timestamp and hope the parser on the other end
 * is smart enough to ignore the time part... (?)
 */
function dateToCloudFormation(x) {
    if (!x) {
        return undefined;
    }
    // tslint:disable-next-line:max-line-length
    return `${x.getUTCFullYear()}-${pad(x.getUTCMonth() + 1)}-${pad(x.getUTCDate())}T${pad(x.getUTCHours())}:${pad(x.getUTCMinutes())}:${pad(x.getUTCSeconds())}`;
}
exports.dateToCloudFormation = dateToCloudFormation;
/**
 * Pad a number to 2 decimal places
 */
function pad(x) {
    if (x < 10) {
        return '0' + x.toString();
    }
    return x.toString();
}
/**
 * Turn a tag object into the proper CloudFormation representation
 */
function cfnTagToCloudFormation(x) {
    return {
        Key: x.key,
        Value: x.value,
    };
}
exports.cfnTagToCloudFormation = cfnTagToCloudFormation;
function listMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
exports.listMapper = listMapper;
function hashMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
exports.hashMapper = hashMapper;
/**
 * Return a union mapper
 *
 * Takes a list of validators and a list of mappers, which should correspond pairwise.
 *
 * The mapper of the first successful validator will be called.
 */
function unionMapper(validators, mappers) {
    if (validators.length !== mappers.length) {
        throw Error('Not the same amount of validators and mappers passed to unionMapper()');
    }
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        for (let i = 0; i < validators.length; i++) {
            if (validators[i](x).isSuccess) {
                return mappers[i](x);
            }
        }
        // Should not be possible because the union must have passed validation before this function
        // will be called, but catch it anyway.
        throw new TypeError('No validators matched in the union()');
    };
}
exports.unionMapper = unionMapper;
// ----------------------------------------------------------------------
// VALIDATORS
//
// These are used while checking that supplied property bags match the expected schema
//
// We have a couple of datatypes that model validation errors and collections of validation
// errors (together forming a tree of errors so that we can trace validation errors through
// an object graph), and validators.
//
// Validators are simply functions that take a value and return a validation results. Then
// we have some combinators to turn primitive validators into more complex validators.
//
/**
 * Representation of validation results
 *
 * Models a tree of validation errors so that we have as much information as possible
 * about the failure that occurred.
 */
class ValidationResult {
    constructor(errorMessage = '', results = new ValidationResults()) {
        this.errorMessage = errorMessage;
        this.results = results;
    }
    get isSuccess() {
        return !this.errorMessage && this.results.isSuccess;
    }
    /**
     * Turn a failed validation into an exception
     */
    assertSuccess() {
        if (!this.isSuccess) {
            let message = this.errorTree();
            // The first letter will be lowercase, so uppercase it for a nicer error message
            message = message.substr(0, 1).toUpperCase() + message.substr(1);
            throw new CfnSynthesisError(message);
        }
    }
    /**
     * Return a string rendering of the tree of validation failures
     */
    errorTree() {
        const childMessages = this.results.errorTreeList();
        return this.errorMessage + (childMessages.length ? `\n  ${childMessages.replace(/\n/g, '\n  ')}` : '');
    }
    /**
     * Wrap this result with an error message, if it concerns an error
     */
    prefix(message) {
        if (this.isSuccess) {
            return this;
        }
        return new ValidationResult(`${message}: ${this.errorMessage}`, this.results);
    }
}
exports.ValidationResult = ValidationResult;
/**
 * A collection of validation results
 */
class ValidationResults {
    constructor(results = []) {
        this.results = results;
    }
    collect(result) {
        // Only collect failures
        if (!result.isSuccess) {
            this.results.push(result);
        }
    }
    get isSuccess() {
        return this.results.every(x => x.isSuccess);
    }
    errorTreeList() {
        return this.results.map(child => child.errorTree()).join('\n');
    }
    /**
     * Wrap up all validation results into a single tree node
     *
     * If there are failures in the collection, add a message, otherwise
     * return a success.
     */
    wrap(message) {
        if (this.isSuccess) {
            return exports.VALIDATION_SUCCESS;
        }
        return new ValidationResult(message, this);
    }
}
exports.ValidationResults = ValidationResults;
// Singleton object to save on allocations
exports.VALIDATION_SUCCESS = new ValidationResult();
/**
 * Return whether this object can be validated at all
 *
 * True unless it's undefined or a CloudFormation intrinsic
 */
function canInspect(x) {
    // Note: using weak equality on purpose, we also want to catch undefined
    return (x != null && !isCloudFormationIntrinsic(x));
}
exports.canInspect = canInspect;
// CloudFormation validators for primitive types
function validateString(x) {
    if (canInspect(x) && typeof x !== 'string') {
        return new ValidationResult(`${JSON.stringify(x)} should be a string`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateString = validateString;
function validateNumber(x) {
    if (canInspect(x) && typeof x !== 'number') {
        return new ValidationResult(`${JSON.stringify(x)} should be a number`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateNumber = validateNumber;
function validateBoolean(x) {
    if (canInspect(x) && typeof x !== 'boolean') {
        return new ValidationResult(`${JSON.stringify(x)} should be a boolean`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateBoolean = validateBoolean;
function validateDate(x) {
    if (canInspect(x) && !(x instanceof Date)) {
        return new ValidationResult(`${JSON.stringify(x)} should be a Date`);
    }
    if (x !== undefined && isNaN(x.getTime())) {
        return new ValidationResult('got an unparseable Date');
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateDate = validateDate;
function validateObject(x) {
    if (canInspect(x) && typeof x !== 'object') {
        return new ValidationResult(`${JSON.stringify(x)} should be an 'object'`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateObject = validateObject;
function validateCfnTag(x) {
    if (!canInspect(x)) {
        return exports.VALIDATION_SUCCESS;
    }
    if (x.key == null || x.value == null) {
        return new ValidationResult(`${JSON.stringify(x)} should have a 'key' and a 'value' property`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateCfnTag = validateCfnTag;
/**
 * Return a list validator based on the given element validator
 */
function listValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        if (!x.forEach) {
            return new ValidationResult(`${JSON.stringify(x)} should be a list`);
        }
        for (let i = 0; i < x.length; i++) {
            const element = x[i];
            const result = elementValidator(element);
            if (!result.isSuccess) {
                return result.prefix(`element ${i}`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.listValidator = listValidator;
/**
 * Return a hash validator based on the given element validator
 */
function hashValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        for (const key of Object.keys(x)) {
            const result = elementValidator(x[key]);
            if (!result.isSuccess) {
                return result.prefix(`element '${key}'`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.hashValidator = hashValidator;
/**
 * Decorate a validator with a message clarifying the property the failure is for.
 */
function propertyValidator(propName, validator) {
    return (x) => {
        return validator(x).prefix(propName);
    };
}
exports.propertyValidator = propertyValidator;
/**
 * Return a validator that will fail if the passed property is not present
 *
 * Does not distinguish between the property actually not being present, vs being present but 'null'
 * or 'undefined' (courtesy of JavaScript), which is generally the behavior that we want.
 *
 * Empty strings are considered "present"--don't know if this agrees with how CloudFormation looks
 * at the world.
 */
function requiredValidator(x) {
    if (x == null) {
        return new ValidationResult('required but missing');
    }
    return exports.VALIDATION_SUCCESS;
}
exports.requiredValidator = requiredValidator;
/**
 * Require a property from a property bag.
 *
 * @param props  the property bag from which a property is required.
 * @param name   the name of the required property.
 * @param typeName the name of the construct type that requires the property
 *
 * @returns the value of ``props[name]``
 *
 * @throws if the property ``name`` is not present in ``props``.
 */
function requireProperty(props, name, context) {
    const value = props[name];
    if (value == null) {
        throw new Error(`${context.toString()} is missing required property: ${name}`);
    }
    // Possibly add type-checking here...
    return value;
}
exports.requireProperty = requireProperty;
/**
 * Validates if any of the given validators matches
 *
 * We add either/or words to the front of the error mesages so that they read
 * more nicely. Example:
 *
 *   Properties not correct for 'FunctionProps'
 *     codeUri: not one of the possible types
 *       either: properties not correct for 'S3LocationProperty'
 *         bucket: required but missing
 *         key: required but missing
 *         version: required but missing
 *       or: '3' should be a 'string'
 *
 */
function unionValidator(...validators) {
    return (x) => {
        const results = new ValidationResults();
        let eitherOr = 'either';
        for (const validator of validators) {
            const result = validator(x);
            if (result.isSuccess) {
                return result;
            }
            results.collect(result.prefix(eitherOr));
            eitherOr = 'or';
        }
        return results.wrap('not one of the possible types');
    };
}
exports.unionValidator = unionValidator;
/**
 * Return whether the indicated value represents a CloudFormation intrinsic.
 *
 * CloudFormation intrinsics are modeled as objects with a single key, which
 * look like: { "Fn::GetAtt": [...] } or similar.
 */
function isCloudFormationIntrinsic(x) {
    if (!(typeof x === 'object')) {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || keys[0].substr(0, 4) === 'Fn::';
}
// Cannot be public because JSII gets confused about es5.d.ts
class CfnSynthesisError extends Error {
    constructor() {
        super(...arguments);
        this.type = 'CfnSynthesisError';
    }
}
//# sourceMappingURL=data:application/json;base64,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