"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogicalIDs = void 0;
/**
 * Class that keeps track of the logical IDs that are assigned to resources
 *
 * Supports renaming the generated IDs.
 */
class LogicalIDs {
    constructor() {
        /**
         * The rename table (old to new)
         */
        this.renames = {};
        /**
         * All assigned names (new to old, may be identical)
         *
         * This is used to ensure that:
         *
         * - No 2 resources end up with the same final logical ID, unless they were the same to begin with.
         * - All renames have been used at the end of renaming.
         */
        this.reverse = {};
    }
    /**
     * Rename a logical ID from an old ID to a new ID
     */
    addRename(oldId, newId) {
        if (oldId in this.renames) {
            throw new Error(`A rename has already been registered for '${oldId}'`);
        }
        this.renames[oldId] = newId;
    }
    /**
     * Return the renamed version of an ID or the original ID.
     */
    applyRename(oldId) {
        let newId = oldId;
        if (oldId in this.renames) {
            newId = this.renames[oldId];
        }
        // If this newId has already been used, it must have been with the same oldId
        if (newId in this.reverse && this.reverse[newId] !== oldId) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Two objects have been assigned the same Logical ID: '${this.reverse[newId]}' and '${oldId}' are now both named '${newId}'.`);
        }
        this.reverse[newId] = oldId;
        validateLogicalId(newId);
        return newId;
    }
    /**
     * Throw an error if not all renames have been used
     *
     * This is to assure that users didn't make typoes when registering renames.
     */
    assertAllRenamesApplied() {
        const keys = new Set();
        Object.keys(this.renames).forEach(keys.add.bind(keys));
        Object.keys(this.reverse).map(newId => {
            keys.delete(this.reverse[newId]);
        });
        if (keys.size !== 0) {
            const unusedRenames = Array.from(keys.values());
            throw new Error(`The following Logical IDs were attempted to be renamed, but not found: ${unusedRenames.join(', ')}`);
        }
    }
}
exports.LogicalIDs = LogicalIDs;
const VALID_LOGICALID_REGEX = /^[A-Za-z][A-Za-z0-9]{1,254}$/;
/**
 * Validate logical ID is valid for CloudFormation
 */
function validateLogicalId(logicalId) {
    if (!VALID_LOGICALID_REGEX.test(logicalId)) {
        throw new Error(`Logical ID must adhere to the regular expression: ${VALID_LOGICALID_REGEX.toString()}, got '${logicalId}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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