"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const crypto = require("crypto");
const fs = require("fs-extra");
const os = require("os");
const path = require("path");
const assets_1 = require("./assets");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
const STAGING_TMP = '.cdk.staging';
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
    }
    synthesize(session) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(session.assembly.outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Move bundling directory to staging directory
            fs.moveSync(this.bundleDir, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Temp staging directory in the working directory
        const stagingTmp = path.join('.', STAGING_TMP);
        fs.ensureDirSync(stagingTmp);
        // Create temp directory for bundling inside the temp staging directory
        const bundleDir = path.resolve(fs.mkdtempSync(path.join(stagingTmp, 'asset-bundle-')));
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            options.image._run({
                command: options.command,
                user,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : AssetStaging.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${AssetStaging.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                // Hash the hash to make sure we can use it in a file/directory name.
                // The resulting hash will also have the same length as for the other hash types.
                return crypto.createHash('sha256').update(props.assetHash).digest('hex');
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
//# sourceMappingURL=data:application/json;base64,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