"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchSubmitJob = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_b = props.containerOverrides) === null || _b === void 0 ? void 0 : _b.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        var _a;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: (_a = this.props.payload) === null || _a === void 0 ? void 0 : _a.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a, _b;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: (_b = containerOverrides.memory) === null || _b === void 0 ? void 0 : _b.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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