"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUp = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const { root } = path.parse(directory);
    if (directory === root && !fs.existsSync(path.join(directory, name))) {
        return undefined;
    }
    if (fs.existsSync(path.join(directory, name))) {
        return directory;
    }
    return findUp(name, path.dirname(directory));
}
exports.findUp = findUp;
//# sourceMappingURL=data:application/json;base64,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