"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const fs = require("fs");
const path = require("path");
const bundling_1 = require("./bundling");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a, _b;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = findEntry(id, props.entry);
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : defaultRunTime;
        // Look for the closest package.json starting in the directory of the entry
        // file. We need to restore it after bundling.
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(entry));
        try {
            super(scope, id, {
                ...props,
                runtime,
                code: bundling_1.Bundling.parcel({
                    entry,
                    runtime,
                    ...props,
                }),
                handler: `index.${handler}`,
            });
        }
        finally {
            // We can only restore after the code has been bound to the function
            packageJsonManager.restore();
        }
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
//# sourceMappingURL=data:application/json;base64,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