"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (lambci)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_b = options.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_c = options.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_d = options.nodeModules) !== null && _d !== void 0 ? _d : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_e = options.sourceMaps) !== null && _e !== void 0 ? _e : false,
                    minify: (_f = options.minify) !== null && _f !== void 0 ? _f : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = `parcel build ${containerEntryPath.replace(/\\/g, '/')} --target cdk-lambda${options.cacheDir ? ' --cache-dir /parcel-cache' : ''}`;
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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