"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// tslint:disable-next-line:max-line-length
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
//# sourceMappingURL=data:application/json;base64,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