"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || this.node.uniqueId;
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.volumes }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({ produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    get executionRole() {
        return this._executionRole;
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Adds a firelens log router to the task definition.
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The IPC resource namespace to use for the containers in the task.
 */
var IpcMode;
(function (IpcMode) {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not
     * shared with other containers in a task or on the container instance
     */
    IpcMode["NONE"] = "none";
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on
     * the same container instance share the same IPC resources with the host Amazon EC2 instance.
     */
    IpcMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     */
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * The process namespace to use for the containers in the task.
 */
var PidMode;
(function (PidMode) {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     */
    PidMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     */
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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