"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3DeployAction = exports.CacheControl = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /** The 'must-revalidate' cache control directive. */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /** The 'no-cache' cache control directive. */
    static noCache() { return new CacheControl('no-cache'); }
    /** The 'no-transform' cache control directive. */
    static noTransform() { return new CacheControl('no-transform'); }
    /** The 'public' cache control directive. */
    static setPublic() { return new CacheControl('public'); }
    /** The 'private' cache control directive. */
    static setPrivate() { return new CacheControl('private'); }
    /** The 'proxy-revalidate' cache control directive. */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /** The 'max-age' cache control directive. */
    static maxAge(t) { return new CacheControl(`max-age: ${t.toSeconds()}`); }
    /** The 's-max-age' cache control directive. */
    static sMaxAge(t) { return new CacheControl(`s-maxage: ${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive,
     * in case our support is missing a method for a particular directive.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 */
class S3DeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,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